#!/usr/bin/env bash
# ==============================================================================
# 服务自检脚本 (Linux 版本，单机运行)
# 目标：在服务器本机运行，无需 SSH 连接逻辑；实现与 check_server_health.ps1 主要功能一致。
#
# 输出：
# - logs/health_check_YYYYmmdd_HHMMSS.log
# - Reports/health_report_<ip>_YYYYmmdd_HHMMSS.md
#
# 说明：
# - 默认以 root 执行（涉及 docker/iptables/firewalld/读日志等）。
# - 修复动作依赖同目录 issue_handler.sh（如 fix_ntp_config / fix_dns_config / fix_port_access 等）。
# ==============================================================================

set -euo pipefail

# ------------------------------
# 基础配置
# ------------------------------
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SCRIPT_VERSION="1.0.0"
SSH_TIMEOUT=30 # 占位：与 ps1 一致，这里不使用

LOG_DIR="$SCRIPT_DIR/logs"
REPORT_DIR="$SCRIPT_DIR/Reports"
TS="$(date '+%Y%m%d_%H%M%S')"
LOG_FILE="$LOG_DIR/health_check_${TS}.log"

mkdir -p "$LOG_DIR" "$REPORT_DIR"

# ------------------------------
# 日志函数
# ------------------------------
log() {
  local level="$1"; shift
  local msg="${*:-}"
  local now
  now="$(date '+%Y-%m-%d %H:%M:%S')"

  if [[ -z "$msg" ]]; then
    echo ""
    return 0
  fi

  local line="[$now] [$level] $msg"
  echo "$line" | tee -a "$LOG_FILE" >/dev/null
}

# ✅ 新增：阶段标题（让执行过程更“显眼”地打印）
section() {
  local title="$*"
  log INFO "=================================================================="
  log INFO "$title"
  log INFO "=================================================================="
}

# ---- helper: 将多行压缩成一行，避免 Markdown 反引号被换行破坏
_squash_ws_one_line() {
  # stdin -> stdout
  tr '\r\n\t' '   ' | sed 's/  */ /g; s/^ *//; s/ *$//'
}

# ---- helper: 安全取值，空则返回 N/A
_kv_or_na() {
  local v="$1"
  if [[ -n "$v" ]]; then echo "$v"; else echo "N/A"; fi
}

get_resolv_conf_raw() {
  if [[ -r /etc/resolv.conf ]]; then
    sed 's/\r$//' /etc/resolv.conf
  else
    echo "N/A (cannot read /etc/resolv.conf)"
  fi
}

# ------------------------------
# 报告汇总（增强：同时支持关键值/状态）
# ------------------------------
REPORT_LINES=()

report_add() { REPORT_LINES+=("$*"); }

report_add_block() {
  while [[ $# -gt 0 ]]; do
    REPORT_LINES+=("$1"); shift
  done
}

# 可选：埋点到报告里（key/value 级别的数据）
declare -A REPORT_KV=()

report_kv_set() {
  local k="$1"; shift
  local v="${*:-}"
  REPORT_KV["$k"]="$v"
}

report_kv_get() {
  local k="$1"
  echo "${REPORT_KV[$k]:-}"
}

# ------------------------------
# 工具函数
# ------------------------------
command_exists() { command -v "$1" >/dev/null 2>&1; }

get_primary_ip() {
  local ip=""
  if command_exists ip; then
    ip="$(ip route get 1 2>/dev/null | awk '{for(i=1;i<=NF;i++){if($i=="src"){print $(i+1); exit}}}')"
  fi
  if [[ -z "$ip" ]] && command_exists hostname; then
    ip="$(hostname -I 2>/dev/null | awk '{print $1}')"
  fi
  echo "${ip:-unknown}"
}

# 执行修复脚本：同目录 issue_handler.sh
run_issue_handler() {
  local action="$1"
  local platform="${2:-auto}"
  local extra="${3:-}"

  local issue="$SCRIPT_DIR/issue_handler.sh"
  if [[ ! -f "$issue" ]]; then
    log ERROR "[修复] 未找到 issue_handler.sh：$issue"
    return 1
  fi

  chmod +x "$issue" 2>/dev/null || true

  # dos2unix 可选
  if command_exists dos2unix; then
    dos2unix "$issue" >/dev/null 2>&1 || true
  fi

  local cmd="$issue --action $action --platform $platform $extra"
  log WARN "[修复] 执行：$cmd"
  if bash -c "$cmd" >>"$LOG_FILE" 2>&1; then
    log SUCCESS "[修复] 执行成功：$action"
    return 0
  else
    log ERROR "[修复] 执行失败：$action（详见日志：$LOG_FILE）"
    return 1
  fi
}

# ------------------------------
# 1) 平台识别
# ------------------------------
detect_platform() {
  if [[ -d /data/services ]]; then
    echo "new"
  else
    echo "old"
  fi
}

# ------------------------------
# 2) 系统识别（容器）
# ------------------------------
detect_systems() {
  local out=""
  if command_exists docker; then
    out="$(docker ps --format '{{.Names}}' 2>/dev/null || true)"
  fi

  local has_ujava=0 has_upython=0 has_upython_voice=0
  local ujava_name="" upython_name="" upython_voice_name=""

  while IFS= read -r c; do
    [[ -z "$c" ]] && continue
    if [[ "$c" =~ ^ujava[0-9]*$ ]]; then
      has_ujava=1; ujava_name="$c"
    elif [[ "$c" =~ ^upython[0-9]*$ ]] && [[ ! "$c" =~ voice ]]; then
      has_upython=1; upython_name="$c"
    elif [[ "$c" =~ ^upython_voice[0-9]*$ ]]; then
      has_upython_voice=1; upython_voice_name="$c"
    fi
  done <<< "$out"

  # 输出为 KEY=VALUE，供调用方 eval
  cat <<EOF
HAS_UJAVA=$has_ujava
HAS_UPYTHON=$has_upython
HAS_UPYTHON_VOICE=$has_upython_voice
UJAVA_CONTAINER=${ujava_name:-}
UPYTHON_CONTAINER=${upython_name:-}
UPYTHON_VOICE_CONTAINER=${upython_voice_name:-}
EOF
}

# ------------------------------
# 3) 服务检测配置（对齐 ps1）
# ------------------------------
declare -A UJAVA_SERVICES=(
  ["auth"]="ubains-auth.jar"
  ["gateway"]="ubains-gateway.jar"
  ["system"]="ubains-modules-system.jar"
  ["meeting2.0"]="ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
  ["meeting3.0"]="ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
  ["mqtt"]="ubains-meeting-mqtt-1.0-SNAPSHOT.jar"
  ["quartz"]="ubains-meeting-quartz-1.0-SNAPSHOT.jar"
  ["message"]="ubains-meeting-message-scheduling-1.0-SNAPSHOT.jar"
)

declare -A UJAVA_HOST_SERVICES=(
  ["extapi"]="ubains-meeting-api-1.0-SNAPSHOT.jar"
)

# 传统平台 ujava 容器内（nginx + meeting）
declare -A UJAVA_OLD_CONTAINER_SERVICES=(
  ["nginx"]="nginx: master process"
  ["meeting"]="ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
)

# ------------------------------
# 4) 进程检测：ujava（容器/宿主机）
# ------------------------------
count_proc_in_container() {
  local container="$1"
  local pattern="$2"
  local extra_grep="${3:-}"

  local cmd="docker exec $container sh -c \"ps aux 2>/dev/null | grep -v grep | grep '$pattern' ${extra_grep} | wc -l\""
  # shellcheck disable=SC2086
  local n
  n="$(eval "$cmd" 2>/dev/null | tail -n 1 | tr -d '\r' || true)"
  [[ "$n" =~ ^[0-9]+$ ]] || n=0
  echo "$n"
}

count_proc_host() {
  local pattern="$1"
  local extra_grep="${2:-}"

  local n
  # shellcheck disable=SC2009
  n="$(ps aux 2>/dev/null | grep -v grep | grep "$pattern" ${extra_grep} | wc -l | tail -n 1 | tr -d '\r' || true)"
  [[ "$n" =~ ^[0-9]+$ ]] || n=0
  echo "$n"
}

test_ujava_services() {
  local container="${1:-}"  # 可空：表示宿主机检测
  local platform="${2:-new}"

  local results=()
  log INFO "========== 检测 ujava 服务 ($( [[ -n "$container" ]] && echo "容器:$container" || echo "宿主机" )) =========="

  for svc in "${!UJAVA_SERVICES[@]}"; do
    local jar="${UJAVA_SERVICES[$svc]}"
    local n=0

    if [[ -n "$container" ]]; then
      if [[ "$svc" == "meeting2.0" ]]; then
        n="$(count_proc_in_container "$container" "$jar" "| grep 'java-meeting2.0'")"
      elif [[ "$svc" == "meeting3.0" ]]; then
        n="$(count_proc_in_container "$container" "$jar" "| grep 'java-meeting3.0'")"
      else
        n="$(count_proc_in_container "$container" "$jar")"
      fi

      # 容器内未命中则 fallback 宿主机（与 ps1 一致的兜底）
      if [[ "$n" -eq 0 ]]; then
        if [[ "$svc" == "meeting2.0" ]]; then
          n="$(count_proc_host "$jar" "| grep 'java-meeting2.0'")"
        elif [[ "$svc" == "meeting3.0" ]]; then
          n="$(count_proc_host "$jar" "| grep 'java-meeting3.0'")"
        else
          n="$(count_proc_host "$jar")"
        fi
      fi
    else
      if [[ "$svc" == "meeting2.0" ]]; then
        n="$(count_proc_host "$jar" "| grep 'java-meeting2.0'")"
      elif [[ "$svc" == "meeting3.0" ]]; then
        n="$(count_proc_host "$jar" "| grep 'java-meeting3.0'")"
      else
        n="$(count_proc_host "$jar")"
      fi
    fi

    if [[ "$n" -gt 0 ]]; then
      log SUCCESS "  [OK] $svc ($jar): 运行中"
      results+=("$svc|$jar|running")
    else
      log ERROR "  [FAIL] $svc ($jar): 未运行"
      results+=("$svc|$jar|stopped")
    fi
  done

  printf "%s\n" "${results[@]}"
}

test_ujava_host_services() {
  log INFO "========== 检测 ujava 宿主机服务 (extapi) =========="
  local results=()
  for svc in "${!UJAVA_HOST_SERVICES[@]}"; do
    local jar="${UJAVA_HOST_SERVICES[$svc]}"
    local n
    n="$(count_proc_host "$jar")"
    if [[ "$n" -gt 0 ]]; then
      log SUCCESS "  [OK] $svc ($jar): 运行中"
      results+=("$svc|$jar|running")
    else
      log ERROR "  [FAIL] $svc ($jar): 未运行"
      results+=("$svc|$jar|stopped")
    fi
  done
  printf "%s\n" "${results[@]}"
}

test_ujava_old_container_services() {
  local container="$1"
  log INFO "========== 检测传统平台 ujava 容器内服务 ($container) =========="
  local results=()

  for svc in "${!UJAVA_OLD_CONTAINER_SERVICES[@]}"; do
    local pattern="${UJAVA_OLD_CONTAINER_SERVICES[$svc]}"
    local n
    n="$(count_proc_in_container "$container" "$pattern")"
    if [[ "$n" -gt 0 ]]; then
      log SUCCESS "  [OK] $svc ($pattern): 运行中"
      results+=("$svc|$pattern|running")
    else
      log ERROR "  [FAIL] $svc ($pattern): 未运行"
      results+=("$svc|$pattern|stopped")
    fi
  done
  printf "%s\n" "${results[@]}"
}

# ------------------------------
# 5) 端口检测（容器内）
# ------------------------------
container_port_listening() {
  local container="$1"
  local port="$2"
  # netstat/ss 任一可用
  local cmd="docker exec $container sh -c \"(netstat -tlnp 2>/dev/null || ss -tlnp 2>/dev/null) | grep -E ':[[:space:]]*${port}[[:space:]]'\""
  # shellcheck disable=SC2086
  if eval "$cmd" >/dev/null 2>&1; then
    return 0
  fi
  return 1
}

test_container_ports() {
  local container="$1"
  local service_type="$2"
  shift 2
  local ports=("$@") # 形如 "11211|memcached|xxx"

  log INFO "========== 检测 ${service_type} 容器内端口 ($container) =========="
  local results=()
  for p in "${ports[@]}"; do
    local port proc desc
    IFS="|" read -r port proc desc <<<"$p"
    if container_port_listening "$container" "$port"; then
      log SUCCESS "  [OK] 端口 $port ($desc): 监听中"
      results+=("$port|$proc|$desc|listening")
    else
      log ERROR "  [FAIL] 端口 $port ($desc): 未监听"
      results+=("$port|$proc|$desc|closed")
    fi
  done
  printf "%s\n" "${results[@]}"
}

# 新统一平台端口要求（对齐 ps1 的 UpythonPorts / UpythonVoicePorts）
UPYTHON_PORTS_NEW=(
  "11211|memcached|Memcached 缓存服务"
  "8000|uwsgi|uWSGI 应用服务"
)

UPYTHON_VOICE_PORTS_NEW=(
  "1883|mosquitto|MQTT Broker 服务"
  "8000|uwsgi|uWSGI 应用服务"
  "9001|mosquitto|MQTT WebSocket 服务"
  "11211|memcached|Memcached 缓冲服务"
  "8080|nginx|Nginx 代理服务 (8080)"
  "80|nginx|Nginx Web 服务 (80)"
)

# 传统平台 upython 端口要求
UPYTHON_PORTS_OLD=(
  "8081|nginx|Nginx 代理服务 (8081)"
  "8443|nginx|Nginx HTTPS 服务 (8443)"
  "8000|uwsgi|uWSGI 应用服务"
  "8002|httpd|Apache HTTPD 服务"
  "11211|memcached|Memcached 缓冲服务"
)

# ------------------------------
# 6) DNS 检测 + 修复 + 复检（增强：记录 details 到报告）
# ------------------------------
DNS_TEST_DOMAINS=("www.baidu.com" "www.qq.com" "www.aliyun.com")

get_dns_nameservers() {
  if [[ -r /etc/resolv.conf ]]; then
    grep -E '^nameserver' /etc/resolv.conf 2>/dev/null | awk '{print $2}' | head -n 5 | paste -sd',' - || true
  else
    echo ""
  fi
}

# ------------------------------
# 6) DNS 检测 + 修复 + 复检（增强：记录 details 到报告 + 记录 raw）
# ------------------------------
test_dns() {
  log INFO "========== 检测 DNS 解析功能 =========="

  local resolv_raw dns_servers
  resolv_raw="$(get_resolv_conf_raw)"
  dns_servers="$(get_dns_nameservers)"

  report_kv_set "dns.resolv_conf" "$resolv_raw"
  report_kv_set "dns.nameservers" "${dns_servers:-N/A}"

  if [[ -n "$dns_servers" ]]; then
    log SUCCESS "  DNS配置: nameserver=${dns_servers}"
  else
    log WARN "  DNS配置: 未检测到 nameserver 或无法读取 /etc/resolv.conf"
  fi

  local ok=0 total="${#DNS_TEST_DOMAINS[@]}"
  local detail_lines=()

  for d in "${DNS_TEST_DOMAINS[@]}"; do
    if command_exists nslookup && nslookup "$d" >/dev/null 2>&1; then
      log SUCCESS "  [OK] $d : 解析成功(nslookup)"
      ok=$((ok+1))
      detail_lines+=("$d|OK|nslookup")
    elif command_exists host && host "$d" >/dev/null 2>&1; then
      log SUCCESS "  [OK] $d : 解析成功(host)"
      ok=$((ok+1))
      detail_lines+=("$d|OK|host")
    else
      log ERROR "  [FAIL] $d : 解析失败"
      detail_lines+=("$d|FAIL|-")
    fi
  done

  if [[ "${#detail_lines[@]}" -gt 0 ]]; then
    report_kv_set "dns.details" "$(printf "%s\n" "${detail_lines[@]}")"
  fi

  # raw（像日志）
  local raw=""
  raw+="[resolv.conf]\n${resolv_raw}\n\n"
  raw+="DNS配置: nameserver=${dns_servers:-N/A}\n"
  for l in "${detail_lines[@]}"; do
    local dd rr tt
    IFS="|" read -r dd rr tt <<<"$l"
    if [[ "$rr" == "OK" ]]; then
      raw+="[OK] ${dd} : 解析成功(${tt})\n"
    else
      raw+="[FAIL] ${dd} : 解析失败\n"
    fi
  done
  report_kv_set "dns.raw" "$(printf "%b" "$raw")"

  if [[ "$ok" -eq "$total" ]]; then
    echo "OK"
  elif [[ "$ok" -gt 0 ]]; then
    echo "PARTIAL"
  else
    echo "FAIL"
  fi
}

repair_dns_if_needed() {
  local status="$1"
  if [[ "$status" == "OK" ]]; then
    return 0
  fi
  log WARN "[DNS] 检测到 DNS 异常（$status），触发修复：fix_dns_config"
  run_issue_handler "fix_dns_config" "auto" "--non-interactive --yes" || true

  log INFO "[DNS] 修复后复检..."
  local post
  post="$(test_dns)"
  report_kv_set "dns.recheck" "$post"

  if [[ "$post" == "OK" ]]; then
    log SUCCESS "[DNS] 复检成功：DNS 已恢复"
  else
    log WARN "[DNS] 复检仍异常：$post（需人工排查）"
  fi
}

# ------------------------------
# 7) 资源分析（CPU/内存/磁盘/防火墙）
# ------------------------------
get_cpu_usage() {
  # top 或 mpstat 任取
  local usage="0"
  if command_exists top; then
    usage="$(LC_ALL=C top -bn1 2>/dev/null | awk -F'[:, ]+' '/Cpu\(s\)/{print $3+$5; exit}' || true)"
  fi
  if [[ -z "$usage" || "$usage" == "0" ]] && command_exists mpstat; then
    usage="$(mpstat 1 1 2>/dev/null | awk 'END{print 100-$NF}' || true)"
  fi
  echo "${usage:-0}"
}

get_mem_usage_gb_pct() {
  # 输出：total_gb,used_gb,pct
  local line=""
  if command_exists free; then
    line="$(LC_ALL=C free -m 2>/dev/null | awk '$1=="Mem:"{total=$2; avail=$7; if(avail==""||avail==0){used=$3}else{used=total-avail}; pct=(total>0)?used*100/total:0; printf "%.2f,%.2f,%.1f\n", total/1024, used/1024, pct }' || true)"
  fi
  if [[ -z "$line" ]] && [[ -r /proc/meminfo ]]; then
    local total_kb=0 avail_kb=0
    total_kb="$(awk '/MemTotal/{print $2; exit}' /proc/meminfo 2>/dev/null || echo 0)"
    avail_kb="$(awk '/MemAvailable/{print $2; exit}' /proc/meminfo 2>/dev/null || echo 0)"
    [[ "$avail_kb" -eq 0 ]] && avail_kb="$(awk '/MemFree/{print $2; exit}' /proc/meminfo 2>/dev/null || echo 0)"
    local used_kb=$(( total_kb - avail_kb ))
    local pct=0
    [[ "$total_kb" -gt 0 ]] && pct=$(( used_kb * 100 / total_kb ))
    local tot_gb=$(( total_kb / 1024 / 1024 ))
    local use_gb=$(( used_kb / 1024 / 1024 ))
    line="${tot_gb},${use_gb},${pct}"
  fi
  echo "${line:-0,0,0}"
}

# ------------------------------
# 7) 资源分析（CPU/内存/磁盘/防火墙）—增强：记录 raw
# ------------------------------
test_resources() {
  log INFO "========== 服务器资源分析 =========="

  local os=""
  if [[ -f /etc/os-release ]]; then
    os="$(grep -E '^(NAME|VERSION)=' /etc/os-release | head -n 2 | paste -sd' ' - | sed 's/NAME=//;s/VERSION=//;s/"//g' || true)"
  fi
  [[ -z "$os" ]] && os="$(uname -a 2>/dev/null || echo unknown)"
  log INFO "  操作系统: $os"
  report_kv_set "res.os" "$os"

  local cpu cores
  cpu="$(get_cpu_usage)"
  cores="$(command_exists nproc && nproc || grep -c processor /proc/cpuinfo 2>/dev/null || echo 0)"
  log INFO "  CPU 使用率: ${cpu}% (核心数: $cores)"
  report_kv_set "res.cpu_usage" "${cpu}%"
  report_kv_set "res.cpu_cores" "$cores"

  local mem_line total used pct
  mem_line="$(get_mem_usage_gb_pct)"
  IFS="," read -r total used pct <<<"$mem_line"
  log INFO "  内存使用: ${used}GB / ${total}GB (${pct}%)"
  report_kv_set "res.mem_total_gb" "${total}GB"
  report_kv_set "res.mem_used_gb" "${used}GB"
  report_kv_set "res.mem_pct" "${pct}%"

  log INFO "  磁盘使用情况:"
  local disk_lines=""
  if command_exists df; then
    disk_lines="$(df -h 2>/dev/null | awk 'NR==1{next} $1 ~ "^/dev/" {printf "%s|%s|%s|%s|%s\n",$6,$2,$3,$4,$5}' | head -n 20 || true)"
    if [[ -n "$disk_lines" ]]; then
      while IFS= read -r l; do [[ -n "$l" ]] && log INFO "    $l"; done <<< "$disk_lines"
      report_kv_set "res.disk" "$disk_lines"
    else
      report_kv_set "res.disk" "N/A"
    fi
  else
    log WARN "    df 不可用，跳过"
    report_kv_set "res.disk" "N/A"
  fi

  local fw_active=0 fw_type="unknown" fw_open="" fw_raw=""
  if command_exists systemctl && systemctl is-active firewalld >/dev/null 2>&1; then
    fw_active=1; fw_type="firewalld"
    local ports services
    ports="$(firewall-cmd --list-ports 2>/dev/null || true)"
    services="$(firewall-cmd --list-services 2>/dev/null || true)"

    # ✅ fw.open 强制压成单行，避免 md 反引号被换行破坏
    fw_open="$( { printf "%s\n" "$ports"; printf "%s\n" "$services"; } | _squash_ws_one_line )"

    fw_raw="$( { echo "[firewall-cmd --list-ports]"; printf "%s\n" "$ports"; echo; echo "[firewall-cmd --list-services]"; printf "%s\n" "$services"; } | sed 's/\r$//' )"
  elif command_exists iptables && iptables -L INPUT >/dev/null 2>&1; then
    fw_type="iptables"
    fw_raw="$(iptables -S 2>/dev/null || iptables -L -n -v 2>/dev/null || true)"
    fw_open="$(printf "%s\n" "$fw_raw" | grep -oE 'dpt:[0-9]+' | cut -d: -f2 | sort -u | paste -sd',' - || true)"
    [[ -n "$fw_open" ]] && fw_active=1 || fw_active=0
  fi

  report_kv_set "fw.type" "$fw_type"
  report_kv_set "fw.active" "$fw_active"
  report_kv_set "fw.open" "$(_kv_or_na "$fw_open")"
  [[ -n "$fw_raw" ]] && report_kv_set "fw.raw" "$fw_raw"

  if [[ "$fw_active" -eq 1 ]]; then
    log INFO "  防火墙状态: 已启用 ($fw_type)"
    [[ -n "$fw_open" ]] && log INFO "  开放端口/服务: $fw_open"
  else
    log WARN "  防火墙状态: 未启用或未知 ($fw_type)"
  fi

  local raw=""
  raw+="操作系统: ${os}\n"
  raw+="CPU 使用率: ${cpu}% (核心数: ${cores})\n"
  raw+="内存使用: ${used}GB / ${total}GB (${pct}%)\n"
  raw+="磁盘使用情况:\n"
  if [[ -n "$disk_lines" && "$disk_lines" != "N/A" ]]; then
    while IFS="|" read -r m t u a p; do
      [[ -z "$m" ]] && continue
      raw+="  ${m}|${t}|${u}|${a}|${p}\n"
    done <<< "$disk_lines"
  else
    raw+="  N/A\n"
  fi
  raw+="防火墙状态: ${fw_active} (${fw_type})\n"
  raw+="开放端口/服务: ${fw_open}\n"
  report_kv_set "res.raw" "$(printf "%b" "$raw")"
}

# ------------------------------
# 8) NTP 检测（增强：写入 raw）
# ------------------------------
test_ntp() {
  log INFO "========== NTP 服务检测 =========="

  local impl="unknown"
  local diff="N/A"
  local raw=""

  # 服务状态
  local chrony_active=0 ntp_active=0
  if command_exists systemctl; then
    systemctl is-active chronyd >/dev/null 2>&1 && chrony_active=1 || true
    systemctl is-active ntpd   >/dev/null 2>&1 && ntp_active=1 || true
  fi

  if [[ "$chrony_active" -eq 1 ]]; then impl="chronyd"; fi
  if [[ "$ntp_active" -eq 1 ]]; then impl="ntpd"; fi

  # 尝试用 chrony 获取 offset（比你之前 date/local_ts 更靠谱）
  if [[ "$impl" == "chronyd" ]] && command_exists chronyc; then
    raw+="[chronyc tracking]\n"
    raw+="$(chronyc tracking 2>/dev/null || true)\n\n"
    # 从 tracking 中取 Last offset（可能是 +/-0.0000xxx seconds）
    local off
    off="$(chronyc tracking 2>/dev/null | awk -F: '/Last offset/ {gsub(/^[ \t]+/,"",$2); print $2}' | head -n1 || true)"
    if [[ -n "$off" ]]; then
      diff="$off"
    fi
  elif [[ "$impl" == "ntpd" ]] && command_exists ntpq; then
    raw+="[ntpq -p]\n"
    raw+="$(ntpq -p 2>/dev/null || true)\n\n"
  fi

  report_kv_set "ntp.impl" "$impl"
  report_kv_set "ntp.diff_seconds" "$diff"
  report_kv_set "ntp.raw" "${raw:-N/A}"

  if [[ "$impl" == "unknown" ]]; then
    log WARN "[NTP] 未检测到 chronyd/ntpd 正在运行"
    echo "NOT_RUNNING"
    return 0
  fi

  log INFO "[NTP] impl=$impl diff=$diff"
  echo "OK"
}

repair_ntp_if_needed() {
  local status="$1"
  if [[ "$status" == "OK" ]]; then
    return 0
  fi
  log WARN "[NTP] 状态=$status，触发修复：fix_ntp_config"
  run_issue_handler "fix_ntp_config" "auto" "--ntp-auto" || true

  log INFO "[NTP] 修复后复检..."
  local post
  post="$(test_ntp)"
  report_kv_set "ntp.recheck" "$post"

  if [[ "$post" == "OK" ]]; then
    log SUCCESS "[NTP] 复检成功：已恢复正常"
  else
    log WARN "[NTP] 复检仍异常：$post"
  fi
}

# ------------------------------
# 9) 文件权限检测（增强：统计缺失项 + 写入报告）
# ------------------------------
PERM_REPORT_INCLUDE_LISTING=1  # 1=在报告中列出存在项的ls -l；0=只统计

check_file_permissions() {
  local platform="$1"
  log INFO "========== 文件权限检测 (平台: $platform) =========="

  local targets=()
  if [[ "$platform" == "new" ]]; then
    targets+=( "/data/services/api/auth/auth-sso-auth/run.sh" "/data/services/api/auth/auth-sso-gatway/run.sh" "/data/services/api/auth/auth-sso-system/run.sh"
               "/data/services/api/java-meeting/java-meeting2.0/run.sh" "/data/services/api/java-meeting/java-meeting3.0/run.sh" "/data/services/api/java-meeting/java-meeting-extapi/run.sh"
               "/data/services/api/java-meeting/java-message-scheduling/run.sh" "/data/services/api/java-meeting/java-mqtt/run.sh" "/data/services/api/java-meeting/java-quartz/run.sh"
               "/data/services/api/start.sh" "/data/services/scripts/*.sh"
               "/data/third_party/paperless/run.sh" "/data/third_party/paperless/start.sh"
               "/data/third_party/wifi-local/config.ini" "/data/third_party/wifi-local/startDB.sh" "/data/third_party/wifi-local/wifi*"
               "/etc/rc.d/rc.local"
               "/data/middleware/nginx/config/*.conf" "/data/middleware/emqx/config/*.conf"
               "/data/services/api/python-cmdb/*.sh" "/data/services/api/python-voice/*.sh" )
  else
    targets+=( "/var/www/java/api-java-meeting2.0/run.sh" "/var/www/java/external-meeting-api/run.sh" "/var/www/java/start.sh"
               "/var/www/html/start.sh"
               "/var/www/wifi-local/config.ini" "/var/www/wifi-local/startDB.sh" "/var/www/wifi-local/wifi*"
               "/var/www/paperless/run.sh" "/var/www/paperless/start.sh"
               "/var/www/redis/redis-*.conf"
               "/var/www/emqx/config/*.conf"
               "/etc/rc.d/rc.local" )
  fi

  local miss=0 exist=0
  local report_list=()
  local report_missing=()

  for p in "${targets[@]}"; do
    if ls -l $p >/dev/null 2>&1; then
      exist=$((exist+1))
      if [[ "$PERM_REPORT_INCLUDE_LISTING" -eq 1 ]]; then
        # 只取前 50 行，防止 *.conf 太多
        while IFS= read -r l; do
          [[ -n "$l" ]] && report_list+=("$l")
          log SUCCESS "[PERM] $l"
        done < <(ls -l $p 2>/dev/null | awk '/^[-dl]/{print}' | head -n 50)
      else
        log SUCCESS "[PERM] EXISTS $p"
      fi
    else
      log WARN "[PERM] MISS $p"
      miss=$((miss+1))
      report_missing+=("$p")
    fi
  done

  log INFO "[PERM] 检测结束：存在项=${exist} 缺失项=${miss}"
  report_kv_set "perm.exist_count" "$exist"
  report_kv_set "perm.miss_count" "$miss"
  if [[ "${#report_missing[@]}" -gt 0 ]]; then
    report_kv_set "perm.missing" "$(printf "%s\n" "${report_missing[@]}")"
  fi
  if [[ "${#report_list[@]}" -gt 0 ]]; then
    report_kv_set "perm.listing" "$(printf "%s\n" "${report_list[@]}")"
  fi
}

# ------------------------------
# 10) 容器信息收集（增强：写入报告KV）
# ------------------------------
collect_container_info() {
  log INFO "========== 容器信息收集 =========="

  if ! command_exists docker; then
    log WARN "Docker 不可用，跳过容器信息收集"
    report_kv_set "docker.available" "false"
    return 0
  fi
  report_kv_set "docker.available" "true"

  local all
  all="$(docker ps -a --format '{{.ID}} {{.Names}} {{.Status}}' 2>/dev/null || true)"

  local running=() stopped=()
  while IFS= read -r line; do
    [[ -z "$line" ]] && continue
    local id name status
    id="$(awk '{print $1}' <<<"$line")"
    name="$(awk '{print $2}' <<<"$line")"
    status="$(cut -d' ' -f3- <<<"$line")"
    if [[ "$status" =~ ^Up ]]; then
      running+=("$name|$id|$status")
    else
      stopped+=("$name|$id|$status")
    fi
  done <<< "$all"

  if [[ "${#running[@]}" -gt 0 ]]; then
    report_kv_set "docker.running" "$(printf "%s\n" "${running[@]}")"
  fi
  if [[ "${#stopped[@]}" -gt 0 ]]; then
    report_kv_set "docker.stopped" "$(printf "%s\n" "${stopped[@]}")"
  fi

  log INFO "----- 运行中的容器 -----"
  for x in "${running[@]}"; do
    IFS="|" read -r name id status <<<"$x"
    log INFO "  [RUNNING] $name ($id) - $status"
  done
  log INFO "----- 未运行的容器 -----"
  for x in "${stopped[@]}"; do
    IFS="|" read -r name id status <<<"$x"
    log INFO "  [STOPPED] $name ($id) - $status"
  done

  local out_base="$SCRIPT_DIR/output/$(get_primary_ip)"
  mkdir -p "$out_base/logs/redis" "$out_base/logs/emqx"
  report_kv_set "middleware.export_base" "$out_base"

  local platform="$1"
  local redis_log="/var/www/redis/data/redis.log"
  local emqx_log="/var/www/emqx/log/emqx.log.1"
  if [[ "$platform" == "new" ]]; then
    redis_log="/data/middleware/redis/data/redis.log"
    emqx_log="/data/middleware/emqx/log/emqx.log.1"
  fi

  local redis_export="" emqx_export=""
  if [[ -f "$redis_log" ]]; then
    redis_export="$out_base/logs/redis/redis.log"
    cp -f "$redis_log" "$redis_export" 2>/dev/null || true
    log SUCCESS "[Redis] redis.log 已导出：$redis_export"
  else
    log WARN "[Redis] redis.log 不存在：$redis_log"
  fi
  if [[ -f "$emqx_log" ]]; then
    emqx_export="$out_base/logs/emqx/emqx.log.1"
    cp -f "$emqx_log" "$emqx_export" 2>/dev/null || true
    log SUCCESS "[Emqx] emqx.log.1 已导出：$emqx_export"
  else
    log WARN "[Emqx] emqx.log.1 不存在：$emqx_log"
  fi
  report_kv_set "redis.export" "${redis_export:-N/A}"
  report_kv_set "emqx.export" "${emqx_export:-N/A}"

  # Redis 异常判定
  local redis_running=0 uredis_running=0 uredis_stopped=0
  for x in "${running[@]}"; do
    IFS="|" read -r name _ _ <<<"$x"
    [[ "$name" =~ [Rr][Ee][Dd][Ii][Ss] ]] && redis_running=1
    [[ "$name" == "uredis" ]] && uredis_running=1
  done
  for x in "${stopped[@]}"; do
    IFS="|" read -r name _ _ <<<"$x"
    [[ "$name" == "uredis" ]] && uredis_stopped=1
  done

  report_kv_set "redis.redis_running" "$redis_running"
  report_kv_set "redis.uredis_running" "$uredis_running"
  report_kv_set "redis.uredis_stopped" "$uredis_stopped"

  if [[ "$uredis_running" -eq 0 && "$uredis_stopped" -eq 1 && "$redis_running" -eq 0 ]]; then
    log ERROR "[Redis] 判定 Redis 容器异常：uredis 未运行且无其他 redis 容器运行，触发修复"
    report_kv_set "redis.exception" "true"
    run_issue_handler "redis_container_exception" "auto" "--non-interactive --yes" || true
    if docker ps --format '{{.Names}}' | grep -w uredis >/dev/null 2>&1; then
      log SUCCESS "[Redis] 复检成功：uredis 已运行"
      report_kv_set "redis.recheck" "OK"
    else
      log WARN "[Redis] 复检失败：uredis 仍未运行（需人工排查）"
      report_kv_set "redis.recheck" "FAIL"
    fi
  else
    log INFO "[Redis] 未检测到需要自动修复的 Redis 容器异常"
    report_kv_set "redis.exception" "false"
  fi
}

# ------------------------------
# 11) 日志导出（增强：记录导出文件列表）
# ------------------------------
export_logs() {
  local platform="$1"
  local has_ujava="$2"
  local has_upython="$3"
  local ujava_container="$4"
  local upython_container="$5"

  log INFO "========== 服务日志导出 =========="

  local export_dir="$SCRIPT_DIR/logs_$(get_primary_ip)_${TS}"
  mkdir -p "$export_dir"

  # 收集导出结果（写入报告KV）
  local exported_files=()

  copy_if_exists() {
    local src="$1"
    local dst="$2"
    if [[ -f "$src" ]]; then
      cp -f "$src" "$dst" 2>/dev/null || true
      exported_files+=("$(basename "$dst")|$src")
      return 0
    fi
    return 1
  }

  if [[ "$platform" == "new" && "$has_ujava" -eq 1 ]]; then
    # ...existing code...（保持不变，但把 cp 的地方尽量改用 copy_if_exists；不改也能跑）
    :
  fi

  if [[ "$platform" == "old" ]]; then
    if [[ "$has_ujava" -eq 1 ]]; then
      copy_if_exists "/var/www/java/api-java-meeting2.0/logs/ubains-INFO-AND-ERROR.log" "$export_dir/对内后端_ubains-INFO-AND-ERROR.log" || true
      copy_if_exists "/var/www/java/external-meeting-api/logs/ubains-INFO-AND-ERROR.log" "$export_dir/对外后端_ubains-INFO-AND-ERROR.log" || true
    fi

    if [[ "$has_upython" -eq 1 ]]; then
      copy_if_exists "/var/www/html/log/error.log" "$export_dir/运维集控_error.log" || true
      copy_if_exists "/var/www/html/log/uinfo.log" "$export_dir/运维集控_uinfo.log" || true
      copy_if_exists "/var/www/html/log/uwsgi.log" "$export_dir/运维集控_uwsgi.log" || true
    fi
  fi

  if [[ "${#exported_files[@]}" -gt 0 ]]; then
    report_kv_set "log_export.dir" "$export_dir"
    report_kv_set "log_export.files" "$(printf "%s\n" "${exported_files[@]}")"
  else
    report_kv_set "log_export.dir" "$export_dir"
    report_kv_set "log_export.files" ""
  fi

  log SUCCESS "日志导出目录: $export_dir"
  echo "$export_dir"
}

# ------------------------------
# 12) Android 自检（增强：记录状态/导出目录）
# ------------------------------
android_self_check() {
  log INFO "========== 安卓设备自检 (PRD 15) =========="

  if ! command_exists adb; then
    log WARN "[Android] 未检测到 adb，跳过"
    report_kv_set "android.status" "SKIP(no_adb)"
    return 0
  fi

  read -r -p "请输入安卓设备IP（留空跳过安卓自检）: " dev_ip
  dev_ip="${dev_ip// /}"

  if [[ -z "$dev_ip" || "$dev_ip" =~ ^(n|no|skip)$ ]]; then
    log INFO "[Android] 已跳过安卓自检"
    report_kv_set "android.status" "SKIP"
    return 0
  fi

  read -r -p "请输入安卓设备端口 [默认 5555]: " dev_port
  dev_port="${dev_port// /}"
  dev_port="${dev_port:-5555}"
  if ! [[ "$dev_port" =~ ^[0-9]+$ ]]; then
    log WARN "[Android] 端口输入无效：$dev_port，使用默认 5555"
    dev_port="5555"
  fi

  local target="${dev_ip}:${dev_port}"
  report_kv_set "android.target" "$target"

  log INFO "[Android] 连接设备: adb connect $target"
  if ! adb connect "$target" >/dev/null 2>&1; then
    log ERROR "[Android] 连接失败: $target"
    report_kv_set "android.status" "FAIL(connect)"
    return 0
  fi
  log SUCCESS "[Android] 连接成功: $target"

  local out_dir="$SCRIPT_DIR/logs/android/${target//[:\/]/_}_${TS}"
  mkdir -p "$out_dir"
  report_kv_set "android.export_dir" "$out_dir"

  adb -s "$target" pull "/sdcard/Android/data/com.ubains.local.gviewer/files/" "$out_dir/files" >/dev/null 2>&1 || true

  if adb -s "$target" shell "test -d /sdcard/Android/data/com.ubains.local.gviewer/cache" >/dev/null 2>&1; then
    adb -s "$target" pull "/sdcard/Android/data/com.ubains.local.gviewer/cache/" "$out_dir/cache" >/dev/null 2>&1 || true
  fi

  adb disconnect "$target" >/dev/null 2>&1 || true
  report_kv_set "android.status" "OK"
  log SUCCESS "[Android] 日志导出目录: $out_dir"
}

# ------------------------------
# 14) 配置文件 IP 检测（补齐：按 PRD 8；只记录异常/需关注项；合法项不打印）
# 规则：
# - 允许：目标服务器IP / 127.0.0.1 / 172.17.0.1
# - 仅打印出：出现“疑似 IP”但不在允许列表内的行
# - 不修改任何文件
# 输出：
# - cfgip.status = OK/WARN/FAIL
# - cfgip.server_ip
# - cfgip.scan_root (平台根)
# - cfgip.finding.count
# - cfgip.finding.lines（代码块原样输出，格式：file:line:content）
# ------------------------------
CFGIP_ALLOWED_EXTRA=("127.0.0.1" "172.17.0.1")

_cfgip_allowed() {
  local ip="$1"
  local server_ip="$2"
  [[ "$ip" == "$server_ip" ]] && return 0
  local x
  for x in "${CFGIP_ALLOWED_EXTRA[@]}"; do
    [[ "$ip" == "$x" ]] && return 0
  done
  return 1
}

_cfgip_add_file_if_exists() {
  local f="$1"
  [[ -f "$f" ]] && printf "%s\n" "$f"
}

_cfgip_add_glob_files() {
  local dir="$1"; shift
  local ext f
  [[ -d "$dir" ]] || return 0

  shopt -s nullglob
  for ext in "$@"; do
    for f in "$dir"/*."$ext"; do
      [[ -f "$f" ]] && printf "%s\n" "$f"
    done
  done
  shopt -u nullglob
}

_cfgip_add_find_files() {
  # 用法：_cfgip_add_find_files "/data/services/web/pc/pc-vue2-main" 5000 js json conf
  local dir="$1"; shift
  local max="${1:-5000}"; shift
  local exts=("$@")

  [[ -d "$dir" ]] || return 0
  [[ "${#exts[@]}" -gt 0 ]] || exts=(js json conf)

  local cond="" ext
  for ext in "${exts[@]}"; do
    cond="${cond} -name '*.${ext}' -o"
  done
  cond="${cond% -o}"

  find "$dir" \
    -type d \( -name node_modules -o -name dist -o -name build -o -name .git \) -prune -o \
    -type f \( $cond \) -print 2>/dev/null | head -n "$max"
}

_cfgip_list_files_new() {
  {
    _cfgip_add_glob_files "/data/services/api/auth/auth-sso-auth/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/data/services/api/auth/auth-sso-gatway/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/data/services/api/auth/auth-sso-system/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/data/services/api/java-meeting/java-meeting2.0/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/data/services/api/java-meeting/java-meeting3.0/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/data/services/api/java-meeting/java-meeting-extapi/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/data/services/api/java-meeting/java-message-scheduling/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/data/services/api/java-meeting/java-mqtt/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/data/services/api/java-meeting/java-quartz/config" yml yaml properties js json conf

    _cfgip_add_glob_files "/data/services/api/python-cmdb" yml yaml properties js json conf

    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-ai" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-backstage" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-editor" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-main" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-meetingControl" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-meetngV2" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-meetngV3" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-moniter" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-platform" 5000 js json conf
    _cfgip_add_find_files "/data/services/web/pc/pc-vue2-voice" 5000 js json conf

    _cfgip_add_glob_files "/data/middleware/nginx/config" conf
  } | awk 'NF' | sort -u
}

_cfgip_list_files_old() {
  {
    _cfgip_add_glob_files "/var/www/java/api-java-meeting2.0/config" yml yaml properties js json conf
    _cfgip_add_glob_files "/var/www/java/external-meeting-api/config" yml yaml properties js json conf

    _cfgip_add_file_if_exists "/var/www/java/ubains-web-2.0/static/config.json"
    _cfgip_add_file_if_exists "/var/www/java/ubains-web-admin/static/config.json"
    _cfgip_add_file_if_exists "/var/www/java/ubains-web-h5/static/h5/config.js"
    _cfgip_add_file_if_exists "/var/www/java/ubains-web-h5/static/h5/config.json"

    _cfgip_add_glob_files "/var/www/java/nginx-conf.d" conf

    _cfgip_add_glob_files "/var/www/html/conf" conf json js
    _cfgip_add_glob_files "/var/www/html/nginx-conf" conf
    _cfgip_add_file_if_exists "/var/www/html/web-vue-rms/static/config.json"
    _cfgip_add_file_if_exists "/var/www/html/web-vue-h5/static/h5/config.json"
    _cfgip_add_file_if_exists "/var/www/html/web-vue-h5/static/h5/config.js"
  } | awk 'NF' | sort -u
}

# ---- helper: 永远用安全格式输出（避免 printf %sn 之类拼写问题）
p() {
  local s
  for s in "$@"; do
    printf '%s\n' "$s"
  done
}

test_config_ip() {
  local platform="$1"
  local server_ip="$2"

  section "配置文件 IP 检测"
  log INFO "[CFGIP] 开始扫描配置文件中的 IP（只输出异常/需关注项）"

  report_kv_set "cfgip.server_ip" "$server_ip"
  report_kv_set "cfgip.status" "OK"

  local scan_root
  if [[ "$platform" == "new" ]]; then
    scan_root="/data"
  else
    scan_root="/var/www"
  fi
  report_kv_set "cfgip.scan_root" "$scan_root"

  local ip_re='([0-9]{1,3}\.){3}[0-9]{1,3}'
  local findings=()
  local count=0

  # 关键修复点：
  # 1) 不把文件列表落到变量里（避免 CRLF/不可见字符污染后续处理）
  # 2) 对读到的路径做强过滤：去 CR、跳过含冒号、跳过空白异常
  local list_cmd=""
  if [[ "$platform" == "new" ]]; then
    list_cmd="_cfgip_list_files_new"
  else
    list_cmd="_cfgip_list_files_old"
  fi

  while IFS= read -r f; do
    # 去掉潜在的 CR（Windows 换行）
    f="${f%$'\r'}"
    [[ -z "$f" ]] && continue

    # 防御：过滤掉意外混入的 "file:line:content" 或其他带冒号的行
    [[ "$f" == *":"* ]] && continue

    # 防御：路径不应包含制表符等
    [[ "$f" =~ [[:space:]] ]] && {
      # 如果你确认你的路径不会有空格，直接跳过更安全
      continue
    }

    [[ -f "$f" ]] || continue

    # 文本过滤（可选但建议）
    if command_exists file; then
      if ! file "$f" 2>/dev/null | grep -qiE 'text|json|xml|yaml|ascii|utf-8'; then
        continue
      fi
    fi

    local hits
    hits="$(grep -nE "$ip_re" "$f" 2>/dev/null || true)"
    [[ -z "$hits" ]] && continue

    while IFS= read -r line; do
      line="${line%$'\r'}"
      [[ -z "$line" ]] && continue

      local ln content
      ln="${line%%:*}"
      content="${line#*:}"

      # ✅ 关键：清理控制字符，避免写报告时 printf/解析错位
      content="${content//$'\r'/}"
      content="${content//$'\0'/}"

      local ips
      ips="$(grep -oE "$ip_re" <<<"$content" | sort -u || true)"
      [[ -z "$ips" ]] && continue

      local bad=0 ip
      while IFS= read -r ip; do
        ip="${ip%$'\r'}"
        [[ -z "$ip" ]] && continue
        if ! _cfgip_allowed "$ip" "$server_ip"; then
          bad=1
          break
        fi
      done <<<"$ips"

      if [[ "$bad" -eq 1 ]]; then
        count=$((count+1))
        findings+=("${f}:${ln}:${content}")
        log WARN "[CFGIP] 异常IP行: ${f}:${ln}:${content}"
      fi
    done <<<"$hits"
  done < <($list_cmd)

  report_kv_set "cfgip.finding.count" "$count"
  if [[ "$count" -gt 0 ]]; then
    report_kv_set "cfgip.status" "WARN"
    # 改成 helper，避免引号问题
    report_kv_set "cfgip.finding.lines" "$(p "${findings[@]}")"
  else
    report_kv_set "cfgip.finding.lines" ""
  fi

  report_add ""
  report_add "## 配置文件 IP 检测"
  report_add "- status: $(report_kv_get "cfgip.status")"
  report_add "- finding_count: $count"
}

# ------------------------------
# 15) 定时任务查询（补齐：PRD 13）
# - 只检测、记录，不改动
# - 输出：
#   sched.status = OK/WARN
#   sched.crontab.lines = 原样输出
#   sched.expected.miss = 缺失项（如果能识别平台）
# ------------------------------
test_scheduled_tasks() {
  section "定时任务查询（crontab）"
  local platform="$1"

  local cron
  cron="$(crontab -l 2>/dev/null || true)"

  if [[ -z "$cron" ]]; then
    log WARN "[CRON] 当前用户无 crontab 或无法读取"
    report_kv_set "sched.status" "WARN"
    report_kv_set "sched.crontab.lines" "N/A"
    return 0
  fi

  log INFO "[CRON] crontab -l："
  while IFS= read -r l; do
    [[ -n "$l" ]] && log INFO "  $l"
  done <<< "$cron"

  report_kv_set "sched.crontab.lines" "$cron"

  local expected=()
  expected+=("0 13 * * * bash /usr/local/docker/UbainsmysqlBakUp.sh")
  if [[ "$platform" == "new" ]]; then
    expected+=("*/3 * * * * /data/services/scripts/ujava2-startup.sh")
  fi

  local miss=()
  local e
  for e in "${expected[@]}"; do
    if grep -Fq "$e" <<<"$cron"; then
      :
    else
      miss+=("$e")
    fi
  done

  if [[ "${#miss[@]}" -gt 0 ]]; then
    report_kv_set "sched.status" "WARN"
    report_kv_set "sched.expected.miss" "$(printf "%s\n" "${miss[@]}")"
    log WARN "[CRON] 缺失预期任务："
    for e in "${miss[@]}"; do log WARN "  $e"; done
  else
    report_kv_set "sched.status" "OK"
    report_kv_set "sched.expected.miss" ""
    log SUCCESS "[CRON] 预期任务已存在"
  fi

  report_add ""
  report_add "## 定时任务查询"
  report_add "- status: $(report_kv_get "sched.status")"
}

# ------------------------------
# 16) 容器异常：Emqx（补齐：PRD 12）
# 规则：
# - 如果 uemqx 未运行 && 没有其它包含 emqx 的运行容器 => 异常
# - 执行 issue_handler.sh --action emqx_container_exception
# - 复检 uemqx 是否 Up
# - 输出：
#   emqx.exception, emqx.recheck
# ------------------------------
check_emqx_container_exception() {
  local running_txt stopped_txt
  running_txt="$(report_kv_get "docker.running")"
  stopped_txt="$(report_kv_get "docker.stopped")"

  # docker 不可用直接跳过
  if [[ "$(report_kv_get "docker.available")" != "true" ]]; then
    report_kv_set "emqx.exception" "SKIP(no_docker)"
    return 0
  fi

  local has_emqx_running=0 uemqx_running=0 uemqx_stopped=0

  while IFS= read -r l; do
    [[ -z "$l" ]] && continue
    local name="${l%%|*}"
    [[ "$name" == "uemqx" ]] && uemqx_running=1
    [[ "$name" =~ [Ee][Mm][Qq][Xx] ]] && has_emqx_running=1
  done <<<"$running_txt"

  while IFS= read -r l; do
    [[ -z "$l" ]] && continue
    local name="${l%%|*}"
    [[ "$name" == "uemqx" ]] && uemqx_stopped=1
  done <<<"$stopped_txt"

  if [[ "$uemqx_running" -eq 0 && "$uemqx_stopped" -eq 1 && "$has_emqx_running" -eq 0 ]]; then
    log ERROR "[Emqx] 判定 Emqx 容器异常：uemqx 未运行且无其他 emqx 容器运行，触发修复"
    report_kv_set "emqx.exception" "true"
    run_issue_handler "emqx_container_exception" "auto" "--non-interactive --yes" || true

    if docker ps --format '{{.Names}}' | grep -w uemqx >/dev/null 2>&1; then
      log SUCCESS "[Emqx] 复检成功：uemqx 已运行"
      report_kv_set "emqx.recheck" "OK"
    else
      log WARN "[Emqx] 复检失败：uemqx 仍未运行（需人工排查）"
      report_kv_set "emqx.recheck" "FAIL"
    fi
  else
    log INFO "[Emqx] 未检测到需要自动修复的 Emqx 容器异常"
    report_kv_set "emqx.exception" "false"
  fi
}

# ------------------------------
# 17) 现场数据备份（对齐 ps1：目录复制 + 可选 mysqldump + 打包）
# - 本机版不做“下载到本地”，只在本机生成 tar.gz 并写入报告
# - 输出：
#   bak.status = OK/WARN/FAIL/SKIP
#   bak.dir = 备份目录
#   bak.archive = 打包文件
#   bak.items = 备份项列表
#   bak.db.dump = 数据库备份文件（如有）
# ------------------------------
data_backup() {
  local platform="$1"
  local ts="$2"

  section "现场数据备份（DataBackup）"
  local base="/home/bakup"
  local dir="${base}/health_bak_${ts}"
  local items=()
  local warnings=0

  mkdir -p "$dir" || {
    log ERROR "[BAK] 无法创建备份目录：$dir"
    report_kv_set "bak.status" "FAIL"
    report_kv_set "bak.dir" "$dir"
    return 1
  }

  report_kv_set "bak.dir" "$dir"

  copy_tree_if_exists() {
    local src="$1"
    local name="$2"
    local dst="${dir}/${name}"
    if [[ -e "$src" ]]; then
      log INFO "[BAK] 复制：$src -> $dst"
      mkdir -p "$(dirname "$dst")" 2>/dev/null || true
      if command_exists rsync; then
        rsync -a --delete --exclude 'node_modules' --exclude '.git' "$src" "$dst" >>"$LOG_FILE" 2>&1 || {
          log WARN "[BAK] rsync 失败：$src"
          warnings=$((warnings+1))
          return 1
        }
      else
        cp -a "$src" "$dst" >>"$LOG_FILE" 2>&1 || {
          log WARN "[BAK] cp -a 失败：$src"
          warnings=$((warnings+1))
          return 1
        }
      fi
      items+=("$name|$src")
      return 0
    fi
    log WARN "[BAK] 不存在，跳过：$src"
    warnings=$((warnings+1))
    return 1
  }

  # 目录/文件清单（按 ps1 的传统平台思路）
  if [[ "$platform" == "old" ]]; then
    copy_tree_if_exists "/var/www/java"  "var_www_java"  || true
    copy_tree_if_exists "/var/www/html"  "var_www_html"  || true
    copy_tree_if_exists "/var/www/redis" "var_www_redis" || true
    copy_tree_if_exists "/var/www/emqx"  "var_www_emqx"  || true
  else
    # 新平台：按实际目录结构做一个合理对齐（可按需调整）
    copy_tree_if_exists "/data/services"           "data_services"   || true
    copy_tree_if_exists "/data/middleware"         "data_middleware" || true
    copy_tree_if_exists "/data/third_party"        "data_third_party"|| true
    copy_tree_if_exists "/etc"                     "etc"             || true
  fi

  # 数据库备份（如有 umysql 容器）
  local db_dump="N/A"
  if command_exists docker && docker ps --format '{{.Names}}' | grep -w umysql >/dev/null 2>&1; then
    local dump_file="${dir}/mysql_dump_${ts}.sql"
    log INFO "[BAK] 检测到 umysql 容器，尝试 mysqldump -> $dump_file"
    if docker exec umysql sh -c 'command -v mysqldump >/dev/null 2>&1' >/dev/null 2>&1; then
      # 这里尽量“全库导出”。账号密码差异较大，先尝试无密码/默认root，失败则记录WARN
      if docker exec umysql sh -c "mysqldump -uroot --all-databases --single-transaction --quick --lock-tables=false" >"$dump_file" 2>>"$LOG_FILE"; then
        db_dump="$dump_file"
        log SUCCESS "[BAK] mysqldump 成功：$dump_file"
      else
        log WARN "[BAK] mysqldump 失败（可能需要密码/账号），已跳过（详见日志）"
        rm -f "$dump_file" 2>/dev/null || true
        warnings=$((warnings+1))
      fi
    else
      log WARN "[BAK] umysql 容器内无 mysqldump，跳过数据库导出"
      warnings=$((warnings+1))
    fi
  else
    log INFO "[BAK] 未检测到 umysql 容器，跳过数据库导出"
  fi
  report_kv_set "bak.db.dump" "$db_dump"

  # 打包
  local archive="${base}/health_bak_${ts}.tar.gz"
  log INFO "[BAK] 打包：$archive"
  if tar -czf "$archive" -C "$dir" . >>"$LOG_FILE" 2>&1; then
    log SUCCESS "[BAK] 打包完成：$archive"
    report_kv_set "bak.archive" "$archive"
  else
    log ERROR "[BAK] 打包失败：$archive"
    report_kv_set "bak.archive" "N/A"
    report_kv_set "bak.status" "FAIL"
    return 1
  fi

  if [[ "${#items[@]}" -gt 0 ]]; then
    report_kv_set "bak.items" "$(printf "%s\n" "${items[@]}")"
  else
    report_kv_set "bak.items" ""
  fi

  if [[ "$warnings" -gt 0 ]]; then
    report_kv_set "bak.status" "WARN"
  else
    report_kv_set "bak.status" "OK"
  fi
  return 0
}

# ------------------------------
# 18) 报告生成（Markdown）—增强：把 KV 也写入
# ------------------------------
write_report() {
  local server_ip="$1"
  local platform="$2"
  local systems_txt="$3"
  local report_file="$4"

  : > "$report_file" || return 1
  w() {
    local s
    for s in "$@"; do
      printf '%s\n' "$s" >>"$report_file"
    done
  }

  w "<!-- REPORT_GENERATOR: sh-write_report-v4(unified)  $(date '+%F %T') -->"
  w ""
  w "# 服务自检报告"
  w ""
  w "- 服务器地址: ${server_ip}"
  w "- 平台类型: $( [[ "$platform" == "new" ]] && echo "新统一平台" || echo "传统平台" )"
  w "- 检测时间: $(date '+%Y-%m-%d %H:%M:%S')"
  w "- 脚本版本: $SCRIPT_VERSION"
  w ""
  
  # ✅ 新增：将旧的 REPORT_LINES（摘要/过程性输出）写入报告，避免丢失
  if [[ "${#REPORT_LINES[@]}" -gt 0 ]]; then
    w "## 过程摘要（REPORT_LINES）"
    w ""
    w "> 该段来自脚本过程汇总（report_add），用于补齐 ps1 风格的摘要输出。"
    w ""
    local rl
    for rl in "${REPORT_LINES[@]}"; do
      w "$rl"
    done
    w ""
  fi

  w "## 系统识别"
  w ""
  w '```'
  printf "%s\n" "$systems_txt" >> "$report_file"
  w '```'
  w ""

  # ✅ 摘要：只放结论，不放长明细（避免重复、不易读）
  w "## 检测结果摘要"
  w ""
  w "- DNS: \`$(report_kv_get "dns.status")\` (nameserver: $(report_kv_get "dns.nameservers"))"
  w "- CPU: \`$(report_kv_get "res.cpu_usage")\`  MEM: \`$(report_kv_get "res.mem_used_gb")/$(report_kv_get "res.mem_total_gb") ($(report_kv_get "res.mem_pct"))\`"
  w "- DISK: 详见“检测详情 -> 资源分析/磁盘”（关注 /home 这类高占用挂载点）"
  w "- Firewall: type=\`$(report_kv_get "fw.type")\` active=\`$(report_kv_get "fw.active")\` open=\`$(report_kv_get "fw.open")\`"
  w "- NTP: \`$(report_kv_get "ntp.status")\` (impl: $(report_kv_get "ntp.impl"), diff: $(report_kv_get "ntp.diff_seconds"))"
  w "- Perm: miss_count=\`$(report_kv_get "perm.miss_count")\`"
  w "- Docker: \`$(report_kv_get "docker.available")\`"
  w ""

  w "## 检测详情"
  w ""

  # DNS 详情（raw）
  w "### DNS 检测（详细）"
  w "- 初次检测结果: \`$(report_kv_get "dns.status")\`"
  w "- nameserver: \`$(report_kv_get "dns.nameservers")\`"
  local dns_raw
  dns_raw="$(report_kv_get "dns.raw")"
  if [[ -n "$dns_raw" ]]; then
    w ""
    w '```'
    printf "%s\n" "$dns_raw" >> "$report_file"
    w '```'
  fi
  w ""

  # 资源 raw + disk 表格
  w "### 资源分析（详细）"
  local res_raw
  res_raw="$(report_kv_get "res.raw")"
  if [[ -n "$res_raw" ]]; then
    w '```'
    printf "%s\n" "$res_raw" >> "$report_file"
    w '```'
  fi
  local disk
  disk="$(report_kv_get "res.disk")"
  if [[ -n "$disk" && "$disk" != "N/A" ]]; then
    w ""
    w "| 挂载点 | 总量 | 已用 | 可用 | 使用率 |"
    w "|---|---:|---:|---:|---:|"
    while IFS="|" read -r m t u a p; do
      [[ -z "$m" ]] && continue
      w "| $m | $t | $u | $a | $p |"
    done <<< "$disk"
  fi
  w ""

  # 防火墙：open 单行 + raw 代码块
  w "### 防火墙（详细）"
  w "- type: \`$(report_kv_get "fw.type")\`"
  w "- active: \`$(report_kv_get "fw.active")\`"
  w "- open: \`$(report_kv_get "fw.open")\`"
  local fw_raw
  fw_raw="$(report_kv_get "fw.raw")"
  if [[ -n "$fw_raw" ]]; then
    w ""
    w '```'
    printf "%s\n" "$fw_raw" >> "$report_file"
    w '```'
  fi
  w ""

  # NTP：保证不空 + raw
  w "### NTP 检测（详细）"
  w "- 初次检测结果: \`$(report_kv_get "ntp.status")\`"
  w "- impl: \`$(report_kv_get "ntp.impl")\`"
  w "- diff: \`$(report_kv_get "ntp.diff_seconds")\`"
  local ntp_raw
  ntp_raw="$(report_kv_get "ntp.raw")"
  if [[ -n "$ntp_raw" && "$ntp_raw" != "N/A" ]]; then
    w ""
    w '```'
    printf "%s\n" "$ntp_raw" >> "$report_file"
    w '```'
  fi
  w ""

  # 权限：listing + missing
  w "### 文件权限检测（详细）"
  w "- exist_count: \`$(report_kv_get "perm.exist_count")\`"
  w "- miss_count: \`$(report_kv_get "perm.miss_count")\`"
  local perm_listing perm_missing
  perm_listing="$(report_kv_get "perm.listing")"
  perm_missing="$(report_kv_get "perm.missing")"
  if [[ -n "$perm_listing" ]]; then
    w ""
    w "**存在项（ls -l）：**"
    w '```'
    printf "%s\n" "$perm_listing" >> "$report_file"
    w '```'
  fi
  if [[ -n "$perm_missing" ]]; then
    w ""
    w "**缺失项：**"
    w '```'
    printf "%s\n" "$perm_missing" >> "$report_file"
    w '```'
  fi
  w ""

  # 容器（原样输出）
  w "### 容器与中间件（详细）"
  w "- docker: \`$(report_kv_get "docker.available")\`"
  local dr ds
  dr="$(report_kv_get "docker.running")"
  ds="$(report_kv_get "docker.stopped")"
  if [[ -n "$dr" ]]; then
    w ""
    w "**运行容器（name|id|status）：**"
    w '```'
    printf "%s\n" "$dr" >> "$report_file"
    w '```'
  fi
  if [[ -n "$ds" ]]; then
    w ""
    w "**停止容器（name|id|status）：**"
    w '```'
    printf "%s\n" "$ds" >> "$report_file"
    w '```'
  fi
  w "- redis.log export: \`$(report_kv_get "redis.export")\`"
  w "- emqx.log export: \`$(report_kv_get "emqx.export")\`"
  w "- redis container exception: \`$(report_kv_get "redis.exception")\`"
  local redis_recheck
  redis_recheck="$(report_kv_get "redis.recheck")"
  [[ -n "$redis_recheck" ]] && w "- redis recheck: \`$redis_recheck\`"
  w ""
  
  w "### 配置文件 IP 检测（详细）"
  w "- status: \`$(report_kv_get "cfgip.status")\`"
  w "- server_ip: \`$(report_kv_get "cfgip.server_ip")\`"
  w "- scan_root: \`$(report_kv_get "cfgip.scan_root")\`"
  w "- finding_count: \`$(report_kv_get "cfgip.finding.count")\`"
  local cfg_lines
  cfg_lines="$(report_kv_get "cfgip.finding.lines")"
  if [[ -n "$cfg_lines" ]]; then
    w ""
    w '```'
    printf "%s\n" "$cfg_lines" >>"$report_file"
    w '```'
  else
    w "- findings: （无异常IP行）"
  fi
  w ""

  w "### 定时任务查询（详细）"
  w "- status: \`$(report_kv_get "sched.status")\`"
  local cron_lines miss_lines
  cron_lines="$(report_kv_get "sched.crontab.lines")"
  miss_lines="$(report_kv_get "sched.expected.miss")"
  if [[ -n "$cron_lines" && "$cron_lines" != "N/A" ]]; then
    w ""
    w "**crontab -l：**"
    w '```'
    printf "%s\n" "$cron_lines" >>"$report_file"
    w '```'
  fi
  if [[ -n "$miss_lines" ]]; then
    w ""
    w "**缺失的预期任务：**"
    w '```'
    printf "%s\n" "$miss_lines" >>"$report_file"
    w '```'
  fi
  w ""

  w "### Emqx 容器异常（详细）"
  w "- exception: \`$(report_kv_get "emqx.exception")\`"
  local emqx_recheck
  emqx_recheck="$(report_kv_get "emqx.recheck")"
  [[ -n "$emqx_recheck" ]] && w "- recheck: \`$emqx_recheck\`"
  w ""

  # 说明
  w "## 说明"
  w ""
  w "- 详细过程日志见：\`${LOG_FILE}\`"
}

# ------------------------------
# 主流程
# ------------------------------
main() {
  section "服务自检开始 (Linux 版本，本机运行)"
  log INFO "脚本版本: $SCRIPT_VERSION"
  log INFO "脚本路径: $SCRIPT_DIR"
  log INFO "日志文件: $LOG_FILE"

  section "平台识别/系统识别"
  local server_ip platform
  server_ip="$(get_primary_ip)"
  platform="$(detect_platform)"
  log INFO "[平台识别] 平台类型=$( [[ "$platform" == "new" ]] && echo "新统一平台(/data/services)" || echo "传统平台(/var/www)" )"

  report_add "## 平台识别"
  report_add "- 平台类型: $( [[ "$platform" == "new" ]] && echo "新统一平台(/data/services)" || echo "传统平台(/var/www)" )"

  local sys_kv
  sys_kv="$(detect_systems)"
  eval "$sys_kv"
  log INFO "[系统识别] HAS_UJAVA=$HAS_UJAVA HAS_UPYTHON=$HAS_UPYTHON HAS_UPYTHON_VOICE=$HAS_UPYTHON_VOICE containers=(ujava=$UJAVA_CONTAINER upython=$UPYTHON_CONTAINER upython_voice=$UPYTHON_VOICE_CONTAINER)"

  report_add ""
  report_add "## 系统识别（摘要）"
  report_add_block '```' "$sys_kv" '```'

  # 1) 服务检测
  section "服务检测"
  local ujava_container_results=""
  local ujava_host_results=""
  local upython_results=""
  local upython_voice_results=""

  if [[ "$platform" == "new" ]]; then
    if [[ "$HAS_UJAVA" -eq 1 && -n "${UJAVA_CONTAINER:-}" ]]; then
      ujava_container_results="$(test_ujava_services "$UJAVA_CONTAINER" "$platform" || true)"
      ujava_host_results="$(test_ujava_host_services || true)"

      if echo "$ujava_host_results" | grep -q "extapi|ubains-meeting-api-1.0-SNAPSHOT.jar|stopped"; then
        log WARN "[EXT] 检测到对外服务未运行，触发修复: fix_external_service_disconnect"
        run_issue_handler "fix_external_service_disconnect" "$platform" "--non-interactive --yes" || true
        log INFO "[EXT] 修复后复检 extapi..."
        test_ujava_host_services >/dev/null || true
      fi
    else
      log WARN "未检测到 ujava 容器，改为宿主机检测 ujava 进程"
      ujava_container_results="$(test_ujava_services "" "$platform" || true)"
      ujava_host_results="$(test_ujava_host_services || true)"
    fi

    if [[ "$HAS_UPYTHON" -eq 1 && -n "${UPYTHON_CONTAINER:-}" ]]; then
      upython_results="$(test_container_ports "$UPYTHON_CONTAINER" "upython" "${UPYTHON_PORTS_NEW[@]}" || true)"
    fi
    if [[ "$HAS_UPYTHON_VOICE" -eq 1 && -n "${UPYTHON_VOICE_CONTAINER:-}" ]]; then
      upython_voice_results="$(test_container_ports "$UPYTHON_VOICE_CONTAINER" "upython_voice" "${UPYTHON_VOICE_PORTS_NEW[@]}" || true)"
    fi
  else
    log INFO "传统平台：使用传统平台检测逻辑"

    if [[ "$HAS_UJAVA" -eq 1 && -n "${UJAVA_CONTAINER:-}" ]]; then
      ujava_container_results="$(test_ujava_old_container_services "$UJAVA_CONTAINER" || true)"
    else
      log WARN "未检测到 ujava 容器，跳过容器内服务检测"
    fi

    ujava_host_results="$(test_ujava_host_services || true)"
    if echo "$ujava_host_results" | grep -q "extapi|ubains-meeting-api-1.0-SNAPSHOT.jar|stopped"; then
      log WARN "[EXT] 检测到对外服务未运行，触发修复: fix_external_service_disconnect"
      run_issue_handler "fix_external_service_disconnect" "$platform" "--non-interactive --yes" || true
      log INFO "[EXT] 修复后复检 extapi..."
      test_ujava_host_services >/dev/null || true
    fi

    if [[ "$HAS_UPYTHON" -eq 1 && -n "${UPYTHON_CONTAINER:-}" ]]; then
      upython_results="$(test_container_ports "$UPYTHON_CONTAINER" "upython(传统平台)" "${UPYTHON_PORTS_OLD[@]}" || true)"
    fi
    if [[ "$HAS_UPYTHON_VOICE" -eq 1 && -n "${UPYTHON_VOICE_CONTAINER:-}" ]]; then
      upython_voice_results="$(test_container_ports "$UPYTHON_VOICE_CONTAINER" "upython_voice" "${UPYTHON_VOICE_PORTS_NEW[@]}" || true)"
    fi
  fi

  report_add ""
  report_add "## 服务检测"
  if [[ -n "${ujava_container_results:-}" ]]; then
    report_add "- ujava 检测结果："
    report_add_block '```' "$ujava_container_results" '```'
  else
    report_add "- ujava 检测结果：无（未检测或无容器/进程）"
  fi

  if [[ -n "${ujava_host_results:-}" ]]; then
    report_add "- extapi 检测结果："
    report_add_block '```' "$ujava_host_results" '```'
  else
    report_add "- extapi 检测结果：无"
  fi

  if [[ -n "${upython_results:-}" ]]; then
    report_add "- upython 端口检测结果："
    report_add_block '```' "$upython_results" '```'
  else
    report_add "- upython 端口检测结果：无"
  fi

  if [[ -n "${upython_voice_results:-}" ]]; then
    report_add "- upython_voice 端口检测结果："
    report_add_block '```' "$upython_voice_results" '```'
  fi

  # 2) DNS
  section "DNS 检测"
  local dns_status
  dns_status="$(test_dns)"
  report_kv_set "dns.status" "$dns_status"
  repair_dns_if_needed "$dns_status"

  report_add ""
  report_add "## DNS 检测"
  report_add "- 初次检测结果: $dns_status"
  local dns_recheck
  dns_recheck="$(report_kv_get "dns.recheck")"
  [[ -n "$dns_recheck" ]] && report_add "- 修复后复检结果: $dns_recheck"
  report_add "- nameserver: $(report_kv_get "dns.nameservers")"

  # 3) 资源
  section "资源分析"
  test_resources

  report_add ""
  report_add "## 资源分析"
  report_add "- OS: $(report_kv_get "res.os")"
  report_add "- CPU: $(report_kv_get "res.cpu_usage") (cores: $(report_kv_get "res.cpu_cores"))"
  report_add "- MEM: $(report_kv_get "res.mem_used_gb")/$(report_kv_get "res.mem_total_gb") ($(report_kv_get "res.mem_pct"))"
  report_add "- DISK: 详见下方“检测详情 -> 资源分析/磁盘”"
  report_add "- 防火墙: type=$(report_kv_get "fw.type") active=$(report_kv_get "fw.active") open=$(report_kv_get "fw.open")"

  # 4) 配置文件 IP 检测（✅ 已实现：替换“未实现”占位）
  test_config_ip "$platform" "$server_ip"
  # （test_config_ip 内部已做 section + report_add 摘要，这里不再重复写）

  # 5) NTP
  section "NTP 检测"
  local ntp_status
  ntp_status="$(test_ntp)"
  report_kv_set "ntp.status" "$ntp_status"
  repair_ntp_if_needed "$ntp_status"

  report_add ""
  report_add "## NTP 检测"
  report_add "- 初次检测结果: $ntp_status"
  report_add "- impl: $(report_kv_get "ntp.impl")"
  report_add "- diff_seconds: $(report_kv_get "ntp.diff_seconds")"
  local ntp_recheck
  ntp_recheck="$(report_kv_get "ntp.recheck")"
  [[ -n "$ntp_recheck" ]] && report_add "- 修复后复检结果: $ntp_recheck"

  # 6) 文件权限
  section "文件权限检测"
  check_file_permissions "$platform"

  report_add ""
  report_add "## 文件权限检测"
  report_add "- exist_count: $(report_kv_get "perm.exist_count")"
  report_add "- miss_count: $(report_kv_get "perm.miss_count")"

  # 7) 现场数据备份
  section "现场数据备份（可选）"
  read -r -p "是否执行现场数据备份? (y/n) [默认:n]: " bak
  if [[ "${bak:-n}" =~ ^[Yy]$ ]]; then
    if data_backup "$platform" "$TS"; then
      log SUCCESS "[BAK] 现场数据备份完成：$(report_kv_get "bak.archive")"
    else
      log WARN "[BAK] 现场数据备份失败/部分失败（详见日志）"
    fi
    report_add ""
    report_add "## 现场数据备份"
    report_add "- status: $(report_kv_get "bak.status")"
    report_add "- dir: $(report_kv_get "bak.dir")"
    report_add "- archive: $(report_kv_get "bak.archive")"
  else
    log INFO "跳过现场数据备份"
    report_kv_set "bak.status" "SKIP"
    report_kv_set "bak.dir" "N/A"
    report_kv_set "bak.archive" "N/A"
    report_add ""
    report_add "## 现场数据备份"
    report_add "- 用户选择跳过"
  fi

  # 8) 日志导出
  section "日志导出（可选）"
  read -r -p "是否导出服务日志到本地目录? (y/n) [默认:n]: " ex
  local export_dir=""
  if [[ "${ex:-n}" =~ ^[Yy]$ ]]; then
    export_dir="$(export_logs "$platform" "$HAS_UJAVA" "$HAS_UPYTHON" "${UJAVA_CONTAINER:-}" "${UPYTHON_CONTAINER:-}" || true)"
    report_add ""
    report_add "## 日志导出"
    report_add "- 导出目录: $export_dir"
  else
    log INFO "跳过日志导出"
    report_add ""
    report_add "## 日志导出"
    report_add "- 用户选择跳过"
  fi

  # 9) 容器信息 + Redis/Emqx
  section "容器与中间件"
  collect_container_info "$platform"
  # ✅ 补齐：Emqx 异常检测（你已经实现但 main 没调用）
  check_emqx_container_exception

  report_add ""
  report_add "## 容器与中间件"
  report_add "- docker: $(report_kv_get "docker.available")"
  report_add "- redis.log export: $(report_kv_get "redis.export")"
  report_add "- emqx.log export: $(report_kv_get "emqx.export")"
  report_add "- redis container exception: $(report_kv_get "redis.exception")"
  local redis_recheck
  redis_recheck="$(report_kv_get "redis.recheck")"
  [[ -n "$redis_recheck" ]] && report_add "- redis recheck: $redis_recheck"
  # ✅ 新增摘要：Emqx
  report_add "- emqx container exception: $(report_kv_get "emqx.exception")"
  local emqx_recheck
  emqx_recheck="$(report_kv_get "emqx.recheck")"
  [[ -n "$emqx_recheck" ]] && report_add "- emqx recheck: $emqx_recheck"

  # 10) Android
  section "Android 自检（可选）"
  android_self_check

  report_add ""
  report_add "## Android 自检"
  report_add "- status: $(report_kv_get "android.status")"
  local at
  at="$(report_kv_get "android.target")"
  [[ -n "$at" ]] && report_add "- target: $at"
  local adir
  adir="$(report_kv_get "android.export_dir")"
  [[ -n "$adir" ]] && report_add "- export_dir: $adir"

  # 10.5) 定时任务查询（✅ 补齐，放在报告生成前）
  test_scheduled_tasks "$platform"
  # （test_scheduled_tasks 内部已 section + report_add 摘要，这里不重复）

  # 11) 生成报告
  section "生成 Markdown 报告"
  local report_file="$REPORT_DIR/health_report_${server_ip//[^0-9A-Za-z\.\-_]/_}_${TS}.md"
  write_report "$server_ip" "$platform" "$sys_kv" "$report_file"
  log SUCCESS "Markdown 报告已生成: $report_file"
  echo "[REPORT_FILE] $report_file"
}

# ✅ 确保文件末尾仍然调用 main
main "$@"