#!/usr/bin/env bash
set -euo pipefail

# -------------------- Version --------------------
# 版本号写死在脚本内，便于日志追溯（需要时手工递增）
SCRIPT_VERSION="1.0.0"
# -------------------- /Version --------------------

log() { printf '[%s] %s\n' "$(date '+%F %T')" "$*" >&2; }

usage() {
  cat <<'EOF'
program_update.sh (PRD 3.3 Backup)

Usage:
  ./program_update.sh --platform "新统一平台|传统平台" \
                      --system "会议预定系统|运维集控系统|讯飞转录系统" \
                      --update "前端更新|后端更新|全量更新" \
                      [--workdir /home/Update/]

Behavior (3.3):
  - mkdir -p /home/Backup/Bak<timestamp>
  - backup frontend/backend main files based on PRD mapping
  - backup database by finding mysql container and running mysqldump
  - tar backup folder to /home/Backup/Bak<timestamp>.tar.gz
  - print BACKUP_TAR path for caller to download
EOF
}

print_version() {
  echo "program_update.sh version: ${SCRIPT_VERSION}"
}

PLATFORM=""
SYSTEM=""
UPDATE_TYPE=""
WORKDIR="/home/Update"
RESTART_ONLY=0

while [[ $# -gt 0 ]]; do
  case "$1" in
    --platform) PLATFORM="${2:-}"; shift 2;;
    --system) SYSTEM="${2:-}"; shift 2;;
    --update) UPDATE_TYPE="${2:-}"; shift 2;;
    --workdir) WORKDIR="${2:-}"; shift 2;;
    --restart-only) RESTART_ONLY=1; shift 1;;
    --version) print_version; exit 0;;
    -h|--help) usage; exit 0;;
    *) log "Unknown arg: $1"; usage; exit 2;;
  esac
done

if [[ -z "$PLATFORM" || -z "$SYSTEM" || -z "$UPDATE_TYPE" ]]; then
  usage
  exit 2
fi

# 启动即记录版本与调用参数（方便截图/回溯）
log "program_update.sh version=$SCRIPT_VERSION"
log "Args: platform=$PLATFORM system=$SYSTEM update=$UPDATE_TYPE workdir=$WORKDIR restart_only=$RESTART_ONLY"

# 仅在非 RESTART_ONLY 时初始化备份目录（避免 restart-only 还创建备份目录）
if [[ "$RESTART_ONLY" -eq 0 ]]; then
  TS="$(date '+%Y%m%d_%H%M%S')"
  BACKUP_ROOT="/home/Backup"
  BACKUP_DIR="${BACKUP_ROOT}/Bak${TS}"
  BACKUP_TAR="${BACKUP_ROOT}/Bak${TS}.tar.gz"

  mkdir -p "$BACKUP_DIR"
  log "Backup dir: $BACKUP_DIR"
fi

# PRD 指定数据库密码（固定值）
MYSQL_ROOT_PASSWORD='dNrprU&2S'

# glob 没匹配到时展开为空，避免把 "*.jar" 当成字面量
shopt -s nullglob

# 只备份关键文件/目录（支持 glob）
# 用法：backup_globs "label" "/source/base" "/backup/dest" "glob1" "glob2" ...
backup_globs() {
  local label="$1"; shift
  local base="$1"; shift
  local dest="$1"; shift

  mkdir -p "$dest"

  if [[ ! -d "$base" ]]; then
    log "WARN: base not found, skip: $label ($base)"
    return 0
  fi

  log "Backup [$label]: base=$base -> $dest"

  local matched=0
  local g p
  for g in "$@"; do
    for p in "$base"/$g; do
      matched=1
      cp -a "$p" "$dest/"
    done
  done

  if [[ "$matched" -eq 0 ]]; then
    log "WARN: no files matched for [$label] under $base"
  fi
}

# 传统平台映射（按 PRD 4.1：仅备份关键文件）
DB_NAME=""

# 为了适配“会议预定系统前端两套、后端两套”，这里最多给两组规则
FRONT_1_BASE=""; FRONT_1_GLOBS=()
FRONT_2_BASE=""; FRONT_2_GLOBS=()
BACK_1_BASE="";  BACK_1_GLOBS=()
BACK_2_BASE="";  BACK_2_GLOBS=()

case "$PLATFORM" in
  "传统平台")
    case "$SYSTEM" in
      "会议预定系统")
        DB_NAME="ubains"

        # 前台前端：/var/www/java/ubains-web-2.0  *.js static/ index.html
        FRONT_1_BASE="/var/www/java/ubains-web-2.0"
        FRONT_1_GLOBS=("index.html" "static" "*.js")

        # 后台前端：/var/www/java/ubains-web-admin  index.html static/
        FRONT_2_BASE="/var/www/java/ubains-web-admin"
        FRONT_2_GLOBS=("index.html" "static")

        # 对内后端：/var/www/java/api-java-meeting2.0  *.jar（只匹配以 .jar 结尾，避免 jar2/jarbak/jarBak）
        BACK_1_BASE="/var/www/java/api-java-meeting2.0"
        BACK_1_GLOBS=("*.jar")

        # 对外后端：/var/www/java/external-meeting-api *.jar
        BACK_2_BASE="/var/www/java/external-meeting-api"
        BACK_2_GLOBS=("*.jar")
        ;;
      "运维集控系统")
        DB_NAME="devops"

        # 前端：/var/www/html/web-vue-rms  static/ index.html
        FRONT_1_BASE="/var/www/html/web-vue-rms"
        FRONT_1_GLOBS=("index.html" "static")

        # 后端：/var/www/html  cmdb/ UbainsDevOps/
        BACK_1_BASE="/var/www/html"
        BACK_1_GLOBS=("cmdb" "UbainsDevOps")
        ;;
      "讯飞转录系统")
        DB_NAME="devops"

        # 前端：/var/www/html/web-vue-voice  static/ index.html
        FRONT_1_BASE="/var/www/html/web-vue-voice"
        FRONT_1_GLOBS=("index.html" "static")

        # 后端：/var/www/html  UbainsDevOps/
        BACK_1_BASE="/var/www/html"
        BACK_1_GLOBS=("UbainsDevOps")
        ;;
      *)
        log "Unsupported system: $SYSTEM"
        exit 3
        ;;
    esac
    ;;
  "新统一平台")
    log "ERROR: 新统一平台的路径映射未在 PRD 中提供，暂无法执行备份。请补充 4.2 映射后再启用。"
    exit 3
    ;;
  *)
    log "Unsupported platform: $PLATFORM"
    exit 3
    ;;
esac

# -------------------- workdir 自动探测（适配解压多一层 update/） --------------------
resolve_workdir() {
  local base="${1%/}"

  # 约定：检查这些候选目录（base 本身优先，其次常见子目录）
  local candidates=(
    "$base"
    "$base/update"
    "$base/Update"
    "$base/package"
    "$base/dist"
  )

  # 根据系统类型给出“应该存在的源目录名集合”，用于判定候选目录是否正确
  local expected=()
  case "$SYSTEM" in
    "会议预定系统")
      expected=("ubains-web-2.0" "ubains-web-admin" "api-java-meeting2.0" "external-meeting-api")
      ;;
    "运维集控系统")
      expected=("web-vue-rms" "cmdb" "UbainsDevOps")
      ;;
    "讯飞转录系统")
      expected=("web-vue-voice" "UbainsDevOps")
      ;;
    *)
      expected=()
      ;;
  esac

  local c hit
  for c in "${candidates[@]}"; do
    [[ -d "$c" ]] || continue

    # 如果没有 expected（未知系统），就直接返回 base
    if [[ ${#expected[@]} -eq 0 ]]; then
      echo "$c"
      return 0
    fi

    # 命中任意一个关键目录就认为是正确 workdir（更宽松，避免必须全部存在）
    hit=0
    for e in "${expected[@]}"; do
      if [[ -d "$c/$e" ]]; then
        hit=1
        break
      fi
    done

    if [[ "$hit" -eq 1 ]]; then
      echo "$c"
      return 0
    fi
  done

  # 找不到就退回原始 base
  echo "$base"
}

if [[ "$RESTART_ONLY" -eq 0 ]]; then
  WORKDIR="$(resolve_workdir "$WORKDIR")"
  log "WORKDIR_RESOLVED=$WORKDIR"

# 3.3 备份前端/后端（按更新类型）
if [[ "$UPDATE_TYPE" == "前端更新" || "$UPDATE_TYPE" == "全量更新" ]]; then
  mkdir -p "$BACKUP_DIR/frontend"

  if [[ -n "$FRONT_1_BASE" ]]; then
    backup_globs "frontend-1" "$FRONT_1_BASE" "$BACKUP_DIR/frontend/$(basename "$FRONT_1_BASE")" "${FRONT_1_GLOBS[@]}"
  fi
  if [[ -n "$FRONT_2_BASE" ]]; then
    backup_globs "frontend-2" "$FRONT_2_BASE" "$BACKUP_DIR/frontend/$(basename "$FRONT_2_BASE")" "${FRONT_2_GLOBS[@]}"
  fi
fi

if [[ "$UPDATE_TYPE" == "后端更新" || "$UPDATE_TYPE" == "全量更新" ]]; then
  mkdir -p "$BACKUP_DIR/backend"

  if [[ -n "$BACK_1_BASE" ]]; then
    backup_globs "backend-1" "$BACK_1_BASE" "$BACKUP_DIR/backend/$(basename "$BACK_1_BASE")" "${BACK_1_GLOBS[@]}"
  fi
  if [[ -n "$BACK_2_BASE" ]]; then
    backup_globs "backend-2" "$BACK_2_BASE" "$BACKUP_DIR/backend/$(basename "$BACK_2_BASE")" "${BACK_2_GLOBS[@]}"
  fi
fi

# 备份数据库（按 PRD：查 MySQL 容器名 -> 进入容器 -> mysqldump）
backup_db() {
  local db="$1"
  [[ -z "$db" ]] && return 0

  if ! command -v docker >/dev/null 2>&1; then
    log "WARN: docker not found, skip DB backup."
    return 0
  fi

  local mysql_container=""
  mysql_container="$(docker ps --format '{{.Names}}' | grep -Eiw 'mysql|mariadb' | head -n 1 || true)"
  if [[ -z "$mysql_container" ]]; then
    mysql_container="$(docker ps --format '{{.Names}} {{.Image}}' | grep -Ei 'mysql|mariadb' | awk '{print $1}' | head -n 1 || true)"
  fi

  if [[ -z "$mysql_container" ]]; then
    log "WARN: MySQL container not found, skip DB backup."
    return 0
  fi

  mkdir -p "$BACKUP_DIR/db"
  local out="${BACKUP_DIR}/db/${db}_${TS}.sql"

  log "DB backup: container=$mysql_container db=$db -> $out"
  docker exec "$mysql_container" sh -lc "mysqldump -uroot -p'${MYSQL_ROOT_PASSWORD}' '${db}'" > "$out"
}

backup_db "$DB_NAME"

else
  log "RESTART_ONLY=1 -> skip WORKDIR resolve/backup/db/update"
fi
# -------------------- 3.4 更新服务（传统平台） --------------------
# 精准覆盖：仅把 sourceDir 中匹配的文件/目录覆盖到 targetDir（不删除 targetDir 里其它内容）
# 用法：sync_overwrite_globs "/src" "/dst" "label" "glob1" "glob2" ...
sync_overwrite_globs() {
  local sourceDir="$1"; shift
  local targetDir="$1"; shift
  local label="$1"; shift

  if [[ ! -d "$sourceDir" ]]; then
    log "ERROR: 更新源目录不存在：$sourceDir ($label)"
    return 10
  fi
  mkdir -p "$targetDir"

  log "UPDATE (globs) [$label]: $sourceDir -> $targetDir ; items=($*)"

  local matched=0
  local g p
  shopt -s dotglob nullglob
  for g in "$@"; do
    for p in "$sourceDir"/$g; do
      matched=1
      # 覆盖同名项：先删再拷（避免 static 目录冲突）
      rm -rf "$targetDir/$(basename "$p")"
      cp -a "$p" "$targetDir/"
    done
  done
  shopt -u dotglob nullglob

  if [[ "$matched" -eq 0 ]]; then
    log "ERROR: 未匹配到任何可更新内容：$sourceDir ($label)"
    return 11
  fi
}

# 前端安全覆盖（只更新主文件；强制保留目标目录 Bak*/bak*/new）
# 规则：
# - 覆盖：index.html、根目录 *.js
# - static：只同步 static/ 内部内容（不删除 static 目录本身）
# - 保护：目标目录下 Bak*/bak*/new 在更新过程中会被临时移出并恢复（兜底防误删）
sync_frontend_preserve() {
  local sourceDir="$1"
  local targetDir="$2"
  local label="$3"

  if [[ ! -d "$sourceDir" ]]; then
    log "ERROR: 更新源目录不存在：$sourceDir ($label)"
    return 10
  fi
  mkdir -p "$targetDir"

  log "UPDATE (frontend-preserve) [$label]: $sourceDir -> $targetDir (preserve: Bak*/bak*/new)"

  # ---- 0) 兜底保护：把目标目录的 Bak*/bak*/new 临时移走，避免被任何覆盖/清空逻辑误删 ----
  local preserveRoot="/tmp/program_update_preserve_$(date +%Y%m%d_%H%M%S)_$$"
  mkdir -p "$preserveRoot"

  local movedAny=0
  shopt -s nullglob
  local d bn
  for d in "$targetDir"/Bak* "$targetDir"/bak* "$targetDir"/new; do
    if [[ -d "$d" ]]; then
      bn="$(basename "$d")"
      log "PRESERVE: move out $d -> $preserveRoot/$bn"
      rm -rf "$preserveRoot/$bn" 2>/dev/null || true
      mv "$d" "$preserveRoot/$bn"
      movedAny=1
    fi
  done
  shopt -u nullglob

  # ---- 1) 覆盖 index.html（如果存在）----
  if [[ -f "$sourceDir/index.html" ]]; then
    cp -a "$sourceDir/index.html" "$targetDir/index.html"
  fi

  # ---- 2) 覆盖根目录 *.js（如果存在）----
  shopt -s nullglob
  local f
  for f in "$sourceDir"/*.js; do
    cp -a "$f" "$targetDir/"
  done
  shopt -u nullglob

  # ---- 3) 同步 static（只同步内容，不 rm -rf static）----
  if [[ -d "$sourceDir/static" ]]; then
    mkdir -p "$targetDir/static"

    # 优先 rsync（不使用 --delete；排除 Bak/bak/new）
    if command -v rsync >/dev/null 2>&1; then
      rsync -a \
        --exclude 'Bak*/' --exclude 'bak*/' --exclude 'new/' \
        "${sourceDir%/}/static/" "${targetDir%/}/static/"
    else
      shopt -s dotglob nullglob
      for f in "$sourceDir/static/"*; do
        # 源里如果也带 Bak/bak/new 目录，跳过
        if [[ -d "$f" ]]; then
          case "$(basename "$f")" in
            Bak*|bak*|new) continue ;;
          esac
        fi
        rm -rf "$targetDir/static/$(basename "$f")"
        cp -a "$f" "$targetDir/static/"
      done
      shopt -u dotglob nullglob
    fi
  fi

  # ---- 4) 恢复被保护的目录 ----
  if [[ "$movedAny" -eq 1 ]]; then
    shopt -s nullglob
    for d in "$preserveRoot"/*; do
      bn="$(basename "$d")"
      # 如果更新包里恰好也有同名目录（不应该），优先保留原来的备份目录
      if [[ -e "$targetDir/$bn" ]]; then
        log "PRESERVE: target already has $bn, keep preserved copy at $preserveRoot/$bn"
        continue
      fi
      log "PRESERVE: restore $bn -> $targetDir/$bn"
      mv "$d" "$targetDir/$bn"
    done
    shopt -u nullglob

    # 清理空目录
    rmdir "$preserveRoot" 2>/dev/null || true
  fi
}

# 后端目录安全覆盖（适用于 cmdb / UbainsDevOps 这类“目录型后端”）
# - 覆盖 sourceDir 内的内容到 targetDir
# - 不清空 targetDir（不使用 rsync --delete）
# - 保护：不影响 targetDir 下 Bak*/bak*/new
sync_backend_dir_preserve() {
  local sourceDir="$1"
  local targetDir="$2"
  local label="$3"

  if [[ ! -d "$sourceDir" ]]; then
    log "ERROR: 更新源目录不存在：$sourceDir ($label)"
    return 10
  fi
  mkdir -p "$targetDir"

  if [[ -z "$(ls -A "$sourceDir" 2>/dev/null || true)" ]]; then
    log "ERROR: 更新源目录为空：$sourceDir ($label)"
    return 11
  fi

  log "UPDATE (backend-dir-preserve) [$label]: $sourceDir -> $targetDir (preserve: Bak*/bak*/new)"

  # 1) 优先 rsync（不 delete；排除 Bak/bak/new）
  if command -v rsync >/dev/null 2>&1; then
    rsync -a \
      --exclude 'Bak*/' --exclude 'bak*/' --exclude 'new/' \
      "${sourceDir%/}/" "${targetDir%/}/"
    return 0
  fi

  # 2) 无 rsync：逐项覆盖复制（禁止删除 Bak/bak/new）
  local item name
  shopt -s dotglob nullglob
  for item in "$sourceDir"/*; do
    name="$(basename "$item")"
    case "$name" in
      Bak*|bak*|new) continue ;;
    esac

    # 仅删除将要覆盖的同名项（且上面已排除 Bak/bak/new）
    if [[ -e "$targetDir/$name" ]]; then
      rm -rf "$targetDir/$name"
    fi
    cp -a "$item" "$targetDir/"
  done
  shopt -u dotglob nullglob
}


# 宿主机对外后端启动/重启逻辑（external-meeting-api）
# - 不依赖 run.sh 可执行位（用 sh 解释执行；run.sh 为 #!/bin/sh）
# - 执行后等待最多 30 秒做一次“进程检测”（基于 jar 名关键字）
restart_external_meeting_api_host() {
  local app_dir="/var/www/java/external-meeting-api"
  local run_sh="${app_dir}/run.sh"

  log "Restart external-meeting-api (host):"
  log "  cd $app_dir"
  log "  sh ./run.sh"

  if [[ ! -d "$app_dir" ]]; then
    log "WARN: external app dir not found: $app_dir"
    return 0
  fi
  if [[ ! -f "$run_sh" ]]; then
    log "WARN: external run.sh not found: $run_sh"
    return 0
  fi

  (
    set +e
    cd "$app_dir" || exit 1
    sh ./run.sh
    ec=$?
    log "External sh ./run.sh exitCode=$ec"
    exit 0
  ) || true

  # --- verify (best-effort) ---
  local i
  for i in {1..30}; do
    if ps -ef | grep -F 'ubains-meeting-api-1.0-SNAPSHOT.jar' | grep -v grep >/dev/null; then
      log "External service process detected."
      return 0
    fi
    sleep 1
  done

  log "WARN: External service process NOT detected after 30s."
  log "WARN: ps snapshot:"
  ps -ef | grep -E 'ubains-meeting-api|external-meeting-api|java' | grep -v grep | head -n 20 || true
}

# ========== 3.5 重启服务并验证更新是否成功 ==========
restart_services_traditional() {
  case "$SYSTEM" in
    "会议预定系统")
      if [[ "$UPDATE_TYPE" == "前端更新" ]]; then
        log "Skip restart (frontend update only)."
        return 0
      fi

      log "Restart services: 会议预定系统"

      # ========== 1) 对外后端：宿主机执行 run.sh ==========
      # 对外后端（宿主机）重启
      restart_external_meeting_api_host

      # ========== 2) 对内后端：优先进入 java 容器内执行 run.sh ==========
      log "PRD 3.5 internal (preferred in java container ujava*):"
      log "  docker exec -it ujavaX bash"
      log "  cd /var/www/java/api-java-meeting2.0"
      log "  bash ./run.sh; tail -f logs/ubains-INFO-AND-ERROR.log"

      local java_container=""
      if command -v docker >/dev/null 2>&1; then
        java_container="$(docker ps --format '{{.Names}}' | grep -E '^ujava[0-9]*$' | head -n 1 || true)"
        if [[ -z "$java_container" ]]; then
          java_container="$(docker ps -a --format '{{.Names}}' | grep -E '^ujava[0-9]*$' | head -n 1 || true)"
        fi

        if [[ -n "$java_container" ]]; then
          log "Detected java container: $java_container"
          log "Restart (in container $java_container): /var/www/java/api-java-meeting2.0/run.sh (force bash)"
          docker exec "$java_container" bash -lc "cd /var/www/java/api-java-meeting2.0 && bash ./run.sh" || true

          log "Verify process (in container $java_container): ps -ef | grep api-java-meeting2.0"
          docker exec "$java_container" bash -lc "ps -ef | grep -E 'api-java-meeting2\.0|meeting' | grep -v grep | head -n 5" || true
        else
          log "WARN: java container (ujava*) not found, fallback to host restart"
          if [[ -d "/var/www/java/api-java-meeting2.0" && -f "/var/www/java/api-java-meeting2.0/run.sh" ]]; then
            log "Restart (host fallback): /var/www/java/api-java-meeting2.0/run.sh (force bash)"
            ( cd /var/www/java/api-java-meeting2.0 && bash ./run.sh ) || true
          else
            log "WARN: api-java-meeting2.0 run.sh not found (host fallback)"
          fi
        fi
      else
        log "WARN: docker not found, fallback to host restart"
        if [[ -d "/var/www/java/api-java-meeting2.0" && -f "/var/www/java/api-java-meeting2.0/run.sh" ]]; then
          log "Restart (host fallback): /var/www/java/api-java-meeting2.0/run.sh (force bash)"
          ( cd /var/www/java/api-java-meeting2.0 && bash ./run.sh ) || true
        else
          log "WARN: api-java-meeting2.0 run.sh not found (host fallback)"
        fi
      fi

      # ========== 3) 验证日志 ==========
      if [[ -f "/var/www/java/external-meeting-api/logs/ubains-INFO-AND-ERROR.log" ]]; then
        log "Verify log (external): tail -n 50 /var/www/java/external-meeting-api/logs/ubains-INFO-AND-ERROR.log"
        tail -n 50 /var/www/java/external-meeting-api/logs/ubains-INFO-AND-ERROR.log || true
      else
        log "WARN: external log file not found: /var/www/java/external-meeting-api/logs/ubains-INFO-AND-ERROR.log"
      fi

      if [[ -f "/var/www/java/api-java-meeting2.0/logs/ubains-INFO-AND-ERROR.log" ]]; then
        log "Verify log (internal): tail -n 50 /var/www/java/api-java-meeting2.0/logs/ubains-INFO-AND-ERROR.log"
        tail -n 50 /var/www/java/api-java-meeting2.0/logs/ubains-INFO-AND-ERROR.log || true
      else
        log "WARN: internal log file not found: /var/www/java/api-java-meeting2.0/logs/ubains-INFO-AND-ERROR.log"
      fi
      ;;

    "运维集控系统"|"讯飞转录系统")
      # PRD 3.5：后端更新需要重启 upython；前端更新无需重启
      if [[ "$UPDATE_TYPE" == "前端更新" ]]; then
        log "Skip restart (frontend update only)."
        return 0
      fi

      log "Restart services: $SYSTEM -> docker restart upython"
      if command -v docker >/dev/null 2>&1; then
        if docker ps -a --format '{{.Names}}' | grep -qx 'upython'; then
          docker restart upython
        else
          log "WARN: container upython not found"
        fi
      else
        log "WARN: docker not found, cannot restart upython"
      fi

      # 验证日志（仅提示，不阻塞）
      if [[ -f "/var/www/html/log/uinfo.log" ]]; then
        log "Verify log: tail -n 50 /var/www/html/log/uinfo.log"
        tail -n 50 /var/www/html/log/uinfo.log || true
      else
        log "WARN: log file not found: /var/www/html/log/uinfo.log"
      fi
      ;;
    *)
      log "WARN: 未配置 $SYSTEM 的重启逻辑，跳过重启。"
      ;;
  esac
}

# 在 update 之后调用 3.5
run_post_update_actions() {
  case "$PLATFORM" in
    "传统平台")
      restart_services_traditional
      ;;
    *)
      log "WARN: 平台 $PLATFORM 暂未实现 3.5 重启逻辑"
      ;;
  esac
}

# ====== 调试模式：只跑 3.5（放在函数定义之后）======
if [[ "$RESTART_ONLY" -eq 1 ]]; then
  log "RESTART_ONLY=1 -> skip backup/update/tar; only run PRD 3.5 restart"
  run_post_update_actions
  exit 0
fi


# 依据系统+更新类型执行
do_update_traditional() {
  case "$SYSTEM" in
    "会议预定系统")
      local src_front1="${WORKDIR%/}/ubains-web-2.0"
      local src_front2="${WORKDIR%/}/ubains-web-admin"
      local src_back1="${WORKDIR%/}/api-java-meeting2.0"
      local src_back2="${WORKDIR%/}/external-meeting-api"

      local dst_front1="/var/www/java/ubains-web-2.0"
      local dst_front2="/var/www/java/ubains-web-admin"
      local dst_back1="/var/www/java/api-java-meeting2.0"
      local dst_back2="/var/www/java/external-meeting-api"

      if [[ "$UPDATE_TYPE" == "前端更新" || "$UPDATE_TYPE" == "全量更新" ]]; then
        sync_frontend_preserve "$src_front1" "$dst_front1" "会议预定-前台前端"
        sync_frontend_preserve "$src_front2" "$dst_front2" "会议预定-后台前端"
      fi

      if [[ "$UPDATE_TYPE" == "后端更新" || "$UPDATE_TYPE" == "全量更新" ]]; then
        # 只更新 *.jar（严格只匹配以 .jar 结尾，避免 jar2/jarbak/jarBak）
        sync_overwrite_globs "$src_back1" "$dst_back1" "会议预定-对内后端" "*.jar"
        sync_overwrite_globs "$src_back2" "$dst_back2" "会议预定-对外后端" "*.jar"
      fi
      ;;

    "运维集控系统")
      local src_front="${WORKDIR%/}/web-vue-rms"
      local src_back_cmdb="${WORKDIR%/}/cmdb"
      local src_back_devops="${WORKDIR%/}/UbainsDevOps"

      local dst_front="/var/www/html/web-vue-rms"
      local dst_back_root="/var/www/html"

      if [[ "$UPDATE_TYPE" == "前端更新" || "$UPDATE_TYPE" == "全量更新" ]]; then
        # 只覆盖主文件：index.html + static/（不清空目录；保留 Bak*/bak*/new）
        sync_frontend_preserve "$src_front" "$dst_front" "运维集控-前端"
      fi

      if [[ "$UPDATE_TYPE" == "后端更新" || "$UPDATE_TYPE" == "全量更新" ]]; then
        # 目录型后端：覆盖更新包中同名项，不清空 targetDir；保留 Bak*/bak*/new
        sync_backend_dir_preserve "$src_back_cmdb"   "${dst_back_root%/}/cmdb"          "运维集控-后端-cmdb"
        sync_backend_dir_preserve "$src_back_devops" "${dst_back_root%/}/UbainsDevOps" "运维集控-后端-UbainsDevOps"
      fi
      ;;

    "讯飞转录系统")
      # 前端目标：/var/www/html/uvoice/web-vue-uvoice/
      # 后端目标：/var/www/html
      # 后端主服务文件夹：UbainsDevOps
      local src_front="${WORKDIR%/}/web-vue-uvoice"
      local src_back_devops="${WORKDIR%/}/UbainsDevOps"

      local dst_front="/var/www/html/uvoice/web-vue-uvoice"
      local dst_back_devops="/var/www/html/UbainsDevOps"

      if [[ "$UPDATE_TYPE" == "前端更新" || "$UPDATE_TYPE" == "全量更新" ]]; then
        # 前端只覆盖主文件（index.html / *.js / static），不动 Bak*/bak*/new
        sync_frontend_preserve "$src_front" "$dst_front" "讯飞转录-前端"
      fi

      if [[ "$UPDATE_TYPE" == "后端更新" || "$UPDATE_TYPE" == "全量更新" ]]; then
        # 后端目录型覆盖：目标在 /var/www/html/UbainsDevOps；不清空、不 delete；排除 Bak*/bak*/new
        sync_backend_dir_preserve "$src_back_devops" "$dst_back_devops" "讯飞转录-后端-UbainsDevOps"
      fi
      ;;

    *)
      log "ERROR: 不支持的系统类型（传统平台）：$SYSTEM"
      return 20
      ;;
  esac
}

case "$PLATFORM" in
  "传统平台")
    log "Start update (传统平台): system=$SYSTEM update=$UPDATE_TYPE workdir=$WORKDIR"
    do_update_traditional
    log "Update done (传统平台)."

    # 3.5 重启并验证
    run_post_update_actions
    ;;
  "新统一平台")
    log "ERROR: 新统一平台 3.4 更新规则 PRD 未提供（目前仅实现传统平台）。"
    exit 30
    ;;
  *)
    log "ERROR: Unsupported platform: $PLATFORM"
    exit 31
    ;;
esac

# 打包备份目录（仅非 restart-only）
if [[ "$RESTART_ONLY" -eq 0 ]]; then
  log "Packing backup to: $BACKUP_TAR"
  tar -czvf "$BACKUP_TAR" "$BACKUP_DIR" >/dev/null

  log "Backup done."
  echo "BACKUP_DIR=$BACKUP_DIR"
  echo "BACKUP_TAR=$BACKUP_TAR"
fi