import datetime
import logging
import queue
import threading
import time

import schedule

from Base.base import run_automation_test


# ==============================
# 日志配置（建议你按需改级别/输出位置）
# ==============================
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(message)s",
)


# ==============================
# 任务队列 + 工作线程
# ==============================
task_queue: "queue.Queue[tuple]" = queue.Queue()


def run_task(task, *args, **kwargs) -> None:
    """把任务放进队列，由 worker 异步执行"""
    task_queue.put((task, args, kwargs))
    logging.debug("任务已加入队列: %s args=%s kwargs=%s", getattr(task, "__name__", str(task)), args, kwargs)


def worker() -> None:
    """工作线程：不断从队列取任务执行"""
    while True:
        task, args, kwargs = task_queue.get()
        try:
            logging.info("开始执行任务: %s", getattr(task, "__name__", str(task)))
            result = task(*args, **kwargs)
            if result:
                logging.info("任务返回: %s", result)
        except Exception as e:
            logging.error("执行任务时发生错误: %s", e, exc_info=True)
        finally:
            task_queue.task_done()
            logging.debug("任务完成: %s", getattr(task, "__name__", str(task)))


def start_workers(num_workers: int) -> None:
    for _ in range(num_workers):
        threading.Thread(target=worker, daemon=True).start()


# ==============================
# 时间段控制（禁用窗口）#
# ==============================
def is_in_time_range(now: datetime.time, start: datetime.time, end: datetime.time) -> bool:
    """判断 now 是否在 [start, end) 区间内（不跨天场景）"""
    return start <= now < end


def run_task_if_allowed(task, *args, block_ranges=None, **kwargs) -> None:
    """
    在指定时间段内跳过任务（不入队）

    block_ranges: [(start_time, end_time), ...]
      e.g. [(time(7,0), time(9,0))]
    """
    now_time = datetime.datetime.now().time()
    if block_ranges:
        for start_t, end_t in block_ranges:
            if is_in_time_range(now_time, start_t, end_t):
                logging.info(
                    "任务已跳过(禁用时段 %s~%s): %s",
                    start_t.strftime("%H:%M"),
                    end_t.strftime("%H:%M"),
                    getattr(task, "__name__", str(task)),
                )
                return
    run_task(task, *args, **kwargs)


# ==============================
# Schedule 配置
# ==============================
REPORT_URL_PREFIX = "http://nat.ubainsyun.com:31136"


def setup_schedules() -> None:
    # 1) 每天 10:00 执行“预定系统测试”
    schedule.every().day.at("10:00").do(
        run_task,
        run_automation_test,
        report_title="预定系统测试报告",
        report_url_prefix=REPORT_URL_PREFIX,
        test_case="预定系统测试",
        ding_type="标准版巡检",
    )

    # 2) 每半小时执行一次“预定快速测试”（07:00~09:00 不执行）
    schedule.every(30).minutes.do(
        run_task_if_allowed,
        run_automation_test,
        report_title="预定系统快速测试报告",
        report_url_prefix=REPORT_URL_PREFIX,
        test_case="预定系统快速测试",
        ding_type="标准版巡检",
        block_ranges=[(datetime.time(7, 0), datetime.time(9, 0))],
    )

    # 3) 展厅巡检（工作日 07:42）
    for weekday in ("monday", "tuesday", "wednesday", "thursday", "friday"):
        getattr(schedule.every(), weekday).at("07:42").do(
            run_task,
            run_automation_test,
            report_title="展厅巡检测试报告",
            report_url_prefix=REPORT_URL_PREFIX,
            test_case="展厅巡检",
            ding_type="展厅巡检",
        )


def main() -> None:
    # 启动工作线程
    start_workers(3)

    # 注册定时任务
    setup_schedules()

    logging.info("Scheduler started. Running pending jobs...")
    try:
        while True:
            schedule.run_pending()
            time.sleep(1)
    except KeyboardInterrupt:
        logging.info("Scheduler interrupted by user.")
    except Exception as e:
        logging.error("Unexpected error: %s", e, exc_info=True)


if __name__ == "__main__":
    main()