#!/bin/bash

# 宿主机上的脚本：检查 EMQX，如果容器未运行则重启 uemqx 容器
LOG_FILE="/var/log/monitor_emqx_service.log"

log() {
    local message
    message="$(date '+%Y-%m-%d %H:%M:%S') - $1"
    echo "$message"
    # 同时写入日志文件
    echo "$message" >> "$LOG_FILE"
}

check_emqx() {
    # 检查 uemqx 容器是否正在运行
    if docker ps --format '{{.Names}}' | grep -Fxq "uemqx"; then
        # 通过 docker exec 检查 EMQX 进程是否在容器内运行
        if docker exec uemqx pgrep -f "emqx" >/dev/null 2>&1; then
            # 检查EMQX是否在运行状态（使用EMQX自带的命令）
            if docker exec uemqx emqx_ctl status >/dev/null 2>&1; then
                # 额外验证EMQX是否真正可用
                status_output=$(docker exec uemqx emqx_ctl status 2>&1)
                if echo "$status_output" | grep -q "is started"; then
                    log "EMQX 服务状态正常"
                    return 0
                else
                    log "警告: emqx_ctl 命令执行成功，但返回意外状态"
                    return 1
                fi
            else
                log "警告: 容器正在运行，EMQX 进程存在，但 emqx_ctl 命令执行失败"
                return 1
            fi
        else
            log "警告: 容器正在运行，但在容器内未找到 EMQX 进程"
            return 1
        fi
    else
        log "信息: uemqx 容器未运行"
        return 1
    fi
}

restart_emqx_container() {
    log "EMQX 未运行。正在尝试重启容器 'uemqx'..."

    # 检查容器是否存在（不仅仅是运行状态）
    if docker ps -a --format '{{.Names}}' | grep -Fxq "uemqx"; then
        # 停止可能挂起的容器
        docker stop uemqx >/dev/null 2>&1 || true
        
        # 容器存在但未运行，尝试启动
        if docker start uemqx; then
            log "成功: EMQX 容器已成功启动。"
            
            # 等待几秒让服务启动
            log "信息: 等待 30 秒让 EMQX 服务完全启动..."
            for i in {1..30}; do
                echo -n "." >> "$LOG_FILE"
                sleep 1
            done
            echo "" >> "$LOG_FILE"
            
            # 检查启动后容器是否仍在运行
            if docker ps --format '{{.Names}}' | grep -Fxq "uemqx"; then
                log "信息: EMQX 容器现在正在运行。"
                
                # 再次检查EMQX进程是否已启动
                if docker exec uemqx pgrep -f "emqx" >/dev/null 2>&1; then
                    log "信息: EMQX 进程在容器内正在运行。"
                    
                    # 等待一段时间后再次检查服务状态
                    log "信息: 等待 10 秒后检查服务状态..."
                    sleep 10
                    
                    if docker exec uemqx emqx_ctl status >/dev/null 2>&1; then
                        status_output=$(docker exec uemqx emqx_ctl status 2>&1)
                        if echo "$status_output" | grep -q "is started"; then
                            log "信息: EMQX 服务状态正常。"
                            return 0
                        else
                            log "错误: EMQX 进程运行中，但服务状态异常。"
                            return 1
                        fi
                    else
                        log "错误: EMQX 进程运行中，但服务状态检查失败。"
                        return 1
                    fi
                else
                    log "错误: EMQX 进程在容器内未运行。"
                    return 1
                fi
            else
                log "错误: EMQX 容器在启动后不久就停止了。"
                return 1
            fi
        else
            log "错误: 无法启动 EMQX 容器。"
            return 1
        fi
    else
        log "错误: 容器 'uemqx' 不存在！"
        log "信息: 您可能需要手动重新创建 EMQX 容器。"
        return 1
    fi
}

# 主逻辑
log "开始检查 EMQX 服务状态..."
if check_emqx; then
    log "EMQX 正在运行且状态正常。"
else
    log "EMQX 无响应或容器未运行。"
    restart_emqx_container
    
    # 检查重启后是否正常工作
    if check_emqx; then
        log "EMQX 已成功重启，现在状态正常。"
    else
        log "错误: 重启尝试后 EMQX 仍无响应。"
        log "信息: 请使用以下命令检查容器日志: docker logs uemqx"
    fi
fi