#!/bin/bash

# ==================== 配置区 ====================
CONTAINER_NAME="umysql"
DB_USER="root"
HOST_BACKUP_DIR="/opt/mysql"   # 宿主机备份目录
LOG_FILE="/var/log/backup_mysql_databases.log"
RETENTION_DAYS=30

TARGET_DBS=("devops" "devops_voice" "huazhao2" "nacos_mysql" "offline" "ubains" "wifi" "voice" "ubains_nacos_config" "ubains_sso")

DATE=$(date +"%Y%m%d")
CONTAINER_TMP_DIR="/tmp/mysql_backup_$$.sql.gz"  # 容器内临时目录（带PID防冲突）

# ==================== 日志函数 ====================
log() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" >> "$LOG_FILE"
}

get_mysql_password() {
    docker exec "$CONTAINER_NAME" printenv MYSQL_ROOT_PASSWORD 2>/dev/null
}

# ==================== 初始化 ====================
mkdir -p "$HOST_BACKUP_DIR/$DATE"
touch "$LOG_FILE"
log "===== 开始备份任务 (容器: $CONTAINER_NAME) ====="

# 在容器内创建临时目录
docker exec "$CONTAINER_NAME" mkdir -p "$CONTAINER_TMP_DIR"
if [ $? -ne 0 ]; then
    log "❌ 无法在容器内创建临时目录，请检查容器是否运行。"
    exit 1
fi

DB_PASSWORD=$(get_mysql_password)

if [ -z "$DB_PASSWORD" ]; then
    docker exec "$CONTAINER_NAME" rm -rf "$CONTAINER_TMP_DIR"
    exit 1
fi

# 测试连接
if docker exec "$CONTAINER_NAME" mysql -u"$DB_USER" -p"$DB_PASSWORD" -Nse "SELECT 1;" >/dev/null 2>&1; then
    log "✅ 使用密码成功连接数据库"
else
    log "❌ 使用密码连接数据库失败"
    docker exec "$CONTAINER_NAME" rm -rf "$CONTAINER_TMP_DIR"
    exit 1
fi

# 获取容器中实际存在的数据库列表
EXISTING_DBS_RAW=$(docker exec "$CONTAINER_NAME" mysql -u"$DB_USER" -p"$DB_PASSWORD" -Nse "SHOW DATABASES;" 2>/dev/null)
mapfile -t EXISTING_DBS <<< "$EXISTING_DBS_RAW"

declare -A DB_EXISTS
for db in "${EXISTING_DBS[@]}"; do
    DB_EXISTS["$db"]=1
done

# ==================== 备份每个目标数据库 ====================
SUCCESS_COUNT=0
for db in "${TARGET_DBS[@]}"; do
    if [[ -n "${DB_EXISTS[$db]}" ]]; then
        CONTAINER_DUMP_FILE="$CONTAINER_TMP_DIR/${db}.sql.gz"
        HOST_DUMP_FILE="$HOST_BACKUP_DIR/$DATE/${db}.sql.gz"

        log "开始备份数据库: $db"

        # 在容器内执行 mysqldump + gzip
        docker exec "$CONTAINER_NAME" \
            sh -c "mysqldump -u'$DB_USER' -p'$DB_PASSWORD' --single-transaction --routines --triggers --events '$db' | gzip > '$CONTAINER_DUMP_FILE'"

        if [ $? -eq 0 ] && docker exec "$CONTAINER_NAME" [ -s "$CONTAINER_DUMP_FILE" ]; then
            # 从容器复制到宿主机
            docker cp "$CONTAINER_NAME:$CONTAINER_DUMP_FILE" "$HOST_DUMP_FILE"
            if [ $? -eq 0 ] && [ -s "$HOST_DUMP_FILE" ]; then
                log "✅ 备份成功: $db -> $HOST_DUMP_FILE"
                ((SUCCESS_COUNT++))
            else
                log "❌ 复制到宿主机失败: $db"
                [ -f "$HOST_DUMP_FILE" ] && rm -f "$HOST_DUMP_FILE"
            fi
        else
            log "❌ 容器内备份失败: $db"
        fi

        # 立即删除容器内的临时文件（无论成功与否）
        docker exec "$CONTAINER_NAME" rm -f "$CONTAINER_DUMP_FILE"
    else
        log "⚠️ 跳过: 数据库 $db 不存在"
    fi
done

# 清理容器内临时目录
docker exec "$CONTAINER_NAME" rm -rf "$CONTAINER_TMP_DIR"

log "本次共成功备份 $SUCCESS_COUNT 个数据库"

# ==================== 清理超过30天的旧备份 ====================
log "开始清理超过 $RETENTION_DAYS 天的旧备份..."
find "$HOST_BACKUP_DIR" -maxdepth 1 -type d -name "????????" -mtime +$RETENTION_DAYS -exec rm -rf {} + 2>/dev/null

log "===== 备份任务结束 ====="
