#!/bin/bash

# ==================== 配置区域 ====================
# 可在此处配置要监控的容器名称，默认为ujava2
CONTAINER_NAME="${UJAVA_CONTAINER_NAME:-${1:-ujava2}}"
LOG_FILE="/var/log/monitor_ujava_service.log"

# ==================== 日志函数 ====================
log() {
    local message
    message="$(date '+%Y-%m-%d %H:%M:%S') - $1"
    echo "$message"
    # 同时写入日志文件
    echo "$message" >> "$LOG_FILE"
}

check_ujava() {
    # 检查指定的 ujava 容器是否正在运行
    if docker ps --format '{{.Names}}' | grep -Fxq "$CONTAINER_NAME"; then
        # 获取运行中的Java进程列表
        running_processes=$(docker exec "$CONTAINER_NAME" ps aux | grep -E "java.*\.jar|ubains" | grep -v grep)
        
        if [ -n "$running_processes" ]; then
            log "检测到容器中有Java进程在运行"
            log "运行中的Java进程: $(echo "$running_processes" | wc -l) 个"
            
            # 定义服务目录和JAR文件的对应关系
            declare -A service_map
            service_map["/var/www/java/auth-sso-auth"]="ubains-auth.jar"
            service_map["/var/www/java/auth-sso-gatway"]="ubains-gateway.jar"
            service_map["/var/www/java/auth-sso-system"]="ubains-modules-system.jar"
            service_map["/var/www/java/api-java-meeting2.0"]="ubains-meeting-inner-api-1.0-SNAPSHOT"
            service_map["/var/www/java/external-meeting-api"]="ubains-meeting-api-1.0-SNAPSHOT.jar"
            service_map["/var/www/java/api-dubbo-meeting-control"]="ubains-dubbo-meeting-control"
            service_map["/var/www/java/api-dubbo-smc-three"]="ubains-dubbo-smc-three-0.0.1"
            
            missing_services=()
            active_services=()
            
            # 检查每个服务是否在运行
            for dir in "${!service_map[@]}"; do
                jar_file="${service_map[$dir]}"
                if echo "$running_processes" | grep -q "$jar_file"; then
                    active_services+=("$jar_file")
                else
                    missing_services+=("$dir:$jar_file")
                fi
            done
            
            if [ ${#missing_services[@]} -eq 0 ]; then
                log "所有预期的Java进程都在运行中 (共 ${#active_services[@]} 个)"
                return 0
            else
                log "警告: 以下服务未运行: ${missing_services[*]}"
                log "信息: 以下服务正在运行: ${active_services[*]}"
                return 1
            fi
        else
            log "警告: 容器运行中，但未检测到Java进程"
            # 返回1表示服务不正常
            return 1
        fi
    else
        log "信息: 容器 $CONTAINER_NAME 未运行"
        return 1
    fi
}

restart_missing_services() {
    log "检测到服务缺失，正在尝试启动缺失的服务..."

    # 定义服务目录和JAR文件的对应关系
    declare -A service_map
    service_map["/var/www/java/auth-sso-auth"]="ubains-auth.jar"
    service_map["/var/www/java/auth-sso-gatway"]="ubains-gateway.jar"
    service_map["/var/www/java/auth-sso-system"]="ubains-modules-system.jar"
    service_map["/var/www/java/api-java-meeting2.0"]="ubains-meeting-inner-api-1.0-SNAPSHOT"
    service_map["/var/www/java/external-meeting-api"]="ubains-meeting-api-1.0-SNAPSHOT.jar"
    service_map["/var/www/java/api-dubbo-meeting-control"]="ubains-dubbo-meeting-control"
    service_map["/var/www/java/api-dubbo-smc-three"]="ubains-dubbo-smc-three-0.0.1"

    # 统计启动了多少服务
    started_count=0
    
    # 遍历每个服务，检查是否运行，如果没有则启动
    for dir in "${!service_map[@]}"; do
        jar_file="${service_map[$dir]}"
        
        # 检查服务是否已经运行
        running_processes=$(docker exec "$CONTAINER_NAME" ps aux | grep -E "java.*\.jar|ubains" | grep -v grep)
        
        if ! echo "$running_processes" | grep -q "$jar_file"; then
            log "启动服务: $jar_file 在目录 $dir"
            
            # 检查JAR文件是否存在
            if docker exec -w "$dir" "$CONTAINER_NAME" test -f "$jar_file\.jar"; then
                # 直接执行Java命令启动服务，而不是运行run.sh
                docker exec -d -w "$dir" "$CONTAINER_NAME" sh -c "nohup java -Xms1024m -Xmx1024m -jar -DAPP_CONFIG=./config $jar_file\.jar > log.out 2>&1 &"
                log "已尝试启动 $jar_file"
                ((started_count++))
            elif docker exec -w "$dir" "$CONTAINER_NAME" test -f "$jar_file-0.0.1-SNAPSHOT\.jar"; then
                # 对于带版本号的JAR文件
                docker exec -d -w "$dir" "$CONTAINER_NAME" sh -c "nohup java -Xms1024m -Xmx1024m -jar -DAPP_CONFIG=./config $jar_file-0.0.1-SNAPSHOT\.jar > log.out 2>&1 &"
                log "已尝试启动 $jar_file-0.0.1-SNAPSHOT.jar"
                ((started_count++))
            elif docker exec -w "$dir" "$CONTAINER_NAME" test -f "run.sh"; then
                # 如果JAR文件不存在但run.sh存在，安全地执行run.sh（传递参数以避免默认执行start）
                docker exec -d -w "$dir" "$CONTAINER_NAME" sh -c "./run.sh start"
                log "已尝试使用run.sh启动 $jar_file"
                ((started_count++))
            else
                log "错误: JAR文件和run.sh脚本在 $dir 目录中都不存在"
            fi
        else
            log "服务 $jar_file 已经在运行"
        fi
    done
    
    if [ $started_count -gt 0 ]; then
        log "成功启动了 $started_count 个服务，等待服务启动..."
        return 0
    else
        log "没有需要启动的服务"
        return 0
    fi
}

wait_for_services_to_start() {
    log "开始监控服务启动状态，最多等待10分钟..."
    
    # 循环检查服务是否启动成功，最多等待10次，每次等待1分钟
    for i in {1..10}; do
        log "等待第 $i 分钟，检查服务是否已启动..."
        
        # 检查是否还有缺失的服务
        running_processes=$(docker exec "$CONTAINER_NAME" ps aux | grep -E "java.*\.jar|ubains" | grep -v grep)
        
        missing_services=()
        # 定义服务目录和JAR文件的对应关系
        declare -A service_map
        service_map["/var/www/java/auth-sso-auth"]="ubains-auth.jar"
        service_map["/var/www/java/auth-sso-gatway"]="ubains-gateway.jar"
        service_map["/var/www/java/auth-sso-system"]="ubains-modules-system.jar"
        service_map["/var/www/java/api-java-meeting2.0"]="ubains-meeting-inner-api-1.0-SNAPSHOT"
        service_map["/var/www/java/external-meeting-api"]="ubains-meeting-api-1.0-SNAPSHOT.jar"
        service_map["/var/www/java/api-dubbo-meeting-control"]="ubains-dubbo-meeting-control"
        service_map["/var/www/java/api-dubbo-smc-three"]="ubains-dubbo-smc-three-0.0.1"
        
        # 检查每个服务是否在运行
        for dir in "${!service_map[@]}"; do
            jar_file="${service_map[$dir]}"
            if ! echo "$running_processes" | grep -q "$jar_file"; then
                missing_services+=("$dir:$jar_file")
            fi
        done
        
        if [ ${#missing_services[@]} -eq 0 ]; then
            log "恭喜！所有服务都已成功启动。"
            return 0
        else
            log "仍有 ${#missing_services[@]} 个服务未启动: ${missing_services[*]}"
        fi
        
        # 等待1分钟
        sleep 60
    done
    
    log "已等待10分钟，仍有服务未能启动，将重启整个容器。"
    return 1
}

restart_ujava_container() {
    log "容器 $CONTAINER_NAME 服务不正常。正在尝试重启容器 '$CONTAINER_NAME'..."

    # 检查容器是否存在（不仅仅是运行状态）
    if docker ps -a --format '{{.Names}}' | grep -Fxq "$CONTAINER_NAME"; then
        # 停止可能挂起的容器
        docker stop "$CONTAINER_NAME" >/dev/null 2>&1 || true
        
        # 容器存在但未运行，尝试启动
        if docker start "$CONTAINER_NAME"; then
            log "成功: 容器 $CONTAINER_NAME 已成功启动。"
            
            # 等待几分钟让Java服务启动
            log "信息: 等待 90 秒让 Java 服务完全启动..."
            for i in {1..90}; do
                echo -n "." >> "$LOG_FILE"
                sleep 1
            done
            echo "" >> "$LOG_FILE"
            
            # 检查启动后容器是否仍在运行
            if docker ps --format '{{.Names}}' | grep -Fxq "$CONTAINER_NAME"; then
                log "信息: 容器 $CONTAINER_NAME 现在正在运行。"
                
                # 启动缺失的服务
                restart_missing_services
                
                # 检查Java进程是否已启动
                log "信息: 检查服务状态..."
                
                if check_ujava; then
                    log "信息: 容器 $CONTAINER_NAME 服务状态正常。"
                    return 0
                else
                    log "错误: 容器 $CONTAINER_NAME 进程运行中，但服务状态异常。"
                    return 1
                fi
            else
                log "错误: 容器 $CONTAINER_NAME 在启动后不久就停止了。"
                return 1
            fi
        else
            log "错误: 无法启动容器 $CONTAINER_NAME。"
            return 1
        fi
    else
        log "错误: 容器 '$CONTAINER_NAME' 不存在！"
        log "信息: 您可能需要手动重新创建 $CONTAINER_NAME 容器。"
        return 1
    fi
}

# 主逻辑
log "开始检查容器 $CONTAINER_NAME 服务状态..."
if check_ujava; then
    log "容器 $CONTAINER_NAME 正在运行且状态正常。"
else
    log "容器 $CONTAINER_NAME 服务不完整或容器未运行。"
    
    # 如果容器在运行但服务不完整，尝试启动缺失的服务
    if docker ps --format '{{.Names}}' | grep -Fxq "$CONTAINER_NAME"; then
        # 先获取当前运行的服务状态
        running_processes=$(docker exec "$CONTAINER_NAME" ps aux | grep -E "java.*\.jar|ubains" | grep -v grep)
        
        # 如果没有任何Java进程在运行，可能是所有服务都崩溃了，直接重启容器
        if [ -z "$running_processes" ]; then
            log "警告: 检测到没有任何Java进程在运行，尝试重启整个容器。"
            restart_ujava_container
        else
            # 否则，尝试启动缺失的服务
            restart_missing_services
            
            # 检查重启后是否正常工作
            if check_ujava; then
                log "容器 $CONTAINER_NAME 服务已恢复，现在状态正常。"
            else
                log "容器 $CONTAINER_NAME 服务仍然不正常，尝试重启整个容器。"
                restart_ujava_container
            fi
        fi
    else
        # 如果容器不在运行，直接重启容器
        restart_ujava_container
    fi
    
    # 最终检查
    if check_ujava; then
        log "容器 $CONTAINER_NAME 已成功恢复，现在状态正常。"
    else
        log "错误: 尝试恢复后 $CONTAINER_NAME 仍无响应。"
        log "信息: 请使用以下命令检查容器日志: docker logs $CONTAINER_NAME"
    fi
fi