#!/bin/bash
#------------------------------工具类模块---------------------------------------------------
# 日志打印函数
LOG_FILE="/var/log/new_auto_script.log"
mkdir -p "$(dirname "$LOG_FILE")" 2>/dev/null
# 用户输入的东西也同步记录到日志信息里面
function log() {
    # 获取当前时间
    local timestamp=$(date "+%Y-%m-%d %H:%M:%S")
    # 日志级别对应的颜色代码
    local INFO_COLOR="\033[32m"      # 绿色
    local WARN_COLOR="\033[33m"      # 黄色
    local ERROR_COLOR="\033[31m"     # 红色
    local DEBUG_COLOR="\033[36m"     # 青色
    local RESET_COLOR="\033[0m"      # 重置颜色
    # 参数判断
    if [ $# -lt 2 ]; then
        echo "Usage: log <level> <message>"
        return 1
    fi
    local level=$1
    local message=$2
    # 根据日志级别选择颜色
    case $level in
        "INFO")
            color=$INFO_COLOR
            ;;
        "WARN")
            color=$WARN_COLOR
            ;;
        "ERROR")
            color=$ERROR_COLOR
            ;;
        "DEBUG")
            color=$DEBUG_COLOR
            ;;
        *)
            echo "Invalid log level: $level"
            return 1
            ;;
    esac
    # 输出带颜色的日志到屏幕
    echo -e "${color}[$timestamp] [$level] $message${RESET_COLOR}"
    # 追加纯文本日志到文件
    echo "[$timestamp] [$level] $message" >> "$LOG_FILE"
}



#------------------------------检测模块-----------------------------------------------------

#------------------------------服务器检测-start---------------------------------------------
function detectNetType() {
  log "INFO" "准备获取所有网口信息，排除 loopback 和虚拟接口"
  INTERFACES=$(ls /sys/class/net | grep -vE 'lo|docker|br-|veth|virbr|vmnet')
  # 遍历每个网口，检查是否配置了 IP 地址，并输出网口信息
  log "INFO" "检测到的网口信息如下："
for iface in $INTERFACES; do
    log "INFO" "-----------------------------"
    log "INFO" "接口: $iface"

    log "INFO" "  检查接口 $iface 是否配置了 IP 地址"
    IP_ADDR=$(ip addr show "$iface" | grep "inet " | awk '{print $2}')

    if [[ -n "$IP_ADDR" ]]; then
        log  "INFO" "  配置的 IP 地址: $IP_ADDR"
    else
        log  "INFO"  "  没有配置 IP 地址"
    fi

    # 获取接口的连接状态（carrier）
    log "INFO" "  检查接口 $iface 的连接状态"
    if [[ -f /sys/class/net/$iface/carrier ]]; then
        CARRIER=$(cat /sys/class/net/$iface/carrier)
        [[ "$CARRIER" == "1" ]] && LINK="connected" || LINK="disconnected"
    else
        LINK="unknown"
    fi
    log "INFO" "  状态: $LINK"

    # 使用 ethtool 获取接口的类型（光口、电口等）
    log "INFO" "  获取接口 $iface 的类型"
    ETHTOOL_OUTPUT=$(ethtool "$iface" 2>/dev/null)
    TYPE=$(echo "$ETHTOOL_OUTPUT" | grep -i "Port:" | awk '{print tolower($2)}')
    [[ -z "$TYPE" ]] && TYPE="unknown" && TYPE="other"

    # 获取速率和驱动信息
    log "INFO" "  获取接口 $iface 的速率和驱动信息"
    SPEED=$(echo "$ETHTOOL_OUTPUT" | grep -i "Speed:" | awk '{print $2}')
    [[ -z "$SPEED" ]] && SPEED="unknown"
    DRIVER=$(ethtool -i "$iface" 2>/dev/null | grep "driver:" | awk '{print $2}')
    [[ -z "$DRIVER" ]] && DRIVER="unknown"

    # 获取 MAC 地址
    log "INFO" "  获取接口 $iface 的 MAC 地址"
    MAC=$(cat /sys/class/net/$iface/address)

#    转换 TYPE 为中文描述
    case "$TYPE" in
        "copper")
            TYPE="电口"
            ;;
        "fiber")
            TYPE="光口"
            ;;
        "other")
            TYPE="其他类型"
            ;;
        *)
            TYPE="未知类型"
            ;;
    esac
    # 输出接口信息
    log "INFO" "  光电口类型: ${TYPE:-unknown}"
    log "INFO" "  速率: $SPEED"
    log "INFO"  "  MAC 地址: $MAC"
    log "INFO" "  驱动: $DRIVER"
done
}


function checkOS() {
log "INFO" "开始检测服务器环境"
log "INFO" "**************《请检查当前系统是否适配》**************************************************"
# 检测当前的操作系统是否为x86架构
log "INFO" "开始检查当前操作系统架构"
if [ "$(uname -m)" != "x86_64" ]; then
    log "ERROR" "当前操作系统不是x86架构，请使用x86架构的操作系统"
    exit 1
else
    log "INFO" "当前操作系统为x86架构，继续执行脚本"
fi


server_available_mem=`free -g |awk 'NR==2' | awk '{print$7}'`
server_max_mem=`free -g|awk 'NR==2' |awk '{print$2}'`

log "INFO" "当前系统最大内存为:${server_max_mem}GB 可用内存为:${server_available_mem}GB"

#判断服务器的最大内存值小于约定的内存大小min_memory_value=15，就返回。否则成功。
log "INFO" "开始检查服务器最大内存"
if [ $server_max_mem -lt $min_memory_value ];then
  log "WARN" "当前系统最大内存为:${server_max_mem}G,建议将内存至少添加为: ${min_memory_value}G"
    #  需要用户确认信息，支持默认空格和y或Y，并日志记录用户输入情况
    read -p "是否继续执行脚本？(y/n): " yn
    if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
#        记录用户如情况
        log "INFO" "用户选择继续执行脚本,输入为${yn} "
    else
        log "ERROR" "脚本执行终止，请调整服务器内存后重新运行脚本"
        exit 1
    fi
else
  log "INFO" "检查当前系统最大内存为:${server_max_mem}G 满足系统运行！"
fi
#todo:考虑把交换关闭掉

#判断服务器的内存值小于约定的内存大小min_memory_available=10，就返回。否则成功。
log "INFO" "开始检查服务器可用内存"
if [ $server_available_mem -lt $min_memory_available ];then
      log  "WARN"  "当前系统docker可用内存为:${server_available_mem}G,建议将内存至少添加为： ${min_memory_available}G ！"
      #  需要用户确认信息，支持默认空格和y或Y，并日志记录用户输入情况
      read -p "是否继续执行脚本？(y/n): " yn
      if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
      #   记录用户如情况
          log "INFO" "用户选择继续执行脚本,输入为${yn} "
      else
          log "ERROR" "脚本执行终止，请调整服务器内存后重新运行脚本"
          exit 1
      fi
else
      log  "INFO"  "检查当前系统docker可用内存为:${server_available_mem}G 满足系统运行!"
fi

#判断服务的根分区的硬盘大小是否超过预设的硬盘资源。成功继续，否则退出。
log "INFO" "开始检查服务器硬盘空间"
server_disk_available=`df -B G /|awk '/\//{print$4}' | awk '{sub(/.{1}$/,"")}1'`

if [ $server_disk_available -lt $min_disk_available ];then
    log "WARN"  "当前系统的硬盘空间为:${server_disk_available}G,建议将硬盘空间至少添加为: ${min_disk_available}G!"
        #  需要用户确认信息，支持默认空格和y或Y，并日志记录用户输入情况
        read -p "是否继续执行脚本？(y/n): " yn
        if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
        #   记录用户如情况
            log "INFO" "用户选择继续执行脚本,输入为${yn} "
        else
            log "ERROR" "脚本执行终止，请调整服务器磁盘后重新运行脚本"
            exit 1
        fi
else
    log "INFO"  "检查当前系统硬盘空间为:${server_disk_available}G,满足系统运行!"
fi

# 检测当前用户是否有root权限
log "INFO" "开始检查当前用户权限"
if [ "$(id -u)" -ne 0 ]; then
    log "ERROR" "当前用户没有root权限，请使用root用户执行脚本，获者使用sudo命令执行脚本"
    exit 1
else
    log "INFO" "当前用户有root权限，继续执行脚本"
fi
log "INFO" "当前用户权限检查完成"

# 检测当前服务器是否存在多张网卡并且ipv4不唯一的需要列出来
log "INFO" "开始检查服务器网卡信息"
network_interfaces=$(ip -o -4 addr show | awk '{print $2}' | sort -u)
if [ -z "$network_interfaces" ]; then
    log "ERROR" "当前服务器没有检测到任何网卡，请检查网络配置"
    exit 1
else
    log "INFO" "当前服务器检测到的网卡信息如下：$network_interfaces"
#    获取对应的IP信息
    for interface in $network_interfaces; do
        ip_address=$(ip -o -4 addr show "$interface" | awk '{print $4}' | cut -d/ -f1)
        if [ -n "$ip_address" ]; then
            log "INFO" "网卡 $interface 的 IP 地址为: $ip_address"
            IP=$ip_address
        else
            log "WARN" "网卡 $interface 没有配置 IPv4 地址"
        fi
    done
fi
log "INFO" "服务器网卡信息检查完成"

log "INFO" "开始检测服务器网口类型"
detectNetType
log "INFO" "服务器网口类型检测完成"
# 用户确认以后继续往下
read -p "请确认网口信息是否正确,特别注意需要仔细观察你的项目是要光口还是电口 ，输入 y 或 Y 确认，输入 n 或 N 退出脚本: " yn
if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
    log "INFO" "用户确认网口信息正确，继续执行脚本"
else
    log "ERROR" "用户确认网口信息不正确，脚本执行终止"
    exit 1
fi


  
echo -e "\033[36m 
*******************************************************
*                  可适配的操作系统(x86)              *
*                                                     *
*                     cento7.5                        *                  
*                     cento7.6                        *
*                     cento7.9                        *
*                     统信系统                        *
*                     鲲鹏系统                        *
*                     Ubuntu                          *
*                     redhat7                         *
*                     银河麒麟V10                     *
*                                                     *
*******************************************************
         \033[0m"

echo -e "\033[36m\n检测到当前操作系统如下，请确认是否适配：\n         \033[0m"
# 优先获取PRETTY_NAME，没有则拼接NAME和VERSION
log "INFO" "开始获取当前操作系统版本信息"
if grep -q '^PRETTY_NAME=' /etc/os-release; then
    os_version=$(grep '^PRETTY_NAME=' /etc/os-release | cut -d= -f2 | tr -d '"')
else
    name=$(grep '^NAME=' /etc/os-release | cut -d= -f2 | tr -d '"')
    version=$(grep '^VERSION=' /etc/os-release | cut -d= -f2 | tr -d '"')
    os_version="$name $version"
fi
log "INFO" "当前操作系统版本为: $os_version"

case "$os_version" in
    *"CentOS Linux 7"*|*"UOS"*|*"Kylin V10"*|*"Ubuntu"*|*"Red Hat Enterprise Linux Server 7"*)
        log "INFO" "当前操作系统版本适配，继续执行脚本" ;;
    *)
        log "ERROR" "当前操作系统版本不适配，请使用CentOS 7、UOS、麒麟V10、Ubuntu或Red Hat Enterprise Linux Server 7"
        #  需要用户确认信息，支持默认空格和y或Y，并日志记录用户输入情况
        read -p "是否继续执行脚本？(y/n): " yn
        if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
        #   记录用户如情况
            log "INFO" "用户选择继续执行脚本,输入为${yn} "
        else
            log "ERROR" "脚本执行终止，请调整服务器操作系统后重新运行脚本"
            exit 1
        fi;;
esac
log "INFO" "当前操作系统版本适配，继续执行脚本"

#检查服务器时间和时区
log "INFO" "开始检查服务器时间和时区"
log "INFO" "**************《请检查当前时间，时区是否正确》*****************************************************************"
log "INFO" "当前服务器时区、时间为: $(date)"

read -p "确认无误请按 y 或 Y 或直接回车，不正确请按 n 修改时间:     " yn
if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
    log "INFO" "下面进入系统部署。。。"
    sleep 3
else
    read -p "输入当前日期，格式如2021/02/06——" da
    echo "="
    date -s $da
    read -p "输入当前时间，格式如18:00:00——" sa
    echo "="
    date -s $sa
    timedatectl set-timezone Asia/Shanghai
    clock -w
    echo "==================================================="
    echo "查看修改后的时间"
    date
fi
}
log "INFO" "服务器检测完成"

#------------------------------服务器检测-end------------------------------------------------------------------------------------------------------------------------
#todo: 对文件的检测每次部署的时候根据功能，这里检测的文件是要不一样的，这个先记录，不着急实现
#------------------------------部署文件检测-start--------------------------------------------------------------------------------------------------------------------
#==============================================================#
# 函数名: File_detection_java
# 功能:   检测新统一平台系统所需部署目录是否完整存在
# 说明:   本函数用于在部署前校验关键服务目录是否存在
#         若任一目录缺失，将记录错误并终止脚本
#         支持未来根据不同部署模式动态调整检测列表
# 作者:   [ubains]
# 日期:   $(date +%Y-%m-%d)
#--------------------------------------------------------------#
# 参数:   无
# 返回:   0: 所有文件检测通过
#        1: 存在缺失文件，终止执行
#--------------------------------------------------------------#
# 日志级别:
#   INFO: 正常流程
#   ERROR: 错误信息
#==============================================================#
function File_detection_java() {
    log "INFO" "=================================================================="
    log "INFO" "开始检测新统一平台系统部署文件是否完整"
    log "INFO" "=================================================================="

    # 定义待检测的服务模块及其相对路径（可根据部署模式动态调整）
    # 格式: "模块名称:路径"
    local services=(
        "预定系统-api/java-meeting2.0:/data/services/api/java-meeting/java-meeting2.0"
        "预定系统-extapi:/data/services/api/java-meeting/java-meeting-extapi"
        "消息调度:/data/services/api/java-meeting/java-message-scheduling"
        "MQTT消息:/data/services/api/java-meeting/java-mqtt"
        "Quartz定时任务:/data/services/api/java-meeting/java-quartz"
        "运维集控服务:/data/services/api/python-cmdb"
        "语音转录服务:/data/services/api/python-voice"
        "SSO认证服务:/data/services/api/auth/auth-sso-auth"
        "SSO网关服务:/data/services/api/auth/auth-sso-gatway"
        "SSO系统管理:/data/services/api/auth/auth-sso-system"
        "Dubbo-华为云:/data/services/api/dubbo/dubbo-cloudLink"
        "Dubbo-钉钉:/data/services/api/dubbo/dubbo-dingding"
        "Dubbo-会议控制:/data/services/api/dubbo/dubbo-meeting-control"
        "Dubbo-宝利通:/data/services/api/dubbo/dubbo-polycom"
        "Dubbo-服务调用:/data/services/api/dubbo/dubbo-serviceCall"
        "Dubbo-SMC3.0:/data/services/api/dubbo/dubbo-smc-three"
        "Dubbo-SMC2.0:/data/services/api/dubbo/dubbo-smc-two"
        "Dubbo-Teams:/data/services/api/dubbo/dubbo-teams"
        "Dubbo-腾讯会议:/data/services/api/dubbo/dubbo-tencent-meeting"
        "Dubbo-小鱼易连:/data/services/api/dubbo/dubbo-xylink"
        "前端-微应用:/data/services/web/pc/pc-vue2-main"
        "前端-AI助手:/data/services/web/pc/pc-vue2-ai"
        "前端-统一后台:/data/services/web/pc/pc-vue2-backstage"
        "前端-编辑器:/data/services/web/pc/pc-vue2-editor"
        "前端-预定:/data/services/web/pc/pc-vue2-meetngV2"
        "前端-统一平台:/data/services/web/pc/pc-vue2-meetngV3"
        "前端-门户:/data/services/web/pc/pc-vue2-platform"
        "前端-运维集控:/data/services/web/pc/pc-vue2-moniter"
        "前端-语音转录:/data/services/web/pc/pc-vue2-voice"
        "前端-预定移动端:/data/services/web/h5/h5-uniapp-meeting"
        "前端-运维集控移动端:/data/services/web/h5/h5-uniapp-moniter"
        "前端-统一平台移动端:/data/services/web/h5/h5-uniapp-platform"
        "全局脚本:/data/services/scripts"
        "备份目录:/data/bakup"
        "容器目录:/data/dockers"
        "日志目录:/data/logs"
        "中间件目录:/data/middleware"
        "安全认证目录:/data/security"
        "文件存储目录:/data/storage"
        "第三方配套件目录:/data/third_party"
    )

    # 统计总数和失败数
    local total=${#services[@]}
    local failed=0
    local missing_dirs=()

    log "INFO" "共检测 $total 个服务模块的部署目录..."

    # 遍历每个服务进行目录检测
    for service in "${services[@]}"; do
        # 分割模块名和路径
        local module_name="${service%%:*}"
        local relative_path="${service#*:}"
        local full_path="$auto_java/$relative_path"

        log "INFO" "正在检测 [$module_name] => $full_path"

        if [[ -d "$full_path" ]]; then
            log "INFO" "✅ [$module_name] 目录校验通过"
        else
            log "ERROR" "❌ [$module_name] 目录不存在: $full_path"
            missing_dirs+=("[$module_name] $full_path")
            ((failed++))
        fi
        sleep 0.3  # 可选：避免日志刷屏过快
    done

    # 输出最终检测结果
    if [[ $failed -eq 0 ]]; then
        log "INFO" "🎉 部署文件检测完成：所有 $total 个目录均存在，检测成功！"
        return 0
    else
        log "ERROR" "⛔ 部署文件检测失败：共 $failed 个目录缺失"
        for dir in "${missing_dirs[@]}"; do
            log "ERROR" "   缺失: $dir"
        done
        log "ERROR" "请重新上传缺失文件后，再执行部署脚本。"
        exit 1  # 终止脚本
    fi
}
#------------------------------部署文件检测-end--------------------------------------------------------------------------------------------------------------------


#创建用户(预定安装需要:该用户与系统启动用户有关)
function add_user()
{
log "INFO" "创建用户user"
user=user
group=user

log "INFO" "判断是否存在用户$user"
egrep "^$user" /etc/passwd >& /dev/null
if [ $? -ne 0 ]; then
    log "INFO" "用户$user不存在，开始创建用户"
    username="user"
    password="Ubains@135"
    useradd -m "$username"
    echo "$username:$password" | chpasswd
    log "INFO" "用户$user创建成功"
else
    log "INFO" "用户user已创建! "
fi
}


#------------------------------开放防火墙端口-start------------------------------------------------------------------------------------------------------------------------

#--------开放服务端口---------------------
#==============================================================#
# 函数名: firewalldjava
# 功能:   开放 Java 服务所需防火墙端口（TCP/UDP）
# 说明:   自动检测 firewalld 状态，若开启则检查并开放以下端口：
#           - TCP: 22(SSH), 443(HTTPS), 8848/9848(Nacos), 1883(MQTT), 8999(应用)
#           - UDP: 123(NTP), 323(Telegraf/监控) —— 用于门口屏等设备
#         使用 --permanent 持久化规则，并在最后重新加载防火墙。
# 作者:   [ubains]
# 日期:   $(date +%Y-%m-%d)
#--------------------------------------------------------------#
# 参数:   无
# 返回:   0: 成功（或 firewalld 未运行）
#        1: firewalld 运行但操作失败
#--------------------------------------------------------------#
# 依赖:   firewall-cmd, sudo（若使用 sudoset）
# 注意:   确保 $sudoset 已定义（如: sudoset="sudo"）
#==============================================================#
function firewalldjava() {
    log "INFO" "=================================================================="
    log "INFO" "开始配置防火墙：开放服务所需端口"
    log "INFO" "=================================================================="

    # 定义 TCP 和 UDP 需要开放的端口
    local tcp_ports=(22 443 8848 9848 1883 8883)
    local udp_ports=(123 323)

    # 检查 firewalld 是否正在运行
    log "INFO" "正在检查 firewalld 服务状态..."
    if ! $sudoset firewall-cmd --state >/dev/null 2>&1; then
        log "WARN" "firewalld 未运行或未安装，跳过端口配置。"
        log "INFO" "如需启用，请运行: sudo systemctl start firewalld && sudo systemctl enable firewalld"
        return 0
    fi

    log "INFO" "✅ firewalld 已启用，开始配置端口..."

    local failed=0
    local changes_made=0

    # ------------------- 开放 TCP 端口 -------------------
    log "INFO" "🔍 检查并配置 TCP 端口: ${tcp_ports[*]}"
    for port in "${tcp_ports[@]}"; do
        if $sudoset firewall-cmd --zone=public --query-port="$port"/tcp >/dev/null 2>&1; then
            log "INFO" "✅ TCP/$port 已开放，跳过"
        else
            log "INFO" "🔧 正在开放 TCP/$port ..."
            if $sudoset firewall-cmd --zone=public --add-port="$port"/tcp --permanent >/dev/null 2>&1; then
                log "INFO" "🎉 TCP/$port 开放成功（持久化）"
                ((changes_made++))
            else
                log "ERROR" "⛔ TCP/$port 开放失败，请手动检查"
                ((failed++))
            fi
        fi
    done

    # ------------------- 开放 UDP 端口 -------------------
    log "INFO" "🔍 检查并配置 UDP 端口: ${udp_ports[*]}"
    for port in "${udp_ports[@]}"; do
        if $sudoset firewall-cmd --zone=public --query-port="$port"/udp >/dev/null 2>&1; then
            log "INFO" "✅ UDP/$port 已开放，跳过"
        else
            log "INFO" "🔧 正在开放 UDP/$port ..."
            if $sudoset firewall-cmd --zone=public --add-port="$port"/udp --permanent >/dev/null 2>&1; then
                log "INFO" "🎉 UDP/$port 开放成功（持久化）"
                ((changes_made++))
            else
                log "ERROR" "⛔ UDP/$port 开放失败，请手动检查"
                ((failed++))
            fi
        fi
    done

    # ------------------- 重新加载防火墙 -------------------
    if [[ $changes_made -gt 0 ]]; then
        log "INFO" "🔄 检测到配置变更，正在重新加载 firewalld ..."
        if $sudoset firewall-cmd --reload >/dev/null 2>&1; then
            log "INFO" "✅ 防火墙重载成功！所有端口已生效。"
        else
            log "ERROR" "⛔ 防火墙重载失败，请手动执行: firewall-cmd --reload"
            return 1
        fi
    else
        log "INFO" "🟢 所有端口均已开放，无需重新加载。"
    fi

    # ------------------- 输出当前开放端口 -------------------
    log "INFO" "📋 当前已开放的端口列表:"
    $sudoset firewall-cmd --list-port 2>/dev/null || log "WARN" "无法获取端口列表"

    # ------------------- 最终状态 -------------------
    if [[ $failed -eq 0 ]]; then
        log "INFO" "✅ 防火墙配置完成：所有端口检查通过。"
        return 0
    else
        log "ERROR" "❌ 防火墙配置失败：共 $failed 个端口未能开放。"
        return 1
    fi
}
#------------------------------开放防火墙端口-end------------------------------------------------------------------------------------------------------------------------


#------------------------------服务安装-start------------------------------------------------------------------------------------------------------------------------

#==============================================================#
# 函数名: docker_x86
# 功能:   检查并安装 Docker（x86_64 离线安装）
# 说明:   用于在无网络或受限环境中自动部署 Docker 引擎
#         若 Docker 已安装则跳过；否则从本地 tar 包离线安装
#         安装内容包括：
#           - 解压 docker 二进制文件到 /usr/bin
#           - 配置 systemd 服务文件
#           - 启动并设置开机自启
# 作者:   [ubains]
# 日期:   $(date +%Y-%m-%d)
#--------------------------------------------------------------#
# 参数:   无
# 返回:   0: 成功（已安装或跳过）
#        1: 安装失败
#--------------------------------------------------------------#
# 依赖:   tar, systemctl, sudo（通过 $sudoset）
# 注意:   请确保以下资源已提前上传：
#           - /data/temp/docker-20.10.7.tgz
#           - /data/temp/docker.service
#==============================================================#
function docker_x86() {
    log "INFO" "=================================================================="
    log "INFO" "开始检查并安装 Docker (x86_64 离线模式)"
    log "INFO" "=================================================================="

    # ------------------- 定义变量 -------------------
    local docker_version="29.1.3"
    local temp_dir="/data/temp"
    local tar_file="${temp_dir}/docker-${docker_version}.tgz"
    local service_file="${temp_dir}/docker.service"
    local docker_bin="/usr/bin/docker"

    # ------------------- 检查 Docker 是否已安装 -------------------
    log "INFO" "🔍 正在检查 Docker 是否已安装..."
    if command -v docker >/dev/null 2>&1; then
        local version=$(docker --version 2>&1)
        log "INFO" "✅ Docker 已安装: $version"
        log "INFO" "💡 若需重装，请先手动卸载。"
        return 0
    else
        log "WARN" "❌ Docker 未安装，准备从离线包安装..."
    fi

    # ------------------- 检查依赖和文件是否存在 -------------------
    log "INFO" "🔧 检查安装依赖和文件..."

    if [[ ! -f "$tar_file" ]]; then
        log "ERROR" "⛔ Docker 离线包不存在: $tar_file"
        log "ERROR" "请将 docker-${docker_version}.tgz 上传至 $temp_dir 目录"
        return 1
    fi

    if [[ ! -f "$service_file" ]]; then
        log "ERROR" "⛔ Docker systemd 服务文件不存在: $service_file"
        log "ERROR" "请将 docker.service 配置文件上传至 $temp_dir 目录"
        return 1
    fi

    # ------------------- 开始安装 Docker -------------------
    log "INFO" "📥 开始解压 Docker ${docker_version} ..."

    if $sudoset tar -xzf "$tar_file" -C "$temp_dir"; then
        log "INFO" "✅ 解压成功"
    else
        log "ERROR" "⛔ 解压失败，请检查压缩包完整性"
        return 1
    fi

    log "INFO" "📦 安装 Docker 二进制文件到 /usr/bin/ ..."
    if $sudoset cp "$temp_dir/docker"/* /usr/bin/ && $sudoset chmod +x /usr/bin/docker*; then
        log "INFO" "✅ Docker 核心命令安装完成"
    else
        log "ERROR" "⛔ Docker 二进制文件复制失败"
        return 1
    fi

    # ------------------- 配置 systemd 服务 -------------------
    log "INFO" "⚙️ 配置 Docker systemd 服务 ..."

    $sudoset rm -f /etc/systemd/system/docker.service
    if $sudoset cp "$service_file" /etc/systemd/system/docker.service; then
        $sudoset chmod 644 /etc/systemd/system/docker.service
        log "INFO" "✅ 服务文件已部署"
    else
        log "ERROR" "⛔ 服务文件复制失败"
        return 1
    fi

    # ------------------- 启动 Docker 服务 -------------------
    log "INFO" "🚀 正在启动 Docker 服务 ..."

    $sudoset systemctl daemon-reload

    if $sudoset systemctl start docker; then
        log "INFO" "✅ Docker 服务启动成功"
    else
        log "ERROR" "⛔ Docker 启动失败，请检查日志: journalctl -u docker"
        return 1
    fi

    if $sudoset systemctl enable docker.service; then
        log "INFO" "✅ Docker 已设置为开机自启"
    else
        log "WARN" "⚠️  设置开机自启失败，可能影响后续使用"
    fi

    # ------------------- 验证安装 -------------------
    log "INFO" "🔍 验证 Docker 安装结果..."
    if command -v docker >/dev/null 2>&1 && docker info >/dev/null 2>&1; then
        log "INFO" "🎉 Docker 安装并启动成功！版本信息:"
        docker --version
        return 0
    else
        log "ERROR" "⛔ Docker 安装后验证失败，请手动排查"
        return 1
    fi
}


#==============================================================#
# 函数名: mysql_x86
# 功能:   部署基于 Docker 的 MySQL 服务（x86 架构）
# 说明:   1. 检查本地是否已运行 umysql 容器
#         2. 若无，则从离线 tar 加载镜像并启动容器
#         3. 初始化数据库：创建用户、数据库、授权
#         4. 配置定时备份脚本（每天 13:00）
# 作者:   [ubains]
# 日期:   $(date +%Y-%m-%d)
#--------------------------------------------------------------#
# 参数:   无
# 返回:   0: 成功
#         1: 失败
#--------------------------------------------------------------#
# 依赖:   docker, crond (或 cronie), sudo
# 注意:   请确保以下资源已上传：
#           - /data/temp/umysql.tar.gz
#           - $PWD/UbainsmysqlBakUp.sh
#         并确保宿主机 /data/middleware/mysql/ 目录存在
#==============================================================#
function mysql_x86() {
    # --- 配置参数 ---
    local container_name="umysql"
    local image_tar="/data/temp/umysql.tar.gz"
    local image_name="139.9.60.86:5000/umysql:v5.2"
    local image_id="4c5574ba4b04"
    local mysql_root_password="dNrprU&2S"
    local mysql_port="8306"
    local host_data_dir="/data/middleware/mysql"
    local backup_script_src="$PWD/UbainsmysqlBakUp.sh"
    local backup_script_dst="/usr/local/docker/UbainsmysqlBakUp.sh"
    local cron_user="root"

    log "INFO" "=================================================================="
    log "INFO" "开始部署 MySQL 服务 (Docker 版, x86)"
    log "INFO" "=================================================================="

    # 自动判断是否需要 sudo
    local SUDO=""
    if [[ $(id -u) -ne 0 ]]; then
        if command -v sudo >/dev/null; then
            SUDO="sudo"
        else
            log "ERROR" "⛔ 当前不是 root 用户，且系统未安装 sudo，无法执行操作"
            return 1
        fi
    fi

    # ------------------- 检查容器是否已运行 -------------------
    log "INFO" "🔍 检查容器 '$container_name' 是否已运行..."
    if $SUDO docker ps --format '{{.Names}}' | grep -qw "$container_name"; then
        log "INFO" "✅ 容器 '$container_name' 已在运行"
        return 0
    fi

    # ------------------- 检查镜像是否存在 -------------------
    log "INFO" "🔍 检查镜像 $image_name 是否存在..."
    if $SUDO docker images --format '{{.Repository}}:{{.Tag}}' | grep -qw "$image_name"; then
        log "INFO" "✅ 镜像 $image_name 已存在"
    else
        log "WARN" "❌ 镜像 $image_name 不存在，开始加载离线包..."

        if [[ ! -f "$image_tar" ]]; then
            log "ERROR" "⛔ 镜像文件不存在: $image_tar"
            return 1
        fi

        if $SUDO docker load -i "$image_tar"; then
            log "INFO" "🎉 镜像加载成功"
        else
            log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
            return 1
        fi

    fi

    # ------------------- 启动容器 -------------------
    log "INFO" "🚀 正在启动 MySQL 容器: $container_name"
    $SUDO docker run -d \
        --name "$container_name" \
        --mac-address="02:42:ac:11:00:05" \
        -p "${mysql_port}:3306" \
        -v "${host_data_dir}/conf/my.cnf:/etc/mysql/my.cnf" \
        -v "${host_data_dir}/conf.d:/etc/mysql/conf.d" \
        -v "${host_data_dir}/log:/var/log/mysql" \
        -v "${host_data_dir}/data:/var/lib/mysql" \
        -v "/etc/localtime:/etc/localtime:ro" \
        --restart=always \
        -e "MYSQL_ROOT_PASSWORD=$mysql_root_password" \
        "$image_name"

    if [[ $? -ne 0 ]]; then
        log "ERROR" "⛔ 容器启动失败，请检查端口占用或磁盘权限"
        log "INFO" "📄 查看最近容器日志以排查问题..."
        $SUDO docker logs "$container_name" 2>/dev/null || true
        return 1
    fi
    
    sleep 120

    log "INFO" "✅ 容器启动成功，等待 MySQL 服务就绪..."
    
        # ------------------- 初始化数据库：创建用户、数据库、授权、导入数据 -------------------
    log "INFO" "🔧 初始化数据库：创建用户、数据库、授权..."

    # 先拷贝 nacos_mysql.sql 到容器内
    log "INFO" "📁 拷贝 /data/temp/nacos_mysql.sql 到容器 '$container_name' 内..."
    if $SUDO docker cp "/data/temp/nacos_mysql.sql" "$container_name:/tmp/nacos_mysql.sql"; then
        log "INFO" "✅ SQL 文件已成功拷贝到容器内"
    else
        log "ERROR" "⛔ 无法拷贝 nacos_mysql.sql 到容器，请检查文件路径或容器状态"
        return 1
    fi
    
    # 先拷贝 devops_voice.sql 到容器内
    log "INFO" "📁 拷贝 /data/temp/devops_voice.sql 到容器 '$container_name' 内..."
    if $SUDO docker cp "/data/temp/devops_voice.sql" "$container_name:/tmp/devops_voice.sql"; then
        log "INFO" "✅ SQL 文件已成功拷贝到容器内"
    else
        log "ERROR" "⛔ 无法拷贝 devops_voice.sql 到容器，请检查文件路径或容器状态"
        return 1
    fi

    # 先拷贝 huazhao2.sql 到容器内
    log "INFO" "📁 拷贝 /data/temp/huazhao2.sql 到容器 '$container_name' 内..."
    if $SUDO docker cp "/data/temp/huazhao2.sql" "$container_name:/tmp/huazhao2.sql"; then
        log "INFO" "✅ SQL 文件已成功拷贝到容器内"
    else
        log "ERROR" "⛔ 无法拷贝 huazhao2.sql 到容器，请检查文件路径或容器状态"
        return 1
    fi

    # 先拷贝 offline.sql 到容器内
    log "INFO" "📁 拷贝 /data/temp/offline.sql 到容器 '$container_name' 内..."
    if $SUDO docker cp "/data/temp/offline.sql" "$container_name:/tmp/offline.sql"; then
        log "INFO" "✅ SQL 文件已成功拷贝到容器内"
    else
        log "ERROR" "⛔ 无法拷贝 offline.sql 到容器，请检查文件路径或容器状态"
        return 1
    fi

    # 定义要执行的 SQL 命令序列（注意顺序）
    local init_sql=(
        "CREATE USER IF NOT EXISTS 'nacos'@'%' IDENTIFIED BY 'nacos2025';"
        "CREATE USER IF NOT EXISTS 'mysqluser'@'%' IDENTIFIED BY 'dNrprU&2S';"
        "CREATE DATABASE IF NOT EXISTS nacos_mysql CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
        "CREATE DATABASE IF NOT EXISTS devops CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
        "CREATE DATABASE IF NOT EXISTS devops_voice CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
        "CREATE DATABASE IF NOT EXISTS offline CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
        "CREATE DATABASE IF NOT EXISTS huazhao2 CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
        "GRANT ALL PRIVILEGES ON nacos_mysql.* TO 'nacos'@'%';"
        "GRANT ALL PRIVILEGES ON devops.* TO 'mysqluser'@'%';"
        "FLUSH PRIVILEGES;"
    )

    # 执行前半部分 SQL
    for sql in "${init_sql[@]}"; do
        $SUDO docker exec "$container_name" mysql -uroot -p"$mysql_root_password" -e "$sql"
        if [[ $? -ne 0 ]]; then
            log "ERROR" "⛔ 执行SQL语句失败: $sql"
            log "INFO" "📄 输出容器最近日志..."
            $SUDO docker logs "$container_name" | tail -30
            return 1
        fi
    done

    # 执行 source 导入 nacos_mysql.sql 到 nacos_mysql 数据库
    log "INFO" "📥 正在导入 nacos_mysql.sql 到数据库 nacos_mysql..."
    $SUDO docker exec "$container_name" mysql -uroot -p"$mysql_root_password" -D nacos_mysql -e "source /tmp/nacos_mysql.sql"

    if [[ $? -eq 0 ]]; then
        log "INFO" "✅ nacos_mysql.sql 数据导入成功"
        # 在导入数据后执行 IP 替换
        log "INFO" "🔄 正在更新配置中的 IP 地址..."
        $SUDO docker exec "$container_name" mysql -uroot -p"$mysql_root_password" -D nacos_mysql -e \
            "UPDATE config_info SET content = REPLACE(content, '192.168.5.234', '"$server_ip"') 
            WHERE data_id IN ('auth-sso-auth-dev.yml','auth-sso-gateway-dev.yml','auth-sso-system-dev.yml');"
        
        log "INFO" "🔄 正在保存配置..."
        $SUDO docker exec "$container_name" mysql -uroot -p"$mysql_root_password" -D nacos_mysql -e \
            "FLUSH PRIVILEGES;"

        if [[ $? -eq 0 ]]; then
            log "INFO" "✅ 配置保存成功"
        else
            log "ERROR" "⛔ 配置保存失败"
            return 1
        fi

    else
        log "ERROR" "⛔ 数据导入失败，请检查 nacos_mysql.sql 文件内容或权限"
        log "INFO" "📄 输出容器日志以排查问题..."
        $SUDO docker logs "$container_name" | tail -50
        return 1
    fi
    sleep 60

    # 执行 source 导入 devops_voice.sql 到 devops_voice 数据库
    log "INFO" "📥 正在导入 devops_voice.sql 到数据库 devops_voice..."
    $SUDO docker exec "$container_name" mysql -uroot -p"$mysql_root_password" -D devops_voice -e "source /tmp/devops_voice.sql"

    if [[ $? -eq 0 ]]; then
        log "INFO" "✅ devops_voice.sql 数据导入成功"
    else
        log "ERROR" "⛔ 数据导入失败，请检查 devops_voice.sql 文件内容或权限"
        log "INFO" "📄 输出容器日志以排查问题..."
        $SUDO docker logs "$container_name" | tail -50
        return 1
    fi
    sleep 60


    # 执行 source 导入 huazhao2.sql 到 huazhao2 数据库
    log "INFO" "📥 正在导入 huazhao2.sql 到数据库 huazhao2..."
    $SUDO docker exec "$container_name" mysql -uroot -p"$mysql_root_password" -D huazhao2 -e "source /tmp/huazhao2.sql"

    if [[ $? -eq 0 ]]; then
        log "INFO" "✅ huazhao2.sql 数据导入成功"
    else
        log "ERROR" "⛔ 数据导入失败，请检查 huazhao2.sql 文件内容或权限"
        log "INFO" "📄 输出容器日志以排查问题..."
        $SUDO docker logs "$container_name" | tail -50
        return 1
    fi
    sleep 60

    # 执行 source 导入 offline.sql 到 offline 数据库
    log "INFO" "📥 正在导入 offline.sql 到数据库 offline..."
    $SUDO docker exec "$container_name" mysql -uroot -p"$mysql_root_password" -D offline -e "source /tmp/offline.sql"

    if [[ $? -eq 0 ]]; then
        log "INFO" "✅ offline.sql 数据导入成功"
    else
        log "ERROR" "⛔ 数据导入失败，请检查 offline.sql 文件内容或权限"
        log "INFO" "📄 输出容器日志以排查问题..."
        $SUDO docker logs "$container_name" | tail -50
        return 1
    fi
    sleep 60

    # ------------------- 提升用户权限 -------------------
    for user in "root" "mysqluser"; do
        $SUDO docker exec "$container_name" mysql -u"root" -p"$mysql_root_password" -e \
            "UPDATE mysql.user SET Grant_priv='Y', Super_priv='Y' WHERE User='$user' AND Host='%'; FLUSH PRIVILEGES;" 2>/dev/null
    done
    log "INFO" "✅ 用户权限提升完成"

    # ------------------- 部署备份脚本 -------------------
    log "INFO" "📦 部署备份脚本到容器..."
    $SUDO mkdir -p "$(dirname "$backup_script_dst")"
    if $SUDO cp -f "$backup_script_src" "$backup_script_dst"; then
        $SUDO chmod +x "$backup_script_dst"
        log "INFO" "✅ 备份脚本已复制到: $backup_script_dst"
    else
        log "ERROR" "⛔ 备份脚本复制失败，请检查源文件是否存在"
        return 1
    fi

    # ------------------- 配置定时任务 -------------------
    log "INFO" "⏰ 配置定时备份任务 (每天 13:00)"
    local cron_job="0 13 * * * bash $backup_script_dst"

    if $SUDO crontab -u "$cron_user" -l 2>/dev/null | grep -Fq "$backup_script_dst"; then
        log "INFO" "✅ 定时任务已存在"
    else
        ( $SUDO crontab -u "$cron_user" -l 2>/dev/null; echo "$cron_job" ) | $SUDO crontab -u "$cron_user" -
        if [[ $? -eq 0 ]]; then
            log "INFO" "✅ 定时任务添加成功"
        else
            log "ERROR" "⛔ 定时任务添加失败"
            return 1
        fi
    fi

    # ------------------- 重启 crond -------------------
    log "INFO" "🔄 重启 crond 服务..."
    if command -v systemctl >/dev/null; then
        $SUDO systemctl restart crond || $SUDO systemctl restart cron
    else
        $SUDO service crond restart || $SUDO service cron restart
    fi
    log "INFO" "✅ crond 服务已重启"

    # ------------------- 最终状态输出 -------------------
    log "INFO" "🎉 MySQL 部署完成！"
    $SUDO docker ps --filter "name=$container_name" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"

    return 0
}

#x86架构安装mqtt
function emqx_x86()
{
    # ------------------- 定义变量 -------------------
    local container_name="uemqx"
    local image_tar="/data/temp/uemqx5.8.7.tar.gz"
    local image_name="emqx/emqx:5.8.7"
    local host_config_dir="/data/middleware/emqx/config"
    local host_dir="/data/middleware/emqx"

    log "INFO" "=================================================================="
    log "INFO" "开始部署 EMQX 服务 (Docker 版, x86)"
    log "INFO" "=================================================================="

    # ------------------- 检查容器是否已运行 -------------------
    log "INFO" "🔍 检查 EMQX 容器是否已运行..."
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "$container_name"; then
        log "INFO" "✅ 容器 '$container_name' 已在运行。"
        return 0
    fi

    # ------------------- 检查镜像是否存在 -------------------
    log "INFO" "🔍 检查 EMQX 镜像是否存在..."
    if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "$image_name"; then
        log "INFO" "✅ 镜像 $image_name 已存在。"
    else
        log "WARN" "❌ 镜像 $image_name 不存在，开始加载离线包..."

        if [[ ! -f "$image_tar" ]]; then
            log "ERROR" "⛔ 镜像文件不存在: $image_tar"
            return 1
        fi

        if $sudoset docker load -i "$image_tar"; then
            log "INFO" "🎉 镜像加载成功"
        else
            log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
            return 1
        fi

    fi

    # EMQX 容器使用用户 1000:1000
    $sudoset chown -R 1000:1000 "$host_dir"/{data,log}
    log "INFO" "✅ 目录创建完成，权限已设置"

    # ------------------- 启动 EMQX 容器 -------------------
    log "INFO" "🚀 正在启动 EMQX 容器: $container_name ..."
    $sudoset docker run -d \
        --name "$container_name" \
        --mac-address="02:42:ac:12:00:06" \
        --privileged \
        --restart=always \
        -p 1883:1883 \
        -p 8083:8083 \
        -p 8883:8883 \
        -v "$host_config_dir/emqx.conf:/opt/emqx/etc/emqx.conf:ro" \
        -v "$host_config_dir/auth-built-in-db-bootstrap.csv:/opt/emqx/etc/auth-built-in-db-bootstrap.csv:ro" \
        -v "$host_config_dir/acl.conf:/opt/emqx/etc/acl.conf:ro" \
	    -v "$host_dir/data:/opt/emqx/data" \
        -v "$host_dir/log:/opt/emqx/log" \
        "$image_name"

    if [ $? -ne 0 ]; then
        log "ERROR" "⛔ 容器启动失败"
        return 1
    fi

    log "INFO" "✅ EMQX 容器启动成功，等待初始化..."
    sleep 5

    # ------------------- 验证容器状态 -------------------
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "$container_name"; then
        log "INFO" "🎉 EMQX 部署完成！"
        $sudoset docker ps --filter "name=$container_name" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    else
        log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs $container_name"
        return 1
    fi

    return 0
}


function redis_x86()
{
    # ------------------- 定义变量 -------------------
    local container_name="uredis"
    local image_tar="/data/temp/redis8.2.2.tar.gz"
    local image_name="139.9.60.86:5000/redis:v3"
    local image_id="3bd8c109f88b"
    local redis_conf_host="/data/middleware/redis/config/redis.conf"
    local redis_data_host="/data/middleware/redis/data"
    local redis_port="6379"
    
    log "INFO" "=================================================================="
    log "INFO" "开始部署 Redis 服务 (Docker 版, x86)"
    log "INFO" "=================================================================="

    # ------------------- 检查容器是否已运行 -------------------
    log "INFO" "🔍 检查 Redis 容器是否已运行..."
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "$container_name"; then
        log "INFO" "✅ 容器 '$container_name' 已在运行。"
        return 0
    fi

    # ------------------- 自动配置 sysctl -------------------
    log "INFO" "🔧 配置系统参数: /etc/sysctl.conf"
    local sysctl_config="
#新增
net.core.somaxconn = 511
vm.overcommit_memory = 1
"
    # 检查是否已存在配置，避免重复添加
    if ! grep -Fq "#新增" /etc/sysctl.conf; then
        echo "$sysctl_config" | $sudoset tee -a /etc/sysctl.conf > /dev/null
        if [ $? -eq 0 ]; then
            log "INFO" "✅ sysctl 配置已追加"
        else
            log "ERROR" "⛔ 写入 /etc/sysctl.conf 失败，请检查权限"
            return 1
        fi
    else
        log "INFO" "✅ sysctl 配置已存在，跳过添加"
    fi

    # 应用配置
    $sudoset sysctl -p > /dev/null 2>&1
    if [ $? -eq 0 ]; then
        log "INFO" "✅ sysctl 参数已生效"
    else
        log "WARN" "⚠️ sysctl -p 执行可能有警告，继续执行"
    fi

    # ------------------- 检查镜像是否存在 -------------------
    log "INFO" "🔍 检查 Redis 镜像是否存在..."
    if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "$image_name"; then
        log "INFO" "✅ 镜像 $image_name 已存在。"
    else
        log "WARN" "❌ 镜像 $image_name 不存在，开始加载离线包..."

        if [[ ! -f "$image_tar" ]]; then
            log "ERROR" "⛔ 镜像文件不存在: $image_tar"
            return 1
        fi

        if $sudoset docker load -i "$image_tar"; then
            log "INFO" "🎉 镜像加载成功"
        else
            log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
            return 1
        fi

    fi

    # ------------------- 启动 Redis 容器 -------------------
    log "INFO" "🚀 正在启动 Redis 容器: $container_name ..."
    $sudoset docker run -d \
        --name "$container_name" \
        --security-opt seccomp=unconfined \
        --network host \
        -v "$redis_conf_host:/etc/redis/redis.conf" \
        -v "$redis_data_host:/data" \
        -v "/etc/localtime:/etc/localtime:ro" \
        --restart=always \
        "$image_name" \
        redis-server /etc/redis/redis.conf --appendonly yes

    if [ $? -ne 0 ]; then
        log "ERROR" "⛔ 容器启动失败"
        return 1
    fi

    log "INFO" "✅ Redis 容器启动成功，等待初始化..."
    sleep 5

    # ------------------- 验证状态 -------------------
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "$container_name"; then
        log "INFO" "🎉 Redis 部署完成！"
        $sudoset docker ps --filter "name=$container_name" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    else
        log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs $container_name"
        return 1
    fi

    return 0
}

function java_x86()
{
    log "INFO" "=================================================================="
    log "INFO" "开始部署 Java 服务 (Docker 版, x86)"
    log "INFO" "=================================================================="

    # ------------------- 定义变量 -------------------
    local container_name="ujava2"
    local image_tar="/data/temp/java1.8.0_472.tar.gz"
    local image_name="139.9.60.86:5000/ujava:v6"
    local image_id="a8ba5fa12b8e"

    # 主机目录映射
    local host_api="/data/services/api"
    local host_web="/data/services/web"
    local host_nginx_log="/data/middleware/nginx/nginx_log"
    local host_fdfs_data="/data/storage/storage/data"

    # 端口映射（可根据需要增减）
    local port_args=(
        "-p 8085:8085"
        "-p 8993:8993" "-p 8994:8994" "-p 8995:8995"
        "-p 8999:8999"
        "-p 8719:8719" "-p 8720:8720"
        "-p 9204:9204" "-p 9200:9200" "-p 9201:9201"
        "-p 9905:9905" "-p 9911:9911" "-p 9908:9908"
        "-p 9906:9906" "-p 9907:9907" "-p 9909:9909" "-p 9910:9910"
        "-p 30880:30880" "-p 30881:30881" "-p 30882:30882"
        "-p 30883:30883" "-p 30884:30884"
    )

    # ------------------- 检查容器是否已运行 -------------------
    log "INFO" "🔍 检查 Java 容器是否已运行..."
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
        log "WARN" "✅ 容器 '$container_name' 已在运行。自动退出部署操作！！！"
        return 0
    fi

    # ------------------- 检查镜像是否存在 -------------------
    log "INFO" "🔍 检查 Java 镜像是否存在..."
    if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "$image_name"; then
        log "WARN" "✅ 镜像 $image_name 已存在。"
    else
        log "INFO" "❌ 镜像 $image_name 不存在，开始加载离线包..."

        if [[ ! -f "$image_tar" ]]; then
            log "ERROR" "⛔ 镜像文件不存在: $image_tar"
            return 1
        fi

        if $sudoset docker load -i "$image_tar"; then
            log "INFO" "🎉 镜像加载成功"
        else
            log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
            return 1
        fi

    fi

    # ------------------- 启动 Java 容器 -------------------
    log "INFO" "🚀 正在启动 Java 容器: $container_name ..."

    $sudoset docker run -itd --privileged -v /data/services/api:/var/www/java/api -v /data/services/web:/var/www/java/web -v /data/middleware/nginx/nginx_log:/usr/local/nginx/logs -v /etc/localtime:/etc/localtime:ro -v /var/fdfs/storage/data:/var/fdfs/storage/data -p 8085:8085 -p 8993:8993 -p 8994:8994 -p 8995:8995 -p 8999:8999 -p 8719:8719 -p 8720:8720 -p 9204:9204 -p 9200:9200 -p 9201:9201 -p 9905:9905 -p 9911:9911 -p 9908:9908 -p 9906:9906 -p 9907:9907 -p 9909:9909 -p 9910:9910 -p 30880:30880 -p 30881:30881 -p 30882:30882 -p 30883:30883 -p 30884:30884 --restart=always --mac-address="02:42:ac:11:00:02" --name ujava2 $image_name /var/www/java/api/start.sh

    if [ $? -ne 0 ]; then
        log "ERROR" "⛔ 容器启动失败"
        return 1
    fi

    log "INFO" "✅ Java 容器启动成功，等待初始化..."
    sleep 8

    # ------------------- 验证容器状态 -------------------
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
        log "INFO" "🎉 Java 服务部署完成！"
        $sudoset docker ps --filter "name=$container_name" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    else
        log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs $container_name"
        return 1
    fi

    return 0
}

function python_x86() {
    log "INFO" "=================================================================="
    log "INFO" "开始部署 Python  容器 (upython) - x86"
    log "INFO" "=================================================================="

    # ------------------- 定义变量 -------------------
    local image_tar="/data/temp/python_v15.tar.gz"
    local image_tag="139.9.60.86:5000/upython:v15"
    local container_name="upython"
    local host_api_dir="/data/services/api/python-cmdb"
    local container_start_cmd="/var/www/html/start.sh"

    # ------------------- 检查容器是否已存在 -------------------
    if $sudoset docker ps -a --format '{{.Names}}' | grep -wq "^$container_name$"; then
        # 容器已存在
        if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
            log "WARN" "✅ 容器 '$container_name' 已在运行。自动退出部署操作！！！"
            return 0
        else
            log "WARN" "🔄 容器 '$container_name' 存在但未运行，正在启动..."
            $sudoset docker start "$container_name"
            sleep 3
            if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
                log "INFO" "✅ 容器 '$container_name' 已成功启动"
                return 0
            else
                log "ERROR" "⛔ 容器启动失败，请检查日志: docker logs $container_name"
                return 1
            fi
        fi
    fi

    # ------------------- 检查挂载目录 -------------------
    if [[ ! -d "$host_api_dir" ]]; then
        log "ERROR" "⛔ 应用目录不存在: $host_api_dir，请先上传代码"
        return 1
    fi

    if [[ ! -f "$host_api_dir/start.sh" ]]; then
        log "ERROR" "⛔ 启动脚本缺失: $host_api_dir/start.sh"
        return 1
    fi

    # ------------------- 加载镜像 -------------------
    if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -Fq "$image_tag"; then
        log "INFO" "✅ 镜像 $image_tag 已存在，跳过加载"
    else
        log "INFO" "📦 正在从离线包加载镜像: $image_tar"
        if [[ -f "$image_tar" ]]; then
            $sudoset docker load -i "$image_tar"
            if [[ $? -ne 0 ]]; then
                log "ERROR" "⛔ 镜像加载失败，请检查文件完整性: $image_tar"
                return 1
            fi
            log "INFO" "🎉 镜像加载成功: $image_tag"
        else
            log "ERROR" "⛔ 镜像文件不存在: $image_tar"
            return 1
        fi
    fi

    # ------------------- 启动新容器 -------------------
    log "INFO" "🚀 正在创建并启动容器 '$container_name'..."

    $sudoset docker run -d \
        --name "$container_name" \
        --restart=always \
        --mac-address="02:42:ac:11:00:07" \
        --privileged \
        -p 8000:8000 \
        -p 8002:8002 \
        -v "$host_api_dir:/var/www/html" \
        -v "/etc/localtime:/etc/localtime:ro" \
        "$image_tag" \
        "$container_start_cmd"

    if [[ $? -ne 0 ]]; then
        log "ERROR" "⛔ 容器创建失败，请检查参数或资源"
        return 1
    fi

    log "INFO" "✅ 容器 '$container_name' 创建并启动成功"

    # ------------------- 状态验证 -------------------
    sleep 3
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
        log "INFO" "🟢 Python 服务运行中 → 访问端口: 8000, 8002"
    else
        log "ERROR" "⛔ 容器启动后退出，请查看日志: docker logs $container_name"
        return 1
    fi

    return 0
}

#播放器系统
#x86架构python安装
function python_player_x86() {
    echo -e "\033[36m* 正在检查Python服务是否安装......\033[0m"

    # 检查Docker镜像和容器状态
    local image_exists=$(docker images | grep -q python && echo "true" || echo "false")
    local container_running=$(docker ps | grep -q uplayer && echo "true" || echo "false")
    local player_dir="/var/www/player"
    local source_player_dir="/home/player_auto_install/player"
    local docker_image_path="/home/player_auto_install/server_bag/upython_player.tar.gz"
    local start_script="/var/www/player/start.sh"

    if [[ "$image_exists" == "true" && "$container_running" == "true" ]]; then
        echo -e "\033[32m* 检查到Python已安装！正在安装下一个服务....\033[0m"
        return 0
    else
        echo -e "\033[33m* 检查到Python服务未安装，正在安装服务....\033[0m"

        # 加载Docker镜像并验证其存在
        if [ ! -f "$docker_image_path" ]; then
            echo -e "\033[31m* 错误: Docker镜像文件 $docker_image_path 不存在。\033[0m"
            return 1
        fi

        if ! docker load -i "$docker_image_path"; then
            echo -e "\033[31m* 加载Docker镜像失败。\033[0m"
            return 1
        fi

        # 标记Docker镜像
        if ! docker tag e449dd1ea3af 139.9.60.86:5000/upython:v11; then
            echo -e "\033[31m* 标记Docker镜像失败。\033[0m"
            return 1
        fi

        # 检查player目录是否存在，如果不存在则复制
        if [ ! -d "$source_player_dir" ]; then
            echo -e "\033[31m* 错误: 源目录 $source_player_dir 不存在。\033[0m"
            return 1
        fi

        if [ ! -d "$player_dir" ]; then
	    mkdir -p /var/www/player
            if ! cp -rp "$source_player_dir" /var/www/; then
                echo -e "\033[31m* 复制player目录失败。\033[0m"
                return 1
            else
                echo -e "\033[32m* 成功复制 player 目录。\033[0m"
		chmod_files=(
			"$player_dir/start.sh"
            		"$player_dir/manage.py"
            		"$player_dir/uuwsgi.ini"
            		"$player_dir/uplayer"
            		"$player_dir/UbainsDevOps"
        		)
        	chmod 755 -R "${chmod_files[@]}"
            fi
        else
            echo -e "\033[32m* 目录 $player_dir 已存在，跳过复制。\033[0m"
        fi

        # 确认启动脚本存在
        if [ ! -f "$start_script" ]; then
            echo -e "\033[31m* 错误: 启动脚本 $start_script 不存在。\033[0m"
            return 1
        fi
        # 启动容器，并指定固定的MAC地址
        if ! docker run -itd \
            --mac-address="02:42:ac:12:00:05" \
            -p 18082:8082 -p 11883:1883 -p 8000:8000 -p 9001:9001 -p 8002:8002\
            -v /var/www/player:/var/www/html \
            -v /etc/localtime:/etc/localtime:ro \
            --restart=always \
            --privileged \
            --name=uplayer \
            139.9.60.86:5000/upython:v11 \
            /root/start.sh; then
            echo -e "\033[31m* 启动Python容器失败。\033[0m"
            return 1
        else
            echo -e "\033[32m* Python服务安装完成并启动成功！\033[0m"
        fi
    fi
}

#x86架构ntp安装
function ntp_x86()
{
#  判断是否为centos7 否则跳出函数
if [ ! -f /etc/redhat-release ]; then
    log "ERROR" "当前系统不是CentOS 7，无法安装NTP服务"
    return 1
fi
log "INFO" "检查ntp服务......"
$sudoset systemctl status ntpd |grep running
#ps -aux | grep ntp
if [ $? -eq  0 ]; then
    log "INFO" "检查到ntp已安装！"
else
    log "WARN" "检查到ntp未安装，正在安装..."
    $sudoset cd $auto_java/server_bag/ntp
    $sudoset rpm -Uvh *.rpm --nodeps --force
    $sudoset systemctl enable ntpd
    $sudoset systemctl enable ntpdate
    $sudoset systemctl start ntpd
    $sudoset systemctl status ntpd
    $sudoset mv /etc/ntp.conf    /etc/ntp.confbak
    $sudoset cp $auto_java/server_bag/ntp/ntp.conf   /etc/
    $sudoset systemctl restart  ntpd
    $sudoset systemctl disable  chronyd
    log "INFO" "完成ntp服务的安装"
fi
}

# ========================================
# 🐍 函数：python_voice_x86
# 描述：部署 upython 容器服务（语音平台后端）
# 功能：
#   - 检查是否已有 upython_voice 容器运行
#   - 若无，则加载镜像、打标签并启动容器
#   - 挂载新路径（生产环境路径）
#   - 端口映射：8003:8000, 8004:8002, 8080:8080
#   - 固定 MAC 地址
# ========================================
function python_voice_x86() {
    log "INFO" "=================================================================="
    log "INFO" "开始部署 Python_voice  容器 (upython_voice) - x86"
    log "INFO" "=================================================================="

    local service_name="Python 语音后端服务"
    local container_name="upython_voice"
    local image_tar="/data/offline_auto_unifiedPlatform/data/temp/uvoice3.tar.gz"
    local source_image_id="2a69026b2899"
    local target_image_tag="139.9.60.86:5000/upython:v13"

    # ✅ 更新为实际的生产挂载路径
    local mounts=(
        "/data/services/api/python-voice:/var/www/html/api"
        "/data/services/web/pc/pc-vue2-voice:/var/www/html/web"
        "/data/third_party/iFlyTrans/iflytek/iListen/voicePath:/home/iflytek/iListen/voicePath"
        "/data/third_party/iFlyTrans/iflytek/lingtOnlineHZ/online/:/home/iflytek/lingtOnlineHZ/online"
        "/data/third_party/iFlyTrans/ubains/lingtOnlineHZ/online:/home/ubains/lingtOnlineHZ/online"
        "/data/third_party/iFlyTrans/iflytek/huiyi/:/home/iflytek/huiyi/"
        "/data/services/api/python-voice/start.sh:/root/start.sh"  # 添加到这里
    )

    log "INFO" "=================================================="
    log "INFO" "🚀 正在部署：$service_name"
    log "INFO" "容器名称: $container_name"
    log "INFO" "镜像文件: $image_tar"
    log "INFO" "=================================================="

    # 1. 检查容器是否已在运行
    if docker ps --format '{{.Names}}' | grep -qw "^$container_name$"; then
        log "WARN" "✅ 容器 '$container_name' 已在运行，退出自动部署！！！"
        return 0
    fi

    # 2. 检查容器是否已存在（即使未运行）
    if docker ps -a --format '{{.Names}}' | grep -qw "^$container_name$"; then
        log "WARN" "⛔ 容器 '$container_name' 已存在，退出自动部署！！！"
        return 0
    fi

    # 赋予启动脚本执行权限
    local host_start_script="/data/services/api/python-voice/start.sh"
    if [ -f "$host_start_script" ]; then
        log "INFO" "🔧 修复宿主机启动脚本权限..."
        chmod +x "$host_start_script"
        log "INFO" "✅ 宿主机脚本权限已修复: $host_start_script"
    else
        log "WARN" "⚠️  宿主机启动脚本不存在: $host_start_script"
    fi

    # 3. 检查目标镜像是否存在
    if docker images --format '{{.Repository}}:{{.Tag}}' | grep -qw "^$target_image_tag$"; then
        log "WARN" "✅ 镜像已存在: $target_image_tag"
    else
        # 4. 检查镜像压缩包是否存在
        if [ ! -f "$image_tar" ]; then
            log "INFO" "❌ 镜像文件不存在: $image_tar"
            return 1
        fi

        log "INFO" "🐳 正在加载 upython 镜像..."
        if ! docker load -i "$image_tar"; then
            log "ERROR" "❌ 镜像加载失败，请检查文件完整性"
            return 1
        fi
        log "INFO" "✅ 镜像加载完成"

    fi

    # 5. 构建挂载参数 - 正确的方式
    local mount_args=()
    for mount in "${mounts[@]}"; do
        mount_args+=("-v")
        mount_args+=("$mount")
    done

    # 6. 启动容器
    log "INFO" "🚀 正在启动容器 '$container_name'..."
    
    docker run -itd \
        --name "$container_name" \
        --mac-address="02:42:ac:11:00:24" \
        --restart=always \
        -p 8003:8000 \
        -p 8004:8002 \
        -p 8080:8080 \
        "${mount_args[@]}" \
        "$target_image_tag"

    if [ $? -ne 0 ]; then
        log "ERROR" "❌ 容器启动失败，请检查端口占用或权限问题"
        return 1
    fi

    # 7. 验证容器状态
    sleep 3
    if docker ps --format '{{.Names}}' | grep -qw "^$container_name$"; then
        local cid=$(docker ps --filter "name=$container_name" --format "{{.ID}}")
        log "INFO" "✅ 容器启动成功！"
        log "INFO" "🏷  名称: $container_name"
        log "INFO" "🔢  ID: $cid"
        log "INFO" "🌐  端口: 8003→8000, 8004→8002, 8080→8080"
        log "INFO" "📁  共挂载 ${#mounts[@]} 个目录"
    else
        log "ERROR" "❌ 容器启动后退出，请执行 'docker logs $container_name' 查看日志"
        return 1
    fi

    log "INFO" "🎉 $service_name 部署成功！"
    return 0
}

#部署宿主机java服务

# ========================================
# ☕ 函数：deploy_jdk_host
# 描述：在宿主机上部署JDK（tar.gz格式）
# 功能：
#   - 检查JDK安装包是否存在
#   - 解压JDK到指定目录
#   - 配置Java环境变量（JAVA_HOME, CLASSPATH, PATH）
#   - 验证JDK安装
# 参数：
#   $1: JDK安装包路径（tar.gz文件）
#   $2: JDK安装目录（可选，默认为/opt/java）
# ========================================
function deploy_jdk_host() {
    local jdk_tar_path="/data/temp/jdk-8u472-linux-x64.tar.gz"
    local install_base_dir="${2:-/opt/java}"
    
    log "INFO" "=================================================="
    log "INFO" "☕ 正在部署宿主机JDK"
    log "INFO" "   JDK安装包: $jdk_tar_path"
    log "INFO" "   安装目录: $install_base_dir"
    log "INFO" "=================================================="
    
    # 1. 检查JDK安装包是否存在
    if [ -z "$jdk_tar_path" ]; then
        log "ERROR" "❌ 未指定JDK安装包路径"
        log "INFO" "用法: deploy_jdk_host <jdk_tar.gz路径> [安装目录]"
        return 1
    fi
    
    if [ ! -f "$jdk_tar_path" ]; then
        log "ERROR" "❌ JDK安装包不存在: $jdk_tar_path"
        return 1
    fi
    
    # 检查是否为tar.gz格式
    if [[ ! "$jdk_tar_path" =~ \.tar\.gz$ ]]; then
        log "ERROR" "❌ 文件格式错误，需要tar.gz格式: $jdk_tar_path"
        return 1
    fi
    
    log "INFO" "✅ JDK安装包检查通过"
    
    # 2. 检查是否已安装JDK，并进行备份
    local old_java_home=""
    local old_java_version=""
    local backup_dir="/opt/java_backup_$(date +%Y%m%d_%H%M%S)"
    
    if command -v java &> /dev/null; then
        old_java_version=$(java -version 2>&1 | head -n1)
        log "INFO" "📋 检测到系统已安装Java: $old_java_version"
        
        # 获取旧的JAVA_HOME
        if [ -n "$JAVA_HOME" ] && [ -d "$JAVA_HOME" ]; then
            old_java_home="$JAVA_HOME"
        else
            # 尝试从java命令路径推断JAVA_HOME
            local java_path=$(which java 2>/dev/null)
            if [ -n "$java_path" ]; then
                java_path=$(readlink -f "$java_path" 2>/dev/null || echo "$java_path")
                # 移除/bin/java部分
                old_java_home=$(dirname "$(dirname "$java_path")" 2>/dev/null)
            fi
        fi
        
        # 备份旧的JDK目录
        if [ -n "$old_java_home" ] && [ -d "$old_java_home" ] && [ "$old_java_home" != "$install_base_dir" ]; then
            log "INFO" "💾 正在备份旧JDK目录: $old_java_home"
            mkdir -p "$backup_dir"
            if [ $? -eq 0 ]; then
                cp -r "$old_java_home" "$backup_dir/old_jdk_$(basename "$old_java_home")" 2>/dev/null
                if [ $? -eq 0 ]; then
                    log "INFO" "✅ 旧JDK目录已备份到: $backup_dir"
                else
                    log "WARN" "⚠️  旧JDK目录备份失败，继续安装"
                fi
            fi
        fi
        
        # 备份/etc/profile中的Java配置
        log "INFO" "💾 正在备份/etc/profile中的Java配置..."
        mkdir -p "$backup_dir"
        if [ -f /etc/profile ]; then
            cp /etc/profile "$backup_dir/profile.backup" 2>/dev/null
            if [ $? -eq 0 ]; then
                log "INFO" "✅ /etc/profile已备份到: $backup_dir/profile.backup"
            fi
        fi
        
        # 记录旧Java信息到备份目录
        if [ -d "$backup_dir" ]; then
            {
                echo "备份时间: $(date '+%Y-%m-%d %H:%M:%S')"
                echo "旧Java版本: $old_java_version"
                echo "旧JAVA_HOME: $old_java_home"
            } > "$backup_dir/old_java_info.txt" 2>/dev/null
        fi
        
        log "INFO" "🔄 将自动更新Java环境..."
    fi
    
    # 3. 创建安装目录
    mkdir -p "$install_base_dir"
    if [ $? -ne 0 ]; then
        log "ERROR" "❌ 无法创建安装目录: $install_base_dir"
        return 1
    fi
    log "INFO" "✅ 创建安装目录: $install_base_dir"
    
    # 4. 解压JDK
    log "INFO" "🗜️  正在解压JDK安装包..."
    cd "$install_base_dir" || {
        log "ERROR" "❌ 无法进入目录: $install_base_dir"
        return 1
    }
    
    # 解压到当前目录
    tar -xzf "$jdk_tar_path" -C "$install_base_dir"
    if [ $? -ne 0 ]; then
        log "ERROR" "❌ JDK解压失败，请检查压缩包完整性"
        return 1
    fi
    log "INFO" "✅ JDK解压完成"
    
    # 5. 查找解压后的JDK目录（通常是jdk8u472-b08或类似格式）
    local jdk_dir=$(find "$install_base_dir" -maxdepth 1 -type d -name "jdk*" | head -n1)
    if [ -z "$jdk_dir" ]; then
        log "ERROR" "❌ 未找到JDK目录，解压后的目录结构可能不正确"
        log "INFO" "请检查解压后的目录结构"
        return 1
    fi
    
    local jdk_home="$jdk_dir"
    local jdk_name=$(basename "$jdk_dir")
    log "INFO" "✅ 找到JDK目录: $jdk_name"
    log "INFO" "   JDK_HOME: $jdk_home"
    
    # 6. 检查JDK目录结构是否完整
    if [ ! -d "$jdk_home/bin" ] || [ ! -f "$jdk_home/bin/java" ]; then
        log "ERROR" "❌ JDK目录结构不完整，缺少bin/java文件"
        return 1
    fi
    log "INFO" "✅ JDK目录结构检查通过"
    
    # 7. 配置Java环境变量
    log "INFO" "🛠️  正在配置Java环境变量..."
    
    # 检查/etc/profile中是否已存在JAVA_HOME配置，并备份后注释掉
    if grep -q "JAVA_HOME\|#set java environment" /etc/profile; then
        log "INFO" "📋 检测到/etc/profile中已存在JAVA_HOME配置，将注释旧配置并添加新配置"
        # 备份旧的配置（如果之前没有备份）
        if [ ! -f "$backup_dir/profile.backup" ]; then
            mkdir -p "$backup_dir"
            cp /etc/profile "$backup_dir/profile.backup" 2>/dev/null
            log "INFO" "✅ /etc/profile已备份到: $backup_dir/profile.backup"
        fi
        
        # 使用awk来注释掉Java配置块（更可靠，处理多行块）
        awk '
        BEGIN { in_java_block = 0 }
        /#set java environment/ {
            in_java_block = 1
            # 如果这行还没被注释，就注释掉
            if ($0 !~ /^[[:space:]]*#/) {
                print "#" $0
            } else {
                print $0
            }
            next
        }
        in_java_block {
            # 如果是Java相关的配置行（JAVA_HOME, CLASSPATH, PATH, export等）
            if ($0 ~ /JAVA_HOME|CLASSPATH|PATH.*JAVA_HOME|export.*JAVA_HOME|export.*CLASSPATH|export.*PATH/) {
                # 如果这行还没被注释，就注释掉
                if ($0 !~ /^[[:space:]]*#/ && $0 !~ /^[[:space:]]*$/) {
                    print "#" $0
                } else {
                    print $0
                }
                next
            }
            # 如果是空行，保持块状态，继续
            if ($0 ~ /^[[:space:]]*$/) {
                print $0
                next
            }
            # 其他非空行，结束Java块
            in_java_block = 0
            print $0
            next
        }
        { print $0 }
        ' /etc/profile > /etc/profile.tmp && mv /etc/profile.tmp /etc/profile
        if [ $? -eq 0 ]; then
            log "INFO" "✅ 旧Java配置已注释"
        else
            log "WARN" "⚠️  注释旧配置时出现问题，继续安装"
        fi
    fi
    
    # 添加新的Java环境变量配置
    cat >> /etc/profile << EOF

#set java environment
JAVA_HOME=$jdk_home
CLASSPATH=.:\$JAVA_HOME/lib/tools.jar
PATH=\$JAVA_HOME/bin:\$PATH
export JAVA_HOME CLASSPATH PATH
EOF
    
    log "INFO" "✅ Java环境变量已写入 /etc/profile"
    
    # 8. 立即在当前会话中生效
    log "INFO" "🔄 正在重新加载 /etc/profile 使环境变量生效..."
    source /etc/profile 2>/dev/null || . /etc/profile 2>/dev/null
    
    # 如果source失败，则手动设置环境变量
    if [ $? -ne 0 ]; then
        log "WARN" "⚠️  source /etc/profile 失败，使用export设置环境变量"
        export JAVA_HOME="$jdk_home"
        export CLASSPATH=".:$JAVA_HOME/lib/tools.jar"
        export PATH="$JAVA_HOME/bin:$PATH"
    fi
    
    log "INFO" "✅ 环境变量已在当前会话中生效"
    
    # 9. 验证Java安装
    log "INFO" "🔍 正在验证Java安装..."
    if java -version >/dev/null 2>&1; then
        local java_version=$(java -version 2>&1 | head -n1)
        local java_path=$(which java)
        log "INFO" "✅ Java环境配置成功！"
        log "INFO" "   Java版本: $java_version"
        log "INFO" "   Java路径: $java_path"
        log "INFO" "   JAVA_HOME: $JAVA_HOME"
    else
        log "ERROR" "❌ Java环境配置失败，请检查路径或权限"
        log "INFO" "   请手动执行: source /etc/profile"
        return 1
    fi
    
    log "INFO" "=================================================="
    log "INFO" "🎉 JDK部署成功！"
    log "INFO" "   安装路径: $jdk_home"
    log "INFO" "   环境变量已配置到 /etc/profile"
    log "INFO" "   提示: 新开终端或执行 'source /etc/profile' 使环境变量生效"
    log "INFO" "=================================================="
    
    return 0
}

# ========================================
# 🚀 函数：start_java_meeting_service
# 描述：启动对外服务并将服务写入服务器自启动
# 功能：
#   - 检查Java环境是否部署成功
#   - 启动java-meeting-extapi服务
#   - 检查服务进程是否启动成功
#   - 将启动命令写入/etc/rc.d/rc.local实现自启动
# ========================================
function start_java_meeting_service() {
    local service_dir="/data/services/api/java-meeting/java-meeting-extapi"
    local run_script="$service_dir/run.sh"
    local log_file="$service_dir/logs/ubains-INFO-AND-ERROR.log"
    local rc_local="/etc/rc.d/rc.local"
    local service_name="java-meeting-extapi"
    
    log "INFO" "=================================================="
    log "INFO" "🚀 正在启动对外服务: $service_name"
    log "INFO" "   服务目录: $service_dir"
    log "INFO" "=================================================="
    
    # 1. 检查Java环境是否部署成功
    log "INFO" "🔍 正在检查Java环境..."
    source /etc/profile 2>/dev/null || . /etc/profile 2>/dev/null
    
    if ! java -version >/dev/null 2>&1; then
        log "ERROR" "❌ Java环境未部署或配置失败"
        log "ERROR" "   请先执行 deploy_jdk_host 函数部署JDK"
        log "ERROR" "   或手动执行: source /etc/profile"
        return 1
    fi
    
    local java_version=$(java -version 2>&1 | head -n1)
    log "INFO" "✅ Java环境检查通过: $java_version"
    
    # 2. 检查服务目录和脚本是否存在
    if [ ! -d "$service_dir" ]; then
        log "ERROR" "❌ 服务目录不存在: $service_dir"
        return 1
    fi
    
    if [ ! -f "$run_script" ]; then
        log "ERROR" "❌ 启动脚本不存在: $run_script"
        return 1
    fi
    
    # 确保脚本有执行权限
    if [ ! -x "$run_script" ]; then
        log "INFO" "🔧 正在为启动脚本添加执行权限..."
        chmod +x "$run_script"
    fi
    
    log "INFO" "✅ 服务目录和脚本检查通过"
    
    # 3. 检查服务是否已经在运行
    cd "$service_dir" || {
        log "ERROR" "❌ 无法进入服务目录: $service_dir"
        return 1
    }
    
    # 尝试通过进程名或日志判断服务是否已运行
    local process_count=$(pgrep -f "java-meeting-extapi\|run.sh" 2>/dev/null | wc -l)
    if [ "$process_count" -gt 0 ]; then
        log "WARN" "⚠️  检测到服务可能已在运行（进程数: $process_count）"
        log "INFO" "   将尝试重新启动服务"
    fi
    
    # 4. 启动服务
    log "INFO" "▶️  正在启动服务..."
    
    # 确保日志目录存在
    mkdir -p "$(dirname "$log_file")"
    
    # 执行启动命令：./run.sh
    log "INFO" "📝 执行启动命令: ./run.sh"
    ./run.sh > "$service_dir/nohup.out" 2>&1 &
    local start_pid=$!
    
    # 等待几秒让服务启动
    log "INFO" "⏳ 等待服务启动（5秒）..."
    sleep 5
    
    # 执行tail -f查看日志（后台运行，避免阻塞）
    log "INFO" "📋 正在查看服务日志: tail -f $log_file"
    if [ -f "$log_file" ]; then
        # 显示最后20行日志
        log "INFO" "📄 最新日志内容（最后20行）:"
        tail -n 20 "$log_file" 2>/dev/null | while read line; do
            log "INFO" "   $line"
        done
        # 后台运行tail -f，输出到nohup
        nohup tail -f "$log_file" >> "$service_dir/tail.log" 2>&1 &
        log "INFO" "✅ 日志监控已启动（后台运行），实时日志输出到: $service_dir/tail.log"
    else
        log "WARN" "⚠️  日志文件尚未创建: $log_file"
        log "INFO" "   服务可能正在启动中，请稍后查看日志"
    fi
    
    # 5. 检查服务进程是否启动成功
    log "INFO" "🔍 正在检查服务进程..."
    
    # 检查run.sh进程
    local run_pid=$(pgrep -f "run.sh" | head -n1)
    if [ -n "$run_pid" ]; then
        log "INFO" "✅ 启动脚本进程已运行（PID: $run_pid）"
    else
        log "WARN" "⚠️  未检测到启动脚本进程，服务可能启动失败"
    fi
    
    # 检查Java进程（通过服务目录或进程名）
    local java_pids=$(pgrep -f "java.*$service_dir\|java-meeting-extapi" 2>/dev/null)
    if [ -n "$java_pids" ]; then
        log "INFO" "✅ Java服务进程已启动"
        echo "$java_pids" | while read pid; do
            log "INFO" "   PID: $pid"
        done
    else
        log "WARN" "⚠️  未检测到Java服务进程，请检查日志: $log_file"
        log "WARN" "   或查看: $service_dir/nohup.out"
    fi
    
    # 检查日志文件是否存在（服务启动后会创建日志文件）
    if [ -f "$log_file" ]; then
        log "INFO" "✅ 日志文件已创建: $log_file"
        # 显示最后几行日志
        log "INFO" "📋 最新日志内容:"
        tail -n 10 "$log_file" 2>/dev/null | while read line; do
            log "INFO" "   $line"
        done
    else
        log "WARN" "⚠️  日志文件尚未创建，服务可能正在启动中"
    fi
    
    # 6. 将启动命令写入/etc/rc.d/rc.local实现自启动
    log "INFO" "📝 正在配置服务自启动..."
    
    # 检查rc.local文件是否存在
    if [ ! -f "$rc_local" ]; then
        log "INFO" "📁 rc.local文件不存在，正在创建..."
        sudo touch "$rc_local"
        sudo chmod +x "$rc_local"
    fi
    
    # 构建启动命令（按照用户要求：cd目录，执行run.sh，tail -f日志）
    # 注意：在rc.local中，tail -f会阻塞，所以使用后台运行
    local startup_cmd="cd $service_dir && ./run.sh > $service_dir/nohup.out 2>&1 &"
    local tail_cmd="tail -f $log_file >> $service_dir/tail.log 2>&1 &"
    
    # 检查是否已经存在相同的启动命令
    if grep -q "$service_dir.*run.sh" "$rc_local" 2>/dev/null; then
        log "INFO" "✅ 自启动配置已存在，跳过添加"
    else
        # 备份rc.local
        if [ -f "$rc_local" ]; then
            sudo cp "$rc_local" "${rc_local}.backup.$(date +%Y%m%d_%H%M%S)" 2>/dev/null
            log "INFO" "✅ 已备份rc.local文件"
        fi
        
        # 添加启动命令到rc.local
        log "INFO" "📝 正在添加启动命令到 $rc_local"
        echo "" | sudo tee -a "$rc_local" > /dev/null
        echo "# Auto-start java-meeting-extapi service - $(date '+%Y-%m-%d %H:%M:%S')" | sudo tee -a "$rc_local" > /dev/null
        echo "cd $service_dir && ./run.sh > $service_dir/nohup.out 2>&1 &" | sudo tee -a "$rc_local" > /dev/null
        echo "tail -f $log_file >> $service_dir/tail.log 2>&1 &" | sudo tee -a "$rc_local" > /dev/null
        
        # 确保rc.local有执行权限
        sudo chmod +x "$rc_local" 2>/dev/null
        
        log "INFO" "✅ 自启动配置已添加到 $rc_local"
        log "INFO" "   启动命令: cd $service_dir && ./run.sh"
        log "INFO" "   日志监控: tail -f $log_file"
    fi
    
    log "INFO" "=================================================="
    log "INFO" "🎉 服务启动流程完成！"
    log "INFO" "   服务目录: $service_dir"
    log "INFO" "   日志文件: $log_file"
    log "INFO" "   自启动配置: $rc_local"
    log "INFO" "   提示: 可以使用 'tail -f $log_file' 查看实时日志"
    log "INFO" "=================================================="
    
    return 0
}

# ========================================
# 🔧 函数：malan
# 描述：配置 malan 服务自启动（写入 /etc/rc.local）
# 功能：
#   - 检查 /etc/rc.local 文件是否存在
#   - 检查是否已存在 malan 启动命令（避免重复添加）
#   - 将启动命令写入 /etc/rc.local
#   - 确保文件有执行权限
# ========================================
function malan() {
    local rc_local="/etc/rc.local"
    local service_dir="/data/middleware/monitor"
    local startup_cmd="cd /data/middleware/monitor/  &&  nohup  ./malan &"
    
    log "INFO" "=================================================="
    log "INFO" "🔧 正在配置 malan 服务自启动"
    log "INFO" "   服务目录: $service_dir"
    log "INFO" "=================================================="
    
    # 1. 检查服务目录是否存在
    if [ ! -d "$service_dir" ]; then
        log "WARN" "⚠️  服务目录不存在: $service_dir"
        log "WARN" "   将尝试创建目录..."
        mkdir -p "$service_dir" 2>/dev/null || {
            log "ERROR" "❌ 无法创建目录: $service_dir"
            return 1
        }
        log "INFO" "✅ 目录已创建: $service_dir"
    fi
    
    # 2. 检查 malan 可执行文件是否存在
    if [ ! -f "$service_dir/malan" ]; then
        log "WARN" "⚠️  malan 可执行文件不存在: $service_dir/malan"
        log "WARN" "   将继续配置自启动，但请确保文件已部署"
    else
        # 确保文件有执行权限
        if [ ! -x "$service_dir/malan" ]; then
            log "INFO" "🔧 正在为 malan 添加执行权限..."
            chmod +x "$service_dir/malan" 2>/dev/null || {
                log "WARN" "⚠️  无法添加执行权限，可能需要 root 权限"
            }
        fi
        log "INFO" "✅ malan 文件检查通过"
    fi
    
    # 3. 检查 /etc/rc.local 文件是否存在
    if [ ! -f "$rc_local" ]; then
        log "INFO" "📁 rc.local 文件不存在，正在创建..."
        touch "$rc_local" 2>/dev/null || {
            log "ERROR" "❌ 无法创建 $rc_local，请检查权限"
            return 1
        }
        # 添加 shebang 行（如果文件为空）
        echo "#!/bin/bash" > "$rc_local"
        log "INFO" "✅ rc.local 文件已创建"
    fi
    
    # 确保 rc.local 有执行权限
    if [ ! -x "$rc_local" ]; then
        log "INFO" "🔧 正在为 rc.local 添加执行权限..."
        chmod +x "$rc_local" 2>/dev/null || {
            log "WARN" "⚠️  无法添加执行权限，可能需要 root 权限"
        }
    fi
    
    # 4. 检查是否已经存在相同的启动命令
    if grep -Fq "cd /data/middleware/monitor/  &&  nohup  ./malan &" "$rc_local" 2>/dev/null; then
        log "INFO" "✅ malan 自启动配置已存在，跳过添加"
        return 0
    fi
    
    # 5. 备份 rc.local
    if [ -f "$rc_local" ]; then
        local backup_file="${rc_local}.backup.$(date +%Y%m%d_%H%M%S)"
        cp "$rc_local" "$backup_file" 2>/dev/null
        if [ $? -eq 0 ]; then
            log "INFO" "✅ 已备份 rc.local 文件: $backup_file"
        else
            log "WARN" "⚠️  备份 rc.local 失败，继续执行"
        fi
    fi
    
    # 6. 添加启动命令到 rc.local
    log "INFO" "📝 正在添加 malan 启动命令到 $rc_local"
    
    # 确保文件末尾有换行符
    if [ -s "$rc_local" ] && [ "$(tail -c 1 "$rc_local")" != "" ]; then
        echo "" >> "$rc_local"
    fi
    
    # 添加注释和启动命令
    echo "# Auto-start malan service - $(date '+%Y-%m-%d %H:%M:%S')" >> "$rc_local"
    echo "$startup_cmd" >> "$rc_local"
    
    if [ $? -eq 0 ]; then
        log "INFO" "✅ malan 自启动配置已添加到 $rc_local"
        log "INFO" "   启动命令: $startup_cmd"
    else
        log "ERROR" "❌ 写入 $rc_local 失败，请检查权限"
        return 1
    fi
    
    # 7. 验证写入的内容
    if grep -Fq "cd /data/middleware/monitor/  &&  nohup  ./malan &" "$rc_local" 2>/dev/null; then
        log "INFO" "✅ 配置验证成功"
    else
        log "WARN" "⚠️  配置验证失败，请手动检查 $rc_local"
    fi
    
    log "INFO" "=================================================="
    log "INFO" "🎉 malan 自启动配置完成！"
    log "INFO" "   配置文件: $rc_local"
    log "INFO" "   提示: 系统重启后 malan 服务将自动启动"
    log "INFO" "=================================================="
    
    return 0
}

#x86架构统信、麒麟系统的ntp安装
function ntp_uos() {
    #    判断如果是centos7就退出安装
    if [ -f /etc/redhat-release ]; then
        log "ERROR" "当前系统是CentOS 7，已经安装NTP服务，无需安装chrony服务"
        return 1
    fi

#    判断文件是否存在ntp1.aliyun.com地址
    if grep -q "ntp1.aliyun.com" /etc/chrony.conf; then
        log "INFO" "NTP配置文件中已存在ntp1.aliyun.com地址，无需重复添加"
        return 0
    fi

    local config_dir="$auto_java/data/temp/ntp/server_bag/ntp"
    local backup="/etc/chrony.confbak"
    local target="/etc/chrony.conf"

    # 切换目录（不需要 sudo）
    cd "$config_dir" || { echo "无法进入目录: $config_dir"; return 1; }

    # 备份原有配置
    if [ -f "$target" ]; then
        log "INFO" "正在备份旧配置文件到 $backup"
        sudo cp "$target" "$backup"
    fi

#    检查是否可以通外网，可以则配置外网的ntp服务器地址，不可以则询问是否有企业ntp服务器地址
    if ! ping -c 1 ntp1.aliyun.com &> /dev/null; then
          log "ERROR" "无法连接到外网NTP服务器 ntp1.aliyun.com，请检查网络连接"
          read -p "是否有企业NTP服务器地址？(y/n): " yn
          if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
              read -p "请输入企业NTP服务器地址: " custom_ntp
              echo "server $custom_ntp iburst" | sudo tee -a "$target"
              echo "allow all" | sudo tee -a "$target"
          else
              log "ERROR" "没有可用的NTP服务器地址，配置默认地址，请联系管理员"
              echo "server ntp1.aliyun.com iburst" | sudo tee -a "$target"
              echo "allow all" | sudo tee -a "$target"
          fi
    else
        log "INFO" "可以连接到外网NTP服务器 ntp1.aliyun.com，继续配置"
        # 如果可以连接到外网NTP服务器，则继续添加默认的NTP服务器地址
          echo "server ntp1.aliyun.com iburst" | sudo tee -a "$target"
          echo "server ntp2.aliyun.com iburst" | sudo tee -a "$target"
          echo "server ntp3.aliyun.com iburst" | sudo tee -a "$target"
          echo "allow all" | sudo tee -a "$target"
    fi


    # 输出同步地址到日志
    log "INFO" "正在写入同步地址到 $target"

    # 重启服务
    log "INFO" "正在重启 chronyd 服务..."
    sudo systemctl daemon-reload
    sudo systemctl restart chronyd
    sudo systemctl enable chronyd
    sudo systemctl status chronyd

    log "INFO" "chronyd 服务已重启并设置为开机自启"
}

# ========================================
# ☕ 函数：iListen_x86
# 描述：部署 iListen Java 语音服务（科大讯飞语音识别核心服务）
# 功能：
#   - 卸载 OpenJDK 11
#   - 安装 JDK 8u361
#   - 配置 Java 环境变量
#   - 复制依赖库文件与程序
#   - 启动 iListen 服务
# 依赖：
#   - 离线包路径: /data/offline_auto_unifiedPlatform/data/temp/iFlyTrans_server_bag/
#   - 服务路径:   /data/third_party/iFlyTrans/iflytek/iListen
# ========================================
function iListen_x86() {
    local service_name="iListen 语音服务"
    local source_bag="/data/offline_auto_unifiedPlatform/data/temp/iFlyTrans_server_bag"
    local jdk_tar="jdk-8u361-linux-x64.tar.gz"
    local deploy_path="/opt/deploy"
    local java_install_dir="$deploy_path/java"
    local jdk_home="$java_install_dir/jdk1.8.0_361"
    local iflytek_home="/data/third_party/iFlyTrans/iflytek"
    local lib64_dest="/usr/lib64"
    local lib_dest="/usr/lib"

    log "INFO" "=================================================="
    log "INFO" "🚀 正在部署：$service_name"
    log "INFO" "   源文件目录: $source_bag"
    log "INFO" "   服务安装路径: $iflytek_home"
    log "INFO" "=================================================="

    # --- 1. 检查服务是否已部署（✅ 改为检查目录和脚本存在，而非 Java）---
    local service_dir="$iflytek_home/iListen"
    local restart_script="$service_dir/restart.sh"

    if [[ -d "$service_dir" ]] && [[ -f "$restart_script" ]]; then
        log "INFO" "✅ 检测到 $service_name 已部署，跳过安装，准备启动..."

        cd "$service_dir" || {
            log "ERROR" "❌ 无法进入服务目录: $service_dir"
            return 1
        }

        log "INFO" "▶️  正在启动 iListen 服务..."
        nohup ./restart.sh > nohup.out 2>&1 &
        sleep 10

        if pgrep -f "iListen" > /dev/null; then
            log "INFO" "✅ iListen 服务已启动（PID: $(pgrep -f iListen | head -n1)）"
        else
            log "WARN" "⚠️  服务启动但未检测到进程，请检查日志: $service_dir/nohup.out"
        fi

        log "INFO" "🎉 $service_name 部署成功！"
        return 0
    fi

    # 2. 卸载旧版 OpenJDK（避免冲突）
    log "INFO" "🔧 正在卸载旧版 OpenJDK..."
    yum -y remove java-11-openjdk* > /dev/null 2>&1 || true
    log "INFO" "✅ OpenJDK 卸载完成（如存在）"

    # 3. 检查依赖文件是否存在
    if [ ! -f "$source_bag/$jdk_tar" ]; then
        log "ERROR" "❌ JDK 安装包不存在: $source_bag/$jdk_tar"
        return 1
    fi

    if [ ! -d "$source_bag/lib" ] || [ ! -d "$source_bag/usrlib" ] || [ ! -d "$source_bag/iflytek" ]; then
        log "ERROR" "❌ 源文件目录缺失，请检查: $source_bag"
        return 1
    fi

    # 4. 创建部署目录
    mkdir -p "$deploy_path" "$java_install_dir"
    if [ $? -ne 0 ]; then
        log "ERROR" "❌ 无法创建部署目录: $deploy_path"
        return 1
    fi
    log "INFO" "✅ 创建部署目录: $deploy_path"

    # 5. 复制并解压 JDK
    log "INFO" "📦 复制 JDK 安装包..."
    cp "$source_bag/$jdk_tar" "$deploy_path/"
    cd "$deploy_path" || {
        log "ERROR" "❌ 无法进入目录: $deploy_path"
        return 1
    }

    log "INFO" "🗜️  正在解压 JDK..."
    tar -xzf "$jdk_tar" -C "$java_install_dir"
    if [ $? -ne 0 ]; then
        log "ERROR" "❌ JDK 解压失败，请检查压缩包完整性"
        return 1
    fi
    log "INFO" "✅ JDK 解压完成: $jdk_home"

    # 6. 配置 Java 环境变量（✅ 按你原来的方式，不做任何改动）
    log "INFO" "🛠️  配置 Java 环境变量..."
    cat >> /etc/profile << 'EOF'

#set java environment
JAVA_HOME=$JAVA_HOME
CLASSPATH=.:$JAVA_HOME/lib/tools.jar
PATH=$JAVA_HOME/bin:$PATH
export JAVA_HOME CLASSPATH PATH
EOF

    # 替换占位符为实际路径
    sed -i "s|\$JAVA_HOME|$jdk_home|g" /etc/profile

    # 立即在当前会话中生效
    export JAVA_HOME="$jdk_home"
    export CLASSPATH=".:$JAVA_HOME/lib/tools.jar"
    export PATH="$JAVA_HOME/bin:$PATH"

    # 验证 Java 安装
    if java -version >/dev/null 2>&1; then
        log "INFO" "✅ Java 环境配置成功: $(java -version 2>&1 | head -n1)"
    else
        log "ERROR" "❌ Java 环境配置失败，请检查路径或权限"
        return 1
    fi

    # 7. 复制库文件和程序
    log "INFO" "📤 正在复制依赖库和程序文件..."

    cp -r "$source_bag/lib/"* "$lib64_dest"/ && echo "✅ 已复制 lib → /usr/lib64"
    cp -r "$source_bag/usrlib/"* "$lib_dest"/ && echo "✅ 已复制 usrlib → /usr/lib"

    if [ -d "/iflytek" ]; then
        rm -rf "/iflytek"
    fi
    cp -r "$source_bag/iflytek" / && echo "✅ 已复制 iflytek 主程序到根目录"

    mkdir -p "$(dirname "$iflytek_home")"
    if [ ! -L "$iflytek_home" ] && [ ! -d "$iflytek_home" ]; then
        ln -s "/iflytek" "$iflytek_home"
        echo "✅ 创建软链接: $iflytek_home → /iflytek"
    fi

    log "INFO" "✅ 所有文件复制完成"

    # 8. 启动服务
    log "INFO" "▶️  正在启动 iListen 服务..."
    cd "$iflytek_home/iListen" || {
        log "ERROR" "❌ 无法进入服务目录: $iflytek_home/iListen"
        return 1
    }

    nohup ./restart.sh > nohup.out 2>&1 &
    sleep 2

    if pgrep -f "iListen" > /dev/null; then
        log "INFO" "✅ iListen 服务已启动（PID: $(pgrep -f iListen | head -n1)）"
    else
        log "WARN" "⚠️  服务启动但未检测到进程，建议检查日志: $iflytek_home/iListen/nohup.out"
    fi

    log "INFO" "🎉 $service_name 部署成功！"
    return 0
}

#安装fastfds
function fastfds_x86()
{
    # ------------------- 定义变量 -------------------
    local image_tar="/data/temp/ufastdfs.tar.gz"
    local image_name="139.9.60.86:5000/ufastdfs:v1"

    local tracker_container="utracker"
    local storage_container="ustorage"

    local tracker_host_dir="/data/storage/tracker"
    local storage_host_dir="/data/storage/storage"

    log "INFO" "=================================================================="
    log "INFO" "开始部署 FastDFS 服务 (Docker 版, x86)"
    log "INFO" "=================================================================="

    # 自动获取本机 IP（优先使用非 loopback 的 IPv4）
    local host_ip
    host_ip=$(ip -4 addr show scope global | grep -oP '(?<=inet\s)\d+(\.\d+){3}' | head -n1)
    if [[ -z "$host_ip" ]]; then
        host_ip=$(hostname -I | awk '{print $1}')
    fi
    if [[ -z "$host_ip" ]]; then
        log "ERROR" "⛔ 无法获取本机 IP 地址"
        return 1
    fi
    log "INFO" "🌐 检测到本机 IP: $host_ip"

    # ------------------- 检查两个容器是否都已运行 -------------------
    log "INFO" "🔍 检查 FastDFS 容器是否已运行..."
    local tracker_running=false
    local storage_running=false

    if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$tracker_container$"; then
        tracker_running=true
        log "INFO" "✅ 容器 '$tracker_container' 已在运行。"
    fi

    if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$storage_container$"; then
        storage_running=true
        log "INFO" "✅ 容器 '$storage_container' 已在运行。"
    fi

    if [[ "$tracker_running" == true && "$storage_running" == true ]]; then
        log "INFO" "🎉 FastDFS 服务已全部运行，跳过安装。"
        return 0
    fi

    # ------------------- 检查镜像是否存在 -------------------
    log "INFO" "🔍 检查 FastDFS 镜像是否存在..."
    if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "$image_name"; then
        log "INFO" "✅ 镜像 $image_name 已存在。"
    else
        log "WARN" "❌ 镜像 $image_name 不存在，开始加载离线包..."

        if [[ ! -f "$image_tar" ]]; then
            log "ERROR" "⛔ 镜像文件不存在: $image_tar"
            return 1
        fi

        if $sudoset docker load -i "$image_tar"; then
            log "INFO" "🎉 镜像加载成功"
        else
            log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
            return 1
        fi
    fi

    # ------------------- 启动 Tracker 容器 -------------------
    if [[ "$tracker_running" == false ]]; then
        log "INFO" "🚀 正在启动 Tracker 服务: $tracker_container ..."
        $sudoset docker run -d \
            --network=host \
            --name "$tracker_container" \
            --restart=always \
            -v "$tracker_host_dir:/var/fdfs" \
            -v "/etc/localtime:/etc/localtime:ro" \
            "$image_name" \
            tracker

        if [ $? -ne 0 ]; then
            log "ERROR" "⛔ Tracker 容器启动失败"
            return 1
        fi
        log "INFO" "✅ Tracker 启动成功"
    fi

    # ------------------- 启动 Storage 容器 -------------------
    if [[ "$storage_running" == false ]]; then
        log "INFO" "🚀 正在启动 Storage 服务: $storage_container ..."
        $sudoset docker run -d \
            --network=host \
            --name "$storage_container" \
            --restart=always \
            -e "TRACKER_SERVER=$host_ip:22122" \
            -v "$storage_host_dir:/var/fdfs" \
            -v "/etc/localtime:/etc/localtime:ro" \
            "$image_name" \
            storage

        if [ $? -ne 0 ]; then
            log "ERROR" "⛔ Storage 容器启动失败"
            return 1
        fi
        log "INFO" "✅ Storage 启动成功"
    fi

    # ------------------- 等待并验证 -------------------
    log "INFO" "⏳ 等待服务初始化..."
    sleep 8

    local all_running=true
    for container in "$tracker_container" "$storage_container"; do
        if ! $sudoset docker ps --format '{{.Names}}' | grep -wq "^$container$"; then
            log "ERROR" "⛔ 容器 '$container' 启动后未运行"
            all_running=false
        fi
    done

    if [[ "$all_running" == true ]]; then
        log "INFO" "🎉 FastDFS 部署完成！"
        $sudoset docker ps --filter "name=utracker" --filter "name=ustorage" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    else
        log "ERROR" "⛔ 部分容器启动失败，请检查日志: docker logs utracker / ustorage"
        return 1
    fi

    return 0
}

#安装ngrok
function ngrok_x86()
{
    # ------------------- 定义变量 -------------------
    local container_name="ungrok"
    local image_tar="/data/temp/ungrok2.tar.gz"
    local image_name="ngrok:v1"
    local image_id="493b3f514dfd"
    local ngrok_script_host="/data/middleware/ngrok/ngrok.sh"
    local ngrok_script_container="/home/ngrok.sh"

    log "INFO" "=================================================================="
    log "INFO" "开始部署 ngrok 服务 (Docker 版, x86)"
    log "INFO" "=================================================================="

    # ------------------- 检查容器是否已运行 -------------------
    log "INFO" "🔍 检查 ngrok 容器是否已运行..."
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
        log "INFO" "✅ 容器 '$container_name' 已在运行。"
        return 0
    fi

    # ------------------- 检查镜像是否存在 -------------------
    log "INFO" "🔍 检查 ngrok 镜像是否存在..."
    if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "^$image_name$"; then
        log "INFO" "✅ 镜像 $image_name 已存在。"
    else
        log "WARN" "❌ 镜像 $image_name 不存在，开始加载离线包..."

        if [[ ! -f "$image_tar" ]]; then
            log "ERROR" "⛔ 镜像文件不存在: $image_tar"
            return 1
        fi

        if $sudoset docker load -i "$image_tar"; then
            log "INFO" "🎉 镜像加载成功"
        else
            log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
            return 1
        fi

        # 打标签
        $sudoset docker tag "$image_id" "$image_name" 2>/dev/null || true
        log "INFO" "🏷️  镜像已标记为 $image_name"
    fi

    # ------------------- 检查 ngrok 启动脚本 -------------------
    if [[ ! -f "$ngrok_script_host" ]]; then
        log "ERROR" "⛔ ngrok 启动脚本不存在: $ngrok_script_host"
        return 1
    fi

    if [[ ! -x "$ngrok_script_host" ]]; then
        log "WARN" "🔧 ngrok 脚本无执行权限，正在添加..."
        $sudoset chmod +x "$ngrok_script_host"
        if [ $? -ne 0 ]; then
            log "ERROR" "⛔ 无法为 $ngrok_script_host 添加执行权限"
            return 1
        fi
    fi
    log "INFO" "✅ ngrok 启动脚本检查通过"

    # ------------------- 启动 ngrok 容器 -------------------
    log "INFO" "🚀 正在启动 ngrok 容器: $container_name ..."

    $SUDO docker run -d \
        --name "$container_name" \
        --restart=always \
        --network=host \
        -v /etc/localtime:/etc/localtime \
        -v "$ngrok_script_host":/home/ngrok.sh \
        "$image_name"

    if [ $? -ne 0 ]; then
        log "ERROR" "⛔ 容器启动失败"
        return 1
    fi

    log "INFO" "✅ ngrok 容器已启动，等待初始化..."
    sleep 5

    # ------------------- 验证容器状态 -------------------
    if $sudoset docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
        log "INFO" "🎉 ngrok 服务部署完成！"
        $sudoset docker ps --filter "name=$container_name" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    else
        log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs $container_name"
        return 1
    fi

    return 0
}

function nacos_x86() {
    # --- 配置变量 ---
    local container_name="unacos"
    local image_tag="nacos-server:v2.5.2"
    local tar_image="/data/temp/nacos-server-v2.5.2.tar.gz"
    local host_data_dir="/data/middleware/nacos"

    # 🔐 新增认证配置
    local nacos_username="nacos"                    # 用户名
    local nacos_password="dNrprU&2S"               # 密码

    log "INFO" "=================================================================="
    log "INFO" "开始部署 Nacos Server (Standalone) - x86"
    log "INFO" "=================================================================="

    # 自动判断 sudo
    local SUDO=""
    if [[ $(id -u) -ne 0 ]]; then
        if command -v sudo >/dev/null; then
            SUDO="sudo"
        else
            log "ERROR" "⛔ 当前不是 root 用户且无 sudo 权限"
            return 1
        fi
    fi

    # --- 1. 检查容器是否已在运行 ---
    if $SUDO docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
        log "INFO" "✅ 容器 '$container_name' 已在运行，跳过部署。"
        log "INFO" "🎉 访问地址: http://\$IP:8848/nacos (默认账号密码: nacos/nacos)"
        return 0
    fi

    # --- 2. 检查镜像是否存在 ---
    if $SUDO docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "^$image_tag$"; then
        log "INFO" "✅ 镜像 '$image_tag' 已存在，跳过加载。"
    else
        log "INFO" "📦 开始加载离线镜像包: $tar_image"

        # 检查离线包是否存在
        if [[ ! -f "$tar_image" ]]; then
            log "ERROR" "⛔ 镜像文件不存在: $tar_image"
            log "INFO" "请确保已上传 nacos-server-v2.5.1.tar.gz 到 /data/temp/"
            return 1
        fi

        # 加载镜像
        if ! $SUDO docker load -i "$tar_image"; then
            log "ERROR" "⛔ 镜像加载失败，请检查文件是否完整或已损坏"
            return 1
        fi

        log "INFO" "✅ 镜像加载成功"
    fi

    # --- 3. 启动 Nacos 容器 ---
    log "INFO" "🚀 正在启动 Nacos 容器..."

    $SUDO docker run -d \
        --name "$container_name" \
        --restart=always \
        -p 8848:8848 \
        -p 9848:9848 \
        -e MODE=standalone \
        -e NACOS_AUTH_ENABLE=true \
        -e NACOS_AUTH_USERNAME="$nacos_username" \
        -e NACOS_AUTH_PASSWORD="$nacos_password" \
        -e NACOS_AUTH_IDENTITY_KEY="nacos" \
        -e NACOS_AUTH_IDENTITY_VALUE="nacos" \
        -v "$host_data_dir:/home/nacos" \
        --mac-address="02:42:ac:11:00:10" \
        "$image_tag"

    if [[ $? -ne 0 ]]; then
        log "ERROR" "⛔ Nacos 容器启动失败"
        return 1
    fi

    log "INFO" "✅ Nacos 容器已启动，名称: $container_name"

    # --- 4. 状态确认（容器是否仍在运行）---
    sleep 5
    if ! $SUDO docker ps --format '{{.Names}}' | grep -wq "^$container_name$"; then
        log "ERROR" "⛔ 容器启动后退出，请检查日志: $SUDO docker logs $container_name"
        return 1
    fi

    # --- 5. 获取本机 IP 地址 ---
    local server_ip=$(ip addr show | grep -E 'inet\s+(192\.168|172\.1[6789]\.|172\.2[0-9]\.|172\.3[01]\.|10\.)' | awk '{print $2}' | cut -d/ -f1 | head -n1)
    
    if [[ -z "$server_ip" ]]; then
        # 回退到 hostname -I
        server_ip=$(hostname -I | awk '{print $1}')
    fi

    # --- 6. 输出访问信息 ---
    if [[ -n "$server_ip" ]]; then
        log "INFO" "✅ Nacos 部署完成！"
        log "🎉 可通过浏览器访问: http://$server_ip:8848/nacos"
        log "💡 默认账号密码: nacos / nacos"
    else
        log "INFO" "🎉 Nacos 部署完成。"
        log "💡 默认账号密码: nacos / nacos"
        log "📌 无法获取服务器 IP，请手动确认访问地址: http://<your-ip>:8848/nacos"
    fi

    return 0
}

function getLatestVersion() {
      log "INFO" "检查中间件最新版本信息"
     #判断是否能够访问外网，不能就跳过联网检查
      if ! curl -s --head "https://www.baidu.com" | grep "200 OK" > /dev/null; then
          log "WARN" "无法访问外网，跳过最新版本检查"
          return
      fi
      #todo:如果访问不到，就跳过,后面改成用自己的接口服务
     log "INFO" "准备获取最新版本信息"
     latest_nginx_version=$(curl -s "https://nginx.org/en/download.html" | grep -oP 'nginx-\K[0-9]+\.[0-9]+\.[0-9]+' | head -n 1)
     latest_redis_version=$(curl -s "https://api.github.com/repos/redis/redis/releases/latest" | \
                         grep -m 1 '"tag_name"' | \
                         grep -oP '[0-9]+\.[0-9]+\.[0-9]+' )
     latest_mysql_version=$(curl -s https://repo.mysql.com/yum/mysql-8.0-community/el/7/x86_64/ \
        | grep -oP 'mysql-community-server-[0-9]+\.[0-9]+\.[0-9]+' \
        | sort -V \
        | tail -n 1 \
        | grep -oP '[0-9]+\.[0-9]+\.[0-9]+'
        )
      log "INFO" "最新版本信息获取成功"
      log "INFO" "准备获取本地版本信息"
     current_nginx_version=$(docker exec ujava2  sh -c  "/usr/local/nginx/sbin/nginx -v " 2>&1 | grep -oP 'nginx/\K[0-9\.]+')
     current_redis_version=$(docker exec uredis  sh -c  "redis-server --version " 2>&1 | grep -oP 'Redis server v=\K[0-9\.]+')
     current_mysql_version=$(docker exec umysql  sh -c  "mysql --version " 2>&1 | grep -oP 'mysql  Ver \K[0-9\.]+')
      log "INFO" "本地版本信息获取成功, 准备进行版本比较"
    #比较不同的版本情况，不一样的给出告警提示
    if [ "$latest_nginx_version" != "$current_nginx_version" ]; then
        log "WARN" "Nginx版本不一致，最新版本: $latest_nginx_version, 当前版本: $current_nginx_version"
    else
        log "INFO" "Nginx版本一致，当前版本: $current_nginx_version"
    fi

    if [ "$latest_redis_version" != "$current_redis_version" ]; then
        log "WARN" "Redis版本不一致，最新版本: $latest_redis_version, 当前版本: $current_redis_version"
    else
        log "INFO" "Redis版本一致，当前版本: $current_redis_version"
    fi
    if [ "$latest_mysql_version" != "$current_mysql_version" ]; then
        log "WARN" "MySQL版本不一致，最新版本: $latest_mysql_version, 当前版本: $current_mysql_version"
    else
        log "INFO" "MySQL版本一致，当前版本: $current_mysql_version"
    fi
    log "INFO" "版本检查完成"
}

function nginx_x86() {
    # ------------------- 定义变量 -------------------
    local temp_dir="/data/temp"
    local nginx_version="1.29.3"
    local nginx_image="nginx:${nginx_version}"
    local nginx_container_name="unginx"
    local nginx_image_tar="${temp_dir}/nginx-${nginx_version}.tar.gz"
    local required_dirs=(
        "/data/middleware/nginx/log"
        "/data/middleware/nginx/data/cache"
        "/data/middleware/nginx/data/html"
        "/data/middleware/nginx/config"
        "/data/services/web"
        "/data/security/nginx_cert"
    )

    log "INFO" "=================================================================="
    log "INFO" "开始部署 Nginx (离线编译安装) - x86"
    log "INFO" "=================================================================="

    # 目录预检查，缺失则创建
    for dir in "${required_dirs[@]}"; do
        if [ ! -d "$dir" ]; then
            log "WARN" "目录 $dir 不存在，自动创建"
            mkdir -p "$dir" || {
                log "ERROR" "目录 $dir 创建失败"
                return 1
            }
        fi
    done

    #创建nginx用户
    groupadd -r nginx
    useradd -r -g nginx -s /sbin/nologin -d /var/cache/nginx nginx
    #检查是否创建成功
    id nginx
    # 创建目录
    mkdir -p /data/middleware/nginx/data/{cache,html}

    # 统一设置属主为 nginx（假设 Nginx 运行用户为 nginx）
    chown -R nginx:nginx /data/middleware/nginx

    # 设置权限
    chmod -R 755 /data/middleware/nginx/data/html    # 静态文件可读
    chmod -R 775 /data/middleware/nginx/data/cache   # 缓存目录需写入
    chmod -R 755 /data/middleware/nginx/log         # 日志目录可读写

    # 校验镜像包
    if [ ! -s "$nginx_image_tar" ]; then
        log "ERROR" "离线包 $nginx_image_tar 不存在或为空"
        return 1
    fi

    # 加载镜像文件
    if ! docker load -i "$nginx_image_tar"; then
        log "ERROR" "镜像加载失败"
        return 1
    fi
    log "INFO" "镜像加载成功"

    # 检查证书文件是否存在，如果不存在则生成自签名证书
    local cert_dir="/data/security/nginx_cert"
    local cert_file="$cert_dir/server.crt"
    local key_file="$cert_dir/server.key"
    
    if [ ! -f "$cert_file" ] || [ ! -f "$key_file" ]; then
        log "WARN" "证书文件不存在，正在生成自签名证书..."
        
        # 进入证书目录
        cd "$cert_dir" || {
            log "ERROR" "无法进入证书目录 $cert_dir"
            return 1
        }
        
        # 生成自签名证书
        if ! openssl req -x509 \
          -newkey rsa:2048 \
          -nodes \
          -keyout server.key \
          -out server.crt \
          -days 36500 \
          -config san.cnf \
          -extensions v3_req; then
            log "ERROR" "证书生成失败"
            return 1
        fi
        
        log "INFO" "证书生成成功"
    else
        log "INFO" "证书文件已存在，跳过生成"
    fi

    # 检查容器是否存在
    if docker ps -a --format '{{.Names}}' | grep -qw "$nginx_container_name"; then
        log "WARN" "容器 $nginx_container_name 已存在，自动退出部署操作！！！"
        return 0
    fi

    # 生成容器
    if ! docker run -d \
        --name "$nginx_container_name" \
        --restart=always \
        --mac-address "02:42:ac:11:00:25" \
        -v /etc/localtime:/etc/localtime:ro \
        -p 443:443 \
        -v /data/middleware/nginx/config:/etc/nginx/conf.d \
        -v /data/middleware/nginx/data/html:/usr/share/nginx/html \
        -v /data/middleware/nginx/log:/var/log/nginx \
        -v /data/middleware/nginx/data/cache:/var/cache/nginx \
        -v /data/services/web:/data/services/web:rw \
        -v /data/security/nginx_cert:/data/security/nginx_cert:ro \
        "$nginx_image"; then
        log "ERROR" "容器 $nginx_container_name 启动失败"
        return 1
    fi

    log "INFO" "容器 $nginx_container_name 启动成功"
    return 0
}

# 无纸化信令服务部署
function paperless_x86() {
    log "INFO" "=================================================================="
    log "INFO" "开始部署 Paperless 服务 (x86)"
    log "INFO" "=================================================================="

    # --- 配置参数 ---
    local container_name="paperless"
    local image_tar="/data/offline_auto_unifiedPlatform/data/temp/paperless.tar"
    local image_tag="paperless:v1"
    local host_data_dir="/data/third_party/paperless"
    local start_script_host="$host_data_dir/start.sh"
    local port_http="62121"
    local port_nginx="62122"
    local mac_address="02:42:ac:11:00:09"

    # 判断是否需要 sudo
    local SUDO=""
    if [[ $(id -u) -ne 0 ]]; then
        if command -v sudo >/dev/null; then
            SUDO="sudo"
        else
            log "ERROR" "当前不是 root 用户，且未安装 sudo，无法继续"
            return 1
        fi
    fi

    # ------------------- 检查容器是否存在（关键变更）-------------------
    log "INFO" "🔍 检查容器 '$container_name' 是否已存在..."
    if $SUDO docker ps -a --format '{{.Names}}' | grep -qw "$container_name"; then
        log "WARN" "⛔ 容器 '$container_name' 已存在（无论运行与否），自动退出部署操作！！！"
        log "WARN" "如需重新部署，请先手动删除容器：docker rm -f $container_name"
        return 0
    else
        log "INFO" "✅ 容器 '$container_name' 不存在，可以安全部署"
    fi

    # ------------------- 检查本地必要文件 -------------------
    log "INFO" "🔍 检查本地部署文件..."
    if [[ ! -f "$image_tar" ]]; then
        log "ERROR" "缺少镜像文件: $image_tar"
        return 1
    fi
    log "INFO" "✅ 本地所需文件检查通过"

    # ------------------- 检查目标目录是否存在 -------------------
    log "INFO" "🔍 检查目标目录 $host_data_dir 是否存在..."
    if [[ ! -d "$host_data_dir" ]]; then
        log "ERROR" "⛔ 目标目录不存在: $host_data_dir"
        log "ERROR" "请确保 /data/third_party/paperless 目录已提前部署好"
        return 1
    else
        log "INFO" "✅ 目标目录 $host_data_dir 存在，继续部署"
    fi

    # 确保关键脚本存在并有执行权限
    if [[ ! -f "$start_script_host" ]]; then
        log "ERROR" "⛔ 启动脚本不存在: $start_script_host"
        return 1
    fi

    $SUDO chmod +x "$start_script_host" "$host_data_dir/run.sh" 2>/dev/null
    log "INFO" "🔧 已设置脚本执行权限"

    # ------------------- 检查镜像是否已加载 -------------------
    log "INFO" "🔍 检查镜像 $image_tag 是否已存在..."
    if $SUDO docker images --format '{{.Repository}}:{{.Tag}}' | grep -Fxq "$image_tag"; then
        log "INFO" "✅ 镜像 $image_tag 已存在"
    else
        log "INFO" "🔄 加载镜像 $image_tar..."
        if $SUDO docker load -i "$image_tar"; then
            log "INFO" "🎉 镜像加载成功"
        else
            log "ERROR" "镜像加载失败，请检查文件完整性"
            return 1
        fi
    fi

    # ------------------- 启动容器 -------------------
    log "INFO" "🚀 正在启动容器 '$container_name'..."
    $SUDO docker run \
        -itd \
        --privileged \
        --name "$container_name" \
        --mac-address="$mac_address" \
        -p "${port_http}:${port_http}" \
        -p "${port_nginx}:${port_nginx}" \
        -v "$start_script_host:/root/start.sh" \
        -v "$host_data_dir:/var/www/paperless" \
        -v "/etc/localtime:/etc/localtime:ro" \
        --restart=always \
        "$image_tag" \
        /root/start.sh

    if [[ $? -ne 0 ]]; then
        log "ERROR" "⛔ 容器启动失败"
        return 1
    fi

    # ------------------- 验证容器是否正常运行 -------------------
    sleep 5
    if $SUDO docker ps --format '{{.Names}}' | grep -qw "$container_name"; then
        log "INFO" "✅ 容器启动成功"
    else
        log "ERROR" "⛔ 容器启动失败或立即退出"
        log "ERROR" "📄 查看日志: docker logs $container_name"
        $SUDO docker logs "$container_name"
        return 1
    fi

    return 0
}

function cardtable_x86() {
    local service_name="电子桌牌服务 (cardtable)"
    local container_name="cardtable"
    local image_tar="/data/offline_auto_unifiedPlatform/data/temp/uos-cardtable.tar"          # 建议统一放 /data/temp
    local image_tag="uos-cardtable:v1"
    local work_dir="/data/third_party/wifi-local"
    local db_dir="$work_dir/db"

    log "INFO" "=================================================================="
    log "INFO" "📦 正在部署：$service_name"
    log "INFO" "   容器名称: $container_name"
    log "INFO" "   镜像文件: $image_tar"
    log "INFO" "   工作目录: $work_dir"
    log "INFO" "=================================================================="

    # 1. 检查镜像压缩包是否存在
    if [ ! -f "$image_tar" ]; then
        log "ERROR" "❌ 镜像文件不存在: $image_tar"
        log "ERROR" "💡 请确认已上传镜像至 /data/temp/ 目录"
        return 1
    fi
    log "INFO" "✅ 镜像文件已找到: $image_tar"

    # 2. 加载 Docker 镜像
    log "INFO" "🐳 正在加载 Docker 镜像..."
    docker load -i "$image_tar"
    if [ $? -ne 0 ]; then
        log "ERROR" "❌ 镜像加载失败，请检查文件完整性或 Docker 服务状态"
        return 1
    fi
    log "INFO" "✅ 镜像加载成功: $image_tag"

    # 3. 检查工作目录是否存在
    if [ ! -d "$work_dir" ]; then
        log "ERROR" "❌ 工作目录不存在: $work_dir"
        log "ERROR" "💡 请先创建目录并放入配置文件"
        return 1
    fi
    log "INFO" "✅ 工作目录已存在: $work_dir"

    # 4. 设置文件权限（config、脚本、数据库组件）
    log "INFO" "🔐 正在设置文件执行权限..."
    chmod 755 "$work_dir/config.ini" "$work_dir/startDB.sh" "$work_dir/wifi"
    if [ $? -ne 0 ]; then
        log "ERROR" "❌ 权限设置失败：$work_dir 下的文件"
        return 1
    fi

    chmod 755 "$db_dir/mongo" "$db_dir/mongo.config" "$db_dir/mongod"
    if [ $? -ne 0 ]; then
        log "ERROR" "❌ 权限设置失败：$db_dir 下的 MongoDB 组件"
        return 1
    fi
    log "INFO" "✅ 文件权限设置完成"

    # 5. 检查容器是否已存在，若存在则先停止并删除
    if docker ps -a --format '{{.Names}}' | grep -qw "^$container_name$"; then
        log "WARN" "🔄 检测到已有容器 '$container_name'，自动退出部署操作！！！"
        return 0
    fi

    # 6. 创建并运行新容器（使用 host 网络）
    log "INFO" "🚀 正在创建并启动容器 '$container_name'..."
    docker run -itd \
        --name="$container_name" \
        --privileged \
        --network=host \
        --restart=always \
        -v "$work_dir:/home/wifi-local" \
        "$image_tag" \
        /root/start.sh

    if [ $? -ne 0 ]; then
        log "ERROR" "❌ 容器启动失败，请检查镜像、网络或挂载权限"
        return 1
    fi

    # 7. 验证容器是否正常运行
    sleep 3
    if docker ps --format '{{.Names}}' | grep -qw "^$container_name$"; then
        local container_id=$(docker ps --filter "name=$container_name" --format "{{.ID}}")
        log "INFO" "✅ 容器启动成功！"
        log "INFO" "   🏷  名称: $container_name"
        log "INFO" "   🔢  ID: $container_id"
        log "INFO" "   🌐  网络模式: host"
        log "INFO" "   📁  挂载: $work_dir → /home/wifi-local"
        log "INFO" "   🔁  重启策略: always"
    else
        log "ERROR" "❌ 容器启动后立即退出，请使用 'docker logs $container_name' 查看日志"
        return 1
    fi

    return 0
}

#------------------------------服务安装-end------------------------------------------------------------------------------------------------------------------------

#------------------------------文件上传&更改ip-start------------------------------------------------------------------------------------------------------------------------
#-----------------------------------新统一平台系统--------------------------------------------
function check_files() {
    log "INFO" "=================================================================="
    log "INFO" "开始检查并部署 data 目录文件"
    log "INFO" "=================================================================="

    # ------------------- 定义变量 -------------------
    local source_dir="./data"                    # 脚本同级目录下的 data
    local target_dir="/data"

    # ------------------- 检查目标目录是否已完整存在 -------------------
    local -a required_dirs=("bakup" "dockers" "logs" "middleware" "security" "services" "storage" "temp" "third_party")
    local all_dirs_exist=true
    
    if [[ -d "$target_dir" ]]; then
        log "INFO" "📁 检测到目标目录已存在: $target_dir"
        log "INFO" "🔍 检查必需子目录是否完整..."
        
        for dir in "${required_dirs[@]}"; do
            if [[ ! -d "$target_dir/$dir" ]]; then
                log "WARN" "⚠️  缺少目录: $target_dir/$dir"
                all_dirs_exist=false
            else
                log "INFO" "✅ 目录就位: $target_dir/$dir"
            fi
        done
        
        if $all_dirs_exist; then
            log "INFO" "🎉 所有必需目录已存在，跳过文件部署"
            
            # 直接执行 IP 替换脚本，传入新的IP值
            local replace_script="./replace_ip_interactive.sh"
            if [[ -f "$replace_script" && -x "$replace_script" ]]; then
                log "INFO" "🔄 正在执行 IP 替换脚本: $replace_script $server_ip"
                if "$replace_script" "$server_ip"; then
                    log "INFO" "✅ IP 替换脚本执行成功"
                else
                    log "ERROR" "⛔ IP 替换脚本执行失败"
                    return 1
                fi
            elif [[ -f "$replace_script" ]]; then
                log "ERROR" "⛔ 脚本 $replace_script 存在但不可执行，请运行: chmod +x $replace_script"
                return 1
            else
                log "ERROR" "⛔ 脚本 $replace_script 不存在"
                return 1
            fi
            
            log "INFO" "✅ 文件检查完成，准备进入服务安装阶段..."
            sleep 2
            return 0
        else
            log "INFO" "📦 检测到有目录缺失，需要进行文件同步"
        fi
    else
        log "INFO" "📁 目标目录 $target_dir 不存在，需要创建和部署"
        all_dirs_exist=false
    fi

    # ------------------- 如果有任何一个目录不存在，继续执行文件部署逻辑 -------------------
    # ------------------- 检查源目录 -------------------
    if [[ ! -d "$source_dir" ]]; then
        log "ERROR" "⛔ 源目录不存在: $source_dir"
        log "INFO" "请确保当前目录下有 'data' 文件夹，且包含所需内容"
        return 1
    fi

    log "INFO" "📁 检测到源目录: $source_dir"

    # ------------------- 检查目标目录是否已存在 -------------------
    if [[ -d "$target_dir" ]]; then
        log "WARN" "⚠️  目录 $target_dir 已存在，将进行增量同步（保留原数据）"
        log "INFO" "📦 正在同步文件: $source_dir/ → $target_dir/"
        
        # 使用 rsync 增量同步（推荐）或 cp -r
        if command -v rsync &>/dev/null; then
            log "INFO" "🔄 使用 rsync 进行同步..."
            # 捕获标准输出和错误输出
            if rsync -av "$source_dir/" "$target_dir/" 2>&1 | while read -r line; do
                log "INFO" "📄 $line"
            done; then
                log "INFO" "✅ rsync 同步完成"
            else
                log "ERROR" "❌ rsync 同步失败"
                return 1
            fi
        else
            log "WARN" "⚠️  系统未安装 rsync，使用 cp 命令进行同步"
            log "INFO" "📋 开始复制文件列表..."
            local copy_success=true
            find "$source_dir" -type f | while read -r file; do
                relative_path="${file#$source_dir/}"
                target_file="$target_dir/$relative_path"
                mkdir -p "$(dirname "$target_file")" 2>/dev/null
                if cp -f "$file" "$target_file" 2>/dev/null; then
                    log "INFO" "✅ 已复制: $relative_path"
                else
                    log "ERROR" "❌ 复制失败: $relative_path"
                    copy_success=false
                fi
            done
            
            # 复制空目录
            find "$source_dir" -type d | while read -r dir; do
                relative_path="${dir#$source_dir/}"
                [ -n "$relative_path" ] && mkdir -p "$target_dir/$relative_path" 2>/dev/null
            done
            
            if ! $copy_success; then
                log "ERROR" "⛔ 文件复制过程中出现错误"
                return 1
            fi
        fi
    else
        log "INFO" "📁 目标目录 $target_dir 不存在，正在创建并复制..."
        mkdir -p "$(dirname "$target_dir")" || {
            log "ERROR" "⛔ 无法创建父目录"
            return 1
        }
        
        log "INFO" "📋 开始复制整个 data 目录..."
        # 修复：直接复制到目标位置
        if cp -rfv "$source_dir" "$target_dir" 2>&1 | while read -r line; do
            log "INFO" "📄 $line"
        done; then
            log "INFO" "✅ 成功复制 $source_dir → $target_dir"
        else
            log "ERROR" "⛔ 复制失败，请检查权限"
            return 1
        fi
    fi

    # ------------------- 再次验证关键目录是否存在 -------------------
    log "INFO" "🔍 同步完成后再次验证目录完整性..."
    all_dirs_exist=true
    for dir in "${required_dirs[@]}"; do
        if [[ ! -d "$target_dir/$dir" ]]; then
            log "WARN" "🔍 缺少目录: $target_dir/$dir，部署可能不完整"
            all_dirs_exist=false
        else
            log "INFO" "✅ 目录就位: $target_dir/$dir"
        fi
    done

    if ! $all_dirs_exist; then
        log "ERROR" "⛔ 文件部署后仍然缺少必需目录，请检查源数据完整性"
        return 1
    fi

    # ------------------- 执行 IP 替换脚本 -------------------
    local replace_script="./replace_ip_interactive.sh"
    if [[ -f "$replace_script" && -x "$replace_script" ]]; then
        log "INFO" "🔄 正在执行 IP 替换脚本: $replace_script"
        if "$replace_script"; then
            log "INFO" "✅ IP 替换脚本执行成功"
        else
            log "ERROR" "⛔ IP 替换脚本执行失败"
            return 1
        fi
    elif [[ -f "$replace_script" ]]; then
        log "ERROR" "⛔ 脚本 $replace_script 存在但不可执行，请运行: chmod +x $replace_script"
        return 1
    else
        log "ERROR" "⛔ 脚本 $replace_script 不存在"
        return 1
    fi

    log "INFO" "✅ 文件部署完成：$source_dir → $target_dir"
    log "INFO" "准备进入服务安装阶段..."
    sleep 2
    return 0
}
#------------------------------文件上传&更改ip-end--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

# ========================================
# ✅ 函数：安全添加定时任务（带脚本存在性检查）
# 每 3 分钟执行一次 jvm 监控脚本
# ========================================
function add_crontab_job() {
    local script_path="/data/services/scripts/ujava2-startup.sh"
    local log_path="/var/log/ujava2-cron.log"
    local job="*/3 * * * * $script_path >> $log_path 2>&1"
    local user=$(whoami)
    local crontab_file="/tmp/crontab.tmp.$RANDOM"

    log "INFO" "🔧 正在添加定时任务: $job"

    # 1. 检查 cron 服务是否运行（CentOS/RHEL/Ubuntu 通用）
    if ! systemctl is-active --quiet cron 2>/dev/null && ! systemctl is-active --quiet crond 2>/dev/null; then
        log "WARN" "⚠️  cron 服务未运行，正在尝试启动..."
        if command -v systemctl &>/dev/null; then
            sudo systemctl start crond || sudo systemctl start cron
            sudo systemctl enable crond || sudo systemctl enable cron
        else
            service cron start || service crond start
            chkconfig cron on || chkconfig crond on
        fi
    fi

    # 2. 检查脚本文件是否存在
    if [[ ! -f "$script_path" ]]; then
        log "ERROR" "⛔ 脚本文件不存在: $script_path"
        log "ERROR" "💡 请确认路径正确，并已部署 ujava2-startup.sh"
        return 1
    fi

    # 3. 检查脚本是否具有可执行权限
    if [[ ! -x "$script_path" ]]; then
        log "WARN" "⚠️  脚本无执行权限，正在添加执行权限: chmod +x $script_path"
        chmod +x "$script_path"
        if [[ $? -ne 0 ]]; then
            log "ERROR" "⛔ 无法为 $script_path 添加执行权限，请检查文件权限或使用 root 用户"
            return 1
        fi
    fi

    # 4. 确保日志目录存在
    local log_dir=$(dirname "$log_path")
    if [[ ! -d "$log_dir" ]]; then
        log "INFO" "📁 日志目录不存在，正在创建: $log_dir"
        mkdir -p "$log_dir"
    fi

    # 5. 备份当前 crontab
    crontab -l > "$crontab_file" 2>/dev/null || {
        log "INFO" "📋 当前用户 $user 无现有定时任务，将创建新的"
        > "$crontab_file"
    }

    # 6. 检查是否已存在该任务（避免重复）
    if grep -Fq "$script_path" "$crontab_file"; then
        log "INFO" "✅ 定时任务已存在，跳过添加"
        rm -f "$crontab_file"
        return 0
    fi

    # 7. 追加新任务
    echo "$job" >> "$crontab_file"

    # 8. 写入 crontab
    crontab "$crontab_file"
    if [[ $? -eq 0 ]]; then
        log "INFO" "🎉 定时任务添加成功"
    else
        log "ERROR" "⛔ 定时任务添加失败，请检查权限"
        rm -f "$crontab_file"
        return 1
    fi

    # 9. 清理临时文件
    rm -f "$crontab_file"

    # 10. 显示当前任务（过滤空行和注释）
    log "INFO" "📋 当前定时任务列表："
    crontab -l | grep -v "^$" | grep -v "^#" | sed 's/^/   → /'

    return 0
}

#------------------------------全局配置-start--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
###服务器检测配置项
#mem-check
#最小内存大小G
min_memory_value=8
#最小有效内存大小G
min_memory_available=8
#disk-check
#最小硬盘资源大小G
#min_disk_value=200
#最小有效硬盘大小G
min_disk_available=50

#默认为空，如果为非root用户执行，则需要配置为sudo
sudoset=""
#------------------------------全局配置-end--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
#-------------------------------脚本执行-start--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
sys1="ubains.txt"
sys2="devops.txt"
sys3="ubains_devops.txt"
checkOS
if [ -e "$PWD/$sys1" ]; then
  txt="检测该脚本为【预定系统 部署】"
  version="预定系统版本"
elif [ -e "$PWD/$sys2" ]; then
  txt="检测该脚本为【运维系统 部署】"
  version="运维系统版本"
elif [ -e "$PWD/$sys3" ]; then
  txt="检测该脚本为【预定系统+运维系统 部署】"
  version="预定系统+运维系统版本"
fi

#-------------------------------获取服务器ip-start--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
#文件默认ip
ip='192.168.9.84'
IP=$(hostname -I | awk '{print $1}')
function server_ip()
{
	# 弹出弹框，并根据用户选择执行相应操作
	choice=$(whiptail --title "请选择操作" \
					  --menu "请确认服务器ip是否正确:$IP" 15 50 4 \
					  1 "是" \
					  2 "否" \
					  3 "退出" \
					  3>&1 1>&2 2>&3)

	# 根据用户选择执行相应操作
	case $choice in
		1)
			# 在这里编写操作1的代码逻辑
			;;
		2)
			# 设置对话框的标题和提示信息
			dialog_title="请输入服务器IP："
			dialog_prompt="请输入服务器IP："
			# 使用whiptail显示输入框，并将用户输入的值保存到变量input_value
			IP=$(whiptail --title "$dialog_title" --inputbox "$dialog_prompt" 10 50 3>&1 1>&2 2>&3)
			# 在这里编写操作2的代码逻辑
			;;
		3)
			exit
			# 在这里编写操作3的代码逻辑
			;;
		*)
			echo "无效的选择"
			;;
	esac
}
#-------------------------------获取服务器的ip-end-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#-------------------------------服务部署选择-start--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
function deploy_services() {
    local IP=$(hostname -I | awk '{print $1}' | cut -d' ' -f1)

    # ========================================
    # 🧩 定义服务映射表：支持多个函数
    # 格式：[编号]="函数1 函数2 ..."（用空格分隔）
    # ========================================
    declare -A SERVICE_MAP=(
        [1]="java_x86"
        [2]="python_x86"
        [3]="iListen_x86 python_voice_x86"          # 👈 重点：语音系统 = 多个函数
        [4]="paperless_x86"
        [5]="cardtable_x86"
    )

    declare -A SERVICE_LABEL=(
        [1]="预定系统"
        [2]="运维集控系统"
        [3]="语音转录系统"
        [4]="无纸化信令服务"
        [5]="电子桌牌服务"
    )

    declare -A SERVICE_DESC=(
        [1]="Java 后端服务"
        [2]="Python 运维控制平台"
        [3]="语音识别引擎 + 转录后端"
        [4]="Paperless 信令转发"
        [5]="电子桌牌通信服务"
    )

    log "INFO" "🔍 正在检查 SERVICE_MAP 中引用的所有函数..."

    local all_funcs=()
    for func_list in "${SERVICE_MAP[@]}"; do
        for func_name in $func_list; do  # 注意：这里不要加引号，让其展开
            all_funcs+=("$func_name")
        done
    done

    local missing=0 total=0
    for func_name in "${all_funcs[@]}" ; do
        ((total++))
        if declare -f "$func_name" > /dev/null 2>&1; then
            log "INFO" "✅ 函数已定义: $func_name"
        else
            log "ERROR" "❌ 函数未定义: $func_name"
            ((missing++))
        fi
    done

    if [ $missing -ne 0 ]; then
        log "ERROR" "⛔ 缺少 $missing 个函数，无法继续部署。"
        return 1
    fi

    # 构建 whiptail 选项
    local options=()
    for i in {1..5}; do
        options+=(
            "$i" 
            "${SERVICE_LABEL[$i]}  (${SERVICE_DESC[$i]})" 
            "off"
        )
    done

    # 显示多选菜单（第一级界面）
    local choices
    choices=$(whiptail --title "🔹 服务部署中心" \
                       --backtitle "当前服务器 IP: $IP" \
                       --checklist "\n请选择要部署的服务（使用空格勾选，支持多选）：" \
                       18 80 6 \
                       "${options[@]}" \
                       3>&1 1>&2 2>&3)

    # 第一级界面取消：退出整个部署操作
    if [ $? -ne 0 ]; then
        log "INFO" "用户在第一级界面取消部署"
        whiptail --title "↩️ 退出部署" --msgbox "已取消服务部署操作。" 8 40
        return 1  # 返回1表示用户主动取消
    fi

    if [ -z "$choices" ]; then
        if whiptail --title "ℹ️ 提示" --yesno "未选择任何服务，是否重新选择？" 10 50; then
            # 用户选择重新选择，递归调用
            deploy_services
            return $?
        else
            # 用户选择退出
            whiptail --title "↩️ 退出部署" --msgbox "已取消服务部署操作。" 8 40
            return 1
        fi
    fi

    # 去除多余引号，确保是干净的数字序列
    choices=$(echo "$choices" | sed 's/"//g' | xargs)

    # 显示用户选择的服务
    local selected_list=""
    for choice in $choices; do
        if [[ -n "${SERVICE_LABEL[$choice]}" ]]; then
            selected_list+="${SERVICE_LABEL[$choice]}\n"
        else
            log "ERROR" "⚠️ 未知服务编号: $choice"
        fi
    done

    # 确认部署（第二级界面）
    if ! whiptail --title "✅ 确认部署" \
                  --yesno "您选择了以下服务进行部署：\n\n${selected_list}\n是否确认开始部署？" \
                  14 60; then
        log "INFO" "用户在第二级界面取消，返回服务选择界面"
        whiptail --title "🔄 重新选择" --msgbox "将返回服务选择界面..." 8 40
        # 第二级取消：递归调用返回第一级界面
        deploy_services
        return $?
    fi

    whiptail --title "🚀 开始部署" --msgbox "正在启动部署流程，请查看终端日志..." 10 50

    # --- 开始部署 ---
    local success_count=0
    local total_count=0

    for choice in $choices; do
        if ! [[ "$choice" =~ ^[1-5]$ ]]; then
            log "ERROR" "⚠️ 无效选项: $choice"
            continue
        fi

        local func_list="${SERVICE_MAP[$choice]}"
        local label="${SERVICE_LABEL[$choice]}"

        if [[ -z "$label" ]]; then
            log "ERROR" "⚠️ 未知服务编号: $choice"
            continue
        fi

        log "INFO" "=================================================================="
        log "INFO" "🚀 正在部署: $label"
        log "INFO" "=================================================================="

        # 👇 执行该服务关联的每一个函数
        local service_success=true
        for func_name in $func_list; do  # 注意：不要加引号
            if declare -f "$func_name" > /dev/null; then
                log "INFO" "▶️  执行函数: $func_name"
                "$func_name"
                if [ $? -ne 0 ]; then
                    log "ERROR" "❌ 函数执行失败: $func_name"
                    service_success=false
                    break  # 可选：是否继续执行下一个函数？
                else
                    log "INFO" "✅ $func_name 执行成功"
                fi
            else
                log "ERROR" "❌ 未找到函数: $func_name"
                service_success=false
                break
            fi
        done

        if [ "$service_success" = true ]; then
            log "INFO" "✅ $label 部署成功"
            ((success_count++))
        else
            log "ERROR" "❌ $label 部署失败"
            whiptail --title "❌ 部署失败" \
                    --msgbox "服务 [$label] 部署失败，请检查日志后重试。" \
                    12 60 || echo "警告：无法显示弹窗"
        fi
    done

    # 最终统计
    local fail_count=$(( $(echo $choices | wc -w) - success_count ))
    log "INFO" "=================================================="
    log "INFO" "🎯 部署完成：成功 $success_count 个服务，失败 $fail_count 个"
    log "INFO" "=================================================="

    return $(( fail_count > 0 ? 1 : 0 ))
}
#-------------------------------服务部署选择-end--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#-------------------------------部署系统-------------------------------
    log "INFO" "\033[36m
---------------------------------------------开始部署系统---------------------------------------------------
    \033[0m"
	    server_ip
        server_ip="$IP"
        #自动化脚本路径
        auto_java=$PWD
        add_user
        #检测文件
        File_detection_java
        #开放防火墙
        firewalldjava
        check_files
        # 安装服务
        docker_x86
        mysql_x86
        redis_x86
        emqx_x86
        fastfds_x86
        ngrok_x86
        nacos_x86
        nginx_x86
        ntp_uos
        deploy_jdk_host
        start_java_meeting_service
        malan
        deploy_services
        add_crontab_job
	    #检查中间件版本
        getLatestVersion
#___________________________________________________________________________________

log "INFO" "正在重启docker服务，请耐心等待。"
$sudoset systemctl restart docker
sleep 30
log "INFO" "查看服务是否启动成功——————docker ps"
log "INFO" "系统服务：umysql,uredis,ujava2，uemqx，ustorage，utracker "
$sudoset docker ps