import paho.mqtt.client as mqtt
import logging
import threading
import requests
import json
import os
import time
from queue import Queue
from tqdm.auto import tqdm  # 使用auto版本自动适配环境
import urllib3

# 禁用SSL警告
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# 确保日志级别设置为INFO或更低
logging.basicConfig(
    level=logging.INFO,  # 确保不是 logging.WARNING 或更高
    format='%(asctime)s - %(levelname)s - %(message)s'
)


# 全局配置
message_queue = Queue()
THREAD_COUNT = 110  # 减少线程数量便于观察进度
DOWNLOAD_DIR = "downloads"
os.makedirs(DOWNLOAD_DIR, exist_ok=True)


def test_progress():
    """
    测试进度显示函数

    该函数用于模拟和测试进度显示功能，通过循环打印从10%到100%的进度信息，
    每次间隔0.2秒，用于验证进度显示功能是否正常工作。

    参数:
        无

    返回值:
        无
    """
    logging.info("\n=== 进度测试 ===")

    # 模拟进度从10%到100%的变化过程
    for i in range(1, 11):
        time.sleep(0.2)
        logging.info(f"测试进度: {i * 10}%")

    logging.info("=== 测试完成 ===\n")


def on_connect(client, userdata, flags, rc):
    """
    MQTT连接成功回调函数

    当客户端成功连接到MQTT代理时触发该回调函数。

    Args:
        client: MQTT客户端实例
            - 用于执行MQTT相关操作（如订阅主题）
        userdata: 用户自定义数据
            - 通常为初始化时传入的用户数据
        flags: 连接标志
            - 代理返回的连接标志信息
        rc: 连接结果码
            - 0: 连接成功
            - 其他值: 连接失败（具体错误码参考MQTT协议）
    Returns:
        None
    """
    logging.info("MQTT连接成功")
    # 连接成功后立即订阅指定主题
    client.subscribe("/androidPanel/")


def on_message(client, userdata, msg):
    """
    MQTT消息接收回调函数，处理接收到的消息并加入消息队列

    Args:
        client: MQTT客户端实例
        userdata: 用户自定义数据，通常由Client()设置
        msg: MQTT消息对象，包含topic、payload等信息

    Returns:
        None: 无返回值
    """
    try:
        # 解析JSON格式的消息内容并放入消息队列
        payload = json.loads(msg.payload.decode())
        message_queue.put(payload)
        logging.info(f"消息已加入队列: UDID={payload['udid'][0]}")
    except Exception as e:
        # 捕获并记录消息处理过程中的任何异常
        logging.error(f"消息解析失败: {e}")


def download_file(url, save_path, desc="Downloading", max_retries=3, retry_delay=5):
    """
    增强版文件下载函数，支持断点续传和自动重试机制

    参数:
        url (str): 要下载的文件URL地址
        save_path (str): 本地保存路径
        desc (str): 下载进度描述文本，默认为"Downloading"
        max_retries (int): 最大重试次数，默认为3
        retry_delay (int): 重试间隔时间(秒)，默认为5

    返回值:
        bool: 下载成功返回True，失败返回False

    功能说明:
        1. 支持断点续传，自动检测已下载部分并继续下载
        2. 自动验证文件完整性
        3. 提供详细的下载进度和速度信息
        4. 支持失败自动重试机制
    """
    for attempt in range(max_retries):
        try:
            # 断点续传处理：检查本地是否存在部分下载文件
            file_size = 0
            if os.path.exists(save_path):
                file_size = os.path.getsize(save_path)
                logging.info(f"发现未完成下载文件，尝试续传: {save_path} (已下载: {file_size / 1024 / 1024:.2f}MB)")

            # 设置HTTP请求头，支持断点续传
            headers = {'Range': f'bytes={file_size}-'} if file_size else {}

            # 打印下载基本信息
            logging.info(f"\n开始下载: {url}")
            logging.info(f"保存路径: {os.path.abspath(save_path)}")
            logging.info(f"已下载: {file_size / 1024 / 1024:.2f}MB")

            # 发起HTTP请求并处理响应
            with requests.get(
                    url,
                    headers=headers,
                    stream=True,
                    verify=False,
                    timeout=(30, 300)
            ) as r:
                r.raise_for_status()

                # 计算文件总大小(包括已下载部分)
                total_size = int(r.headers.get('content-length', 0)) + file_size
                if total_size == 0:
                    raise ValueError("无法获取文件总大小")

                logging.info(f"总大小: {total_size / 1024 / 1024:.2f}MB")
                logging.info(f"创建时间: {time.strftime('%Y-%m-%d %H:%M:%S')}")

                # 设置文件写入模式(追加或新建)
                mode = 'ab' if file_size else 'wb'
                downloaded = file_size
                last_update = time.time()

                with open(save_path, mode) as f:
                    # 测试文件可写性
                    test_data = b"file_header_test"
                    f.write(test_data)
                    f.flush()
                    os.fsync(f.fileno())
                    logging.info(f"文件写入测试成功，当前大小: {os.path.getsize(save_path)} bytes")

                    # 分块下载并写入文件
                    for chunk in r.iter_content(chunk_size=8192):
                        if chunk:
                            # 写入文件并立即同步到磁盘
                            f.write(chunk)
                            f.flush()
                            os.fsync(f.fileno())

                            downloaded += len(chunk)

                            # 实时打印文件状态
                            current_size = os.path.getsize(save_path)
                            logging.info(f"\r当前文件大小: {current_size} bytes ({current_size / 1024 / 1024:.2f}MB)",
                                         end="",
                                         flush=True)

                            # 进度显示(每0.1秒更新一次)
                            if time.time() - last_update >= 0.1:
                                percent = (downloaded / total_size) * 100
                                speed = (downloaded - file_size) / (time.time() - last_update) / 1024
                                logging.info(f"\n{desc} 进度: {percent:.1f}% | 速度: {speed:.1f}KB/s")
                                last_update = time.time()

                    # 最终文件完整性验证
                    final_size = os.path.getsize(save_path)
                    logging.info(f"\n下载完成! 最终文件大小: {final_size} bytes")
                    if final_size != total_size:
                        raise ValueError(f"文件大小不匹配: {final_size} != {total_size}")

                return True

        except Exception as e:
            # 异常处理：记录错误并等待重试
            logging.exception(f"下载失败 (尝试 {attempt + 1}/{max_retries}): {str(e)}")
            if attempt < max_retries - 1:
                time.sleep(retry_delay * (attempt + 1))
                continue
            return False


def worker_thread():
    """
    工作线程函数，用于从消息队列中获取任务并处理下载请求

    功能描述:
    - 从消息队列中获取消息载荷(payload)
    - 构建下载请求的URL和请求头
    - 创建会话并设置重试机制
    - 下载文件并显示进度信息
    - 处理各种异常情况并记录日志

    处理流程:
    1. 从消息队列获取任务
    2. 提取授权信息和设备ID
    3. 配置下载请求参数
    4. 创建带有重试机制的HTTP会话
    5. 执行下载并保存文件
    6. 处理异常和重试逻辑
    """
    while True:
        payload = message_queue.get()
        try:
            # 从消息载荷中提取必要信息
            actual_auth = payload["headers"]["Authorization"]
            udid = payload["udid"][0]

            # 配置下载pak请求URL和请求头
            url = "http://192.168.5.229:8999/androidPanel/pack/download?id=29"
            # 配置下载apk请求URL和请求头
            # url = "http://192.168.5.229:8999/androidPanel/pack/download?id=30"

            headers = {
                "Content-Type": "application/x-www-form-urlencoded",
                'Authorization': actual_auth,
                'RandomCode': 'KuYKL30nDA6QmIKeysgbM6qpali4YB1EqklwuOhAtS5Hzoc4fvkvAiTRtcuz',
                'X-SIGN': 'xjcyiFmPtQNNrU6uGxSLeQQWM7+PW9Q445EPAeaU6EH07V3KKvTigeCg5visZaI3IavQhMIhVMLiJEUyPDEfAIyPfNZGPex1C+8fSYoAk9Q=',
                'X-TIMESTAMP': '1717749773',
                'X-RANDOM': 'TbSyCQyC5efAT5dRNs7RxWMg',
                'Connection': 'keep-alive'
            }

            # 准备文件保存路径
            file_name = f"{udid}_downloaded_file.bin"
            save_path = os.path.abspath(os.path.join(DOWNLOAD_DIR, file_name))

            # 创建带有重试机制的HTTP会话
            session = requests.Session()
            adapter = requests.adapters.HTTPAdapter(
                max_retries=3,
                pool_connections=20,
                pool_maxsize=20,
                pool_block=True
            )
            session.mount('http://', adapter)
            session.mount('https://', adapter)

            # 下载文件(带重试机制)
            max_retries = 3
            for attempt in range(max_retries):
                try:
                    # 执行下载请求
                    with session.post(
                            url,
                            headers=headers,
                            stream=True,
                            verify=False,
                            timeout=(30, 300),
                    ) as response:
                        response.raise_for_status()

                        # 处理文件下载
                        total_size = int(response.headers.get('content-length', 0))
                        if total_size == 0:
                            raise ValueError("无法获取文件总大小")

                        os.makedirs(DOWNLOAD_DIR, exist_ok=True)

                        # 下载文件并显示进度
                        downloaded = 0
                        start_time = time.time()

                        with open(save_path, 'wb') as f:
                            for chunk in response.iter_content(chunk_size=8192):
                                if chunk:
                                    f.write(chunk)
                                    downloaded += len(chunk)
                                    if downloaded % (1024 * 1024) == 0:
                                        speed = downloaded / (time.time() - start_time) / 1024
                                        logging.info(
                                            f"{udid} 进度: {downloaded / 1024 / 1024:.1f}MB/"
                                            f"{total_size / 1024 / 1024:.1f}MB | "
                                            f"速度: {speed:.1f}KB/s"
                                        )

                        logging.info(f"{udid} 下载完成! 大小: {downloaded} bytes")
                        break

                except (requests.exceptions.RequestException, ValueError) as e:
                    # 处理下载失败和重试逻辑
                    if attempt == max_retries - 1:
                        logging.error(f"{udid} 下载最终失败: {str(e)}")
                        raise
                    wait_time = (attempt + 1) * 10
                    logging.warning(f"{udid} 下载失败，{wait_time}秒后重试... ({attempt + 1}/{max_retries})")
                    time.sleep(wait_time)

        except Exception as e:
            # 处理线程中的其他异常
            logging.exception(f"{udid} 线程处理异常: {str(e)}")
        finally:
            # 标记任务完成
            message_queue.task_done()


if __name__ == "__main__":
    logging.info("=== 程序启动 ===")
    logging.info(f"当前工作目录: {os.getcwd()}")
    logging.info(f"下载目录: {os.path.abspath(DOWNLOAD_DIR)}")

    # 测试日志系统
    logging.info("日志系统测试 - INFO级别")
    logging.warning("日志系统测试 - WARNING级别")
    # 先测试进度条功能
    test_progress()

    # 启动线程池
    for i in range(THREAD_COUNT):
        threading.Thread(
            target=worker_thread,
            name=f"Worker-{i + 1}",
            daemon=True
        ).start()
    logging.info(f"已启动 {THREAD_COUNT} 个工作线程")

    # MQTT客户端
    client = mqtt.Client()
    client.username_pw_set("mqtt@cmdb", "mqtt@webpassw0RD")
    client.on_connect = on_connect
    client.on_message = on_message

    try:
        client.connect("192.168.5.229", 1883)
        logging.info("MQTT客户端已启动，等待消息...")
        client.loop_forever()
    except KeyboardInterrupt:
        logging.info("程序终止")
    finally:
        client.disconnect()
