# -*- coding: utf-8 -*-
"""
根据模板用例 Excel，在同一个工作簿中新建 Sheet，并从 PRD(Markdown) 自动生成的 JSON 用例写入。

需求对齐（Docs/PRD/_PRD_根据PRD生成用例.md）：
- PRD 文档统一路径：Docs/开发PRD目录（仓库实际目录名可能为 Docs/PRD，下面做了兼容）
- 执行脚本时通过交互型输入需要生成测试用例的PRD文档编号
- 若多选，则生成在同一个 sheet 表中；sheet 名 = 交互输入的“测试用例名称”
- 生成的测试用例文件名：交互输入名称 + 时间戳
"""

import os
import re
import json
import argparse
from copy import copy
from datetime import datetime
from dataclasses import dataclass
from typing import List, Dict

from openpyxl import load_workbook
from openpyxl.styles import Alignment


# ===== 1) 路径与表头 =====
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
REPO_ROOT = os.path.abspath(os.path.join(BASE_DIR, "..", ".."))

# ✅ 兼容两种目录：Docs/开发PRD（PRD里写的） & Docs/PRD（仓库里常见）
PRD_DIR_CANDIDATES = [
    os.path.join(REPO_ROOT, "Docs", "开发PRD"),
    os.path.join(REPO_ROOT, "Docs", "PRD"),
]

TEMPLATE_PATH_DEFAULT = os.path.join(BASE_DIR, "用例文件", "兰州中石化项目测试用例20251203.xlsx")
OUTPUT_JSON_DIR = os.path.join(BASE_DIR, "config")
OUTPUT_XLSX_DIR = os.path.join(BASE_DIR, "用例文件")

headers_order = [
    "序号", "功能模块", "功能类别", "用例编号", "功能描述", "用例等级",
    "功能编号", "用例名称", "预置条件", "操作步骤", "JSON", "预期结果",
    "测试结果", "测试结论", "日志截屏", "备注",
]


def resolve_prd_dir() -> str:
    for d in PRD_DIR_CANDIDATES:
        if os.path.isdir(d):
            return d
    # 默认返回第一个（后续会报错提示）
    return PRD_DIR_CANDIDATES[0]


# ===== 2) PRD 抽取与用例生成（轻量规则）=====
@dataclass
class RequirementItem:
    code: str
    title: str
    detail_lines: List[str]


RE_SECTION = re.compile(r"^\s*(?P<code>\d+(?:\.\d+)*)\s*[、\.\-]\s*(?P<title>.+?)\s*[:：]?\s*$")
RE_MD_HEADING = re.compile(r"^\s*(#{1,6})\s+(?P<title>.+?)\s*$")


def normalize_md(text: str) -> str:
    text = text.replace("\r\n", "\n").replace("\r", "\n")
    text = re.sub(r"```.*?```", "", text, flags=re.S)
    text = re.sub(r"<!--.*?-->", "", text, flags=re.S)
    return text


def clean_line(line: str) -> str:
    line = line.strip()
    line = re.sub(r"^[\-\*\+]\s+", "", line)
    line = re.sub(r"\s+", " ", line)
    return line


def extract_requirement_items(md_text: str) -> List[RequirementItem]:
    lines = md_text.split("\n")
    items: List[RequirementItem] = []

    current_code = ""
    current_title = ""
    current_detail: List[str] = []

    def flush():
        nonlocal current_code, current_title, current_detail
        if current_title:
            detail = [clean_line(x) for x in current_detail if clean_line(x)]
            items.append(RequirementItem(code=current_code, title=clean_line(current_title), detail_lines=detail))
        current_code = ""
        current_title = ""
        current_detail = []

    for raw in lines:
        line = raw.rstrip("\n")

        m = RE_SECTION.match(line)
        if m:
            flush()
            current_code = m.group("code").strip()
            current_title = m.group("title").strip()
            continue

        mh = RE_MD_HEADING.match(line)
        if mh:
            if current_title:
                current_detail.append(line)
            else:
                flush()
                current_code = ""
                current_title = mh.group("title").strip()
            continue

        if current_title:
            current_detail.append(line)

    flush()
    items = [it for it in items if it.title and len(it.title) >= 2]
    return items


def classify_category(title: str, detail_lines: List[str]) -> str:
    text = (title + " " + " ".join(detail_lines)).lower()
    if any(k in text for k in ["异常", "失败", "错误", "告警", "报警", "超时", "断开", "暴涨"]):
        return "异常场景"
    if any(k in text for k in ["安全", "权限", "鉴权", "加密", "脱敏"]):
        return "安全/鉴权"
    if any(k in text for k in ["报告", "输出", "word", "markdown", "邮件", "钉钉", "通知"]):
        return "运维可观测"
    return "功能测试"


def decide_priority(title: str, detail_lines: List[str]) -> str:
    text = title + " " + " ".join(detail_lines)
    if any(k in text for k in ["必须", "报警", "告警", "峰值", "暴涨", "发送", "对接"]):
        return "高"
    if any(k in text for k in ["待实现", "可以", "建议", "优化"]):
        return "中"
    return "中"


def build_steps(detail_lines: List[str]) -> str:
    candidates: List[str] = []
    for ln in detail_lines:
        s = clean_line(ln)
        if not s:
            continue
        if any(k in s for k in ["检查", "监测", "记录", "输出", "发送", "查询", "进入", "生成", "判定", "调用", "执行"]):
            candidates.append(s)

    uniq: List[str] = []
    for c in candidates:
        if c not in uniq:
            uniq.append(c)

    if not uniq:
        return "1. 按 PRD 描述执行该功能/流程; 2. 采集相关日志/输出; 3. 记录实际结果"
    uniq = uniq[:6]
    return "; ".join([f"{i+1}. {x}" for i, x in enumerate(uniq)])


def build_expected(title: str, detail_lines: List[str]) -> str:
    candidates: List[str] = []
    for ln in detail_lines:
        s = clean_line(ln)
        if not s:
            continue
        if any(k in s for k in ["需要", "应", "必须", "输出", "记录", "发送", "判定", "成功", "失败"]):
            candidates.append(s)

    uniq: List[str] = []
    for c in candidates:
        if c not in uniq:
            uniq.append(c)

    if not uniq:
        return f"{title} 按 PRD 约定产出正确结果（请补充具体断言点）"
    uniq = uniq[:6]
    return "; ".join(uniq)


def make_case_id(prefix: str, idx: int) -> str:
    return f"{prefix}-{idx:03d}"


def to_case_record(idx: int, module: str, prefix: str, req: RequirementItem) -> Dict[str, str]:
    record: Dict[str, str] = {
        "序号": idx,
        "功能模块": module,
        "功能类别": classify_category(req.title, req.detail_lines),
        "用例编号": make_case_id(prefix, idx),
        "功能描述": req.title,
        "用例等级": decide_priority(req.title, req.detail_lines),
        "功能编号": req.code or "",
        "用例名称": req.title,
        "预置条件": "1. 环境已部署; 2. 具备执行权限; 3. 必要依赖/配置已准备",
        "操作步骤": build_steps(req.detail_lines),
        "JSON": "",
        "预期结果": build_expected(req.title, req.detail_lines),
        "测试结果": "",
        "测试结论": "",
        "日志截屏": "",
        "备注": "",
    }
    for k in headers_order:
        record.setdefault(k, "")
    return record


def prd_to_cases(prd_path: str, module: str, prefix: str, start_idx: int = 1) -> List[Dict[str, str]]:
    with open(prd_path, "r", encoding="utf-8") as f:
        md = normalize_md(f.read())

    items = extract_requirement_items(md)
    cases: List[Dict[str, str]] = []

    idx = start_idx
    for it in items:
        if any(k in it.title for k in ["说明", "目录", "背景", "概述"]):
            continue
        cases.append(to_case_record(idx=idx, module=module, prefix=prefix, req=it))
        idx += 1

    return cases


# ===== 3) Excel 写入 =====
def find_header_row(template_sheet) -> int:
    must_keys = {"序号", "用例名称", "操作步骤", "预期结果"}
    max_scan = min(30, template_sheet.max_row or 30)

    for r in range(1, max_scan + 1):
        values = []
        for c in range(1, min(40, template_sheet.max_column or 40) + 1):
            v = template_sheet.cell(row=r, column=c).value
            if v is None:
                continue
            values.append(str(v).strip())
        if len(must_keys.intersection(values)) >= 2:
            return r
    return 3


def safe_sheet_name(name: str) -> str:
    name = re.sub(r"[\[\]\:\*\?\/\\]", "_", name)
    return name[:31]


def write_cases_to_sheet(wb, template_sheet, sheet_name: str, cases: List[Dict[str, str]]):
    # 多选时需求：写入同一个 sheet；如果存在则复用并追加，否则创建
    if sheet_name in wb.sheetnames:
        ws = wb[sheet_name]
        start_row = (ws.max_row or 1) + 1
    else:
        ws = wb.create_sheet(sheet_name)
        start_row = 2

        header_row_index = find_header_row(template_sheet)
        for col_idx, cell in enumerate(template_sheet[header_row_index], start=1):
            new_cell = ws.cell(row=1, column=col_idx, value=cell.value)
            if cell.has_style:
                new_cell.font = copy(cell.font)
                new_cell.fill = copy(cell.fill)
                new_cell.border = copy(cell.border)
                new_cell.alignment = copy(cell.alignment)
                new_cell.number_format = cell.number_format
        ws.freeze_panes = "B2"

    # 写入/追加数据
    row = start_row
    for case in cases:
        for col_idx, header in enumerate(headers_order, start=1):
            val = case.get(header, "")
            if header in ("预置条件", "操作步骤") and isinstance(val, str):
                val = val.replace("; ", "\n")
            if header == "JSON":
                val = ""
            ws.cell(row=row, column=col_idx, value=val)
        row += 1

    # 自动换行
    col_idx_pre = headers_order.index("预置条件") + 1
    col_idx_steps = headers_order.index("操作步骤") + 1
    for r in range(1, row):
        ws.cell(row=r, column=col_idx_pre).alignment = Alignment(wrap_text=True, vertical="top")
        ws.cell(row=r, column=col_idx_steps).alignment = Alignment(wrap_text=True, vertical="top")

    # 列宽（仅在首次创建时做一次也可，这里简化：每次都做一次）
    for col in ws.columns:
        max_len = 0
        col_letter = col[0].column_letter
        for c in col:
            v = c.value
            if v is None:
                continue
            l = len(str(v))
            if l > max_len:
                max_len = l
        ws.column_dimensions[col_letter].width = min(max_len + 2, 60)

    return ws


# ===== 4) 多选交互 =====
def list_prd_files(prd_dir: str) -> List[str]:
    if not os.path.isdir(prd_dir):
        return []
    fs = [f for f in os.listdir(prd_dir) if f.lower().endswith(".md")]
    fs.sort()
    return fs


def parse_multi_input(s: str) -> List[str]:
    s = (s or "").strip()
    if not s:
        return []
    parts = re.split(r"[,\s]+", s)
    return [p for p in (x.strip() for x in parts) if p]


def pick_prds_interactively(prd_dir: str) -> List[str]:
    files = list_prd_files(prd_dir)
    if not files:
        raise RuntimeError(f"PRD目录为空或不存在：{prd_dir}")

    print(f"PRD目录：{prd_dir}")
    for i, f in enumerate(files, start=1):
        print(f"{i:>2}. {f}")
    print("支持多选：输入多个序号（如 1,2,5 或 1 2 5）。")

    while True:
        s = input("请输入PRD序号(可多选)：").strip()
        parts = parse_multi_input(s)
        if not parts or not all(p.isdigit() for p in parts):
            print("输入无效，请输入序号（可多选）。")
            continue

        idxs: List[int] = []
        ok = True
        for p in parts:
            n = int(p)
            if not (1 <= n <= len(files)):
                ok = False
                break
            idxs.append(n)
        if not ok:
            print("序号超出范围，请重试。")
            continue

        seen = set()
        paths: List[str] = []
        for n in idxs:
            if n in seen:
                continue
            seen.add(n)
            paths.append(os.path.join(prd_dir, files[n - 1]))
        return paths


def ask_sheet_name_interactively(default_name: str) -> str:
    s = input(f"请输入Sheet名称（回车使用默认：{default_name}）：").strip()
    return safe_sheet_name(s or default_name)


def ask_output_name_interactively(default_name: str) -> str:
    s = input(f"请输入生成的测试用例文件名称（回车使用默认：{default_name}）：").strip()
    return s or default_name


def main():
    parser = argparse.ArgumentParser(description="根据PRD生成用例JSON，并写入测试用例Excel")
    parser.add_argument("--template", default=TEMPLATE_PATH_DEFAULT, help="模板Excel路径")
    parser.add_argument("--module", default="通用模块", help="功能模块字段值（多PRD时统一使用该值）")
    parser.add_argument("--prefix", default="TC", help="用例编号前缀（多PRD时统一使用该值）")
    parser.add_argument("--overwrite", action="store_true", help="是否覆盖保存到模板文件（默认另存为新文件）")
    args = parser.parse_args()

    prd_dir = resolve_prd_dir()

    if not os.path.exists(args.template):
        print("找不到模板文件：", args.template)
        return

    # ✅ 需求：交互选择 PRD（可多选）
    prd_paths = pick_prds_interactively(prd_dir)

    # ✅ 需求：多选时生成在同一个 sheet；sheet 名 = 交互输入的“测试用例名称”
    if len(prd_paths) == 1:
        default_sheet = f"{os.path.splitext(os.path.basename(prd_paths[0]))[0]}_用例"
    else:
        default_sheet = "多PRD_用例"

    sheet_name = ask_sheet_name_interactively(default_sheet)

    # ✅ 需求：输出文件名 = 交互输入名称 + 时间戳
    if len(prd_paths) == 1:
        default_out_base = os.path.splitext(os.path.basename(prd_paths[0]))[0]
    else:
        default_out_base = "MultiPRD"
    out_base = ask_output_name_interactively(default_out_base)

    # 打开模板
    wb = load_workbook(args.template)
    template_sheet = wb.worksheets[0]

    # 多 PRD 合并到同一个 sheet：序号递增，避免重复
    os.makedirs(OUTPUT_JSON_DIR, exist_ok=True)
    total_cases = 0
    next_idx = 1

    for prd_path in prd_paths:
        prd_base = os.path.splitext(os.path.basename(prd_path))[0]

        cases = prd_to_cases(prd_path=prd_path, module=args.module, prefix=args.prefix, start_idx=next_idx)
        if not cases:
            print(f"未从PRD抽取到可生成用例的条目：{prd_path}")
            continue

        # 每个 PRD 仍然单独落 JSON（便于追溯）
        json_path = os.path.join(OUTPUT_JSON_DIR, f"{prd_base}_用例.json")
        with open(json_path, "w", encoding="utf-8") as f:
            json.dump(cases, f, ensure_ascii=False, indent=4)
        print(f"已生成 JSON：{json_path}（{len(cases)}条）")

        # 写入同一个 sheet（追加）
        write_cases_to_sheet(wb, template_sheet, sheet_name, cases)

        total_cases += len(cases)
        next_idx += len(cases)

    # 保存 Excel
    if args.overwrite:
        out_xlsx = args.template
    else:
        os.makedirs(OUTPUT_XLSX_DIR, exist_ok=True)
        ts = datetime.now().strftime("%Y%m%d_%H%M%S")
        out_xlsx = os.path.join(OUTPUT_XLSX_DIR, f"{out_base}_{ts}.xlsx")

    wb.save(out_xlsx)
    print("已生成Excel：", out_xlsx)
    print("Sheet：", sheet_name)
    print("总用例条数：", total_cases)


if __name__ == "__main__":
    main()