#!/bin/bash

# 统信系统安全漏洞修复脚本
# 功能：修复系统安全配置问题
# 作者：Security Team
# 日期：$(date +%Y-%m-%d)

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 日志配置
LOG_FILE="/tmp/security_fix_$(date +%Y%m%d_%H%M%S).log"
SCRIPT_DIR=$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)

# 日志函数
log() {
    local level=$1
    shift
    local message="$*"
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')
    echo "[$timestamp] [$level] $message" | tee -a "$LOG_FILE"
}

# 彩色输出函数
print_title() {
    echo -e "${BLUE}$1${NC}"
}

print_success() {
    echo -e "${GREEN}$1${NC}"
}

print_warning() {
    echo -e "${YELLOW}$1${NC}"
}

print_error() {
    echo -e "${RED}$1${NC}"
}

# 确认函数
confirm_action() {
    local message=$1
    read -p "$message (y/N): " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        log INFO "用户取消操作"
        return 1
    fi
    return 0
}

# 备份文件函数
backup_file() {
    local file_path=$1
    local timestamp=$(date +%Y%m%d_%H%M%S)
    local backup_path="${file_path}_backup_${timestamp}"
    
    if [[ -f "$file_path" ]]; then
        cp "$file_path" "$backup_path"
        log INFO "已创建备份: $backup_path"
        echo "$backup_path"
    else
        log WARN "文件不存在，无法备份: $file_path"
        echo ""
    fi
}

# 修复OpenSSH安全配置
fix_ssh_security() {
    print_title "开始修复OpenSSH安全配置..."
    log INFO "开始修复OpenSSH安全配置"
    
    local ssh_config="/etc/ssh/sshd_config"
    local backup_path=$(backup_file "$ssh_config")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份SSH配置文件，跳过此步骤"
        return 1
    fi
    
    # 创建临时文件
    local temp_file=$(mktemp)
    
    # 读取现有配置并替换或添加所需参数
    while IFS= read -r line; do
        if [[ $line =~ ^[[:space:]]*X11Forwarding[[:space:]] ]]; then
            echo "X11Forwarding no" >> "$temp_file"
        elif [[ $line =~ ^[[:space:]]*MaxAuthTries[[:space:]] ]]; then
            echo "MaxAuthTries 4" >> "$temp_file"
        elif [[ $line =~ ^[[:space:]]*IgnoreRhosts[[:space:]] ]]; then
            echo "IgnoreRhosts yes" >> "$temp_file"
        elif [[ $line =~ ^[[:space:]]*HostbasedAuthentication[[:space:]] ]]; then
            echo "HostbasedAuthentication no" >> "$temp_file"
        elif [[ $line =~ ^[[:space:]]*PermitEmptyPasswords[[:space:]] ]]; then
            echo "PermitEmptyPasswords no" >> "$temp_file"
        elif [[ $line =~ ^[[:space:]]*PermitRootLogin[[:space:]] ]]; then
            echo "PermitRootLogin no" >> "$temp_file"
        elif [[ $line =~ ^[[:space:]]*Protocol[[:space:]] ]]; then
            echo "Protocol 2" >> "$temp_file"
        else
            echo "$line" >> "$temp_file"
        fi
    done < "$ssh_config"
    
    # 检查是否有缺少的参数并添加
    if ! grep -q "^X11Forwarding" "$temp_file"; then
        echo "X11Forwarding no" >> "$temp_file"
    fi
    if ! grep -q "^MaxAuthTries" "$temp_file"; then
        echo "MaxAuthTries 4" >> "$temp_file"
    fi
    if ! grep -q "^IgnoreRhosts" "$temp_file"; then
        echo "IgnoreRhosts yes" >> "$temp_file"
    fi
    if ! grep -q "^HostbasedAuthentication" "$temp_file"; then
        echo "HostbasedAuthentication no" >> "$temp_file"
    fi
    if ! grep -q "^PermitEmptyPasswords" "$temp_file"; then
        echo "PermitEmptyPasswords no" >> "$temp_file"
    fi
    if ! grep -q "^PermitRootLogin" "$temp_file"; then
        echo "PermitRootLogin no" >> "$temp_file"
    fi
    if ! grep -q "^Protocol" "$temp_file"; then
        echo "Protocol 2" >> "$temp_file"
    fi
    
    # 替换原文件
    mv "$temp_file" "$ssh_config"
    
    log INFO "OpenSSH安全配置修复完成"
    print_success "OpenSSH安全配置修复完成"
}

# 修复命令行界面超时设置
fix_timeout_setting() {
    print_title "开始修复命令行界面超时设置..."
    log INFO "开始修复命令行界面超时设置"
    
    local profile_file="/etc/profile"
    local backup_path=$(backup_file "$profile_file")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份profile文件，跳过此步骤"
        return 1
    fi
    
    # 检查是否已有设置，如果有则替换，否则添加
    if grep -q "^TMOUT=" "$profile_file"; then
        sed -i 's/^TMOUT=.*/TMOUT=300/' "$profile_file"
        sed -i '/^export TMOUT/d' "$profile_file"
    else
        echo "" >> "$profile_file"
        echo "# 设置命令行界面超时时间为300秒" >> "$profile_file"
        echo "TMOUT=300" >> "$profile_file"
    fi
    
    if ! grep -q "^export TMOUT" "$profile_file"; then
        echo "export TMOUT" >> "$profile_file"
    fi
    
    # 应用配置
    source "$profile_file"
    
    log INFO "命令行界面超时设置修复完成"
    print_success "命令行界面超时设置修复完成"
}

# 修复PAM认证模块配置
fix_pam_auth() {
    print_title "开始修复PAM认证模块配置..."
    log INFO "开始修复PAM认证模块配置"
    
    local pam_su="/etc/pam.d/su"
    local backup_path=$(backup_file "$pam_su")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份PAM su文件，跳过此步骤"
        return 1
    fi
    
    # 检查是否已有配置，避免重复添加
    if ! grep -q "pam_rootok.so" "$pam_su"; then
        sed -i '1i auth            sufficient      pam_rootok.so' "$pam_su"
    else
        sed -i 's/^auth.*pam_rootok\.so/auth            sufficient      pam_rootok.so/' "$pam_su"
    fi
    
    if ! grep -q "pam_wheel.so" "$pam_su"; then
        sed -i '2i auth            required        pam_wheel.so    group=wheel' "$pam_su"
    else
        sed -i 's/^auth.*pam_wheel\.so.*/auth            required        pam_wheel.so    group=wheel/' "$pam_su"
    fi
    
    log INFO "PAM认证模块配置修复完成"
    print_success "PAM认证模块配置修复完成"
}

# 修复Telnet服务状态
fix_telnet_status() {
    print_title "开始修复Telnet服务状态..."
    log INFO "开始修复Telnet服务状态"
    
    local telnet_file="/etc/xinetd.d/telnet"
    
    if [[ -f "$telnet_file" ]]; then
        local backup_path=$(backup_file "$telnet_file")
        
        if [[ -z "$backup_path" ]]; then
            log ERROR "无法备份telnet配置文件，跳过此步骤"
            return 1
        fi
        
        # 禁用Telnet服务
        sed -i 's/disable.*/disable = yes/' "$telnet_file"
        
        # 重启xinetd服务
        systemctl restart xinetd
        log INFO "Telnet服务已禁用并重启xinetd服务"
        print_success "Telnet服务已禁用"
    else
        log INFO "Telnet服务未安装或配置文件不存在，跳过此步骤"
        print_warning "Telnet服务未安装或配置文件不存在，跳过此步骤"
    fi
}

# 修复SSH服务状态
fix_ssh_service() {
    print_title "开始修复SSH服务状态..."
    log INFO "开始修复SSH服务状态"
    
    # 启动并启用SSH服务
    systemctl enable sshd
    systemctl start sshd
    
    if systemctl is-active --quiet sshd; then
        log INFO "SSH服务已启动并设置为开机自启"
        print_success "SSH服务已启动并设置为开机自启"
    else
        log ERROR "SSH服务启动失败"
        print_error "SSH服务启动失败"
    fi
}

# 修复登录日志记录配置
fix_login_logging() {
    print_title "开始修复登录日志记录配置..."
    log INFO "开始修复登录日志记录配置"
    
    local rsyslog_conf="/etc/rsyslog.conf"
    local backup_path=$(backup_file "$rsyslog_conf")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份rsyslog配置文件，跳过此步骤"
        return 1
    fi
    
    # 检查是否已有配置，避免重复添加
    if ! grep -q "authpriv.* /var/log/secure" "$rsyslog_conf"; then
        echo "authpriv.* /var/log/secure" >> "$rsyslog_conf"
    fi
    
    # 创建并设置日志文件权限
    touch /var/log/secure
    chmod 600 /var/log/secure
    
    # 重启rsyslog服务
    systemctl restart rsyslog
    
    log INFO "登录日志记录配置修复完成"
    print_success "登录日志记录配置修复完成"
}

# 修复用户目录默认访问权限
fix_umask_setting() {
    print_title "开始修复用户目录默认访问权限设置..."
    log INFO "开始修复用户目录默认访问权限设置"
    
    local profile_file="/etc/profile"
    local backup_path=$(backup_file "$profile_file")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份profile文件，跳过此步骤"
        return 1
    fi
    
    # 检查是否已有umask设置
    if grep -q "^umask" "$profile_file"; then
        sed -i 's/^umask.*/umask 027/' "$profile_file"
    else
        echo "" >> "$profile_file"
        echo "# 设置默认权限掩码" >> "$profile_file"
        echo "umask 027" >> "$profile_file"
    fi
    
    # 应用配置
    source "$profile_file"
    
    log INFO "用户目录默认访问权限设置修复完成"
    print_success "用户目录默认访问权限设置修复完成"
}

# 修复口令过期前警告天数
fix_password_warn_age() {
    print_title "开始修复口令过期前警告天数..."
    log INFO "开始修复口令过期前警告天数"
    
    local login_defs="/etc/login.defs"
    local backup_path=$(backup_file "$login_defs")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份login.defs文件，跳过此步骤"
        return 1
    fi
    
    # 检查是否已有设置
    if grep -q "^PASS_WARN_AGE" "$login_defs"; then
        sed -i 's/^PASS_WARN_AGE.*/PASS_WARN_AGE   7/' "$login_defs"
    else
        echo "" >> "$login_defs"
        echo "# 密码过期前警告天数" >> "$login_defs"
        echo "PASS_WARN_AGE   7" >> "$login_defs"
    fi
    
    log INFO "口令过期前警告天数修复完成"
    print_success "口令过期前警告天数修复完成"
}

# 修复密码复杂度策略
fix_password_complexity() {
    print_title "开始修复密码复杂度策略..."
    log INFO "开始修复密码复杂度策略"
    
    local system_auth="/etc/pam.d/system-auth"
    local backup_path=$(backup_file "$system_auth")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份system-auth文件，跳过此步骤"
        return 1
    fi
    
    # 检查是否已有pam_cracklib配置
    if grep -q "pam_cracklib.so" "$system_auth"; then
        sed -i 's/password.*requisite.*pam_cracklib.so.*/password    requisite     pam_cracklib.so try_first_pass retry=3 minlen=8 dcredit=-1 ucredit=-1 ocredit=-1 lcredit=-1/' "$system_auth"
    else
        # 添加密码复杂度配置
        echo "" >> "$system_auth"
        echo "# 密码复杂度策略" >> "$system_auth"
        echo "password    requisite     pam_cracklib.so try_first_pass retry=3 minlen=8 dcredit=-1 ucredit=-1 ocredit=-1 lcredit=-1" >> "$system_auth"
    fi
    
    log INFO "密码复杂度策略修复完成"
    print_success "密码复杂度策略修复完成"
}

# 修复口令更改最小间隔天数
fix_password_min_days() {
    print_title "开始修复口令更改最小间隔天数..."
    log INFO "开始修复口令更改最小间隔天数"
    
    local login_defs="/etc/login.defs"
    local backup_path=$(backup_file "$login_defs")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份login.defs文件，跳过此步骤"
        return 1
    fi
    
    # 检查是否已有设置
    if grep -q "^PASS_MIN_DAYS" "$login_defs"; then
        sed -i 's/^PASS_MIN_DAYS.*/PASS_MIN_DAYS   7/' "$login_defs"
    else
        echo "" >> "$login_defs"
        echo "# 密码更改最小间隔天数" >> "$login_defs"
        echo "PASS_MIN_DAYS   7" >> "$login_defs"
    fi
    
    log INFO "口令更改最小间隔天数修复完成"
    print_success "口令更改最小间隔天数修复完成"
}

# 修复口令生存周期
fix_password_max_days() {
    print_title "开始修复口令生存周期..."
    log INFO "开始修复口令生存周期"
    
    local login_defs="/etc/login.defs"
    local backup_path=$(backup_file "$login_defs")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份login.defs文件，跳过此步骤"
        return 1
    fi
    
    # 检查是否已有设置
    if grep -q "^PASS_MAX_DAYS" "$login_defs"; then
        sed -i 's/^PASS_MAX_DAYS.*/PASS_MAX_DAYS   90/' "$login_defs"
    else
        echo "" >> "$login_defs"
        echo "# 密码最大生存周期" >> "$login_defs"
        echo "PASS_MAX_DAYS   90" >> "$login_defs"
    fi
    
    log INFO "口令生存周期修复完成"
    print_success "口令生存周期修复完成"
}

# 修复重要目录及文件权限
fix_critical_permissions() {
    print_title "开始修复重要目录及文件权限..."
    log INFO "开始修复重要目录及文件权限"
    
    chmod 644 /etc/passwd
    chmod 400 /etc/shadow
    chmod 644 /etc/group
    
    log INFO "重要目录及文件权限修复完成"
    print_success "重要目录及文件权限修复完成"
}

# 修复账户认证失败次数限制
fix_auth_failure_limit() {
    print_title "开始修复账户认证失败次数限制..."
    log INFO "开始修复账户认证失败次数限制"
    
    local system_auth="/etc/pam.d/system-auth"
    local backup_path=$(backup_file "$system_auth")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份system-auth文件，跳过此步骤"
        return 1
    fi
    
    # 添加认证失败限制配置
    if ! grep -q "pam_tally2.so.*deny=5" "$system_auth"; then
        sed -i '/^auth[[:space:]]\+required/a auth        required      pam_tally2.so deny=5 unlock_time=180 onerr=fail no_magic_root' "$system_auth"
    else
        sed -i 's/^auth[[:space:]]\+required[[:space:]]\+pam_tally2.so.*/auth        required      pam_tally2.so deny=5 unlock_time=180 onerr=fail no_magic_root/' "$system_auth"
    fi
    
    if ! grep -q "pam_tally2.so" "$system_auth" | grep -q account; then
        echo "" >> "$system_auth"
        echo "# 账户认证失败次数限制" >> "$system_auth"
        echo "account     required      pam_tally2.so" >> "$system_auth"
    fi
    
    log INFO "账户认证失败次数限制修复完成"
    print_success "账户认证失败次数限制修复完成"
}

# 修复历史命令数量限制
fix_history_limit() {
    print_title "开始修复历史命令数量限制..."
    log INFO "开始修复历史命令数量限制"
    
    local profile_file="/etc/profile"
    local backup_path=$(backup_file "$profile_file")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份profile文件，跳过此步骤"
        return 1
    fi
    
    # 设置历史命令数量限制
    if grep -q "^HISTFILESIZE=" "$profile_file"; then
        sed -i 's/^HISTFILESIZE=.*/HISTFILESIZE=5/' "$profile_file"
    else
        echo "" >> "$profile_file"
        echo "# 历史命令数量限制" >> "$profile_file"
        echo "HISTFILESIZE=5" >> "$profile_file"
    fi
    
    if grep -q "^HISTSIZE=" "$profile_file"; then
        sed -i 's/^HISTSIZE=.*/HISTSIZE=5/' "$profile_file"
    else
        echo "HISTSIZE=5" >> "$profile_file"
    fi
    
    # 应用配置
    source "$profile_file"
    
    log INFO "历史命令数量限制修复完成"
    print_success "历史命令数量限制修复完成"
}

# 修复密码重复使用次数限制
fix_password_reuse_limit() {
    print_title "开始修复密码重复使用次数限制..."
    log INFO "开始修复密码重复使用次数限制"
    
    local system_auth="/etc/pam.d/system-auth"
    local opasswd_file="/etc/security/opasswd"
    local backup_path=$(backup_file "$system_auth")
    
    if [[ -z "$backup_path" ]]; then
        log ERROR "无法备份system-auth文件，跳过此步骤"
        return 1
    fi
    
    # 创建旧密码存储文件
    touch "$opasswd_file"
    chown root:root "$opasswd_file"
    chmod 600 "$opasswd_file"
    
    # 修复或添加密码重复使用限制
    if grep -q "pam_unix.so" "$system_auth"; then
        sed -i 's/\(^password.*sufficient.*pam_unix\.so\).*/\1 remember=5/' "$system_auth"
    else
        echo "" >> "$system_auth"
        echo "# 密码重复使用限制" >> "$system_auth"
        echo "password    sufficient    pam_unix.so remember=5" >> "$system_auth"
    fi
    
    log INFO "密码重复使用次数限制修复完成"
    print_success "密码重复使用次数限制修复完成"
}

# 修复Ctrl+Alt+Delete组合键状态
fix_ctrl_alt_del() {
    print_title "开始修复Ctrl+Alt+Delete组合键状态..."
    log INFO "开始修复Ctrl+Alt+Delete组合键状态"
    
    # 禁用Ctrl+Alt+Delete系统调用
    systemctl mask ctrl-alt-del.target
    
    if [[ $? -eq 0 ]]; then
        log INFO "Ctrl+Alt+Delete组合键已禁用"
        print_success "Ctrl+Alt+Delete组合键已禁用"
    else
        log ERROR "禁用Ctrl+Alt+Delete组合键失败"
        print_error "禁用Ctrl+Alt+Delete组合键失败"
    fi
}

# 重启相关服务
restart_services() {
    print_title "重启相关服务以应用更改..."
    log INFO "重启相关服务以应用更改"
    
    systemctl restart sshd
    systemctl restart rsyslog
    
    log INFO "服务重启完成"
    print_success "服务重启完成"
}

# 输出修复摘要
output_summary() {
    print_title "=================================="
    print_success "安全漏洞修复汇总"
    print_title "=================================="
    
    log INFO "安全漏洞修复完成，请检查日志: $LOG_FILE"
    print_success "安全漏洞修复完成！"
    print_warning "建议重启系统以确保所有更改完全生效。"
}

# 主菜单
show_menu() {
    echo "统信系统安全漏洞修复工具"
    echo "请选择要执行的操作:"
    echo "1) 完整修复所有安全问题"
    echo "2) 选择性修复特定问题"
    echo "3) 退出"
    echo
}

# 选择性修复菜单
selective_fix_menu() {
    while true; do
        echo "选择要修复的安全问题:"
        echo " 1) OpenSSH安全配置"
        echo " 2) 命令行界面超时设置"
        echo " 3) PAM认证模块配置"
        echo " 4) Telnet服务状态"
        echo " 5) SSH服务状态"
        echo " 6) 登录日志记录配置"
        echo " 7) 用户目录默认访问权限"
        echo " 8) 口令过期前警告天数"
        echo " 9) 密码复杂度策略"
        echo "10) 口令更改最小间隔天数"
        echo "11) 口令生存周期"
        echo "12) 重要目录及文件权限"
        echo "13) 账户认证失败次数限制"
        echo "14) 历史命令数量限制"
        echo "15) 密码重复使用次数限制"
        echo "16) Ctrl+Alt+Delete组合键状态"
        echo " b) 返回主菜单"
        echo " a) 全选"
        echo " q) 退出"
        echo
        read -p "请输入选项 (1-16, b, a, q): " option
        
        case $option in
            1) fix_ssh_security ;;
            2) fix_timeout_setting ;;
            3) fix_pam_auth ;;
            4) fix_telnet_status ;;
            5) fix_ssh_service ;;
            6) fix_login_logging ;;
            7) fix_umask_setting ;;
            8) fix_password_warn_age ;;
            9) fix_password_complexity ;;
            10) fix_password_min_days ;;
            11) fix_password_max_days ;;
            12) fix_critical_permissions ;;
            13) fix_auth_failure_limit ;;
            14) fix_history_limit ;;
            15) fix_password_reuse_limit ;;
            16) fix_ctrl_alt_del ;;
            b) break ;;
            a)
                fix_ssh_security
                fix_timeout_setting
                fix_pam_auth
                fix_telnet_status
                fix_ssh_service
                fix_login_logging
                fix_umask_setting
                fix_password_warn_age
                fix_password_complexity
                fix_password_min_days
                fix_password_max_days
                fix_critical_permissions
                fix_auth_failure_limit
                fix_history_limit
                fix_password_reuse_limit
                fix_ctrl_alt_del
                ;;
            q) exit 0 ;;
            *) echo "无效选项，请重新选择" ;;
        esac
    done
}

# 主函数
main() {
    log INFO "开始执行统信系统安全漏洞修复脚本"
    log INFO "脚本位置: $SCRIPT_DIR"
    log INFO "修复开始时间: $(date '+%Y-%m-%d %H:%M:%S')"
    log INFO ""
    
    # 检查是否以root权限运行
    if [[ $EUID -ne 0 ]]; then
        print_error "此脚本需要root权限才能运行，请使用sudo或以root身份运行"
        exit 1
    fi
    
    while true; do
        show_menu
        read -p "请选择操作 (1-3): " choice
        
        case $choice in
            1)
                if confirm_action "确定要修复所有安全问题吗？此操作将修改系统配置文件并创建备份。"; then
                    fix_ssh_security
                    fix_timeout_setting
                    fix_pam_auth
                    fix_telnet_status
                    fix_ssh_service
                    fix_login_logging
                    fix_umask_setting
                    fix_password_warn_age
                    fix_password_complexity
                    fix_password_min_days
                    fix_password_max_days
                    fix_critical_permissions
                    fix_auth_failure_limit
                    fix_history_limit
                    fix_password_reuse_limit
                    fix_ctrl_alt_del
                    
                    restart_services
                    output_summary
                    break
                fi
                ;;
            2)
                selective_fix_menu
                ;;
            3)
                exit 0
                ;;
            *)
                echo "无效选项，请重新选择"
                ;;
        esac
    done
}

# 运行主函数
main "$@"