﻿#Requires -Version 5.1
<#
.SYNOPSIS
    服务自检脚本 (Windows 版本)

.DESCRIPTION
    本脚本用于远程检测服务器上的服务是否正常启动，并返回结果。
    
    功能特性：
    1. 支持预设服务器列表和手动输入（IP/端口/用户名/密码）
    2. 支持自定义 SSH 端口（默认 22）
    3. 自动检测目标服务器平台类型（检测 /data/services 目录）
    4. 自动识别系统类型（ujava/upython/upython_voice 容器）
    5. 根据平台和容器类型检测相应的服务进程和端口
    6. 生成详细的检测报告

.EXAMPLE
    .\check_server_health.ps1
    
.NOTES
    作者: 自动化运维团队
    版本: 1.0.0
    
    ============================================================
    依赖说明（零安装，开箱即用）：
    ============================================================
    
    本脚本需要 plink.exe（PuTTY 工具）来实现自动密码认证。
    
    原因：Windows 原生 SSH 不支持通过命令行传递密码
    
    使用方式（任选其一）：
    
    方式1 (推荐，离线可用): 
       将 plink.exe 放在脚本同目录下
       脚本会自动检测并使用本地的可执行文件
       
       下载地址（可在有网络的电脑下载后拦贝）:
       https://the.earth.li/~sgtatham/putty/latest/w64/plink.exe
    
    方式2: 已安装 PuTTY 到系统 PATH
       脚本会自动检测系统中已安装的 plink
    
    ============================================================
#>

# 严格模式
Set-StrictMode -Version Latest
$ErrorActionPreference = "Continue"

# ================================
# 设置控制台编码为 UTF-8
# ================================
[Console]::OutputEncoding = [System.Text.Encoding]::UTF8
[Console]::InputEncoding = [System.Text.Encoding]::UTF8
$env:PYTHONIOENCODING = "utf-8"
$PSDefaultParameterValues['Out-File:Encoding'] = 'utf8'
$PSDefaultParameterValues['*:Encoding'] = 'utf8'

# ================================
# 全局配置
# ================================
$SCRIPT_DIR = Split-Path -Parent $MyInvocation.MyCommand.Path
$SSH_TIMEOUT = 30

# 脚本版本号（用于日志与报告）
$SCRIPT_VERSION = "1.0.3"

# PuTTY 工具路径
$script:PLINK_PATH = $null
$script:PSCP_PATH = $null
$script:PreferredSSHTool = $null

# ================================
# 日志配置
# ================================
$LOG_DIR = Join-Path $SCRIPT_DIR "logs"
$LOG_TIMESTAMP = Get-Date -Format "yyyyMMdd_HHmmss"
$LOG_FILE = Join-Path $LOG_DIR "health_check_$LOG_TIMESTAMP.log"

# 确保日志目录存在
if (-not (Test-Path $LOG_DIR)) {
    New-Item -ItemType Directory -Path $LOG_DIR -Force | Out-Null
}

# 预设服务器列表
$ServerList = @{
    "1" = @{
        IP = "192.168.5.48"
        User = "root"
        Pass = "Ubains@123"
        Desc = "标准版预定运维服务器"
    };
    "2" = @{
        IP = "192.168.5.67"
        User = "root"
        Pass = "Ubains@123"
        Desc = "阿曼项目预定服务器"
    };
    "3" = @{
        IP = "192.168.5.47"
        User = "root"
        Pass = "Ubains@1234"
        Desc = "标准版预定运维测试发布服务器"
    };
    "4" = @{
        IP = "192.168.5.44"
        User = "root"
        Pass = "Ubains@123"
        Desc = "新统一平台测试服务器"
    }
}

# ================================
# 服务检测配置
# ================================

# ujava 基础服务进程（支持容器内和宿主机检测）
# 使用jar文件名作为匹配模式，不依赖完整路径
$UjavaServices = @{
    "auth"       = "ubains-auth.jar"
    "gateway"    = "ubains-gateway.jar"
    "system"     = "ubains-modules-system.jar"
    "meeting2.0" = "ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
    "meeting3.0" = "ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
    "mqtt"       = "ubains-meeting-mqtt-1.0-SNAPSHOT.jar"
    "quartz"     = "ubains-meeting-quartz-1.0-SNAPSHOT.jar"
    "message"    = "ubains-meeting-message-scheduling-1.0-SNAPSHOT.jar"
}

# ujava 宿主机服务（新统一平台和传统平台）
# 新统一平台路径：/data/services/api/java-meeting/java-meeting-extapi/
# 传统平台路径：可能在 /var/www/ 下
$UjavaHostServices = @{
    "extapi" = "ubains-meeting-api-1.0-SNAPSHOT.jar"
}

# upython 容器内需要监听的端口
$UpythonPorts = @(
    @{ Port = 11211; Process = "memcached"; Description = "Memcached 缓存服务" }
    @{ Port = 8000;  Process = "uwsgi";     Description = "uWSGI 应用服务" }
)

# upython_voice 容器内需要监听的端口
$UpythonVoicePorts = @(
    @{ Port = 1883;  Process = "mosquitto"; Description = "MQTT Broker 服务" }
    @{ Port = 8000;  Process = "uwsgi";     Description = "uWSGI 应用服务" }
    @{ Port = 9001;  Process = "mosquitto"; Description = "MQTT WebSocket 服务" }
    @{ Port = 11211; Process = "memcached"; Description = "Memcached 缓缓冲服务" }
    @{ Port = 8080;  Process = "nginx";     Description = "Nginx 代理服务 (8080)" }
    @{ Port = 80;    Process = "nginx";     Description = "Nginx Web 服务 (80)" }
)

# ================================
# 传统平台配置
# ================================

# 传统平台 ujava 容器内服务（只有nginx和meeting服务）
$UjavaOldPlatformContainerServices = @{
    "nginx"   = "nginx: master process"
    "meeting" = "ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
}

# 传统平台 ujava 宿主机服务
$UjavaOldPlatformHostServices = @{
    "extapi" = "ubains-meeting-api-1.0-SNAPSHOT.jar"
}

# 传统平台 upython 容器内需要监听的端口
$UpythonOldPlatformPorts = @(
    @{ Port = 8081;  Process = "nginx";     Description = "Nginx 代理服务 (8081)" }
    @{ Port = 8443;  Process = "nginx";     Description = "Nginx HTTPS 服务 (8443)" }
    @{ Port = 8000;  Process = "uwsgi";     Description = "uWSGI 应用服务" }
    @{ Port = 8002;  Process = "httpd";     Description = "Apache HTTPD 服务" }
    @{ Port = 11211; Process = "memcached"; Description = "Memcached 缓冲服务" }
)

# DNS 测试域名列表
$DNSTestDomains = @(
    "www.baidu.com"
    "www.qq.com"
    "www.aliyun.com"
)

# ================================
# 日志导出配置
# ================================

# 新统一平台日志路径配置
$NewPlatformLogs = @(
    @{ Name = "auth_log.out"; RemotePath = "/data/services/api/auth/auth-sso-auth/log.out" }
    @{ Name = "gatway_log.out"; RemotePath = "/data/services/api/auth/auth-sso-gatway/log.out" }
    @{ Name = "system_log.out"; RemotePath = "/data/services/api/auth/auth-sso-system/log.out" }
    @{ Name = "对内2.0_ubains-INFO-AND-ERROR.log"; RemotePath = "/data/services/api/java-meeting/java-meeting2.0/logs/ubains-INFO-AND-ERROR.log" }
    @{ Name = "对内3.0_ubains-INFO-AND-ERROR.log"; RemotePath = "/data/services/api/java-meeting/java-meeting3.0/logs/ubains-INFO-AND-ERROR.log" }
    @{ Name = "对外服务_ubains-INFO-AND-ERROR.log"; RemotePath = "/data/services/api/java-meeting/java-meeting-extapi/logs/ubains-INFO-AND-ERROR.log" }
    @{ Name = "信息调度_ubains-INFO-AND-ERROR.log"; RemotePath = "/data/services/api/java-meeting/java-message-scheduling/logs/ubains-INFO-AND-ERROR.log" }
    @{ Name = "MQTT_ubains-INFO-AND-ERROR.log"; RemotePath = "/data/services/api/java-meeting/java-mqtt/logs/ubains-INFO-AND-ERROR.log" }
    @{ Name = "定时任务_ubains-INFO-AND-ERROR.log"; RemotePath = "/data/services/api/java-meeting/java-quartz/logs/ubains-INFO-AND-ERROR.log" }
)

# 传统平台 ujava 日志路径配置
$OldPlatformUjavaLogs = @(
    @{ Name = "对内后端_ubains-INFO-AND-ERROR.log"; RemotePath = "/var/www/java/api-java-meeting2.0/logs/ubains-INFO-AND-ERROR.log" }
    @{ Name = "对外后端_ubains-INFO-AND-ERROR.log"; RemotePath = "/var/www/java/external-meeting-api/logs/ubains-INFO-AND-ERROR.log" }
)

# 传统平台 upython 日志路径配置
$OldPlatformUpythonLogs = @(
    @{ Name = "运维集控_error.log"; RemotePath = "/var/www/html/log/error.log" }
    @{ Name = "运维集控_uinfo.log"; RemotePath = "/var/www/html/log/uinfo.log" }
    @{ Name = "运维集控_uwsgi.log"; RemotePath = "/var/www/html/log/uwsgi.log" }
)

# ================================
# 日志函数
# ================================
function Write-Log {
    param(
        [Parameter(Mandatory=$true)]
        [ValidateSet("INFO", "WARN", "ERROR", "SUCCESS")]
        [string]$Level,
        
        [Parameter(Mandatory=$true)]
        [AllowEmptyString()]
        [string]$Message
    )
    
    # 如果消息为空，直接输出空行到控制台，不记录到日志文件
    if ([string]::IsNullOrEmpty($Message)) {
        Write-Host ""
        return
    }
    
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logLine = "[$timestamp] [$Level] $Message"
    
    $colorMap = @{
        "INFO"    = "Cyan"
        "WARN"    = "Yellow"
        "ERROR"   = "Red"
        "SUCCESS" = "Green"
    }
    $color = $colorMap[$Level]
    Write-Host $logLine -ForegroundColor $color
    
    try {
        $logLine | Out-File -FilePath $LOG_FILE -Append -Encoding utf8
    }
    catch {
        # 日志文件写入失败不影响主流程
    }
}

# ================================
# ujava 系统细分：会议预定系统 vs 统一平台系统（PRD补充）
# 规则：
# - ujava 容器存在时：
#   - 若宿主机存在 /var/www/java/unifiedPlatform => unified
#   - 否则 => meeting
# 说明：仅对传统平台目录体系有意义，但为了统一，这里不强限制平台，按是否存在目录判定即可
# ================================
function Get-UjavaSystemVariant {
    param(
        [Parameter(Mandatory=$true)] [hashtable]$Server
    )

    $cmd = "[ -d /var/www/java/unifiedPlatform ] && echo 'UNIFIED' || echo 'MEETING'"
    $res = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $cmd

    $last = (($res.Output -split "`n") | Where-Object { $_ -match '\S' } | Select-Object -Last 1).Trim()
    if ($last -eq "UNIFIED") { return "unified" }
    return "meeting"
}

# ================================
# 检查依赖
# ================================
function Test-Dependencies {
    Write-Log -Level "INFO" -Message "检查系统依赖..."
    
    $osVersion = [System.Environment]::OSVersion.Version
    Write-Log -Level "INFO" -Message "  Windows 版本: $($osVersion.Major).$($osVersion.Minor).$($osVersion.Build)"
    
    $hasPasswordTool = $false
    
    # 1. 优先检查脚本同目录下的 plink.exe
    $localPlinkPath = Join-Path $SCRIPT_DIR "plink.exe"
    if (Test-Path $localPlinkPath) {
        $script:PLINK_PATH = $localPlinkPath
        Write-Log -Level "INFO" -Message "  plink 已找到 (本地): $localPlinkPath"
        $hasPasswordTool = $true
        $script:PreferredSSHTool = "plink"
    }
    else {
        # 2. 检查远程容器更新目录下的 plink.exe
        $remoteUpdatePlinkPath = Join-Path (Split-Path $SCRIPT_DIR -Parent) "远程容器更新\plink.exe"
        if (Test-Path $remoteUpdatePlinkPath) {
            $script:PLINK_PATH = $remoteUpdatePlinkPath
            Write-Log -Level "INFO" -Message "  plink 已找到 (远程容器更新目录): $remoteUpdatePlinkPath"
            $hasPasswordTool = $true
            $script:PreferredSSHTool = "plink"
        }
        else {
            # 3. 检查系统 PATH 中的 plink
            try {
                $systemPlink = Get-Command plink -ErrorAction Stop
                $script:PLINK_PATH = $systemPlink.Source
                Write-Log -Level "INFO" -Message "  plink 已找到 (系统): $($script:PLINK_PATH)"
                $hasPasswordTool = $true
                $script:PreferredSSHTool = "plink"
            }
            catch {
                Write-Log -Level "WARN" -Message "  plink.exe 未找到"
            }
        }
    }
    
    # 如果没有 plink，检查 sshpass
    if (-not $hasPasswordTool) {
        try {
            $sshpassPath = Get-Command sshpass -ErrorAction Stop
            Write-Log -Level "INFO" -Message "  sshpass 已找到: $($sshpassPath.Source)"
            $hasPasswordTool = $true
            $script:PreferredSSHTool = "sshpass"
        }
        catch {
            Write-Log -Level "WARN" -Message "  sshpass 未找到"
        }
    }
    
    if (-not $hasPasswordTool) {
        Write-Host ""
        Write-Log -Level "ERROR" -Message "  未检测到密码认证工具"
        Write-Host ""
        Write-Log -Level "ERROR" -Message "  请按以下方式解决："
        Write-Host ""
        Write-Log -Level "ERROR" -Message "  方式1 (推荐，离线可用):"
        Write-Log -Level "ERROR" -Message "    将 plink.exe 放在脚本同目录下"
        Write-Log -Level "ERROR" -Message "    当前脚本目录: $SCRIPT_DIR"
        Write-Host ""
        Write-Log -Level "ERROR" -Message "    下载地址:"
        Write-Log -Level "ERROR" -Message "    plink.exe: https://the.earth.li/~sgtatham/putty/latest/w64/plink.exe"
        Write-Host ""
        return $false
    }
    
    # 检查 pscp（用于文件传输/日志导出）
    if ($script:PreferredSSHTool -eq "plink") {
        # 1. 优先检查脚本同目录下的 pscp.exe
        $localPscpPath = Join-Path $SCRIPT_DIR "pscp.exe"
        if (Test-Path $localPscpPath) {
            $script:PSCP_PATH = $localPscpPath
            Write-Log -Level "INFO" -Message "  pscp 已找到 (本地): $localPscpPath"
        }
        else {
            # 2. 检查远程容器更新目录下的 pscp.exe
            $remoteUpdatePscpPath = Join-Path (Split-Path $SCRIPT_DIR -Parent) "远程容器更新\pscp.exe"
            if (Test-Path $remoteUpdatePscpPath) {
                $script:PSCP_PATH = $remoteUpdatePscpPath
                Write-Log -Level "INFO" -Message "  pscp 已找到 (远程容器更新目录): $remoteUpdatePscpPath"
            }
            else {
                # 3. 检查系统 PATH 中的 pscp
                try {
                    $systemPscp = Get-Command pscp -ErrorAction Stop
                    $script:PSCP_PATH = $systemPscp.Source
                    Write-Log -Level "INFO" -Message "  pscp 已找到 (系统): $($script:PSCP_PATH)"
                }
                catch {
                    Write-Log -Level "WARN" -Message "  pscp.exe 未找到，日志导出功能将不可用"
                }
            }
        }
    }
    
    Write-Log -Level "INFO" -Message "系统依赖检查通过 (使用 $script:PreferredSSHTool 进行密码认证)"
    return $true
}

# ================================
# SSH 执行远程命令
# ================================
function Invoke-SSHCommand {
    param(
        [string]$HostName,
        [string]$User,
        [string]$Pass,
        [int]$Port = 22,
        [string]$Command
    )
    
    if ($script:PLINK_PATH -and (Test-Path $script:PLINK_PATH)) {
        $plinkArgs = @(
            "-ssh",
            "-P", $Port,
            "-l", $User,
            "-pw", $Pass,
            "-batch",
            $HostName,
            $Command
        )
        
        $result = & $script:PLINK_PATH @plinkArgs 2>&1
        $exitCode = $LASTEXITCODE
        
        # 处理首次连接主机密钥问题
        if ($exitCode -ne 0 -and ($result -match "host key" -or $result -match "Cannot confirm")) {
            $cmdLine = "echo y | `"$($script:PLINK_PATH)`" -ssh -P $Port -l $User -pw `"$Pass`" $HostName `"$Command`""
            $result = cmd /c $cmdLine 2>&1
            $exitCode = $LASTEXITCODE
        }
    }
    elseif ($script:PreferredSSHTool -eq "sshpass") {
        $env:SSHPASS = $Pass
        $result = & sshpass -e ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null -o ConnectTimeout=$SSH_TIMEOUT -p $Port "$User@$HostName" $Command 2>&1
        $exitCode = $LASTEXITCODE
        $env:SSHPASS = $null
    }
    else {
        Write-Log -Level "ERROR" -Message "未找到 plink 或 sshpass，无法执行 SSH 命令"
        return @{
            Output = "ERROR: No password authentication tool available"
            ExitCode = 1
        }
    }
    
    return @{
        Output = $result
        ExitCode = $exitCode
    }
}

# ================================
# 选择服务器
# ================================
function Select-Server {
    Write-Log -Level "INFO" -Message "可选择的目标服务器："
    Write-Host ""
    foreach ($key in ($ServerList.Keys | Sort-Object)) {
        $server = $ServerList[$key]
        Write-Host "  [$key] $($server.Desc) ($($server.IP) $($server.User))"
    }
    Write-Host "  [0] 手动输入服务器信息"
    Write-Host ""
    
    $serverKey = Read-Host "请输入服务器编号"
    
    if ($serverKey -eq "0") {
        Write-Log -Level "INFO" -Message "进入手动输入模式"
        Write-Host ""
        
        $remoteHost = Read-Host "请输入目标服务器 IP 地址"
        if ([string]::IsNullOrEmpty($remoteHost)) {
            Write-Log -Level "ERROR" -Message "服务器 IP 地址不能为空"
            return $null
        }
        
        $sshPortInput = Read-Host "请输入 SSH 端口号 [默认 22]"
        if ([string]::IsNullOrEmpty($sshPortInput)) {
            $sshPort = 22
        } else {
            $sshPort = [int]$sshPortInput
        }
        
        $remoteUserInput = Read-Host "请输入登录用户名 [默认 root]"
        if ([string]::IsNullOrEmpty($remoteUserInput)) {
            $remoteUser = "root"
        } else {
            $remoteUser = $remoteUserInput
        }
        
        $remotePassSecure = Read-Host "请输入登录密码" -AsSecureString
        $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($remotePassSecure)
        $remotePass = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)
        [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR($BSTR)
        
        if ([string]::IsNullOrEmpty($remotePass)) {
            Write-Log -Level "ERROR" -Message "登录密码不能为空"
            return $null
        }
        
        Write-Log -Level "INFO" -Message "已配置目标服务器: ${remoteUser}@${remoteHost}:${sshPort}"
        
        return @{
            IP = $remoteHost
            User = $remoteUser
            Pass = $remotePass
            Port = $sshPort
            Desc = "手动输入服务器"
        }
    }
    elseif ($ServerList.ContainsKey($serverKey)) {
        $server = $ServerList[$serverKey].Clone()
        $server.Port = 22
        Write-Log -Level "INFO" -Message "已选择 $($server.Desc) ($($server.IP))"
        return $server
    }
    else {
        Write-Log -Level "ERROR" -Message "编号 $serverKey 不存在，请重新运行脚本"
        return $null
    }
}

# ================================
# 测试 SSH 连接
# ================================
function Test-SSHConnection {
    param(
        [hashtable]$Server
    )
    
    Write-Log -Level "INFO" -Message "测试 SSH 连接..."
    
    $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command "echo CONNECTION_OK"
    
    if (($result.ExitCode -ne 0) -or ($result.Output -notmatch "CONNECTION_OK")) {
        Write-Log -Level "ERROR" -Message "SSH 连接失败！"
        Write-Log -Level "ERROR" -Message "输出信息: $($result.Output)"
        Write-Log -Level "ERROR" -Message "请检查: 1) IP地址是否正确 2) 端口是否正确 3) 密码是否正确 4) 网络是否可达"
        return $false
    }
    
    Write-Log -Level "SUCCESS" -Message "SSH 连接测试通过"
    return $true
}

# ================================
# 检测平台类型
# ================================
function Get-PlatformType {
    param(
        [hashtable]$Server
    )
    
    Write-Log -Level "INFO" -Message "自动检测目标服务器平台类型..."
    
    $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command "[ -d /data/services ] && echo 'NEW_PLATFORM' || echo 'OLD_PLATFORM'"
    
    $outputLines = $result.Output -split "`n" | Where-Object { $_ -match '\S' }
    $platformCheck = ($outputLines | Select-Object -Last 1).Trim()
    
    if ($platformCheck -eq "NEW_PLATFORM") {
        Write-Log -Level "SUCCESS" -Message "检测到 /data/services 目录存在，识别为【新统一平台】"
        return "new"
    }
    else {
        Write-Log -Level "SUCCESS" -Message "未检测到 /data/services 目录，识别为【传统平台】"
        return "old"
    }
}

# ================================
# 检测系统类型（容器类型）
# ================================
function Get-SystemType {
    param(
        [hashtable]$Server
    )
    
    Write-Log -Level "INFO" -Message "自动检测系统类型（容器）..."
    
    # 检测运行中的容器
    $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command "docker ps --format '{{.Names}}' 2>/dev/null || echo 'DOCKER_ERROR'"
    
    if ($result.Output -match "DOCKER_ERROR" -or $result.ExitCode -ne 0) {
        Write-Log -Level "WARN" -Message "Docker 命令执行失败，可能未安装 Docker"
        return @{
            HasUjava = $false
            HasUpython = $false
            HasUpythonVoice = $false
            Containers = @()
        }
    }
    
    $containers = $result.Output -split "`n" | Where-Object { $_ -match '\S' } | ForEach-Object { $_.Trim() }
    
    $systemInfo = @{
        HasUjava = $false
        HasUpython = $false
        HasUpythonVoice = $false
        Containers = $containers
        UjavaContainer = $null
        UpythonContainer = $null
        UpythonVoiceContainer = $null
    }
    
    foreach ($container in $containers) {
        if ($container -match '^ujava\d*$') {
            $systemInfo.HasUjava = $true
            $systemInfo.UjavaContainer = $container
            Write-Log -Level "INFO" -Message "  检测到 ujava 容器: $container"
        }
        elseif ($container -match '^upython\d*$' -and $container -notmatch 'voice') {
            $systemInfo.HasUpython = $true
            $systemInfo.UpythonContainer = $container
            Write-Log -Level "INFO" -Message "  检测到 upython 容器: $container -> 运维集控系统"
        }
        elseif ($container -match '^upython_voice\d*$') {
            $systemInfo.HasUpythonVoice = $true
            $systemInfo.UpythonVoiceContainer = $container
            Write-Log -Level "INFO" -Message "  检测到 upython_voice 容器: $container -> 转录系统"
        }
    }
    
    if (-not $systemInfo.HasUjava -and -not $systemInfo.HasUpython -and -not $systemInfo.HasUpythonVoice) {
        Write-Log -Level "WARN" -Message "  未检测到任何已知容器类型"
    }
    
    return $systemInfo
}

# ================================
# 检测 ujava 服务（支持容器内和宿主机）
# ================================
function Test-UjavaServices {
    param(
        [hashtable]$Server,
        [string]$ContainerName = $null,
        [string]$PlatformType = "new"
    )
    
    if ($ContainerName) {
        Write-Host ""
        Write-Log -Level "INFO" -Message "========== 检测 ujava 服务 (容器: $ContainerName) =========="
    }
    else {
        Write-Host ""
        Write-Log -Level "INFO" -Message "========== 检测 ujava 服务 (宿主机) =========="
    }
    
    $results = @()
    
    foreach ($serviceName in $UjavaServices.Keys) {
        $jarFileName = $UjavaServices[$serviceName]
        
        # 构建检测命令
        # 对于meeting2.0和meeting3.0，需要特殊处理，因为它们使用相同的jar文件名
        $checkCmd = $null
        $count = 0
        
        if ($ContainerName) {
            # 尝试在容器内检测
            if ($serviceName -eq "meeting2.0") {
                # meeting2.0 需要匹配路径中包含 java-meeting2.0
                $checkCmd = "docker exec $ContainerName ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | grep 'java-meeting2.0' | wc -l"
            }
            elseif ($serviceName -eq "meeting3.0") {
                # meeting3.0 需要匹配路径中包含 java-meeting3.0
                $checkCmd = "docker exec $ContainerName ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | grep 'java-meeting3.0' | wc -l"
            }
            else {
                # 其他服务直接匹配jar文件名
                $checkCmd = "docker exec $ContainerName ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | wc -l"
            }
        }
        else {
            # 在宿主机检测
            if ($serviceName -eq "meeting2.0") {
                $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | grep 'java-meeting2.0' | wc -l"
            }
            elseif ($serviceName -eq "meeting3.0") {
                $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | grep 'java-meeting3.0' | wc -l"
            }
            else {
                $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | wc -l"
            }
        }
        
        if ($checkCmd) {
            $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd
            
            try {
                $outputLines = $result.Output -split "`n" | Where-Object { $_ -match '^\d+$' }
                if ($outputLines) {
                    $count = [int]($outputLines | Select-Object -Last 1).Trim()
                }
            }
            catch {
                $count = 0
            }
        }
        
        # 如果容器内检测失败，尝试在宿主机检测（仅当有容器时）
        if ($count -eq 0 -and $ContainerName) {
            if ($serviceName -eq "meeting2.0") {
                $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | grep 'java-meeting2.0' | wc -l"
            }
            elseif ($serviceName -eq "meeting3.0") {
                $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | grep 'java-meeting3.0' | wc -l"
            }
            else {
                $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | wc -l"
            }
            
            $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd
            
            try {
                $outputLines = $result.Output -split "`n" | Where-Object { $_ -match '^\d+$' }
                if ($outputLines) {
                    $count = [int]($outputLines | Select-Object -Last 1).Trim()
                }
            }
            catch {
                $count = 0
            }
        }
        
        $status = if ($count -gt 0) { "运行中" } else { "未运行" }
        $statusColor = if ($count -gt 0) { "SUCCESS" } else { "ERROR" }
        
        $results += @{
            Service = $serviceName
            Pattern = $jarFileName
            Status = $status
            Running = ($count -gt 0)
        }
        
        $statusIcon = if ($count -gt 0) { "[OK]" } else { "[FAIL]" }
        $location = if ($ContainerName) { "容器内" } else { "宿主机" }
        Write-Log -Level $statusColor -Message "  $statusIcon $serviceName ($jarFileName) [$location]: $status"
    }
    
    return $results
}

# ================================
# 检测 ujava 宿主机服务（extapi）
# ================================
function Test-UjavaHostServices {
    param(
        [hashtable]$Server
    )
    
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 检测 ujava 宿主机服务 (extapi) =========="
    
    $results = @()
    
    foreach ($serviceName in $UjavaHostServices.Keys) {
        $jarFileName = $UjavaHostServices[$serviceName]
        
        # 在宿主机检查进程，使用jar文件名匹配（不依赖完整路径）
        $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | wc -l"
        $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd
        
        $count = 0
        try {
            $outputLines = $result.Output -split "`n" | Where-Object { $_ -match '^\d+$' }
            if ($outputLines) {
                $count = [int]($outputLines | Select-Object -Last 1).Trim()
            }
        }
        catch {
            $count = 0
        }
        
        $status = if ($count -gt 0) { "运行中" } else { "未运行" }
        $statusColor = if ($count -gt 0) { "SUCCESS" } else { "ERROR" }
        
        $results += @{
            Service = $serviceName
            Pattern = $jarFileName
            Status = $status
            Running = ($count -gt 0)
        }
        
        $statusIcon = if ($count -gt 0) { "[OK]" } else { "[FAIL]" }
        Write-Log -Level $statusColor -Message "  $statusIcon $serviceName ($jarFileName): $status"
    }
    
    return $results
}

# ================================
# 检测传统平台 ujava 容器内服务
# ================================
function Test-UjavaOldPlatformContainerServices {
    param(
        [hashtable]$Server,
        [string]$ContainerName
    )
    
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 检测传统平台 ujava 容器内服务 ($ContainerName) =========="
    
    $results = @()
    
    foreach ($serviceName in $UjavaOldPlatformContainerServices.Keys) {
        $pattern = $UjavaOldPlatformContainerServices[$serviceName]
        
        # 在容器内检查进程
        $checkCmd = "docker exec $ContainerName ps aux 2>/dev/null | grep -v grep | grep '$pattern' | wc -l"
        $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd
        
        $count = 0
        try {
            $outputLines = $result.Output -split "`n" | Where-Object { $_ -match '^\d+$' }
            if ($outputLines) {
                $count = [int]($outputLines | Select-Object -Last 1).Trim()
            }
        }
        catch {
            $count = 0
        }
        
        $status = if ($count -gt 0) { "运行中" } else { "未运行" }
        $statusColor = if ($count -gt 0) { "SUCCESS" } else { "ERROR" }
        
        $results += @{
            Service = $serviceName
            Pattern = $pattern
            Status = $status
            Running = ($count -gt 0)
        }
        
        $statusIcon = if ($count -gt 0) { "[OK]" } else { "[FAIL]" }
        Write-Log -Level $statusColor -Message "  $statusIcon $serviceName ($pattern): $status"
    }
    
    return $results
}

# ================================
# 检测传统平台 ujava 宿主机服务
# ================================
function Test-UjavaOldPlatformHostServices {
    param(
        [hashtable]$Server
    )
    
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 检测传统平台 ujava 宿主机服务 =========="
    
    $results = @()
    
    foreach ($serviceName in $UjavaOldPlatformHostServices.Keys) {
        $jarFileName = $UjavaOldPlatformHostServices[$serviceName]
        
        # 在宿主机检查进程，传统平台路径在 /var/www/java/external-meeting-api
        $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | grep '/var/www/java/external-meeting-api' | wc -l"
        $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd
        
        $count = 0
        try {
            $outputLines = $result.Output -split "`n" | Where-Object { $_ -match '^\d+$' }
            if ($outputLines) {
                $count = [int]($outputLines | Select-Object -Last 1).Trim()
            }
        }
        catch {
            $count = 0
        }
        
        # 如果没找到，尝试只匹配jar文件名（兼容性）
        if ($count -eq 0) {
            $checkCmd = "ps aux 2>/dev/null | grep -v grep | grep '$jarFileName' | wc -l"
            $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd
            
            try {
                $outputLines = $result.Output -split "`n" | Where-Object { $_ -match '^\d+$' }
                if ($outputLines) {
                    $count = [int]($outputLines | Select-Object -Last 1).Trim()
                }
            }
            catch {
                $count = 0
            }
        }
        
        $status = if ($count -gt 0) { "运行中" } else { "未运行" }
        $statusColor = if ($count -gt 0) { "SUCCESS" } else { "ERROR" }
        
        $results += @{
            Service = $serviceName
            Pattern = $jarFileName
            Status = $status
            Running = ($count -gt 0)
        }
        
        $statusIcon = if ($count -gt 0) { "[OK]" } else { "[FAIL]" }
        Write-Log -Level $statusColor -Message "  $statusIcon $serviceName ($jarFileName): $status"
    }
    
    return $results
}

# ================================
# 检测 DNS 解析
# ================================
function Test-DNSResolution {
    param(
        [hashtable]$Server
    )
    
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 检测 DNS 解析功能 =========="
    
    $results = @()
    
    # 1. 检查 /etc/resolv.conf 配置文件
    Write-Log -Level "INFO" -Message "检查 DNS 配置文件..."
    $resolvCheck = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command "cat /etc/resolv.conf 2>/dev/null | grep -E '^nameserver' | head -n 3"
    
    $dnsServers = @()
    if ($resolvCheck.ExitCode -eq 0 -and $resolvCheck.Output) {
        $dnsLines = $resolvCheck.Output -split "`n" | Where-Object { $_ -match 'nameserver' }
        foreach ($line in $dnsLines) {
            if ($line -match 'nameserver\s+(\S+)') {
                $dnsServers += $Matches[1]
            }
        }
        
        if ($dnsServers.Count -gt 0) {
            Write-Log -Level "SUCCESS" -Message "  检测到 DNS 服务器: $($dnsServers -join ', ')"
            $results += @{
                Check   = "DNS配置"
                Status  = "正常"
                Details = "DNS服务器: $($dnsServers -join ', ')"
                Success = $true
            }
        }
        else {
            Write-Log -Level "WARN" -Message "  未检测到 DNS 服务器配置"
            $results += @{
                Check   = "DNS配置"
                Status  = "异常"
                Details = "未找到DNS服务器配置"
                Success = $false
            }
        }
    }
    else {
        Write-Log -Level "WARN" -Message "  无法读取 DNS 配置文件"
        $results += @{
            Check   = "DNS配置"
            Status  = "异常"
            Details = "无法读取 /etc/resolv.conf"
            Success = $false
        }
    }
    
    # 2. 测试 DNS 解析功能
    Write-Log -Level "INFO" -Message "测试 DNS 解析功能..."
    $dnsTestSuccess = 0
    $dnsTestTotal   = $DNSTestDomains.Count
    
    foreach ($domain in $DNSTestDomains) {
        $testCmd    = "nslookup $domain 2>&1 | head -n 5 | grep -E 'Name:|Address:' | head -n 2"
        $testResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $testCmd
        
        if ($testResult.ExitCode -eq 0 -and $testResult.Output -match 'Name:|Address:') {
            $dnsTestSuccess++
            Write-Log -Level "SUCCESS" -Message "  [OK] $domain : 解析成功"
        }
        else {
            # 尝试使用 host 命令
            $testCmd2    = "host $domain 2>&1 | head -n 1"
            $testResult2 = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $testCmd2
            
            if ($testResult2.ExitCode -eq 0 -and $testResult2.Output -match 'has address|has IPv4') {
                $dnsTestSuccess++
                Write-Log -Level "SUCCESS" -Message "  [OK] $domain : 解析成功"
            }
            else {
                Write-Log -Level "ERROR" -Message "  [FAIL] $domain : 解析失败"
            }
        }
    }
    
    $dnsTestStatus = if ($dnsTestSuccess -eq $dnsTestTotal) { "正常" } elseif ($dnsTestSuccess -gt 0) { "部分正常" } else { "异常" }
    $dnsTestColor  = if ($dnsTestSuccess -eq $dnsTestTotal) { "SUCCESS" } elseif ($dnsTestSuccess -gt 0) { "WARN" } else { "ERROR" }
    
    Write-Log -Level $dnsTestColor -Message "  DNS 解析测试结果: $dnsTestSuccess/$dnsTestTotal 成功"
    
    $results += @{
        Check        = "DNS解析"
        Status       = $dnsTestStatus
        Details      = "测试域名解析: $dnsTestSuccess/$dnsTestTotal 成功"
        Success      = ($dnsTestSuccess -gt 0)
        SuccessCount = $dnsTestSuccess
        TotalCount   = $dnsTestTotal
    }
    
    # 3. 测试 ping 连通性（可选，验证DNS解析的IP是否可达）
    Write-Log -Level "INFO" -Message "测试网络连通性..."
    $pingSuccess = 0
    $pingTotal   = 0
    
    foreach ($domain in $DNSTestDomains) {
        $pingCmd    = "ping -c 2 -W 2 $domain 2>&1 | grep -E 'packets transmitted|0% packet loss' | head -n 1"
        $pingResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $pingCmd
        
        $pingTotal++
        if ($pingResult.ExitCode -eq 0 -and $pingResult.Output -match '0% packet loss|packets transmitted') {
            $pingSuccess++
            Write-Log -Level "SUCCESS" -Message "  [OK] $domain : 网络连通正常"
        }
        else {
            Write-Log -Level "WARN" -Message "  [WARN] $domain : 网络连通异常或超时"
        }
    }
    
    if ($pingTotal -gt 0) {
        $pingStatus = if ($pingSuccess -eq $pingTotal) { "正常" } elseif ($pingSuccess -gt 0) { "部分正常" } else { "异常" }
        $pingColor  = if ($pingSuccess -eq $pingTotal) { "SUCCESS" } elseif ($pingSuccess -gt 0) { "WARN" } else { "ERROR" }
        
        Write-Log -Level $pingColor -Message "  网络连通性测试结果: $pingSuccess/$pingTotal 成功"
        
        $results += @{
            Check        = "网络连通性"
            Status       = $pingStatus
            Details      = "Ping测试: $pingSuccess/$pingTotal 成功"
            Success      = ($pingSuccess -gt 0)
            SuccessCount = $pingSuccess
            TotalCount   = $pingTotal
        }
    }

    # 4. 如有 DNS 解析异常，则触发远程修复（参考 NTP，平台固定 auto）
    $needRepair = $false
    foreach ($item in $results) {
        if ($item.Check -eq 'DNS配置' -and -not $item.Success) { $needRepair = $true; break }
        if ($item.Check -eq 'DNS解析' -and -not $item.Success) { $needRepair = $true; break }
    }

    if ($needRepair) {
        Write-Log -Level "WARN" -Message "[DNS] 检测到 DNS 解析异常，准备执行远程修复 (fix_dns_config)"

        try {
            $serverForRepair = @{ IP = $Server.IP; User = $Server.User; Pass = $Server.Pass; Port = $Server.Port }
            $repairRes = Upload_the_repair_script -Server $serverForRepair -Action "fix_dns_config" -Platform "auto" -RemoteDir "/home/repair_scripts"

            # 在结果集中追加一条“修复记录”，方便报告展示
            $repairItem = [ordered]@{
                Check   = "DNS修复"
                Status  = "未执行"
                Details = ""
                Success = $false
            }

            if ($repairRes -and $repairRes['Success']) {
                Write-Log -Level "SUCCESS" -Message "[DNS] 远程 DNS 修复已执行成功 (fix_dns_config)"
                $repairItem.Status  = "已执行"
                $repairItem.Details = "远程脚本执行成功 (fix_dns_config)"
                $repairItem.Success = $true

                # 简单复检：尝试解析一个域名
                Write-Log -Level "INFO" -Message "[DNS] 修复后复检 DNS 解析..."
                $postCmd    = "nslookup www.baidu.com 2>&1 | head -n 5 | grep -E 'Name:|Address:' | head -n 2"
                $postResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $postCmd
                if ($postResult.ExitCode -eq 0 -and $postResult.Output -match 'Name:|Address:') {
                    Write-Log -Level "SUCCESS" -Message "[DNS] 复检成功，DNS 解析已恢复正常 (www.baidu.com)"
                    $repairItem.Details += " | 复检成功 (www.baidu.com)"
                }
                else {
                    Write-Log -Level "WARN" -Message "[DNS] 复检仍失败，请人工进一步排查"
                    $repairItem.Status  = "部分成功"
                    $repairItem.Details += " | 复检仍失败，请人工排查"
                }
            }
            else {
                $errMsg = "未知错误"
                if ($repairRes -is [hashtable]) {
                    if ($repairRes.ContainsKey('Error') -and $repairRes['Error'])   { $errMsg = [string]::Join(' ', $repairRes['Error']) }
                    elseif ($repairRes.ContainsKey('Output') -and $repairRes['Output']) { $errMsg = [string]::Join(' ', $repairRes['Output']) }
                    elseif ($repairRes.ContainsKey('Message') -and $repairRes['Message']) { $errMsg = $repairRes['Message'] }
                } elseif ($repairRes) {
                    $errMsg = $repairRes.ToString()
                }
                Write-Log -Level "ERROR" -Message "[DNS] 远程 DNS 修复执行失败: $errMsg"
                $repairItem.Status  = "失败"
                $repairItem.Details = "远程修复失败: $errMsg"
            }

            $results += $repairItem
        }
        catch {
            Write-Log -Level "ERROR" -Message "[DNS] 调用 Upload_the_repair_script 异常: $($_.Exception.Message)"
            $results += @{
                Check   = "DNS修复"
                Status  = "异常"
                Details = "调用修复脚本异常: $($_.Exception.Message)"
                Success = $false
            }
        }
    }

    return $results
}

# ================================
# 服务器资源分析
# ================================
function Test-ServerResources {
    param(
        [hashtable]$Server
    )
    
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 服务器资源分析 =========="
    
    $results = @{
        OS = $null
        Architecture = $null
        CPU = $null
        Memory = $null
        Disk = @()
        Firewall = $null
    }
    
    # 1. 检测操作系统信息
    Write-Log -Level "INFO" -Message "检测操作系统信息..."
    $osCmd = "cat /etc/os-release 2>/dev/null | grep -E '^(NAME|VERSION)=' | head -n 2 || cat /etc/redhat-release 2>/dev/null || uname -o"
    $osResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $osCmd
    
    if ($osResult.ExitCode -eq 0 -and $osResult.Output) {
        $osInfo = ($osResult.Output -split "`n" | Where-Object { $_ -match '\S' }) -join " | "
        $osInfo = $osInfo -replace 'NAME=|VERSION=|"', ''
        Write-Log -Level "SUCCESS" -Message "  操作系统: $osInfo"
        $results.OS = @{
            Info = $osInfo
            Status = "正常"
            Success = $true
        }
    }
    else {
        Write-Log -Level "WARN" -Message "  无法获取操作系统信息"
        $results.OS = @{
            Info = "未知"
            Status = "未知"
            Success = $false
        }
    }
    
    # 2. 检测服务器架构
    Write-Log -Level "INFO" -Message "检测服务器架构..."
    $archCmd = "uname -m && uname -r"
    $archResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $archCmd
    
    if ($archResult.ExitCode -eq 0 -and $archResult.Output) {
        $archLines = $archResult.Output -split "`n" | Where-Object { $_ -match '\S' }
        $arch = if ($archLines.Count -ge 1) { $archLines[0].Trim() } else { "未知" }
        $kernel = if ($archLines.Count -ge 2) { $archLines[1].Trim() } else { "未知" }
        Write-Log -Level "SUCCESS" -Message "  架构: $arch | 内核: $kernel"
        $results.Architecture = @{
            Arch = $arch
            Kernel = $kernel
            Status = "正常"
            Success = $true
        }
    }
    else {
        Write-Log -Level "WARN" -Message "  无法获取架构信息"
        $results.Architecture = @{
            Arch = "未知"
            Kernel = "未知"
            Status = "未知"
            Success = $false
        }
    }
    
    # 3. 检测 CPU 使用情况
    Write-Log -Level "INFO" -Message "检测 CPU 使用情况..."
    $cpuCmd = "top -bn1 | grep 'Cpu(s)' | awk '{print `$2+`$4}' 2>/dev/null || mpstat 1 1 2>/dev/null | tail -n 1 | awk '{print 100-`$NF}'"
    $cpuResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $cpuCmd
    
    $cpuUsage = 0
    if ($cpuResult.ExitCode -eq 0 -and $cpuResult.Output) {
        $cpuLine = ($cpuResult.Output -split "`n" | Where-Object { $_ -match '^\d' } | Select-Object -First 1)
        if ($cpuLine) {
            try {
                $cpuUsage = [math]::Round([double]$cpuLine.Trim(), 1)
            }
            catch {
                $cpuUsage = 0
            }
        }
    }
    
    # 获取 CPU 核心数
    $cpuCoresCmd = "nproc 2>/dev/null || grep -c processor /proc/cpuinfo"
    $cpuCoresResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $cpuCoresCmd
    $cpuCores = 0
    if ($cpuCoresResult.ExitCode -eq 0 -and $cpuCoresResult.Output) {
        try {
            $cpuCores = [int]($cpuCoresResult.Output -split "`n" | Where-Object { $_ -match '^\d+$' } | Select-Object -First 1).Trim()
        }
        catch {
            $cpuCores = 0
        }
    }
    
    $cpuStatus = if ($cpuUsage -lt 70) { "正常" } elseif ($cpuUsage -lt 90) { "警告" } else { "危险" }
    $cpuColor = if ($cpuUsage -lt 70) { "SUCCESS" } elseif ($cpuUsage -lt 90) { "WARN" } else { "ERROR" }
    
    Write-Log -Level $cpuColor -Message "  CPU 使用率: ${cpuUsage}% (核心数: $cpuCores) [$cpuStatus]"
    $results.CPU = @{
        Usage = $cpuUsage
        Cores = $cpuCores
        Status = $cpuStatus
        Success = ($cpuUsage -lt 90)
    }
    
    # 4. 检测内存使用情况
    Write-Log -Level "INFO" -Message "检测内存使用情况..."
    # 使用 free -m（任一路径），简单 awk 计算，避免复杂转义
    $memCmd = @'
LC_ALL=C (
  /usr/bin/free -m 2>/dev/null || /bin/free -m 2>/dev/null || free -m
) | awk -F"[[:space:]]+" '
$1 == "Mem:" {
  total = $2;
  used_field = $3;
  free_field = $4;
  buffcache = $6;
  avail = $7;
  # 如果 available 字段不存在或为 0，就用 used 字段
  if (avail == "" || avail == 0) {
    used = used_field;
  } else {
    used = total - avail;
  }
  pct = 0;
  if (total > 0) { pct = used * 100 / total; }
  printf "%.2f,%.2f,%.1f\n", total/1024, used/1024, pct;
}'
'@
    $memResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $memCmd

    $memTotal = 0.0; $memUsed = 0.0; $memPercent = 0.0
    if ($memResult.ExitCode -eq 0 -and $memResult.Output) {
        $line = ($memResult.Output -split "`n" | Where-Object { $_ -match '\S' } | Select-Object -First 1)
        Write-Log -Level "INFO" -Message "free 输出(raw): $line"
        if ($line) {
            $parts = ($line.Trim() -replace "`r","") -split ','
            if ($parts.Count -ge 3) {
                try {
                    $ci = [System.Globalization.CultureInfo]::InvariantCulture
                    $memTotal   = [double]::Parse($parts[0], $ci)
                    $memUsed    = [double]::Parse($parts[1], $ci)
                    $memPercent = [double]::Parse($parts[2], $ci)
                } catch {
                    Write-Log -Level "WARN" -Message "free 输出解析失败: $line"
                }
            }
        }
    }

    # Fallback：纯 shell 读取 /proc/meminfo（不依赖 grep/sed/awk）
    if ($memTotal -le 0) {
        $fallbackCmd = @'
total_kb=0; avail_kb=0
while IFS=: read k v; do
  case "$k" in
    "MemTotal")     total_kb=${v//[^0-9]/};;
    "MemAvailable") avail_kb=${v//[^0-9]/};;
    "MemFree")      if [ -z "$avail_kb" ] || [ "$avail_kb" -eq 0 ]; then avail_kb=${v//[^0-9]/}; fi;;
  esac
done < /proc/meminfo
used_kb=$(( total_kb - avail_kb ))
pct=0
if [ "$total_kb" -gt 0 ]; then pct=$(( used_kb * 100 / total_kb )); fi
tot_gb=$(( total_kb / 1024 / 1024 ))
use_gb=$(( used_kb / 1024 / 1024 ))
printf "%d,%d,%d\n" "$tot_gb" "$use_gb" "$pct"
'@
        $fbRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $fallbackCmd
        if ($fbRes.ExitCode -eq 0 -and $fbRes.Output) {
            $fbLine = ($fbRes.Output -split "`n" | Where-Object { $_ -match '\S' } | Select-Object -First 1)
            Write-Log -Level "INFO" -Message "meminfo Fallback 输出(raw): $fbLine"
            if ($fbLine) {
                $fbParts = ($fbLine.Trim() -replace "`r","") -split ','
                if ($fbParts.Count -ge 3) {
                    $memTotal   = [double]$fbParts[0]
                    $memUsed    = [double]$fbParts[1]
                    $memPercent = [double]$fbParts[2]
                }
            }
        }
    }

    if ($memTotal -gt 0) {
        if ($memUsed -lt 0) { $memUsed = 0 }
        if ($memUsed -gt $memTotal) { $memUsed = $memTotal }
        $memPercent = [math]::Round(($memUsed / $memTotal) * 100, 1)
        $memTotal   = [math]::Round($memTotal, 2)
        $memUsed    = [math]::Round($memUsed, 2)
    } else {
        Write-Log -Level "WARN" -Message "内存信息获取失败，目标机缺少可用的 free/awk 或 /proc/meminfo 读取异常"
    }

    $memStatus = if ($memPercent -lt 70) { "正常" } elseif ($memPercent -lt 90) { "警告" } else { "危险" }
    $memColor = if ($memPercent -lt 70) { "SUCCESS" } elseif ($memPercent -lt 90) { "WARN" } else { "ERROR" }
    Write-Log -Level $memColor -Message "  内存使用: ${memUsed}GB / ${memTotal}GB (${memPercent}%) [$memStatus]"
    $results.Memory = @{ Total = $memTotal; Used = $memUsed; Percent = $memPercent; Status = $memStatus; Success = ($memPercent -lt 90) }
    

    # 5. 检测磁盘空间情况
    Write-Log -Level "INFO" -Message "检测磁盘空间情况..."
    $diskCmd = "df -h | grep -E '^/dev/' | awk '{print `$1,`$2,`$3,`$5,`$6}'"
    $diskResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $diskCmd
    
    $diskList = @()
    $diskWarning = $false
    
    if ($diskResult.ExitCode -eq 0 -and $diskResult.Output) {
        $diskLines = $diskResult.Output -split "`n" | Where-Object { $_ -match '\S' }
        foreach ($line in $diskLines) {
            $parts = $line -split '\s+'
            if ($parts.Count -ge 5) {
                $device = $parts[0]
                $size = $parts[1]
                $used = $parts[2]
                $usePercent = $parts[3] -replace '%', ''
                $mountPoint = $parts[4]
                
                try {
                    $usePercentNum = [int]$usePercent
                }
                catch {
                    $usePercentNum = 0
                }
                
                $diskStatus = if ($usePercentNum -lt 70) { "正常" } elseif ($usePercentNum -lt 90) { "警告" } else { "危险" }
                $diskColor = if ($usePercentNum -lt 70) { "SUCCESS" } elseif ($usePercentNum -lt 90) { "WARN" } else { "ERROR" }
                
                if ($usePercentNum -ge 70) {
                    $diskWarning = $true
                }
                
                Write-Log -Level $diskColor -Message "  磁盘 $mountPoint : ${used}/${size} (${usePercent}%) [$diskStatus]"
                
                $diskList += @{
                    Device = $device
                    Size = $size
                    Used = $used
                    Percent = $usePercentNum
                    MountPoint = $mountPoint
                    Status = $diskStatus
                }
            }
        }
    }
    else {
        Write-Log -Level "WARN" -Message "  无法获取磁盘信息"
    }
    
    $results.Disk = $diskList
    
    # 6. 检测防火墙开放端口情况
    Write-Log -Level "INFO" -Message "检测防火墙开放端口..."

    # 先检测防火墙状态
    $firewallStatusCmd = "systemctl is-active firewalld 2>/dev/null || service iptables status 2>/dev/null | head -n 1 || echo 'unknown'"
    $firewallStatusResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $firewallStatusCmd

    $firewallActive = $false
    $firewallType = "unknown"

    # 严格匹配 firewalld 的 is-active
    $statusLine = ($firewallStatusResult.Output | Select-Object -First 1)
    if ($statusLine) { $statusLine = $statusLine.Trim().ToLower() } else { $statusLine = "unknown" }

    if ($statusLine -eq "active") {
        $firewallActive = $true
        $firewallType = "firewalld"
    } elseif ($statusLine -eq "inactive" -or $statusLine -eq "failed" -or $statusLine -eq "unknown") {
        $firewallActive = $false
        $ipLine = ($firewallStatusResult.Output | Select-Object -Last 1)
        if ($ipLine) {
            $ipl = $ipLine.Trim().ToLower()
            if ($ipl -match '\brunning\b' -or $ipl -match '\bok\b') {
                $firewallActive = $true
                $firewallType = "iptables"
            } elseif ($ipl -match 'stopped|not running|inactive|failed') {
                $firewallActive = $false
                $firewallType = "iptables"
            }
        }
    }

    # 获取当前开放端口
    $openPorts = @()
    if ($firewallActive) {
        if ($firewallType -eq "firewalld") {
            $portsCmd = "firewall-cmd --list-ports 2>/dev/null && firewall-cmd --list-services 2>/dev/null"
            $portsResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $portsCmd
            if ($portsResult.ExitCode -eq 0 -and $portsResult.Output) {
                $openPorts = ($portsResult.Output -split "`n" | Where-Object { $_ -match '\S' }) -join ", "
            }
        } else {
            $portsCmd = "iptables -L INPUT -n 2>/dev/null | grep ACCEPT | grep -oP 'dpt:\d+' | cut -d: -f2 | sort -u | head -n 50"
            $portsResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $portsCmd
            if ($portsResult.ExitCode -eq 0 -and $portsResult.Output) {
                $openPorts = ($portsResult.Output -split "`n" | Where-Object { $_ -match '^\d+$' }) -join ", "
            }
        }
    } else {
        $openPorts = "防火墙未启用"
    }

    # 记录初始快照（用于报告时间线）
    $results.Firewall = @{
        Active = $firewallActive
        Type = $firewallType
        OpenPorts = $openPorts
        Status = if ($firewallActive) { "已启用" } else { "未启用" }
        Pre = @{
            Active = $firewallActive
            Type = $firewallType
            OpenPorts = $openPorts
        }
    }

    # 在资源分析阶段就直接打印防火墙概要
    if ($firewallActive) {
        Write-Log -Level "INFO" -Message ("[FIREWALL] 当前状态: 已启用 ({0})" -f $firewallType)
        if ($openPorts -and $openPorts -ne "") {
            Write-Log -Level "INFO" -Message ("[FIREWALL] 开放端口/服务: {0}" -f $openPorts)
        } else {
            Write-Log -Level "INFO" -Message "[FIREWALL] 未检测到具体开放端口列表"
        }
    } else {
        Write-Log -Level "WARN" -Message ("[FIREWALL] 当前状态: 未启用 ({0})" -f $firewallType)
        if ($openPorts -and $openPorts -is [string]) {
            Write-Log -Level "INFO" -Message ("[FIREWALL] 描述: {0}" -f $openPorts)
        }
    }

    # 触发远端修复（仅当未启用或类型未知）
    if (-not $firewallActive -or ($firewallType -eq "unknown")) {
        Write-Log -Level "WARN" -Message "[FIREWALL] 检测到防火墙未启用或状态异常，准备执行远端修复"
        try {
            $serverForRepair = @{ IP = $Server.IP; User = $Server.User; Pass = $Server.Pass; Port = $Server.Port }
            Write-Log -Level "INFO" -Message "[FIREWALL] 触发远端修复: ./issue_handler.sh --action fix_port_access --platform auto --non-interactive"
            $fwRepairRes = Upload_the_repair_script -Server $serverForRepair -Action "fix_port_access" -Platform "auto" -RemoteDir "/home/repair_scripts"
            # 默认标记为已尝试
            $results.Firewall.Repair = @{ Attempted = $true; Succeeded = $false; Message = "fix_port_access (platform=auto)" }

            if ($fwRepairRes -and $fwRepairRes['Success']) {
                Write-Log -Level "SUCCESS" -Message "[FIREWALL] 远端修复已执行成功 (fix_port_access)"
                $results.Firewall.Repair.Succeeded = $true

                # 修复后复检
                $firewallActive = $false; $firewallType = "unknown"
                $firewallStatusResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $firewallStatusCmd
                $statusLine = ($firewallStatusResult.Output | Select-Object -First 1)
                if ($statusLine) { $statusLine = $statusLine.Trim().ToLower() } else { $statusLine = "unknown" }
                if ($statusLine -eq "active") { $firewallActive = $true; $firewallType = "firewalld" }
                else {
                    $ipLine = ($firewallStatusResult.Output | Select-Object -Last 1)
                    if ($ipLine) {
                        $ipl = $ipLine.Trim().ToLower()
                        if ($ipl -match '\brunning\b' -or $ipl -match '\bok\b') { $firewallActive = $true; $firewallType = "iptables" }
                        elseif ($ipl -match 'stopped|not running|inactive|failed') { $firewallActive = $false; $firewallType = "iptables" }
                    }
                }

                # 修复后端口
                $openPorts = @()
                if ($firewallActive) {
                    if ($firewallType -eq "firewalld") {
                        $portsCmd = "firewall-cmd --list-ports 2>/dev/null && firewall-cmd --list-services 2>/dev/null"
                        $portsResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $portsCmd
                        if ($portsResult.ExitCode -eq 0 -and $portsResult.Output) {
                            $openPorts = ($portsResult.Output -split "`n" | Where-Object { $_ -match '\S' }) -join ", "
                        }
                    } else {
                        $portsCmd = "iptables -L INPUT -n 2>/dev/null | grep ACCEPT | grep -oP 'dpt:\d+' | cut -d: -f2 | sort -u | head -n 50"
                        $portsResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $portsCmd
                        if ($portsResult.ExitCode -eq 0 -and $portsResult.Output) {
                            $openPorts = ($portsResult.Output -split "`n" | Where-Object { $_ -match '^\d+$' }) -join ", "
                        }
                    }
                }

                # 更新最终状态
                $results.Firewall.Active = $firewallActive
                $results.Firewall.Type = $firewallType
                $results.Firewall.OpenPorts = $openPorts
                $results.Firewall.Status = if ($firewallActive) { "已启用" } else { "未启用" }
                
                # 修复后复检结果输出
                if ($firewallActive) {
                    Write-Log -Level "INFO" -Message ("[FIREWALL] 修复后状态: 已启用 ({0})" -f $firewallType)
                    if ($openPorts -and $openPorts -ne "") {
                        Write-Log -Level "INFO" -Message ("[FIREWALL] 修复后开放端口/服务: {0}" -f $openPorts)
                    } else {
                        Write-Log -Level "INFO" -Message "[FIREWALL] 修复后仍未检测到具体开放端口列表"
                    }
                } else {
                    Write-Log -Level "WARN" -Message ("[FIREWALL] 修复后状态仍为未启用 ({0})" -f $firewallType)
                }

            } else {
                # 记录失败消息
                $errMsg = "未知错误"
                if ($fwRepairRes -is [hashtable]) {
                    if ($fwRepairRes.ContainsKey('Error') -and $fwRepairRes['Error']) { $errMsg = [string]::Join(' ', $fwRepairRes['Error']) }
                    elseif ($fwRepairRes.ContainsKey('Output') -and $fwRepairRes['Output']) { $errMsg = [string]::Join(' ', $fwRepairRes['Output']) }
                    elseif ($fwRepairRes.ContainsKey('Message') -and $fwRepairRes['Message']) { $errMsg = $fwRepairRes['Message'] }
                } elseif ($fwRepairRes) { $errMsg = $fwRepairRes.ToString() }
                Write-Log -Level "ERROR" -Message "[FIREWALL] 远端修复执行失败: $errMsg"
                $results.Firewall.Repair.Message = "修复失败: $errMsg"
            }
        } catch {
            Write-Log -Level "ERROR" -Message "[FIREWALL] 调用 Upload_the_repair_script 异常: $($_.Exception.Message)"
            $results.Firewall.Repair = @{ Attempted = $true; Succeeded = $false; Message = "异常: $($_.Exception.Message)" }
        }
    }
    
    # 7. 检测系统负载
    Write-Log -Level "INFO" -Message "检测系统负载..."
    $loadCmd = "uptime | awk -F'load average:' '{print `$2}' | tr -d ' '"
    $loadResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $loadCmd
    
    if ($loadResult.ExitCode -eq 0 -and $loadResult.Output) {
        $loadAvg = ($loadResult.Output -split "`n" | Where-Object { $_ -match '\S' } | Select-Object -First 1).Trim()
        $loadParts = $loadAvg -split ','
        if ($loadParts.Count -ge 1) {
            $load1 = $loadParts[0].Trim()
            Write-Log -Level "INFO" -Message "  系统负载 (1/5/15分钟): $loadAvg"
        }
    }
    
    return $results
}

# ================================
# 远端修复对外会议服务断连
# ================================
function Repair-ExternalMeetingService {
    param(
        [Parameter(Mandatory = $true)]
        [hashtable]$Server
    )

    Write-Log -Level "INFO" -Message "[EXT] 触发远端修复: ./issue_handler.sh --action fix_external_service_disconnect"

    # 这里直接使用传入的 hashtable，不再转成 pscustomobject
    $serverForRepair = $Server

    try {
        # 调用上传+执行修复脚本（与 NTP、防火墙相同模式）
        $repairRes = Upload_the_repair_script -Server $serverForRepair -Action "fix_external_service_disconnect" -Platform "auto" -RemoteDir "/home/repair_scripts"
        if ($repairRes -and $repairRes['Success']) {
            Write-Log -Level "SUCCESS" -Message "[EXT] 远端修复已执行成功 (fix_external_service_disconnect)"

            # 修复后复检：检查宿主机 ubains-meeting-api-1.0-SNAPSHOT.jar 是否已启动
            Write-Log -Level "INFO" -Message "[EXT] 修复后复检对外服务进程..."
            $checkCmd = "ps aux | grep -v grep | grep ubains-meeting-api-1.0-SNAPSHOT.jar"
            $post = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd

            $out = if ($post.Output) { ($post.Output -join ' ') } else { '' }

            if ($post.ExitCode -eq 0 -and $out -match 'ubains-meeting-api-1\.0-SNAPSHOT\.jar') {
                Write-Log -Level "SUCCESS" -Message "[EXT] 复检成功，对外服务进程已启动"
                return [pscustomobject]@{
                    Target    = "external-meeting-api"
                    Attempted = $true
                    Success   = $true
                    Detail    = "进程已启动: $out"
                }
            } else {
                Write-Log -Level "WARN" -Message "[EXT] 复检失败，对外服务进程仍未检测到"
                return [pscustomobject]@{
                    Target    = "external-meeting-api"
                    Attempted = $true
                    Success   = $false
                    Detail    = "复检未检测到进程: $out"
                }
            }
        } else {
            Write-Log -Level "ERROR" -Message "[EXT] 远端修复执行失败 (fix_external_service_disconnect)"
            return [pscustomobject]@{
                Target    = "external-meeting-api"
                Attempted = $true
                Success   = $false
                Detail    = "Upload_the_repair_script 返回失败"
            }
        }
    }
    catch {
        Write-Log -Level "ERROR" -Message ("[EXT] 远端修复异常: {0}" -f $_.Exception.Message)
        return [pscustomobject]@{
            Target    = "external-meeting-api"
            Attempted = $true
            Success   = $false
            Detail    = "异常: $($_.Exception.Message)"
        }
    }
}

# ================================
# 服务器容器信息收集
# ================================
function Test-ContainerInformation {
    param(
        [hashtable]$Server,
        [switch]$PrintDetails   # ✅ 新增：是否在控制台打印完整明细
    )

    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 容器信息收集 =========="

    $results = @()

    # 1. 查询所有容器信息（运行 + 未运行）
    $cmd = "docker ps -a --format '{{.ID}} {{.Names}} {{.Status}}'"
    $dockerInfo = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $cmd

    if ($dockerInfo.ExitCode -ne 0) {
        Write-Log -Level "ERROR" -Message "  无法获取容器信息：$($dockerInfo.Error -join ' ')"
        $results += @{
            Check   = "容器信息收集"
            Status  = "失败"
            Details = "docker ps -a 执行失败"
            Success = $false
        }
        return $results
    }

    $runningContainers  = @()
    $stoppedContainers  = @()

    foreach ($line in $dockerInfo.Output) {
        if (-not $line.Trim()) { continue }

        # 简单解析：ID NAME STATUS...
        $parts = $line -split '\s+', 3
        if ($parts.Count -lt 3) { continue }

        $id     = $parts[0]
        $name   = $parts[1]
        $status = $parts[2]

        # 可以在这里按需补充 docker inspect 获取 MAC/端口/启动命令等（如已有可保留）
        $item = [ordered]@{
            Id     = $id
            Name   = $name
            Status = $status
        }

        if ($status -match '^Up') {
            $runningContainers += $item
        } else {
            $stoppedContainers += $item
        }
    }

    # 2.1 生成“逐容器明细”（用于报告 Markdown）
    $containerDetailsMd = New-Object System.Collections.Generic.List[string]

    # 容器总数/运行/停止
    $containerDetailsMd.Add(("- 容器总数: {0}" -f ($runningContainers.Count + $stoppedContainers.Count)))
    $containerDetailsMd.Add(("- 运行中: {0}" -f $runningContainers.Count))
    $containerDetailsMd.Add(("- 已停止: {0}" -f $stoppedContainers.Count))

    # 逐个容器补充信息（镜像/端口/IP/挂载/大小等）
    $allContainers = @()
    $allContainers += $runningContainers
    $allContainers += $stoppedContainers

    foreach ($c in $allContainers) {
        $name = $c.Name

        # 镜像
        $imgCmd = "docker inspect -f '{{.Config.Image}}' $name 2>/dev/null"
        $imgRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $imgCmd
        $img = (($imgRes.Output -split "`n" | Where-Object { $_ -match '\S' } | Select-Object -First 1) -as [string])
        if ([string]::IsNullOrWhiteSpace($img)) { $img = "-" }

        # 状态/健康/重启策略/重启次数
        $stateCmd = "docker inspect -f '{{.State.Status}}|{{if .State.Health}}{{.State.Health.Status}}{{else}}-{{end}}|{{.HostConfig.RestartPolicy.Name}}|{{.RestartCount}}' $name 2>/dev/null"
        $stateRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $stateCmd
        $stateLine = (($stateRes.Output -split "`n" | Where-Object { $_ -match '\S' } | Select-Object -First 1) -as [string])
        $st = "-"; $health = "-"; $restart = "-"; $restartCount = "0"
        if ($stateLine -and $stateLine -match '\|') {
            $parts = $stateLine.Trim() -split '\|', 4
            if ($parts.Count -ge 1) { $st = $parts[0] }
            if ($parts.Count -ge 2) { $health = $parts[1] }
            if ($parts.Count -ge 3) { $restart = $parts[2] }
            if ($parts.Count -ge 4) { $restartCount = $parts[3] }
        }

        # IP（可能多个网络，拼成第一个非空）
        # ✅ 修复：Go template 的 $k/$v 在 PowerShell 双引号内会被当成变量，必须转义为 `$k/`$v
        $ipCmd = "docker inspect -f '{{range `$k,`$v := .NetworkSettings.Networks}}{{if `$v.IPAddress}}{{`$v.IPAddress}} {{end}}{{end}}' $name 2>/dev/null"
        $ipRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $ipCmd
        $ip = (($ipRes.Output -split "`n" | Where-Object { $_ -match '\S' } | Select-Object -First 1) -as [string])
        $ip = ($ip -replace "`r","").Trim()
        if ([string]::IsNullOrWhiteSpace($ip)) { $ip = "-" }

        # 端口
        $portCmd = "docker port $name 2>/dev/null | tr '\n' ',' | sed 's/,$//'"
        $portRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $portCmd
        $ports = (($portRes.Output -split "`n" | Select-Object -First 1) -as [string])
        if ([string]::IsNullOrWhiteSpace($ports)) { $ports = "-" }

        # 网络（network:ip）
        # ✅ 修复：同上，转义 Go template 的 $k/$v
        $netCmd = "docker inspect -f '{{range `$k,`$v := .NetworkSettings.Networks}}{{`$k}}:{{if `$v.IPAddress}}{{`$v.IPAddress}}{{else}}-{{end}} {{end}}' $name 2>/dev/null"
        $netRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $netCmd
        $nets = (($netRes.Output -split "`n" | Where-Object { $_ -match '\S' } | Select-Object -First 1) -as [string])
        $nets = ($nets -replace "`r","").Trim()
        if ([string]::IsNullOrWhiteSpace($nets)) { $nets = "-" }

        # 挂载
        $mntCmd = "docker inspect -f '{{range .Mounts}}{{if .Source}}{{.Source}}{{else}}-{{end}}:{{.Destination}}({{.Mode}}{{if .RW}},rw{{else}},ro{{end}}); {{end}}' $name 2>/dev/null"
        $mntRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $mntCmd
        $mounts = (($mntRes.Output -split "`n" | Select-Object -First 1) -as [string])
        $mounts = ($mounts -replace "`r","").Trim()
        if ([string]::IsNullOrWhiteSpace($mounts)) { $mounts = "-" }

        # 大小（rw / virtual），注意不同 docker 版本输出差异
        $sizeCmd = "docker ps -a --size --filter ""name=^/$name$"" --format ""{{.Size}}"" 2>/dev/null | head -n 1"
        $sizeRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $sizeCmd
        $sizeLine = (($sizeRes.Output -split "`n" | Select-Object -First 1) -as [string])
        $sizeLine = ($sizeLine -replace "`r","").Trim()
        $sizeRw = "-"; $sizeRoot = "-"
        if (-not [string]::IsNullOrWhiteSpace($sizeLine)) {
            # 例如： "12.3MB (virtual 1.23GB)"
            $sizeRw = ($sizeLine -split '\s+\(',2)[0]
            if ($sizeLine -match '\(virtual\s+([^)]+)\)') { $sizeRoot = $Matches[1] }
        }

        $icon = if ($st -eq "running" -or $c.Status -match '^Up') { "✅" } else { "❌" }
        $containerDetailsMd.Add(("- {0} 名称: {1} | 镜像: {2} | 状态: {3} | 健康: {4} | 重启: {5}/{6} | IP: {7}" -f $icon, $name, $img, $st, $health, $restart, $restartCount, $ip))
        $containerDetailsMd.Add(("  - 端口: {0}" -f $ports))
        $containerDetailsMd.Add(("  - 网络: {0}" -f $nets))
        $containerDetailsMd.Add(("  - 挂载: {0}" -f $mounts))
        $containerDetailsMd.Add(("  - 大小: rw={0}, root={1}" -f $sizeRw, $sizeRoot))
        $containerDetailsMd.Add("---")
    }

    # ✅ 2.2 运行时打印（控制台/日志）
    Write-Log -Level "INFO" -Message ("[Docker] 容器总数: {0} | 运行中: {1} | 已停止: {2}" -f `
        ($runningContainers.Count + $stoppedContainers.Count), $runningContainers.Count, $stoppedContainers.Count)

    if ($PrintDetails) {
        # 打印完整明细（含端口/网络/挂载/大小）
        foreach ($line in $containerDetailsMd) {
            Write-Log -Level "INFO" -Message ("[Docker] {0}" -f $line)
        }
    } else {
        # 默认只打印每个容器的“摘要第一行”（不刷屏）
        foreach ($line in $containerDetailsMd) {
            if ($line -match '^- (✅|❌) 名称:') {
                Write-Log -Level "INFO" -Message ("[Docker] {0}" -f $line)
            }
        }
        Write-Log -Level "INFO" -Message "[Docker] (提示) 如需打印端口/挂载/大小等完整明细，请在调用处加 -PrintDetails"
    }

    # 把详细信息作为一条“报告专用记录”塞进 results，供 Show-HealthReport 输出
    $results += @{
        Check    = "容器详情"
        Status   = "完成"
        Details  = ($containerDetailsMd -join "`n")
        Success  = $true
    }

    # =====================================================================
    # Redis 容器异常：日志导出 + 远端修复 + 复检
    # =====================================================================

    # 1) Redis 日志路径根据平台判断
    $platformType = $Global:PlatformType
    $redisLogPath = "/var/www/redis/data/redis.log"
    if ($platformType -eq "new") {
        $redisLogPath = "/data/middleware/redis/data/redis.log"
    }

    # 本地目录
    $baseOutputDir = Join-Path -Path $PSScriptRoot -ChildPath "output"
    $serverDir     = Join-Path -Path $baseOutputDir -ChildPath $Server.IP
    $redisOutDir   = Join-Path -Path $serverDir     -ChildPath "logs\redis"
    if (-not (Test-Path $redisOutDir)) {
        New-Item -Path $redisOutDir -ItemType Directory -Force | Out-Null
    }

    $localRedisLog = Join-Path -Path $redisOutDir -ChildPath "redis.log"
    Write-Log -Level "INFO" -Message "[Redis] 准备导出 redis 日志: $redisLogPath -> $localRedisLog"

    $exportOk = $false
    try {
        $downloadRes = Download-RemoteFile -Server $Server -RemotePath $redisLogPath -LocalPath $localRedisLog -ErrorAction Stop
        if ($downloadRes -and $downloadRes.Success) {
            Write-Log -Level "SUCCESS" -Message "[Redis] redis.log 导出成功，已保存到本地: $localRedisLog"
            $exportOk = $true
        } else {
            Write-Log -Level "WARN" -Message "[Redis] redis.log 导出失败（返回结果不成功），请检查远端文件是否存在：$redisLogPath"
        }
    } catch {
        Write-Log -Level "WARN" -Message "[Redis] redis.log 导出异常：$($_.Exception.Message)"
    }

    $statusText  = if ($exportOk) { "完成" } else { "失败" }
    $detailText  = if ($exportOk) { "redis.log 导出成功 -> $localRedisLog" } else { "redis.log 导出失败或文件不存在 ($redisLogPath)" }

    $results += @{
        Check   = "Redis日志导出"
        Status  = $statusText
        Details = $detailText
        Success = $exportOk
    }

    # 3) Redis 容器异常判定
    $redisRunning   = $runningContainers | Where-Object { $_.Name -match '(?i)redis' }
    $uredisRunning  = $redisRunning      | Where-Object { $_.Name -eq 'uredis' }
    $uredisStopped  = $stoppedContainers | Where-Object { $_.Name -eq 'uredis' }

    $redisNeedRepair = $false
    if (-not $uredisRunning -and $uredisStopped) {
        if (-not $redisRunning) {
            $redisNeedRepair = $true
        }
    }

    # 只有 Redis 需要修复时，才准备 repairItem
    $repairItem = $null
    if ($redisNeedRepair) {
        Write-Log -Level "ERROR" -Message "[Redis] 检测到 Redis 容器异常：uredis 未运行，且无其他 redis 命名容器运行，开始执行远端修复"

        $repairItem = [ordered]@{
            Check   = "Redis容器修复"
            Status  = "未执行"
            Details = ""
            Success = $false
        }

        try {
            $serverForRepair = @{
                IP   = $Server.IP
                User = $Server.User
                Pass = $Server.Pass
                Port = $Server.Port
            }

            $repairRes = Upload_the_repair_script -Server $serverForRepair -Action "redis_container_exception" -Platform "auto" -RemoteDir "/home/repair_scripts"

            if ($repairRes -and $repairRes['Success']) {
                Write-Log -Level "SUCCESS" -Message "[Redis] 远端 Redis 容器修复脚本执行成功 (redis_container_exception)"
                $repairItem.Status  = "已执行"
                $repairItem.Details = "远端脚本执行成功 (redis_container_exception)"
                $repairItem.Success = $true

                # 复检
                Write-Log -Level "INFO" -Message "[Redis] 修复后进行 uredis 容器复检..."
                $checkCmd = "docker ps --format '{{.Names}}' | grep -w 'uredis' || echo 'NO_UREDIS'"
                $recheck  = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd

                if ($recheck.ExitCode -eq 0 -and -not ($recheck.Output -contains 'NO_UREDIS')) {
                    Write-Log -Level "SUCCESS" -Message "[Redis] Redis容器复检成功：uredis 已处于运行状态"
                    $repairItem.Details += " | 复检成功：uredis 已运行"
                } else {
                    Write-Log -Level "WARN" -Message "[Redis] Redis容器复检失败：仍未检测到运行中的 uredis 容器，需要人工排查"
                    $repairItem.Status  = "部分成功"
                    $repairItem.Details += " | 复检失败：uredis 仍未运行，请人工排查"
                }
            } else {
                $errMsg = "未知错误"
                if ($repairRes -is [hashtable]) {
                    if ($repairRes.ContainsKey('Error')   -and $repairRes['Error'])     { $errMsg = [string]::Join(' ', $repairRes['Error']) }
                    elseif ($repairRes.ContainsKey('Output')  -and $repairRes['Output'])  { $errMsg = [string]::Join(' ', $repairRes['Output']) }
                    elseif ($repairRes.ContainsKey('Message') -and $repairRes['Message']) { $errMsg = $repairRes['Message'] }
                } elseif ($repairRes) {
                    $errMsg = $repairRes.ToString()
                }

                Write-Log -Level "ERROR" -Message "[Redis] 远端 Redis 容器修复执行失败：$errMsg"
                $repairItem.Status  = "失败"
                $repairItem.Details = "远程修复失败：$errMsg"
            }
        }
        catch {
            Write-Log -Level "ERROR" -Message "[Redis] 调用 Upload_the_repair_script 异常：$($_.Exception.Message)"
            $repairItem.Status  = "异常"
            $repairItem.Details = "调用修复脚本异常：$($_.Exception.Message)"
        }

        # 把 Redis 修复结果写入 $results
        $results += $repairItem
    }
    else {
        Write-Log -Level "INFO" -Message "[Redis] 未检测到需要自动修复的 Redis 容器异常"
    }

    # =====================================================================
    # Emqx 容器异常：日志导出 + 异常判定（远端修复暂不实现）
    # =====================================================================

    # 1) emqx 日志路径根据平台判断
    $emqxLogPath  = "/var/www/emqx/log/emqx.log.1"
    if ($platformType -eq "new") {
        $emqxLogPath = "/data/middleware/emqx/log/emqx.log.1"
    }

    $emqxOutDir    = Join-Path -Path $serverDir -ChildPath "logs\emqx"
    if (-not (Test-Path $emqxOutDir)) {
        New-Item -Path $emqxOutDir -ItemType Directory -Force | Out-Null
    }

    $localEmqxLog = Join-Path -Path $emqxOutDir -ChildPath "emqx.log.1"
    Write-Log -Level "INFO" -Message "[Emqx] 准备导出 emqx 日志: $emqxLogPath -> $localEmqxLog"

    $emqxExportOk = $false
    try {
        $emqxDl = Download-RemoteFile -Server $Server -RemotePath $emqxLogPath -LocalPath $localEmqxLog -ErrorAction Stop
        if ($emqxDl -and $emqxDl.Success) {
            Write-Log -Level "SUCCESS" -Message "[Emqx] emqx.log.1 导出成功，已保存到本地: $localEmqxLog"
            $emqxExportOk = $true
        } else {
            Write-Log -Level "WARN" -Message "[Emqx] emqx.log.1 导出失败（返回结果不成功），请检查远端文件是否存在：$emqxLogPath"
        }
    } catch {
        Write-Log -Level "WARN" -Message "[Emqx] emqx.log.1 导出异常：$($_.Exception.Message)"
    }

    $emqxStatusText = if ($emqxExportOk) { "完成" } else { "失败" }
    $emqxDetailText = if ($emqxExportOk) { "emqx.log.1 导出成功 -> $localEmqxLog" } else { "emqx.log.1 导出失败或文件不存在 ($emqxLogPath)" }

    $results += @{
        Check   = "Emqx日志导出"
        Status  = $emqxStatusText
        Details = $emqxDetailText
        Success = $emqxExportOk
    }

    # 3) Emqx 容器异常判定
    $emqxRunning  = $runningContainers | Where-Object { $_.Name -match '(?i)emqx' }
    $uemqxRunning = $emqxRunning       | Where-Object { $_.Name -eq 'uemqx' }
    $uemqxStopped = $stoppedContainers | Where-Object { $_.Name -eq 'uemqx' }

    $emqxNeedRepair = $false
    if (-not $uemqxRunning -and $uemqxStopped) {
        if (-not $emqxRunning) {
            $emqxNeedRepair = $true
        }
    }

    if (-not $emqxNeedRepair) {
        Write-Log -Level "INFO" -Message "[Emqx] 未检测到 Emqx 容器异常（或存在其他 emqx 容器运行中）"
    }
    else {
        Write-Log -Level "ERROR" -Message "[Emqx] 检测到 Emqx 容器异常：uemqx 未运行，且无其他 emqx 命名容器运行"
        $results += @{
            Check   = "Emqx容器状态"
            Status  = "异常"
            Details = "检测到 Emqx 容器异常：uemqx 未运行，且无其他 emqx 容器；暂未自动修复，请人工排查"
            Success = $false
        }
    }

    return $results
    
}

function Export-NginxErrorLogFromContainer {
    param(
        [Parameter(Mandatory=$true)] [hashtable]$Server,
        [Parameter(Mandatory=$true)] [string]$ContainerName,
        [Parameter(Mandatory=$true)] [string]$ExportDir,
        [ref]$ExportedFiles,
        [ref]$FailedFiles
    )

    if (-not $script:PSCP_PATH -or -not (Test-Path $script:PSCP_PATH)) {
        Write-Log -Level "ERROR" -Message "[Nginx] pscp.exe 未找到，无法导出 nginx_error.log"
        $FailedFiles.Value += @{
            Name       = "${ContainerName}_nginx_error.log"   # ← 这里
            RemotePath = "/usr/local/nginx/logs/error.log (in-container)"
            Reason     = "pscp.exe 未找到"
        }
        return
    }

    Write-Host ""
    Write-Log -Level "INFO" -Message "[Nginx] 开始导出 Nginx error.log (容器: $ContainerName) ..."

    $remoteTmpDir  = "/tmp"
    $remoteTmpFile = "$remoteTmpDir/nginx_error.log"

    # 本地文件名按容器名区分
    $localFileName = "${ContainerName}_nginx_error.log"       # ← 新增变量
    $localNginxLog = Join-Path $ExportDir $localFileName      # ← 使用新文件名

    # 1) 用单引号 here-string 构造远端脚本，避免 PowerShell 解析 $retCode/$?
    $remoteCmd = @'
if docker ps --format "{{.Names}}" 2>/dev/null | grep -w "__CONTAINER__" >/dev/null 2>&1; then \
  mkdir -p "__TMPDIR__" && \
  echo "[Nginx] docker cp __CONTAINER__:/usr/local/nginx/logs/error.log -> __TMPFILE__" && \
  docker cp "__CONTAINER__:/usr/local/nginx/logs/error.log" "__TMPFILE__" 2>/tmp/nginx_cp_err.log; \
  retCode=$?; \
  if [ $retCode -ne 0 ]; then \
    echo "[Nginx] docker cp 失败，详情："; \
    if [ -f /tmp/nginx_cp_err.log ]; then cat /tmp/nginx_cp_err.log; fi; \
    exit $retCode; \
  fi; \
  echo "[Nginx] 已拷贝到宿主机: __TMPFILE__"; \
else \
  echo "[Nginx] 容器未运行或不存在: __CONTAINER__"; \
  exit 1; \
fi
'@

    # 占位符替换为实际值
    $remoteCmd = $remoteCmd.Replace("__CONTAINER__", $ContainerName).
                             Replace("__TMPDIR__",   $remoteTmpDir).
                             Replace("__TMPFILE__",  $remoteTmpFile)

    # 去掉 Windows 回车，避免 bash: $'\r' 未找到命令
    $remoteCmd = $remoteCmd -replace "`r", ""

    # 2) 远端执行 docker cp 脚本
    $cpRes = Invoke-SSHCommand -HostName $Server.IP `
                               -User $Server.User `
                               -Pass $Server.Pass `
                               -Port $Server.Port `
                               -Command $remoteCmd

    $cpOutput = ($cpRes.Output | Out-String).Trim()
    if ($cpOutput) {
        Write-Log -Level "INFO" -Message ("[Nginx] 远端输出: {0}" -f ($cpOutput -replace '\s+', ' '))
    }

    if ($cpRes.ExitCode -ne 0) {
        Write-Log -Level "ERROR" -Message "[Nginx] 远端 docker cp 执行失败，终止 Nginx 日志导出"
        $FailedFiles.Value += @{
            Name       = "nginx_error.log"
            RemotePath = "/usr/local/nginx/logs/error.log (in-container)"
            Reason     = "docker cp 失败，详见远端输出"
        }
        return
    }

    # 3) 用 pscp 下载 /tmp/nginx_error.log -> 本地
    Write-Log -Level "INFO" -Message "[Nginx] 下载: $remoteTmpFile -> $localNginxLog"

    $pscpArgs = @(
        "-scp",
        "-batch",
        "-P", $Server.Port,
        "-l", $Server.User,
        "-pw", $Server.Pass,
        "$($Server.User)@$($Server.IP):$remoteTmpFile",
        $localNginxLog
    )

    try {
        $dlOut  = & $script:PSCP_PATH @pscpArgs 2>&1
        $dlCode = $LASTEXITCODE

        if ($dlCode -ne 0 -and ($dlOut -match "host key" -or $dlOut -match "Cannot confirm")) {
            $cmdLine = "echo y | `"$($script:PSCP_PATH)`" -scp -batch -P $($Server.Port) -l $($Server.User) -pw `"$($Server.Pass)`" `"$($Server.User)@$($Server.IP):$remoteTmpFile`" `"$localNginxLog`""
            Write-Log -Level "WARN" -Message "[Nginx] 主机密钥提示，自动接受并重试: $cmdLine"
            $dlOut  = cmd /c $cmdLine 2>&1
            $dlCode = $LASTEXITCODE
        }

        if ($dlCode -eq 0 -and (Test-Path $localNginxLog)) {
            $sz   = (Get-Item $localNginxLog).Length
            $szKB = [math]::Round($sz / 1024, 2)
            Write-Log -Level "SUCCESS" -Message "[Nginx] $localFileName 导出成功 ($szKB KB): $localNginxLog"

            $ExportedFiles.Value += @{
                Name       = $localFileName                       # ← 使用容器区分名
                RemotePath = "/usr/local/nginx/logs/error.log (via docker cp)"
                LocalPath  = $localNginxLog
                Size       = $sz
            }
        }
        else {
            Write-Log -Level "ERROR" -Message ("[Nginx] 下载失败，ExitCode={0}" -f $dlCode)
            if ($dlOut) {
                $oneLine = ($dlOut -join " ") -replace '\s+', ' '
                Write-Log -Level "ERROR" -Message ("[Nginx] pscp 输出: {0}" -f $oneLine)
            }
            $FailedFiles.Value += @{
                Name       = $localFileName                       # ← 同样改成容器名版本
                RemotePath = $remoteTmpFile
                Reason     = "下载失败: ExitCode=$dlCode"
            }
        }
    }
    catch {
        Write-Log -Level "ERROR" -Message "[Nginx] 下载异常: $($_.Exception.Message)"
        $FailedFiles.Value += @{
            Name       = $localFileName
            RemotePath = $remoteTmpFile
            Reason     = "异常: $($_.Exception.Message)"
        }
    }
    finally {
        $cleanCmd = "rm -f '$remoteTmpFile' 2>/dev/null || true"
        [void](Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $cleanCmd)
    }
}

# ================================
# 日志导出
# ================================
function Export-ServiceLogs {
    param(
        [hashtable]$Server,
        [string]$PlatformType,
        [hashtable]$SystemInfo
    )
    
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 服务日志导出 =========="
    
    # 检查 pscp 是否可用
    if (-not $script:PSCP_PATH -or -not (Test-Path $script:PSCP_PATH)) {
        Write-Log -Level "ERROR" -Message "pscp.exe 未找到，无法导出日志"
        Write-Log -Level "ERROR" -Message "请将 pscp.exe 放在脚本同目录下"
        Write-Log -Level "ERROR" -Message "下载地址: https://the.earth.li/~sgtatham/putty/latest/w64/pscp.exe"
        return @{
            Success = $false
            ExportedFiles = @()
            FailedFiles = @()
            ExportDir = $null
        }
    }
    
    # 创建导出目录（以服务器IP和时间戳命名）
    $exportTimestamp = Get-Date -Format "yyyyMMdd_HHmmss"
    $exportDirName = "logs_$($Server.IP)_$exportTimestamp"
    $exportDir = Join-Path $SCRIPT_DIR $exportDirName
    
    if (-not (Test-Path $exportDir)) {
        New-Item -ItemType Directory -Path $exportDir -Force | Out-Null
    }
    
    Write-Log -Level "INFO" -Message "日志导出目录: $exportDir"
    
    $exportedFiles = @()
    $failedFiles = @()
    $logsToExport = @()
    
    # 根据平台类型选择要导出的日志
    if ($PlatformType -eq "new") {
        # 新统一平台
        Write-Log -Level "INFO" -Message "平台类型: 新统一平台"
        
        if ($SystemInfo.HasUjava) {
            Write-Log -Level "INFO" -Message "检测到 ujava 容器，准备导出 Java 服务日志..."
            $logsToExport += $NewPlatformLogs
        }
        else {
            Write-Log -Level "WARN" -Message "未检测到 ujava 容器，跳过 Java 服务日志导出"
       
        }
    }
    else {
        # 传统平台
        Write-Log -Level "INFO" -Message "平台类型: 传统平台"
        
        if ($SystemInfo.HasUjava) {
            Write-Log -Level "INFO" -Message "检测到 ujava 容器，准备导出 Java 服务日志..."
            $logsToExport += $OldPlatformUjavaLogs
        }
        else {
            Write-Log -Level "WARN" -Message "未检测到 ujava 容器，跳过 Java 服务日志导出"
        }
        
        if ($SystemInfo.HasUpython) {
            Write-Log -Level "INFO" -Message "检测到 upython 容器，准备导出 Python 服务日志..."
            $logsToExport += $OldPlatformUpythonLogs
        }
        else {
            Write-Log -Level "WARN" -Message "未检测到 upython 容器，跳过 Python 服务日志导出"
        }
    }
    
    if ($logsToExport.Count -eq 0) {
        Write-Log -Level "WARN" -Message "没有需要导出的日志文件"
        return @{
            Success = $true
            ExportedFiles = @()
            FailedFiles = @()
            ExportDir = $exportDir
        }
    }
    
    Write-Log -Level "INFO" -Message "共 $($logsToExport.Count) 个日志文件待导出..."
    Write-Host ""
    
    # 逐个导出日志文件（常规路径：Java/Upython 等）
    foreach ($logConfig in $logsToExport) {
        $localFileName = $logConfig.Name
        $remotePath = $logConfig.RemotePath
        $localPath = Join-Path $exportDir $localFileName
        
        Write-Log -Level "INFO" -Message "正在导出: $localFileName"
        Write-Log -Level "INFO" -Message "  远程路径: $remotePath"
        
        # 先检查远程文件是否存在
        $checkCmd = "[ -f '$remotePath' ] && echo 'EXISTS' || echo 'NOT_EXISTS'"
        $checkResult = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd
        
        if ($checkResult.Output -match 'NOT_EXISTS') {
            Write-Log -Level "WARN" -Message "  [跳过] 文件不存在: $remotePath"
            $failedFiles += @{
                Name = $localFileName
                RemotePath = $remotePath
                Reason = "文件不存在"
            }
            continue
        }
        
        # 使用 pscp 下载文件（显式记录错误，适配大文件）
        $pscpArgs = @(
            "-scp",          # 显式使用 SCP 协议
            "-batch",        # 非交互
            "-P", $Server.Port,
            "-l", $Server.User,
            "-pw", $Server.Pass,
            "$($Server.User)@$($Server.IP):$remotePath",
            $localPath
        )

        try {
            Write-Log -Level "INFO" -Message ("  调用 pscp: {0} {1}" -f $script:PSCP_PATH, ($pscpArgs -join ' '))
            $pscpResult = & $script:PSCP_PATH @pscpArgs 2>&1
            $exitCode = $LASTEXITCODE

            # 如果失败且是主机密钥问题，自动接受后重试一次
            if ($exitCode -ne 0 -and ($pscpResult -match "host key" -or $pscpResult -match "Cannot confirm")) {
                $cmdLine = "echo y | `"$($script:PSCP_PATH)`" -scp -batch -P $($Server.Port) -l $($Server.User) -pw `"$($Server.Pass)`" `"$($Server.User)@$($Server.IP):$remotePath`" `"$localPath`""
                Write-Log -Level "WARN" -Message "  检测到主机密钥提示，自动接受并重试: $cmdLine"
                $pscpResult = cmd /c $cmdLine 2>&1
                $exitCode = $LASTEXITCODE
            }

            if ($exitCode -eq 0 -and (Test-Path $localPath)) {
                $fileSize = (Get-Item $localPath).Length
                $fileSizeKB = [math]::Round($fileSize / 1024, 2)
                Write-Log -Level "SUCCESS" -Message "  [成功] 已导出 ($fileSizeKB KB)"
                $exportedFiles += @{
                    Name       = $localFileName
                    RemotePath = $remotePath
                    LocalPath  = $localPath
                    Size       = $fileSize
                }
            }
            else {
                # 记录完整的 pscp 输出，便于排查大文件失败原因（超时/中断/权限等）
                Write-Log -Level "ERROR" -Message ("  [失败] 导出失败，ExitCode={0}" -f $exitCode)
                if ($pscpResult) {
                    # 将 pscp 输出压缩成一行，避免刷屏
                    $oneLine = ($pscpResult -join " ") -replace '\s+', ' '
                    Write-Log -Level "ERROR" -Message ("  [pscp] 输出: {0}" -f $oneLine)
                }
                $failedFiles += @{
                    Name       = $localFileName
                    RemotePath = $remotePath
                    Reason     = "下载失败: ExitCode=$exitCode; Output=$pscpResult"
                }
            }
        }
        catch {
            Write-Log -Level "ERROR" -Message "  [失败] 导出异常: $($_.Exception.Message)"
            $failedFiles += @{
                Name       = $localFileName
                RemotePath = $remotePath
                Reason     = "异常: $($_.Exception.Message)"
            }
        }
    }

    # =============================
    # 传统平台 nginx 日志导出（需求文档第7点：ujava / upython 都需要）
    # =============================
    if ($PlatformType -eq "old") {

        # ujava 场景：从 ujava 容器导出 nginx error.log
        if ($SystemInfo.HasUjava -and $SystemInfo.UjavaContainer) {
            Export-NginxErrorLogFromContainer -Server $Server -ContainerName $SystemInfo.UjavaContainer -ExportDir $exportDir `
                -ExportedFiles ([ref]$exportedFiles) -FailedFiles ([ref]$failedFiles)
        }

        # upython 场景：从 upython 容器导出 nginx error.log（逻辑完全一致）
        if ($SystemInfo.HasUpython -and $SystemInfo.UpythonContainer) {
            Export-NginxErrorLogFromContainer -Server $Server -ContainerName $SystemInfo.UpythonContainer -ExportDir $exportDir `
                -ExportedFiles ([ref]$exportedFiles) -FailedFiles ([ref]$failedFiles)
        }
    }

    Write-Host ""
    Write-Log -Level "INFO" -Message "日志导出完成: 成功 $($exportedFiles.Count) 个，失败 $($failedFiles.Count) 个"
    
    if ($exportedFiles.Count -gt 0) {
        Write-Log -Level "SUCCESS" -Message "导出目录: $exportDir"
    }
    
    return @{
        Success = ($exportedFiles.Count -gt 0)
        ExportedFiles = $exportedFiles
        FailedFiles = $failedFiles
        ExportDir = $exportDir
    }
}

# ================================
# 检测容器内端口服务
# ================================
function Test-ContainerPorts {
    param(
        [hashtable]$Server,
        [string]$ContainerName,
        [array]$PortList,
        [string]$ServiceType
    )
    
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 检测 $ServiceType 容器内服务 ($ContainerName) =========="
    
    $results = @()
    
    # 获取容器内所有监听端口
    $checkCmd = "docker exec $ContainerName netstat -tlnp 2>/dev/null || docker exec $ContainerName ss -tlnp 2>/dev/null"
    $result = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $checkCmd
    
    $netstatOutput = $result.Output
    
    foreach ($portInfo in $PortList) {
        $port = $portInfo.Port
        $expectedProcess = $portInfo.Process
        $description = $portInfo.Description
        
        # 检查端口是否在监听
        $portPattern = ":$port\s"
        $isListening = $netstatOutput -match $portPattern
        
        $status = if ($isListening) { "监听中" } else { "未监听" }
        $statusColor = if ($isListening) { "SUCCESS" } else { "ERROR" }
        
        $results += @{
            Port = $port
            Process = $expectedProcess
            Description = $description
            Status = $status
            Listening = $isListening
        }
        
        $statusIcon = if ($isListening) { "[OK]" } else { "[FAIL]" }
        Write-Log -Level $statusColor -Message "  $statusIcon 端口 $port ($description): $status"
    }
    
    return $results
}

# ================================
# Android 设备自检（需求 15）
# ================================
$AndroidLogRemoteDir = "/sdcard/Android/data/com.ubains.local.gviewer/files/"
$AndroidDefaultPort  = 5555

# 优先使用脚本目录下的 adb.exe
$script:ADB_PATH = $null
function Resolve-AdbPath {
    # 1) 脚本目录
    $localAdb = Join-Path $SCRIPT_DIR "adb.exe"
    if (Test-Path $localAdb) { return $localAdb }

    # 2) 系统 PATH
    try {
        $cmd = Get-Command adb -ErrorAction Stop
        return $cmd.Source
    } catch {
        return $null
    }
}

function Test-AdbAvailable {
    $script:ADB_PATH = Resolve-AdbPath
    if ([string]::IsNullOrWhiteSpace($script:ADB_PATH)) { return $false }

    # 如果是本地 adb.exe，检查 DLL 是否齐全（避免“只放 adb.exe”运行不了）
    $adbDir = Split-Path $script:ADB_PATH -Parent
    $dll1 = Join-Path $adbDir "AdbWinApi.dll"
    $dll2 = Join-Path $adbDir "AdbWinUsbApi.dll"
    if ((Split-Path $script:ADB_PATH -Leaf) -ieq "adb.exe") {
        if (-not (Test-Path $dll1) -or -not (Test-Path $dll2)) {
            Write-Log -Level "WARN" -Message "[Android] 找到 adb.exe 但缺少依赖 DLL：AdbWinApi.dll/AdbWinUsbApi.dll。请将它们与 adb.exe 放在同一目录"
            return $false
        }
    }
    return $true
}

function Invoke-Adb {
    param(
        [Parameter(Mandatory=$true)] [string[]] $Args
    )

    if (-not $script:ADB_PATH) { $script:ADB_PATH = Resolve-AdbPath }
    $adbExe = if ($script:ADB_PATH) { $script:ADB_PATH } else { "adb" }

    $psi = New-Object System.Diagnostics.ProcessStartInfo
    $psi.FileName = $adbExe
    $psi.Arguments = ($Args -join " ")
    $psi.RedirectStandardOutput = $true
    $psi.RedirectStandardError  = $true
    $psi.UseShellExecute = $false
    $psi.CreateNoWindow = $true

    $p = New-Object System.Diagnostics.Process
    $p.StartInfo = $psi
    [void]$p.Start()
    $out = $p.StandardOutput.ReadToEnd()
    $err = $p.StandardError.ReadToEnd()
    $p.WaitForExit()

    return @{
        ExitCode = $p.ExitCode
        Out      = $out
        Err      = $err
        CmdLine  = ("{0} {1}" -f $adbExe, ($Args -join ' '))
    }
}

function Test-AndroidDeviceHealth {
    param(
        [Parameter(Mandatory=$true)] [string] $ScriptDir
    )

    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 安卓设备自检 (PRD 15) =========="

    $results = @()

    # 0) 依赖检查：adb
    if (-not (Test-AdbAvailable)) {
        Write-Log -Level "WARN" -Message "[Android] 未检测到 adb.exe（请安装 Android Platform Tools 或将 adb 加入 PATH），跳过安卓自检"
        $results += @{
            Check   = "Android自检"
            Status  = "跳过"
            Details = "adb.exe 未找到"
            Success = $false
        }
        return $results
    }

    # 1) 输入设备 IP/端口
    $deviceIp = Read-Host "请输入安卓设备IP（留空则跳过安卓自检）"
    if ([string]::IsNullOrWhiteSpace($deviceIp)) {
        Write-Log -Level "INFO" -Message "[Android] 未输入设备IP，跳过安卓自检"
        $results += @{
            Check   = "Android自检"
            Status  = "跳过"
            Details = "未输入设备IP"
            Success = $true
        }
        return $results
    }

    $portInput = Read-Host ("请输入安卓设备端口 [默认 {0}]" -f $AndroidDefaultPort)
    $port = $AndroidDefaultPort
    if (-not [string]::IsNullOrWhiteSpace($portInput)) {
        try { $port = [int]$portInput } catch { $port = $AndroidDefaultPort }
    }

    $target = ("{0}:{1}" -f $deviceIp.Trim(), $port)
    Write-Log -Level "INFO" -Message ("[Android] 目标设备: {0}" -f $target)

    $connected = $false

    try {
        # 2) 连接设备：adb connect
        $conn = Invoke-Adb -Args @("connect", $target)
        $connOut = (($conn.Out + "`n" + $conn.Err) -replace "`r","").Trim()
        Write-Log -Level "INFO" -Message ("[Android] 执行: {0}" -f $conn.CmdLine)
        if ($connOut) { Write-Log -Level "INFO" -Message ("[Android] 输出: {0}" -f ($connOut -replace '\s+',' ')) }

        # 判定连接成功（兼容多种输出）
        if ($connOut -match "(?i)connected to" -or $connOut -match "(?i)already connected to") {
            $connected = $true
        }

        # adb devices 复核
        if ($connected) {
            $dev = Invoke-Adb -Args @("devices")
            $devTxt = (($dev.Out + "`n" + $dev.Err) -replace "`r","")
            if ($devTxt -match [regex]::Escape($target) + "\s+device") {
                $connected = $true
            } elseif ($devTxt -match [regex]::Escape($target) + "\s+(unauthorized|offline)") {
                $connected = $false
                Write-Log -Level "ERROR" -Message "[Android] 设备处于 unauthorized/offline，请在设备上确认调试授权或检查网络"
            } else {
                $connected = $false
            }
        }

        if (-not $connected) {
            Write-Log -Level "ERROR" -Message "[Android] adb 连接失败，终止后续安卓日志导出"
            $results += @{
                Check   = "Android连接"
                Status  = "失败"
                Details = "adb connect 失败或设备未授权/离线"
                Success = $false
            }
            return $results
        }

        Write-Log -Level "SUCCESS" -Message "[Android] 设备连接成功"
        $results += @{
            Check   = "Android连接"
            Status  = "成功"
            Details = "adb connect 成功: " + $target
            Success = $true
        }

        # 3) 日志拉取：files + cache -> logs/android/<ip_port_ts>/
        $androidLocalDir = Join-Path $ScriptDir "logs\android"
        if (-not (Test-Path $androidLocalDir)) {
            New-Item -ItemType Directory -Path $androidLocalDir -Force | Out-Null
        }

        $ts = Get-Date -Format "yyyyMMdd_HHmmss"
        $safeTarget = (($target -replace '[^\w\.\-:]', '_') -replace ':','_')
        $pullRoot = Join-Path $androidLocalDir ("{0}_{1}" -f $safeTarget, $ts)
        New-Item -ItemType Directory -Path $pullRoot -Force | Out-Null

        function Pull-AndroidDir {
            param(
                [Parameter(Mandatory=$true)][string] $RemoteDir,
                [Parameter(Mandatory=$true)][string] $LocalSubDir,
                [Parameter(Mandatory=$true)][string] $CheckName,
                [Parameter(Mandatory=$false)][switch] $SkipIfMissing   # 新增：目录不存在则跳过
            )

            # 先检查远端目录是否存在（针对 cache：不存在则跳过）
            if ($SkipIfMissing) {
                $remoteDirNoSlash = $RemoteDir.TrimEnd('/')
                $check = Invoke-Adb -Args @("-s", $target, "shell", "test -d `"$remoteDirNoSlash`" && echo __EXISTS__ || echo __MISSING__")
                $checkOut = (($check.Out + "`n" + $check.Err) -replace "`r","").Trim()

                if ($checkOut -match "__MISSING__") {
                    Write-Log -Level "WARN" -Message ("[Android] 远端目录不存在，跳过导出：{0}" -f $RemoteDir)
                    $results += @{
                        Check   = $CheckName
                        Status  = "跳过"
                        Details = "远端目录不存在: " + $RemoteDir
                        Success = $true
                    }
                    return
                }
            }

            $localPath = Join-Path $pullRoot $LocalSubDir
            New-Item -ItemType Directory -Path $localPath -Force | Out-Null

            Write-Log -Level "INFO" -Message ("[Android] 开始拉取: {0} -> {1}" -f $RemoteDir, $localPath)
            $pull = Invoke-Adb -Args @("-s", $target, "pull", $RemoteDir, $localPath)
            $pullOut = (($pull.Out + "`n" + $pull.Err) -replace "`r","").Trim()
            Write-Log -Level "INFO" -Message ("[Android] 执行: {0}" -f $pull.CmdLine)
            if ($pullOut) { Write-Log -Level "INFO" -Message ("[Android] 输出: {0}" -f (($pullOut -split "`n" | Select-Object -First 5) -join " | ")) }

            $ok = $false
            $fileCount = 0
            if ($pull.ExitCode -eq 0 -and (Test-Path $localPath)) {
                $fileCount = (Get-ChildItem -Path $localPath -Recurse -File -ErrorAction SilentlyContinue | Measure-Object).Count
                if ($fileCount -gt 0) { $ok = $true }
                $lvl = if ($ok) { "SUCCESS" } else { "WARN" }
                Write-Log -Level $lvl -Message ("[Android] 拉取完成：{0} 文件数={1}，目录={2}" -f $LocalSubDir, $fileCount, $localPath)
            } else {
                Write-Log -Level "ERROR" -Message ("[Android] adb pull 失败：{0}" -f $RemoteDir)
            }

            $statusText = if ($ok) { "成功" } else { "失败" }
            $detailText = if ($ok) { "已导出到: $localPath (files=$fileCount)" } else { "导出失败/目录不存在/权限不足: $RemoteDir" }

            $results += @{
                Check   = $CheckName
                Status  = $statusText
                Details = $detailText
                Success = $ok
            }
        }

        # 3.1 pull files（必须导出：不存在就算失败）
        Pull-AndroidDir -RemoteDir "/sdcard/Android/data/com.ubains.local.gviewer/files/" -LocalSubDir "files" -CheckName "Android日志导出(files)"

        # 3.2 pull cache（可选：不存在就跳过）
        Pull-AndroidDir -RemoteDir "/sdcard/Android/data/com.ubains.local.gviewer/cache/" -LocalSubDir "cache" -CheckName "Android日志导出(cache)" -SkipIfMissing
    }
    finally {
        # 4) 断开连接（新增）
        # PRD 要求：收集完成后自动断开。即使中途异常也尽量断开，避免残留连接。
        $discOk = $false
        try {
            $disc = Invoke-Adb -Args @("disconnect", $target)
            $discOut = (($disc.Out + "`n" + $disc.Err) -replace "`r","").Trim()
            Write-Log -Level "INFO" -Message ("[Android] 执行: {0}" -f $disc.CmdLine)
            if ($discOut) { Write-Log -Level "INFO" -Message ("[Android] 输出: {0}" -f ($discOut -replace '\s+',' ')) }

            if ($discOut -match "(?i)disconnected" -or $disc.ExitCode -eq 0) {
                $discOk = $true
            }
        } catch {
            $discOk = $false
        }

        $discStatus  = if ($discOk) { "成功" } else { "失败" }
        $discDetails = if ($discOk) { "adb disconnect 成功: $target" } else { "adb disconnect 失败（可能未连接/adb 异常）: $target" }

        $results += @{
            Check   = "Android断开连接"
            Status  = $discStatus
            Details = $discDetails
            Success = $discOk
        }
    }

    return $results
}

# ================================
# 生成检测报告
# ================================
function Show-HealthReport {
    param(
        [hashtable]$Server,
        [string]$PlatformType,
        [hashtable]$SystemInfo,
        [array]$UjavaContainerResults,
        [array]$UjavaHostResults,
        [array]$UpythonResults,
        [array]$UpythonVoiceResults,
        [array]$DNSResults,
        [hashtable]$ResourceResults,
        [hashtable]$LogExportResults,
        [hashtable]$NTPResults,
        [hashtable]$FilePermResults,
        [array]$ContainerInfo,
        [array]$AndroidResults
    )

    if (-not $SCRIPT_DIR -or [string]::IsNullOrWhiteSpace($SCRIPT_DIR)) { $SCRIPT_DIR = (Get-Location).Path }
    $reportDir = Join-Path $SCRIPT_DIR "Reports"
    try { if (-not (Test-Path $reportDir)) { New-Item -ItemType Directory -Path $reportDir -Force | Out-Null } }
    catch { $reportDir = Join-Path $env:TEMP "ubains_reports"; if (-not (Test-Path $reportDir)) { New-Item -ItemType Directory -Path $reportDir -Force | Out-Null } }
    $ts = Get-Date -Format "yyyyMMdd_HHmmss"
    $safeIp = if ($Server -and $Server.IP) { ($Server.IP -replace '[^\w\.\-]', '_') } else { 'unknown' }
    $mdFile = Join-Path $reportDir "health_report_${safeIp}_${ts}.md"
    if (-not $mdFile -or [string]::IsNullOrWhiteSpace($mdFile)) { $mdFile = Join-Path $reportDir ("health_report_" + $ts + ".md") }

    function Get-StateIcon {
        param([string]$Status)
        switch ($Status) {
            "正常"   { "✅" }
            "警告"   { "⚠️" }
            "危险"   { "❌" }
            "偏差"   { "⚠️" }
            "未执行" { "ℹ️" }
            default  { "❌" }
        }
    }

    $md = @()
    $md += "# 服务自检报告"
    $md += ""
    $md += "- 服务器地址: $($Server.IP)"
    $md += "- 服务器描述: $($Server.Desc)"
    $md += "- 平台类型: $(if ($PlatformType -eq 'new') { '新统一平台' } else { '传统平台' })"
    $md += "- 检测时间: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')"
    $md += ""

    Write-Host ""
    Write-Host "==================================================================" -ForegroundColor Cyan
    Write-Host "                    服务自检报告" -ForegroundColor Cyan
    Write-Host "==================================================================" -ForegroundColor Cyan
    Write-Host ""

    # ✅ 系统类型识别：按 ujavaVariant 输出
    $md += "## 系统类型识别"
    if ($SystemInfo.HasUjava) {
        $v = if ($SystemInfo.ContainsKey('UjavaSystemVariant')) { $SystemInfo.UjavaSystemVariant } else { $null }
        $label = if ($v -eq 'unified') { '统一平台系统' } else { '会议预定系统' }
        $md += "- $label (ujava): 已部署 [$($SystemInfo.UjavaContainer)]"
    }
    if ($SystemInfo.HasUpython) { $md += "- 运维集控系统 (upython): 已部署 [$($SystemInfo.UpythonContainer)]" }
    if ($SystemInfo.HasUpythonVoice) { $md += "- 转录系统 (upython_voice): 已部署 [$($SystemInfo.UpythonVoiceContainer)]" }
    if (-not $SystemInfo.HasUjava -and -not $SystemInfo.HasUpython -and -not $SystemInfo.HasUpythonVoice) { $md += "- 未检测到已知系统容器" }
    $md += ""

    # 统计信息
    $totalServices = 0; $runningServices = 0; $failedServices = 0

    # ujava 容器内服务
    if ($UjavaContainerResults -and $UjavaContainerResults.Count -gt 0) {
        Write-Host "【ujava 容器内服务状态】" -ForegroundColor Yellow
        $md += "## ujava 容器内服务状态"
        foreach ($r in $UjavaContainerResults) {
            $totalServices++
            if ($r.Running) { $runningServices++; Write-Host "  [OK] $($r.Service): $($r.Status)" -ForegroundColor Green }
            else { $failedServices++; Write-Host "  [FAIL] $($r.Service): $($r.Status)" -ForegroundColor Red }
            $icon = if ($r.Running) { "✅" } else { "❌" }
            $md += "- $icon $($r.Service) ($($r.Pattern)): $($r.Status)"
        }
        $md += ""
        Write-Host ""
    }

    # ujava 宿主机服务
    if ($UjavaHostResults -and $UjavaHostResults.Count -gt 0) {
        Write-Host "【ujava 宿主机服务状态】" -ForegroundColor Yellow
        $md += "## ujava 宿主机服务状态"
        foreach ($r in $UjavaHostResults) {
            $totalServices++
            if ($r.Running) { $runningServices++; Write-Host "  [OK] $($r.Service): $($r.Status)" -ForegroundColor Green }
            else { $failedServices++; Write-Host "  [FAIL] $($r.Service): $($r.Status)" -ForegroundColor Red }
            $icon = if ($r.Running) { "✅" } else { "❌" }
            $md += "- $icon $($r.Service) ($($r.Pattern)): $($r.Status)"
        }
        $md += ""
        Write-Host ""
    }

    # 对外服务远程修复结果（新统一平台/传统平台通用）
    if ($global:ExternalServiceRepairResult) {
        $r = $global:ExternalServiceRepairResult
        $icon = if ($r.Success) { "✅" } else { "⚠️" }
        $platformLabel = if ($PlatformType -eq 'new') { '新统一平台' } else { '传统平台' }
        $md += "## $platformLabel 对外服务远程修复结果"
        $md += "- $icon 对外服务 (external-meeting-api)"
        $md += "  - 已触发修复: $($r.Attempted)"
        $md += "  - 修复结果: $(if ($r.Success) { '成功' } else { '失败' })"
        if ($r.Detail) {
            $md += "  - 说明: $($r.Detail)"
        }
        $md += ""
    }

    # upython 容器服务
    if ($UpythonResults -and $UpythonResults.Count -gt 0) {
        Write-Host "【upython 容器服务状态】" -ForegroundColor Yellow
        $md += "## upython 容器服务状态"
        foreach ($r in $UpythonResults) {
            $totalServices++
            if ($r.Listening) { $runningServices++; Write-Host "  [OK] 端口 $($r.Port) ($($r.Description)): $($r.Status)" -ForegroundColor Green }
            else { $failedServices++; Write-Host "  [FAIL] 端口 $($r.Port) ($($r.Description)): $($r.Status)" -ForegroundColor Red }
            $icon = if ($r.Listening) { "✅" } else { "❌" }
            $md += "- $icon 端口 $($r.Port) ($($r.Description)): $($r.Status)"
        }
        $md += ""
        Write-Host ""
    }

    # upython_voice 容器服务
    if ($UpythonVoiceResults -and $UpythonVoiceResults.Count -gt 0) {
        Write-Host "【upython_voice 容器服务状态】" -ForegroundColor Yellow
        $md += "## upython_voice 容器服务状态"
        foreach ($r in $UpythonVoiceResults) {
            $totalServices++
            if ($r.Listening) { $runningServices++; Write-Host "  [OK] 端口 $($r.Port) ($($r.Description)): $($r.Status)" -ForegroundColor Green }
            else { $failedServices++; Write-Host "  [FAIL] 端口 $($r.Port) ($($r.Description)): $($r.Status)" -ForegroundColor Red }
            $icon = if ($r.Listening) { "✅" } else { "❌" }
            $md += "- $icon 端口 $($r.Port) ($($r.Description)): $($r.Status)"
        }
        $md += ""
        Write-Host ""
    }

    # DNS 检测
    if ($DNSResults -and $DNSResults.Count -gt 0) {
        Write-Host "【DNS 解析检测结果】" -ForegroundColor Yellow
        $md += "## DNS 解析检测结果"
        foreach ($r in $DNSResults) {
            $statusIcon = if ($r.Success) { "OK" } else { "FAIL" }
            Write-Host "  [$statusIcon] $($r.Check): $($r.Status)"
            $icon = if ($r.Success) { "✅" } else { "❌" }
            $line = "- $icon $($r.Check): $($r.Status)"; if ($r.Details) { $line += "，$($r.Details)" }
            $md += $line
        }
        $md += ""
        Write-Host ""
    }

    # 服务器资源（含异常->修复时间线）
    if ($ResourceResults) {
        Write-Host "【服务器资源分析】" -ForegroundColor Yellow
        $md += "## 服务器资源分析"
        if ($ResourceResults.OS) { Write-Host "  操作系统: $($ResourceResults.OS.Info)" -ForegroundColor Cyan; $md += "- 操作系统: $($ResourceResults.OS.Info)" }
        if ($ResourceResults.Architecture) { Write-Host "  系统架构: $($ResourceResults.Architecture.Arch) | 内核: $($ResourceResults.Architecture.Kernel)" -ForegroundColor Cyan; $md += "- 系统架构: $($ResourceResults.Architecture.Arch) | 内核: $($ResourceResults.Architecture.Kernel)" }
        if ($ResourceResults.CPU) {
            $cpuIcon = Get-StateIcon -Status $ResourceResults.CPU.Status
            Write-Host "  CPU 使用率: $($ResourceResults.CPU.Usage)% (核心数: $($ResourceResults.CPU.Cores)) [$($ResourceResults.CPU.Status)]"
            $md += "- $cpuIcon CPU 使用率: $($ResourceResults.CPU.Usage)% (核心数: $($ResourceResults.CPU.Cores)) [$($ResourceResults.CPU.Status)]"
        }
        if ($ResourceResults.Memory) {
            $memIcon = Get-StateIcon -Status $ResourceResults.Memory.Status
            Write-Host "  内存使用: $($ResourceResults.Memory.Used)GB / $($ResourceResults.Memory.Total)GB ($($ResourceResults.Memory.Percent)%) [$($ResourceResults.Memory.Status)]"
            $md += "- $memIcon 内存使用: $($ResourceResults.Memory.Used)GB / $($ResourceResults.Memory.Total)GB ($($ResourceResults.Memory.Percent)%) [$($ResourceResults.Memory.Status)]"
        }
        if ($ResourceResults.Disk -and $ResourceResults.Disk.Count -gt 0) {
            Write-Host "  磁盘使用情况:"; $md += "- 磁盘使用:"
            foreach ($disk in $ResourceResults.Disk) {
                $diskIcon = switch ($disk.Status) { "正常" {"✅"} "警告" {"⚠️"} default {"❌"} }
                Write-Host "    $($disk.MountPoint): $($disk.Used)/$($disk.Size) ($($disk.Percent)%) [$($disk.Status)]"
                $md += "  - $diskIcon $($disk.MountPoint): $($disk.Used)/$($disk.Size) ($($disk.Percent)%) [$($disk.Status)]"
            }
        }

        # 防火墙详细时间线
        if ($ResourceResults.Firewall) {
            $fwIcon = if ($ResourceResults.Firewall.Active) { "🟢" } else { "🔴" }
            Write-Host "  防火墙状态: $($ResourceResults.Firewall.Status) ($($ResourceResults.Firewall.Type))"
            $md += "- $fwIcon 防火墙状态: $($ResourceResults.Firewall.Status) ($($ResourceResults.Firewall.Type))"
            if ($ResourceResults.Firewall.OpenPorts -and $ResourceResults.Firewall.Active) {
                Write-Host "    开放端口/服务: $($ResourceResults.Firewall.OpenPorts)"
                $md += "  - 开放端口/服务: $($ResourceResults.Firewall.OpenPorts)"
            }

            # 时间线：初始异常 -> 触发修复 -> 修复后
            $hasPre  = $ResourceResults.Firewall.PSObject.Properties['Pre'] -ne $null
            $hasRep  = $ResourceResults.Firewall.PSObject.Properties['Repair'] -ne $null
            if ($hasPre -or $hasRep) {
                $md += "- 修复时间线"
                if ($hasPre -and $ResourceResults.Firewall.Pre) {
                    $pre = $ResourceResults.Firewall.Pre
                    $preIcon = if ($pre.Active) { "🟢" } else { "🔴" }
                    $preType = if ($pre.Type) { $pre.Type } else { "unknown" }
                    $md += "  - $preIcon 初始状态: $(if ($pre.Active) {'已启用'} else {'未启用'}) ($preType)"
                    if ($pre.OpenPorts) { $md += "    - 初始端口/服务: $($pre.OpenPorts)" }
                }
                if ($hasRep -and $ResourceResults.Firewall.Repair) {
                    $rep = $ResourceResults.Firewall.Repair
                    $attempted = ($rep.PSObject.Properties['Attempted'] -and $rep.Attempted)
                    $succeeded = ($rep.PSObject.Properties['Succeeded'] -and $rep.Succeeded)
                    $message   = if ($rep.PSObject.Properties['Message']) { $rep.Message } else { $null }
                    $md += "  - 🛠️ 修复触发: $(if ($attempted) {'已触发'} else {'未触发'})"
                    if ($message) { $md += "    - 修复说明: $message" }
                    $md += "  - $(if ($succeeded) {'✅ 修复结果: 成功'} else {'⚠️ 修复结果: 未确认/失败'})"
                }
                # 修复后对照
                $postIcon = if ($ResourceResults.Firewall.Active) { "🟢" } else { "🔴" }
                $postType = if ($ResourceResults.Firewall.Type) { $ResourceResults.Firewall.Type } else { "unknown" }
                $md += "  - $postIcon 修复后状态: $($ResourceResults.Firewall.Status) ($postType)"
                if ($ResourceResults.Firewall.OpenPorts) { $md += "    - 修复后端口/服务: $($ResourceResults.Firewall.OpenPorts)" }
            }
        }
        $md += ""
        Write-Host ""
    }

    # NTP 服务（异常->修复时间线）
    Write-Host "【NTP 服务检测】" -ForegroundColor Yellow
    $md += "## NTP 服务检测"
    if ($NTPResults) {
        Write-Host "  $($NTPResults.Status): $($NTPResults.Detail)"
        $ntpIcon = Get-StateIcon -Status $NTPResults.Status
        $md += "- 状态: $ntpIcon $($NTPResults.Status)"
        if ($NTPResults.Detail) { $md += "- 详情: $($NTPResults.Detail)" }

        # 解析时间线：初始异常->修复->复检
        $timeline = @()
        if ($NTPResults.Detail -match '未检测到|未运行|不可用|偏差|失败') {
            $timeline += "🔴 初次检测: 异常/偏差"
        }
        if ($NTPResults.Detail -match '已修复') {
            $timeline += "✅ 修复执行: 已自动修复"
        } elseif ($NTPResults.Detail -match '已尝试修复') {
            $timeline += "⚠️ 修复执行: 已尝试修复"
        }
        if ($NTPResults.Detail -match '时间差\s*\d+s') {
            $diffStr = [regex]::Match($NTPResults.Detail, '时间差\s*\d+s').Value
            $timeline += "🟢 复检结果: $diffStr"
        }
        if ($timeline.Count -gt 0) {
            $md += "- 修复时间线"
            foreach ($t in $timeline) { $md += "  - $t" }
        }
    } else {
        Write-Host "  未执行" -ForegroundColor Yellow
        $md += "- 状态: ℹ️ 未执行"
    }
    $md += ""
    Write-Host ""

    # 文件权限
    Write-Host "【文件权限检测】" -ForegroundColor Yellow
    $md += "## 文件权限检测"
    if ($FilePermResults) {
        Write-Host "  $($FilePermResults.Summary)"
        $permHasMiss = $false
        if ($FilePermResults.Lines) { foreach ($l in $FilePermResults.Lines) { if ($l -match '^\s*MISS\s+') { $permHasMiss = $true; break } } }
        $permIcon = if ($permHasMiss) { "⚠️" } else { "✅" }
        $md += "- 总结: $permIcon $($FilePermResults.Summary)"
        if ($FilePermResults.Lines) { $md += "- 明细:"; foreach ($l in $FilePermResults.Lines) { $md += "  - $l" } }
    } else { Write-Host "  未执行" -ForegroundColor Yellow; $md += "- 总结: ℹ️ 未执行" }
    $md += ""

    # 容器信息
    # 容器信息
    $md += "## 容器信息"
    if (-not $ContainerInfo -or $ContainerInfo.Count -eq 0) {
        Write-Host "【容器信息】未发现容器或容器运行时不可用" -ForegroundColor Yellow
        $md += "- 未发现容器或容器运行时不可用"
    } else {
        # ✅ 优先输出“容器详情”（如果 Test-ContainerInformation 已生成）
        $detailItem = $ContainerInfo | Where-Object { $_.Check -eq "容器详情" } | Select-Object -First 1
        if ($detailItem -and $detailItem.Details) {
            $md += ($detailItem.Details -split "`n")
        } else {
            # fallback：保持你原来的简单汇总输出
            $total = $ContainerInfo.Count
            $md += "- 记录条数: $total"
            foreach ($c in $ContainerInfo) {
                $name = $c.Check
                $status = $c.Status
                if (-not $name -and -not $status) { continue }
                $icon = if ($status -match '完成|正常|运行中|Up') { "✅" } else { "⚠️" }
                $md  += ("- {0} {1}: {2}" -f $icon, $name, $status)
            }
        }
    }
    $md += ""

    # 安卓设备自检结果
    if ($AndroidResults -and $AndroidResults.Count -gt 0) {
        Write-Host "【安卓设备自检】" -ForegroundColor Yellow
        $md += "## 安卓设备自检"
        foreach ($r in $AndroidResults) {
            $ok = $false
            if ($r -is [hashtable]) { $ok = [bool]$r.Success } else { $ok = [bool]$r.Success }
            $icon = if ($ok) { "✅" } else { "❌" }
            $name = if ($r -is [hashtable]) { $r.Check } else { $r.Check }
            $st   = if ($r -is [hashtable]) { $r.Status } else { $r.Status }
            $dt   = if ($r -is [hashtable]) { $r.Details } else { $r.Details }
            $line = "- {0} {1}: {2}" -f $icon, $name, $st
            if ($dt) { $line += "，$dt" }
            $md += $line
            Write-Host "  $line"
        }
        $md += ""
        Write-Host ""
    }

    # 总结
    Write-Host "==================================================================" -ForegroundColor Cyan
    Write-Host "【检测总结】" -ForegroundColor Yellow
    Write-Host "  总服务数:   $totalServices"
    Write-Host "  正常运行:   $runningServices" -ForegroundColor Green
    Write-Host "  异常服务:   $failedServices" -ForegroundColor $(if ($failedServices -gt 0) { "Red" } else { "Green" })

    $overallIcon = if ($failedServices -eq 0 -and $totalServices -gt 0) { "✅" } elseif ($failedServices -gt 0) { "❌" } else { "ℹ️" }
    $md += "## 检测总结"
    $md += "- 总服务数: $totalServices"
    $md += "- 正常运行: $runningServices"
    $md += "- 异常服务: $failedServices"
    if ($failedServices -eq 0 -and $totalServices -gt 0) { Write-Host ""; Write-Host "  所有服务运行正常！" -ForegroundColor Green; $md += "- 结论: $overallIcon 所有服务运行正常！" }
    elseif ($failedServices -gt 0) { Write-Host ""; Write-Host "  存在异常服务，请及时处理！" -ForegroundColor Red; $md += "- 结论: $overallIcon 存在异常服务，请及时处理！" }
    elseif ($totalServices -eq 0) { Write-Host ""; Write-Host "  未检测到任何服务" -ForegroundColor Yellow; $md += "- 结论: $overallIcon 未检测到任何服务" }

    # 日志导出
    if ($LogExportResults) {
        Write-Host "【日志导出结果】" -ForegroundColor Yellow
        $md += ""; $md += "## 服务日志导出"
        if ($LogExportResults.Success -and $LogExportResults.ExportedFiles.Count -gt 0) {
            Write-Host "  导出状态: 成功" -ForegroundColor Green
            Write-Host "  导出目录: $($LogExportResults.ExportDir)" -ForegroundColor Cyan
            Write-Host "  成功导出 $($LogExportResults.ExportedFiles.Count) 个文件:" -ForegroundColor Green
            $md += "- 导出状态: ✅ 成功"
            $md += "- 导出目录: $($LogExportResults.ExportDir)"
            $md += "- 成功文件:"
            foreach ($file in $LogExportResults.ExportedFiles) { $sizeKB = [math]::Round($file.Size / 1024, 2); Write-Host "    - $($file.Name) ($sizeKB KB)" -ForegroundColor Gray; $md += "  - $($file.Name) ($sizeKB KB)" }
        } elseif ($LogExportResults.ExportedFiles.Count -eq 0 -and $LogExportResults.FailedFiles.Count -eq 0) {
            Write-Host "  导出状态: 无需导出的日志文件" -ForegroundColor Yellow
            $md += "- 导出状态: ℹ️ 无需导出的日志文件"
        } else {
            Write-Host "  导出状态: 部分失败" -ForegroundColor Yellow
            $md += "- 导出状态: ⚠️ 部分失败"
            if ($LogExportResults.FailedFiles.Count -gt 0) { $md += "- 失败文件:"; foreach ($file in $LogExportResults.FailedFiles) { $md += "  - $($file.Name): $($file.Reason)" } }
        }
        Write-Host ""
    }

    Write-Host "==================================================================" -ForegroundColor Cyan
    Write-Host ""; Write-Host "日志文件: $LOG_FILE"; Write-Host ""

    try { ($md -join "`r`n") | Out-File -FilePath $mdFile -Encoding utf8; Write-Log -Level "SUCCESS" -Message "Markdown 报告已生成: $mdFile" }
    catch { Write-Log -Level "ERROR" -Message "Markdown 写入失败: $($_.Exception.Message)"; Write-Log -Level "ERROR" -Message "报告目录: $reportDir"; Write-Log -Level "ERROR" -Message "文件路径: $mdFile" }
}

# ================================
# 配置文件IP检测 - 传统平台
# ================================
function Check-TraditionalPlatformIPs {
    param (
        [Parameter(Mandatory=$true)][string]$ServerIP,
        [Parameter(Mandatory=$true)][string]$Username,
        [Parameter(Mandatory=$true)][string]$Password,
        [Parameter(Mandatory=$false)][hashtable]$SystemInfo
    )

    Write-Log -Level "INFO" -Message "[CFG] 开始检测传统平台配置文件 IP..."

    $ujavaVariant = "meeting"
    if ($SystemInfo -and $SystemInfo.ContainsKey('UjavaSystemVariant') -and $SystemInfo.UjavaSystemVariant) {
        $ujavaVariant = [string]$SystemInfo.UjavaSystemVariant
    }

    # meeting / unified 路径分支
    $webRoot = "/var/www/java"
    if ($ujavaVariant -eq "unified") {
        $webRoot = "/var/www/java/unifiedPlatform"
    }

    $Paths = @(
        "/var/www/java/api-java-meeting2.0/config",
        "/var/www/java/external-meeting-api/config",
        "$webRoot/ubains-web-2.0/static/config.json",
        "$webRoot/ubains-web-admin/static/config.json",
        # 会议预定系统才有 h5（统一平台如果没有就会自动跳过，不影响，但这里按更严谨处理）
        "/var/www/java/ubains-web-h5/static/h5/config.js",
        "/var/www/java/ubains-web-h5/static/h5/config.json",
        "$webRoot/nginx-conf.d"
    )

    $AllowedIPs = @("172.17.0.1", "127.0.0.1", $ServerIP)

    $hasUnauthorized = $false
    foreach ($Path in $Paths) {
        Write-Log -Level "INFO" -Message ("[CFG] 检测路径: {0}" -f $Path)
        $Command = "grep -Eo '([0-9]{1,3}\.){3}[0-9]{1,3}' $Path/* 2>/dev/null | sort -u"
        $Result  = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command $Command

        if ($Result -and $Result.ExitCode -eq 0 -and $Result.Output) {
            $IPs = $Result.Output -split "`n" | Where-Object { $_ -match '^\d{1,3}(\.\d{1,3}){3}$' }
            foreach ($IP in $IPs) {
                if ($AllowedIPs -notcontains $IP) {
                    $hasUnauthorized = $true
                    Write-Log -Level "WARN" -Message ("[CFG] 未授权 IP: {0} (路径: {1})" -f $IP, $Path)
                }
            }
        }
    }

    if (-not $hasUnauthorized) {
        Write-Log -Level "SUCCESS" -Message "[CFG] 配置文件 IP 均为目标服务器 IP / 本地环回 / 默认网桥 IP 或未配置 IP"
    }
    Write-Log -Level "INFO" -Message "[CFG] 传统平台配置文件 IP 检测完成"
}

# ================================
# 配置文件IP检测 - 新统一平台
# ================================
function Check-NewPlatformIPs {
    param (
        [string]$ServerIP,
        [string]$Username,
        [string]$Password
    )

    Write-Host "开始检测新统一平台配置文件中的IP地址..." -ForegroundColor Yellow

    $Directories = @(
        "/data/services/api/auth/auth-sso-auth/config",
        "/data/services/api/auth/auth-sso-gatway/config",
        "/data/services/api/auth/auth-sso-system/config",
        "/data/services/api/java-meeting/java-meeting2.0/config",
        "/data/services/api/java-meeting/java-meeting3.0/config",
        "/data/services/api/java-meeting/java-meeting-extapi/config",
        "/data/services/api/java-meeting/java-message-scheduling/config",
        "/data/services/api/java-meeting/java-mqtt/config",
        "/data/services/api/java-meeting/java-quartz/config",
        "/data/services/api/python-cmdb",
        "/data/services/web/pc/pc-vue2-ai",
        "/data/services/web/pc/pc-vue2-backstage",
        "/data/services/web/pc/pc-vue2-editor",
        "/data/services/web/pc/pc-vue2-main",
        "/data/services/web/pc/pc-vue2-meetingControl",
        "/data/services/web/pc/pc-vue2-meetngV2",
        "/data/services/web/pc/pc-vue2-meetngV3",
        "/data/services/web/pc/pc-vue2-moniter",
        "/data/services/web/pc/pc-vue2-platform",
        "/data/services/web/pc/pc-vue2-voice",
        "/data/middleware/nginx/config"
    )

    $AllowedIPs = @("172.17.0.1", "127.0.0.1", $ServerIP)
    $hasUnauthorized = $false

    foreach ($Directory in $Directories) {
        $Command = "find $Directory -type f \( -name '*.yml' -o -name '*.properties' -o -name 'config.js' -o -name 'config.json' -o -name 'unified*.conf' \) 2>/dev/null"
        $Result = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command $Command

        if ($Result.ExitCode -eq 0 -and $Result.Output) {
            $Files = $Result.Output -split "`n" | Where-Object { $_ -ne "" }
            foreach ($File in $Files) {
                $GrepCommand = "grep -Eo '([0-9]{1,3}\.){3}[0-9]{1,3}' $File 2>/dev/null | sort -u"
                $IPResult = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command $GrepCommand

                if ($IPResult.ExitCode -eq 0 -and $IPResult.Output) {
                    $IPs = $IPResult.Output -split "`n" | Where-Object { $_ -ne "" }
                    $UnauthorizedIPs = @()
                    foreach ($IP in $IPs) { if ($AllowedIPs -notcontains $IP) { $UnauthorizedIPs += $IP } }
                    if ($UnauthorizedIPs.Count -gt 0) {
                        $hasUnauthorized = $true
                        Write-Warning "文件: $File 未授权IP: $($UnauthorizedIPs -join ', ')"
                    }
                    # 合法IP不打印
                }
            }
        } else {
            Write-Host "未找到配置文件: $Directory" -ForegroundColor Yellow
        }
    }

    if (-not $hasUnauthorized) {
        Write-Host "配置文件IP均为目标服务器IP/本地环回/默认网桥IP" -ForegroundColor Green
    }
    Write-Host "新统一平台配置文件IP检测完成." -ForegroundColor Green
}

# ================================
# 检测 NTP 服务
# ================================
function Check-NTPService {
    param (
        [string]$ServerIP,
        [string]$Username,
        [string]$Password
    )
    $summary = @{ Status = '未执行'; Detail = '' }
    $needRepair = $false
    $serverForRepair = @{ IP = $ServerIP; User = $Username; Pass = $Password; Port = 22 }

    Write-Host "开始检测目标服务器的NTP服务..." -ForegroundColor Yellow

    Write-Log -Level "INFO" -Message "[NTP] 检测服务是否安装 (ntp/chronyd)"
    $Command = "systemctl list-units --type=service | grep -E 'ntp|chronyd'"
    $Result = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command $Command
    Write-Log -Level "INFO" -Message "[NTP] systemctl 输出: $([string]::Join(' ', $Result.Output))"
    if ($Result.ExitCode -eq 0 -and $Result.Output -match "ntp|chronyd") {
        Write-Log -Level "SUCCESS" -Message "[NTP] 已检测到 NTP/Chrony 服务"

        # 新增：检查 chronyd/ntpd 是否处于 active(running)
        $activeCmd = "if systemctl is-active chronyd >/dev/null 2>&1; then echo CHRONYD_ACTIVE; fi; if systemctl is-active ntpd >/dev/null 2>&1; then echo NTPD_ACTIVE; fi"
        $activeRes = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command $activeCmd
        $chronydActive = ($activeRes.Output -match 'CHRONYD_ACTIVE')
        $ntpdActive    = ($activeRes.Output -match 'NTPD_ACTIVE')
        if (-not $chronydActive -and -not $ntpdActive) {
            Write-Log -Level "ERROR" -Message "[NTP] 检测到 chronyd/ntpd 未运行"
            $summary.Status = '异常'; $summary.Detail = 'chronyd/ntpd 未运行'
            $needRepair = $true
        }

        $StatusCommand = "timedatectl status"
        $StatusResult = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command $StatusCommand
        Write-Log -Level "INFO" -Message "[NTP] timedatectl 输出: $([string]::Join(' ', $StatusResult.Output))"
        if ($StatusResult.ExitCode -eq 0) {
            $TimeCommand = "date +%s"; Write-Log -Level "INFO" -Message "[NTP] 读取服务器时间: $TimeCommand"
            $ServerTimeResult = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command $TimeCommand
            Write-Log -Level "INFO" -Message "[NTP] 服务器时间戳: $([string]::Join(' ', $ServerTimeResult.Output))"
            if ($ServerTimeResult.ExitCode -eq 0) {
                $ServerTimeUTC = [int]($ServerTimeResult.Output | Select-Object -First 1).Trim()
                # 本地改为 UTC 秒，避免时区偏差
                $LocalTime = [int][DateTimeOffset]::UtcNow.ToUnixTimeSeconds()
                $TimeDifference = [math]::Abs($ServerTimeUTC - $LocalTime)
                Write-Log -Level "INFO" -Message "[NTP] 本地UTC时间戳: $LocalTime, 差值: $TimeDifference 秒"
                if ($TimeDifference -le 5 -and ($chronydActive -or $ntpdActive)) {
                    Write-Log -Level "SUCCESS" -Message "[NTP] 服务器时间与本地时间一致"
                    $summary.Status = '正常'
                    $summary.Detail = "时间差 ${TimeDifference}s"
                } else {
                    Write-Log -Level "WARN" -Message "[NTP] 时间偏差 ${TimeDifference} 秒"
                    $summary.Status = '偏差'
                    $summary.Detail = "时间差 ${TimeDifference}s"
                    $needRepair = $true
                }
            } else { 
                Write-Log -Level "ERROR" -Message "[NTP] 获取服务器时间失败"
                $summary.Status = '异常'; $summary.Detail = '获取服务器时间失败'
                $needRepair = $true
            }
        } else { 
            Write-Log -Level "ERROR" -Message "[NTP] timedatectl 不可用"
            $summary.Status = '异常'; $summary.Detail = 'timedatectl 不可用'
            $needRepair = $true
        }
    } else {
        Write-Log -Level "WARN" -Message "[NTP] 未检测到 NTP/Chrony 服务"
        $summary.Status = '未安装'; $summary.Detail = '未检测到 ntp/chronyd'
        $needRepair = $true
    }
    Write-Log -Level "INFO" -Message "[NTP] 检测结束"

    # 若异常/未安装/偏差，则上传并执行修复动作
    if ($needRepair) {
        Write-Log -Level "INFO" -Message "[NTP] 触发远端修复: ./issue_handler.sh --action fix_ntp_config --ntp-auto"
        try {
            $repairRes = Upload_the_repair_script -Server $serverForRepair -Action "fix_ntp_config" -Platform "auto" -RemoteDir "/home/repair_scripts"
            if ($repairRes -and $repairRes['Success']) {
                Write-Log -Level "SUCCESS" -Message "[NTP] 远端修复已执行成功 (fix_ntp_config)"
                # 修复后复检 NTP 状态与时间差
                Write-Log -Level "INFO" -Message "[NTP] 修复后复检..."
                $postStatus = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command "timedatectl status"
                $postTime = Invoke-SSHCommand -HostName $ServerIP -User $Username -Pass $Password -Command "date +%s"
                if ($postStatus.ExitCode -eq 0 -and $postTime.ExitCode -eq 0) {
                    $srvTs = [int]($postTime.Output | Select-Object -First 1).Trim()
                    # 本地用 UTC 秒
                    $locTs = [int][DateTimeOffset]::UtcNow.ToUnixTimeSeconds()
                    $diff = [math]::Abs($srvTs - $locTs)
                    Write-Log -Level "INFO" -Message "[NTP] 复检时间差: ${diff}s"
                    if ($diff -le 5) {
                        $summary.Status = '正常'
                        $summary.Detail = "已修复 | 时间差 ${diff}s"
                        Write-Log -Level "SUCCESS" -Message "[NTP] 修复成功，时间已对齐"
                    } else {
                        # 未完全对齐，标记为偏差但已尝试修复
                        $summary.Status = '偏差'
                        $summary.Detail = "已尝试修复 | 当前时间差 ${diff}s"
                        Write-Log -Level "WARN" -Message "[NTP] 修复后仍有偏差 ${diff}s"
                    }
                } else {
                    # 复检失败但修复已执行
                    if ([string]::IsNullOrWhiteSpace($summary.Detail)) {
                        $summary.Detail = "已尝试修复(复检失败)"
                    } else {
                        $summary.Detail = "$($summary.Detail) | 已尝试修复(复检失败)"
                    }
                    Write-Log -Level "WARN" -Message "[NTP] 修复后复检失败"
                }
            } else {
                # 安全读取返回的错误信息
                $errMsg = "未知错误"
                if ($repairRes -is [hashtable]) {
                    if ($repairRes.ContainsKey('Error') -and $repairRes['Error']) {
                        $errMsg = [string]::Join(' ', $repairRes['Error'])
                    } elseif ($repairRes.ContainsKey('Output') -and $repairRes['Output']) {
                        $errMsg = [string]::Join(' ', $repairRes['Output'])
                    } elseif ($repairRes.ContainsKey('Message') -and $repairRes['Message']) {
                        $errMsg = $repairRes['Message']
                    }
                } elseif ($repairRes) {
                    $errMsg = $repairRes.ToString()
                }
                Write-Log -Level "ERROR" -Message "[NTP] 远端修复执行失败: $errMsg"
            }
        } catch {
            # 捕获函数调用异常（不再访问 .Error）
            Write-Log -Level "ERROR" -Message "[NTP] 调用 Upload_the_repair_script 异常: $($_.Exception.Message)"
        }
    }

    return $summary
}

# ================================
# 检测 Check-FilePermissions 服务
# ================================
function Check-FilePermissions {
    param (
        [Parameter(Mandatory=$true)] [hashtable]$Server,
        [Parameter(Mandatory=$true)] [ValidateSet('new','old')] [string]$PlatformType,
        [Parameter(Mandatory=$false)] [hashtable]$SystemInfo
    )

    Write-Log -Level "INFO" -Message "开始文件权限检测 (平台: $PlatformType) ..."

    $targets = @()
    if ($PlatformType -eq 'new') {
        $targets += @(
            "/data/services/api/auth/auth-sso-auth/run.sh",
            "/data/services/api/auth/auth-sso-gatway/run.sh",
            "/data/services/api/auth/auth-sso-system/run.sh",
            "/data/services/api/java-meeting/java-meeting2.0/run.sh",
            "/data/services/api/java-meeting/java-meeting3.0/run.sh",
            "/data/services/api/java-meeting/java-meeting-extapi/run.sh",
            "/data/services/api/java-meeting/java-message-scheduling/run.sh",
            "/data/services/api/java-meeting/java-mqtt/run.sh",
            "/data/services/api/java-meeting/java-quartz/run.sh",
            "/data/services/api/start.sh",
            "/data/services/scripts/*.sh",
            "/data/third_party/paperless/run.sh",
            "/data/third_party/paperless/start.sh",
            "/data/third_party/wifi-local/config.ini",
            "/data/third_party/wifi-local/startDB.sh",
            "/data/third_party/wifi-local/wifi*",
            "/etc/rc.d/rc.local",
            "/data/middleware/nginx/config/*.conf",
            "/data/middleware/emqx/config/*.conf",
            "/data/services/api/python-cmdb/*.sh",
            "/data/services/api/python-voice/*.sh"
        )
    } else {
        # ✅ 传统平台：根据 ujava 系统细分调整路径
        $ujavaVariant = $null
        if ($SystemInfo -and $SystemInfo.ContainsKey('UjavaSystemVariant')) { $ujavaVariant = $SystemInfo.UjavaSystemVariant }

        $targets += @(
            "/var/www/java/api-java-meeting2.0/run.sh",
            "/var/www/java/external-meeting-api/run.sh",
            "/var/www/html/start.sh",
            "/var/www/wifi-local/config.ini",
            "/var/www/wifi-local/startDB.sh",
            "/var/www/wifi-local/wifi*",
            "/var/www/paperless/run.sh",
            "/var/www/paperless/start.sh",
            "/var/www/redis/redis-*.conf",
            "/var/www/emqx/config/*.conf",
            "/etc/rc.d/rc.local"
        )

        # start.sh 路径分支：unified 优先检查 unifiedPlatform/start.sh
        if ($ujavaVariant -eq 'unified') {
            $targets += "/var/www/java/unifiedPlatform/start.sh"
            $targets += "/var/www/java/unifiedPlatform/nginx-conf.d/*.conf"
        } else {
            $targets += "/var/www/java/start.sh"
            $targets += "/var/www/java/nginx-conf.d/*.conf"
        }
    }

    Write-Log -Level "INFO" -Message "[PERM] 目标列表生成 (平台: $PlatformType)"
    foreach ($path in $targets) { Write-Log -Level "INFO" -Message "[PERM] 待检查: $path" }
    $lines = @()
    foreach ($path in $targets) {
        $cmd = "if ls -l $path 2>/dev/null; then echo '__PERM_OK__'; else echo '__PERM_MISS__ $path'; fi"
        Write-Log -Level "INFO" -Message "[PERM] 执行: $cmd"
        $res = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $cmd
        $out = if ($res.Output) { ($res.Output -join " ") } else { "" }
        Write-Log -Level "INFO" -Message "[PERM] 输出: $out"
        if ($out -match "__PERM_OK__") {
            ($res.Output) | Where-Object { $_ -match "^[-dl]" } | ForEach-Object {
                Write-Log -Level 'SUCCESS' -Message "[PERM] 权限: $_"; $lines += $_
            }
        } else { Write-Log -Level 'ERROR' -Message "[PERM] 未找到: $path"; $lines += "MISS $path" }
    }
    Write-Log -Level "INFO" -Message "[PERM] 检测结束: 共 $($lines.Count) 项"

    $summaryText = if ($lines.Count -gt 0) { "已检查 $($lines.Count) 项" } else { "未找到任何匹配项" }
    return @{ Summary = $summaryText; Lines = $lines }
}

# ================================
# 从远端下载文件到本地（基于 pscp）
# ================================
function Download-RemoteFile {
    param(
        [Parameter(Mandatory=$true)] [hashtable]$Server,
        [Parameter(Mandatory=$true)] [string]$RemotePath,
        [Parameter(Mandatory=$true)] [string]$LocalPath,
        [int]$TimeoutSeconds = 600
    )

    if (-not $script:PSCP_PATH -or -not (Test-Path $script:PSCP_PATH)) {
        Write-Log -Level "ERROR" -Message "[DL] pscp.exe 未找到，无法下载文件: $RemotePath"
        return @{ Success = $false; Reason = "pscp not found" }
    }

    $localDir = Split-Path $LocalPath -Parent
    if (-not (Test-Path $localDir)) { New-Item -ItemType Directory -Path $localDir -Force | Out-Null }

    $args = @(
        "-scp",
        "-batch",
        "-P", $Server.Port,
        "-l", $Server.User,
        "-pw", $Server.Pass,
        "$($Server.User)@$($Server.IP):$RemotePath",
        $LocalPath
    )

    Write-Log -Level "INFO" -Message ("[DL] pscp 下载(超时 {0}s): {1} {2}" -f $TimeoutSeconds, $script:PSCP_PATH, ($args -join ' '))

    # 用临时文件接 stdout/stderr，避免卡住无输出时看起来“没打印”
    $tmpOut = Join-Path $env:TEMP ("pscp_out_{0}.log" -f ([guid]::NewGuid().ToString("N")))
    $tmpErr = Join-Path $env:TEMP ("pscp_err_{0}.log" -f ([guid]::NewGuid().ToString("N")))

    try {
        $p = Start-Process -FilePath $script:PSCP_PATH `
                           -ArgumentList $args `
                           -NoNewWindow `
                           -PassThru `
                           -RedirectStandardOutput $tmpOut `
                           -RedirectStandardError  $tmpErr

        $ok = $p.WaitForExit($TimeoutSeconds * 1000)
        if (-not $ok) {
            try { $p.Kill() } catch {}
            Write-Log -Level "ERROR" -Message ("[DL] 下载超时，已终止 pscp (>{0}s): {1}" -f $TimeoutSeconds, $RemotePath)
            $outTxt = (Get-Content -Path $tmpOut -ErrorAction SilentlyContinue | Out-String).Trim()
            $errTxt = (Get-Content -Path $tmpErr -ErrorAction SilentlyContinue | Out-String).Trim()

            if ($outTxt) {
                $one = ($outTxt -replace '\s+',' ').Trim()
                $n = [math]::Min(500, $one.Length)
                Write-Log -Level "ERROR" -Message ("[DL] pscp stdout: {0}" -f $one.Substring(0, $n))
            }
            if ($errTxt) {
                $one = ($errTxt -replace '\s+',' ').Trim()
                $n = [math]::Min(500, $one.Length)
                Write-Log -Level "ERROR" -Message ("[DL] pscp stderr: {0}" -f $one.Substring(0, $n))
            }

            return @{ Success = $false; ExitCode = -1; Reason = "timeout" }
        }

        $code = $p.ExitCode
        $outTxt = (Get-Content -Path $tmpOut -ErrorAction SilentlyContinue | Out-String)
        $errTxt = (Get-Content -Path $tmpErr -ErrorAction SilentlyContinue | Out-String)

        $all = (($outTxt + "`n" + $errTxt) -replace "`r","").Trim()

        # host key：第一次连接需要 y，pscp 在 -batch 下会失败，这里自动接受并重试一次
        if ($code -ne 0 -and $all -match "host key|Cannot confirm") {
            $cmdLine = "echo y | `"$($script:PSCP_PATH)`" -scp -batch -P $($Server.Port) -l $($Server.User) -pw `"$($Server.Pass)`" `"$($Server.User)@$($Server.IP):$RemotePath`" `"$LocalPath`""
            Write-Log -Level "WARN" -Message "[DL] 首次连接主机密钥提示，自动接受并重试一次"

            $all2 = cmd /c $cmdLine 2>&1
            $code2 = $LASTEXITCODE

            if ($code2 -eq 0 -and (Test-Path $LocalPath)) {
                $size = (Get-Item $LocalPath).Length
                Write-Log -Level "SUCCESS" -Message "[DL] 下载成功 ($([math]::Round($size/1024,2)) KB): $LocalPath"
                return @{ Success = $true; Size = $size; Output = $all2 }
            } else {
                $oneLine = (($all2 -join " ") -replace '\s+',' ').Trim()
                Write-Log -Level "ERROR" -Message "[DL] 下载失败(重试) ExitCode=$code2, 输出: $oneLine"
                return @{ Success = $false; ExitCode = $code2; Output = $all2 }
            }
        }

        $code = $null
        try { $code = $p.ExitCode } catch { $code = $null }

        $outTxt = (Get-Content -Path $tmpOut -ErrorAction SilentlyContinue | Out-String)
        $errTxt = (Get-Content -Path $tmpErr -ErrorAction SilentlyContinue | Out-String)
        $all = (($outTxt + "`n" + $errTxt) -replace "`r","").Trim()

        # ✅ 关键：以“文件是否存在且大小>0”作为成功判定（pscp 会把进度输出到 stderr，不代表失败）
        $fileOk = $false
        $size = 0
        if (Test-Path -LiteralPath $LocalPath) {
            try {
                $size = (Get-Item -LiteralPath $LocalPath).Length
                if ($size -gt 0) { $fileOk = $true }
            } catch { $fileOk = $false }
        }

        # host key：第一次连接需要 y（只有在明确失败且输出包含 host key 时才走）
        if (-not $fileOk -and $all -match "host key|Cannot confirm") {
            $cmdLine = "echo y | `"$($script:PSCP_PATH)`" -scp -batch -P $($Server.Port) -l $($Server.User) -pw `"$($Server.Pass)`" `"$($Server.User)@$($Server.IP):$RemotePath`" `"$LocalPath`""
            Write-Log -Level "WARN" -Message "[DL] 首次连接主机密钥提示，自动接受并重试一次"

            $all2 = cmd /c $cmdLine 2>&1
            $code2 = $LASTEXITCODE

            if ((Test-Path -LiteralPath $LocalPath) -and ((Get-Item -LiteralPath $LocalPath).Length -gt 0)) {
                $size2 = (Get-Item -LiteralPath $LocalPath).Length
                Write-Log -Level "SUCCESS" -Message "[DL] 下载成功 ($([math]::Round($size2/1024,2)) KB): $LocalPath"
                return @{ Success = $true; Size = $size2; Output = $all2 }
            } else {
                $oneLine = (($all2 -join " ") -replace '\s+',' ').Trim()
                Write-Log -Level "ERROR" -Message "[DL] 下载失败(重试) ExitCode=$code2, 输出: $oneLine"
                return @{ Success = $false; ExitCode = $code2; Output = $all2 }
            }
        }

        if ($fileOk) {
            Write-Log -Level "SUCCESS" -Message "[DL] 下载成功 ($([math]::Round($size/1024,2)) KB): $LocalPath"
            return @{ Success = $true; Size = $size; ExitCode = $code; Output = $all }
        } else {
            $oneLine = ($all -replace '\s+',' ').Trim()
            if ($oneLine.Length -gt 800) { $oneLine = $oneLine.Substring(0,800) + "..." }
            Write-Log -Level "ERROR" -Message ("[DL] 下载失败 ExitCode={0}, 输出: {1}" -f $code, $oneLine)
            return @{ Success = $false; ExitCode = $code; Output = $all }
        }
    }
    finally {
        Remove-Item -Path $tmpOut,$tmpErr -Force -ErrorAction SilentlyContinue | Out-Null
    }
}

# ================================
# 远程复制辅助函数（pscp 优先）
# ================================
function Copy-File-To-Remote {
    param(
        [Parameter(Mandatory=$true)] [string]$LocalPath,
        [Parameter(Mandatory=$true)] [hashtable]$Server,
        [Parameter(Mandatory=$true)] [string]$RemoteDir
    )
    if (-not (Test-Path -LiteralPath $LocalPath)) {
        Write-Log -Level "ERROR" -Message "本地文件不存在: $LocalPath"
        return $false
    }

    # 1) 用 pscp.exe 直接传输
    if ($script:PSCP_PATH -and (Test-Path $script:PSCP_PATH)) {
        Write-Log -Level "INFO" -Message ("已找到 pscp: {0}" -f $script:PSCP_PATH)
        $pscpArgs = @(
            "-P", $Server.Port,
            "-l", $Server.User,
            "-pw", $Server.Pass,
            "-batch",
            $LocalPath,
            ("{0}@{1}:{2}/" -f $Server.User, $Server.IP, $RemoteDir)
        )
        try {
            $out = & $script:PSCP_PATH @pscpArgs 2>&1
            $code = $LASTEXITCODE
            if ($code -ne 0 -and ($out -match "host key" -or $out -match "Cannot confirm")) {
                $cmdLine = "echo y | `"$($script:PSCP_PATH)`" -P $($Server.Port) -l $($Server.User) -pw `"$($Server.Pass)`" `"$LocalPath`" `"$($Server.User)@$($Server.IP):$RemoteDir/`""
                $out = cmd /c $cmdLine 2>&1
                $code = $LASTEXITCODE
            }
            if ($code -eq 0) {
                Write-Log -Level "SUCCESS" -Message "文件上传成功: $LocalPath -> $($Server.User)@$($Server.IP):$RemoteDir/"
                return $true
            } else {
                Write-Log -Level "ERROR" -Message "pscp 上传失败: $out"
                return $false
            }
        } catch {
            Write-Log -Level "ERROR" -Message "pscp 异常: $($_.Exception.Message)"
            return $false
        }
    }

    # 2) 没有 pscp 时，使用 plink + base64 方式传输（兼容）
    if ($script:PLINK_PATH -and (Test-Path $script:PLINK_PATH)) {
        $fileName = [System.IO.Path]::GetFileName($LocalPath)
        $remoteFile = "$RemoteDir/$fileName"
        Write-Log -Level "INFO" -Message "使用 plink/base64 传输: $fileName"

        $bytes = [System.IO.File]::ReadAllBytes($LocalPath)
        $b64 = [System.Convert]::ToBase64String($bytes)
        # 远端接收命令：创建目录、写入、解码
        $recvCmd = @"
set -e
mkdir -p '$RemoteDir'
cat > /tmp/.upload_${fileName}.b64 <<'__EOF__'
$b64
__EOF__
base64 -d /tmp/.upload_${fileName}.b64 > '$remoteFile'
rm -f /tmp/.upload_${fileName}.b64
"@
        $plinkArgs = @("-ssh","-P",$Server.Port,"-l",$Server.User,"-pw",$Server.Pass,"-batch",$Server.IP,$recvCmd)
        $out = & $script:PLINK_PATH @plinkArgs 2>&1
        $code = $LASTEXITCODE
        if ($code -eq 0) {
            Write-Log -Level "SUCCESS" -Message "文件上传成功: $LocalPath -> $remoteFile"
            return $true
        } else {
            Write-Log -Level "ERROR" -Message "plink/base64 上传失败: $out"
            return $false
        }
    }

    # 3) sshpass + scp（Linux/WSL 环境下）
    try {
        $scpCmd = "sshpass -p `"$($Server.Pass)`" scp -o StrictHostKeyChecking=no -P $($Server.Port) `"$LocalPath`" $($Server.User)@$($Server.IP):`"$RemoteDir/`""
        $out = & powershell -NoProfile -Command $scpCmd
        if ($LASTEXITCODE -eq 0) {
            Write-Log -Level "SUCCESS" -Message "文件上传成功 (sshpass/scp)"
            return $true
        } else {
            Write-Log -Level "ERROR" -Message "sshpass/scp 上传失败: $out"
            return $false
        }
    } catch {
        Write-Log -Level "ERROR" -Message "未找到可用的传输工具 (pscp/plink/sshpass)"
        return $false
    }
}

# ================================
# 上传修复脚本函数
# ================================
function Upload_the_repair_script {
    param(
        [hashtable]$Server,
        [string]$Action,
        [string]$Platform = "auto",
        [string]$RemoteDir = "/home/repair_scripts"
    )
    Write-Log -Level "INFO" -Message "========== 上传修复脚本并执行 =========="
    Write-Log -Level "INFO" -Message "目标: $($Server.User)@$($Server.IP):$($Server.Port) | 动作: $Action | 平台: $Platform"

    # 1) 确保远端目录存在
    $prepCmd = "mkdir -p '$RemoteDir'"
    $prepRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $prepCmd
    $prepExit = if ($prepRes -is [hashtable] -and $prepRes.ContainsKey('ExitCode')) { [int]$prepRes['ExitCode'] } else { -1 }
    if ($prepExit -ne 0) {
        $prepOut = if ($prepRes -is [hashtable] -and $prepRes.ContainsKey('Output')) { $prepRes['Output'] } else { $null }
        $prepErr = if ($prepRes -is [hashtable] -and $prepRes.ContainsKey('Error'))  { $prepRes['Error'] }  else { $null }
        return @{ Success = $false; Step = "MkdirRemote"; ExitCode = $prepExit; Output = $prepOut; Error = $prepErr }
    }

    # 2) 上传 issue_handler.sh 到远端目录
    $localScript = Join-Path (Split-Path -Parent $PSCommandPath) "issue_handler.sh"
    Write-Log -Level "INFO" -Message "本地脚本: $localScript"
    $scpOk = Copy-File-To-Remote -LocalPath $localScript -Server $Server -RemoteDir $RemoteDir
    if (-not $scpOk) {
        return @{ Success = $false; Step = "UploadScript"; Error = "SCP 上传失败" }
    }
    Write-Log -Level "SUCCESS" -Message "脚本上传完成"
    $remoteFile = "$RemoteDir/issue_handler.sh"
    Write-Log -Level "INFO" -Message "远端脚本就绪: $remoteFile"

    # 3) 修复换行并赋权
    $fixCmd = "set -e && cd '$RemoteDir' && dos2unix '$remoteFile' 2>/dev/null || true && chmod +x '$remoteFile'"
    $fixRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $fixCmd
    $fixExit = if ($fixRes -is [hashtable] -and $fixRes.ContainsKey('ExitCode')) { [int]$fixRes['ExitCode'] } else { -1 }
    if ($fixExit -ne 0) {
        $fixOut = if ($fixRes -is [hashtable] -and $fixRes.ContainsKey('Output')) { $fixRes['Output'] } else { $null }
        $fixErr = if ($fixRes -is [hashtable] -and $fixRes.ContainsKey('Error'))  { $fixRes['Error'] }  else { $null }
        return @{ Success = $false; Step = "PrepareRemoteScript"; ExitCode = $fixExit; Output = $fixOut; Error = $fixErr }
    }

    # 4) 远端执行修复动作
    $platArg = if ($Platform -and $Platform.Length -gt 0) { "--platform $Platform" } else { "" }

    # 保持原有逻辑，只新增 redis_container_exception 分支
    $extraArgs = ""
    if ($Action -eq "fix_ntp_config") {
        $extraArgs = "--ntp-auto"
    }
    elseif ($Action -eq "fix_port_access") {
        $extraArgs = "--non-interactive --yes"
    }
    elseif ($Action -eq "fix_external_service_disconnect") {
        # 对外服务修复：非交互 + 默认 yes
        $extraArgs = "--non-interactive --yes"
    }
    elseif ($Action -eq "fix_dns_config") {
        # DNS 修复：非交互
        $extraArgs = "--non-interactive --yes"
    }
    elseif ($Action -eq "redis_container_exception") {
        # Redis 容器异常修复：需求文档第12点，必须非交互 + 默认 yes
        $extraArgs = "--non-interactive --yes"
    }

    $execCmd = @(
        "set -e",
        "cd '$RemoteDir'",
        "./issue_handler.sh --action $Action $platArg $extraArgs"
    ) -join " && "

    Write-Log -Level "INFO" -Message "开始执行远端修复: $execCmd"
    $execRes = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $execCmd

    $exitCode = if ($execRes -is [hashtable] -and $execRes.ContainsKey('ExitCode')) { [int]$execRes['ExitCode'] } else { -1 }
    $output   = if ($execRes -is [hashtable] -and $execRes.ContainsKey('Output'))   { $execRes['Output'] }   else { $null }
    $errorOut = if ($execRes -is [hashtable] -and $execRes.ContainsKey('Error'))    { $execRes['Error'] }    else { $null }

    $succ = ($exitCode -eq 0)
    if ($succ) {
        Write-Log -Level "SUCCESS" -Message "远端修复执行成功: $Action"
    } else {
        Write-Log -Level "ERROR" -Message "远端修复执行失败，ExitCode=$exitCode"
    }

    return @{
        Success    = $succ
        Action     = $Action
        Platform   = $Platform
        RemoteDir  = $RemoteDir
        RemoteFile = $remoteFile
        ExitCode   = $exitCode
        Output     = $output
        Error      = $errorOut
    }
}

# ================================
# 检测 DataBakup 服务
# ================================
function DataBakup {
    param (
        [Parameter(Mandatory=$true)] [hashtable]$Server,
        [Parameter(Mandatory=$true)] [ValidateSet('new','old')] [string]$PlatformType,
        [Parameter(Mandatory=$true)] [hashtable]$SystemInfo
    )

    Write-Log -Level "INFO" -Message "开始现场数据备份 (平台: $PlatformType) ..."

    $bakDir = "/home/bakup"
    $cmds = @(
        "set -e",
        "mkdir -p $bakDir"
    )

    # ✅ 修复：SystemInfo key 对齐 Get-SystemType 的返回结构
    $hasUjava = [bool]($SystemInfo.HasUjava)
    $hasUpython = [bool]($SystemInfo.HasUpython)
    $hasCardtable = ($SystemInfo.ContainsKey('cardtable') -and $SystemInfo['cardtable'])
    $hasPaperless = ($SystemInfo.ContainsKey('paperless') -and $SystemInfo['paperless'])
    $hasUmysql = ($SystemInfo.ContainsKey('umysql') -and $SystemInfo['umysql'])

    $ujavaVariant = "meeting"
    if ($SystemInfo.ContainsKey('UjavaSystemVariant') -and $SystemInfo.UjavaSystemVariant) {
        $ujavaVariant = [string]$SystemInfo.UjavaSystemVariant
    }

    # 仅实现传统平台备份
    if ($PlatformType -eq 'old') {
        # ✅ ujava：按 meeting/unified 分支备份
        if ($hasUjava) {
            if ($ujavaVariant -eq 'unified') {
                $cmds += "[ -d /var/www/java/unifiedPlatform ] && cp -a /var/www/java/unifiedPlatform $bakDir/"
            } else {
                $cmds += "[ -d /var/www/java ] && cp -a /var/www/java $bakDir/"
            }
        }

        if ($hasUpython) {
            $cmds += "[ -d /var/www/html ] && cp -a /var/www/html $bakDir/"
        }

        $cmds += "[ -d /var/www/emqx ] && cp -a /var/www/emqx $bakDir/"
        $cmds += "[ -d /var/www/redis ] && cp -a /var/www/redis $bakDir/"
    } else {
        Write-Log -Level "WARN" -Message "新统一平台备份暂未实现，已跳过"
    }

    # ✅ umysql 判定建议改为：直接远端检测容器，而不是 SystemInfo['umysql']
    if ($PlatformType -eq 'old') {
        $dbUser = "root"; $dbPass = "dNrprU&2S"; $dbs = @("ubains","devops")
        $cmds += "mkdir -p /tmp/bak_sql"
        foreach ($db in $dbs) {
            $cmds += "if docker ps --format '{{.Names}}' | grep -q '^umysql$'; then docker exec umysql sh -c ""mysqldump -u${dbUser} -p'${dbPass}' --single-transaction --quick --lock-tables=false ${db}"" > /tmp/bak_sql/${db}_$(date +%Y%m%d%H%M%S).sql; fi || true"
        }
        $cmds += "if ls /tmp/bak_sql/*.sql >/dev/null 2>&1; then cp -a /tmp/bak_sql/*.sql $bakDir/; fi || true"
    }

    Write-Log -Level "INFO" -Message "[BAK] 创建备份目录: $bakDir"
    # 展示计划复制的目录
    Write-Log -Level "INFO" -Message "[BAK] 平台: $PlatformType, 容器: ujava=$hasUjava upython=$hasUpython cardtable=$hasCardtable paperless=$hasPaperless umysql=$hasUmysql"
    foreach ($c in $cmds) { Write-Log -Level "INFO" -Message "[BAK] 计划执行: $c" }
    $joined = ($cmds -join '; ')
    Write-Log -Level "INFO" -Message "[BAK] 执行远程备份命令"
    $res = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $joined
    $bakOutput = if ($res.Output) { [string]::Join(' ', $res.Output) } else { '' }
    Write-Log -Level "INFO" -Message "[BAK] 备份输出: $bakOutput"
    if ($res.ExitCode -ne 0) { Write-Log -Level "ERROR" -Message "[BAK] 远程备份步骤失败"; return @{ Summary = "失败" } }

    $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"; $tarName = "bakup_${timestamp}.tar.gz"; $tarPath = "/home/$tarName"
    $packCmd = "set -e; tar -czf $tarPath -C /home bakup"
    Write-Log -Level "INFO" -Message "[BAK] 压缩命令: $packCmd"
    $res2 = Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $packCmd
    $packOutput = if ($res2.Output) { [string]::Join(' ', $res2.Output) } else { '' }
    Write-Log -Level "INFO" -Message "[BAK] 压缩输出: $packOutput"
    if ($res2.ExitCode -ne 0 -or $packOutput -match "error") { Write-Log -Level "ERROR" -Message "[BAK] 压缩备份失败"; return @{ Summary = "失败" } }

    if (-not $script:PSCP_PATH) {
        Write-Log -Level "ERROR" -Message "[BAK] 未找到 pscp.exe，无法下载备份文件"; $downloadSummary = "未下载"
    } else {
        # 使用 ASCII 路径避免编码问题
        $localOutDir = Join-Path $SCRIPT_DIR "Downloads"
        if (-not (Test-Path $localOutDir)) { New-Item -ItemType Directory -Path $localOutDir | Out-Null }
        $localFile = Join-Path $localOutDir $tarName

        # 检查磁盘空间（尽量检查，失败则忽略，不中断流程）
        try {
            $qualifier = Split-Path $localOutDir -Qualifier
            if ($qualifier) {
                $drive = Get-PSDrive -Name $qualifier.TrimEnd(':')
                if ($drive -and ($drive.Free -lt 1GB)) {
                    Write-Log -Level "WARN" -Message "[BAK] 本地磁盘可用空间不足 1GB，可能导致下载失败"
                }
            }
        } catch {
            Write-Log -Level "WARN" -Message "[BAK] 无法检测本地磁盘空间，已忽略: $($_.Exception.Message)"
        }

        # 构造下载命令，添加 -batch 防交互
        $pscpCmd = "`"$($script:PSCP_PATH)`" -batch -scp -P $($Server.Port) -pw `"$($Server.Pass)`" $($Server.User)@$($Server.IP):$tarPath `"$localFile`""
        Write-Log -Level "INFO" -Message "[BAK] 下载命令: $pscpCmd"
        $dl = & powershell -NoProfile -Command $pscpCmd
        if ($LASTEXITCODE -ne 0) {
            Write-Log -Level "ERROR" -Message "[BAK] 下载备份失败，尝试使用 TEMP 目录重试"
            $fallbackDir = Join-Path $env:TEMP "ubains_downloads"
            if (-not (Test-Path $fallbackDir)) { New-Item -ItemType Directory -Path $fallbackDir -Force | Out-Null }
            $fallbackFile = Join-Path $fallbackDir $tarName
            $pscpCmd2 = "`"$($script:PSCP_PATH)`" -batch -scp -P $($Server.Port) -pw `"$($Server.Pass)`" $($Server.User)@$($Server.IP):$tarPath `"$fallbackFile`""
            Write-Log -Level "INFO" -Message "[BAK] 重试下载命令: $pscpCmd2"
            $dl2 = & powershell -NoProfile -Command $pscpCmd2
            if ($LASTEXITCODE -ne 0) {
                Write-Log -Level "ERROR" -Message "[BAK] 重试下载失败"
                $downloadSummary = "下载失败"
            } else {
                Write-Log -Level "SUCCESS" -Message "[BAK] 备份已下载: $fallbackFile"
                $downloadSummary = "已下载(TEMP)"
            }
        } else {
            Write-Log -Level "SUCCESS" -Message "[BAK] 备份已下载: $localFile"
            $downloadSummary = "已下载"
        }
    }

    $cleanCmd = "rm -rf $bakDir; rm -f $tarPath; rm -rf /tmp/bak_sql"; Write-Log -Level "INFO" -Message "[BAK] 清理命令: $cleanCmd"
    [void] (Invoke-SSHCommand -HostName $Server.IP -User $Server.User -Pass $Server.Pass -Port $Server.Port -Command $cleanCmd)
    Write-Log -Level "INFO" -Message "[BAK] 清理完成"

    Write-Log -Level "INFO" -Message "[BAK] 备份流程完成"
    return @{ Summary = "完成 ($downloadSummary)" }
}

# ================================
# 主函数
# ================================
function Main {
    Write-Host ""
    Write-Host "==================================================================" -ForegroundColor Cyan
    Write-Host "                    服务自检工具 (Windows 版本)" -ForegroundColor Cyan
    Write-Host "==================================================================" -ForegroundColor Cyan
    Write-Host ""
    
    # 版本号打印记录（控制台 + 日志）
    Write-Log -Level "INFO" -Message ("脚本版本: {0}" -f $SCRIPT_VERSION)
    Write-Log -Level "INFO" -Message ("PowerShell 版本: {0}" -f $PSVersionTable.PSVersion)
    Write-Log -Level "INFO" -Message ("脚本路径: {0}" -f $PSCommandPath)

    Write-Log -Level "INFO" -Message "日志文件: $LOG_FILE"
    Write-Host ""
    
    # 检查依赖
    if (-not (Test-Dependencies)) {
        exit 1
    }
    Write-Host ""
    
    # 选择服务器
    $server = Select-Server
    if ($null -eq $server) {
        exit 2
    }
    Write-Host ""
    
    # 测试 SSH 连接
    if (-not (Test-SSHConnection -Server $server)) {
        exit 3
    }
    Write-Host ""
    
    # 检测平台类型
    $platformType = Get-PlatformType -Server $server
    $Global:PlatformType = $platformType      # 新增：供 Test-ContainerInformation 使用
    Write-Host ""
    
    # 检测系统类型
    $systemInfo = Get-SystemType -Server $server
    Write-Host ""
    # ✅ ujava 系统细分
    if ($systemInfo.HasUjava) {
        $ujavaVariant = Get-UjavaSystemVariant -Server $server
        $systemInfo.UjavaSystemVariant = $ujavaVariant

        if ($ujavaVariant -eq "unified") {
            Write-Log -Level "INFO" -Message "  [系统细分] ujava -> 统一平台系统 (检测到 /var/www/java/unifiedPlatform)"
        } else {
            Write-Log -Level "INFO" -Message "  [系统细分] ujava -> 会议预定系统 (未检测到 /var/www/java/unifiedPlatform)"
        }
    } else {
        $systemInfo.UjavaSystemVariant = "N/A"
    }
    
    # 初始化结果变量
    $ujavaContainerResults = @()
    $ujavaHostResults = @()
    $upythonResults = @()
    $upythonVoiceResults = @()
    
    # 根据检测到的系统类型进行服务检测
    if ($platformType -eq "new") {
        # 新统一平台检测
        
        # ujava 检测
        if ($systemInfo.HasUjava) {
            # 先尝试在容器内检测
            $ujavaContainerResults = Test-UjavaServices -Server $server -ContainerName $systemInfo.UjavaContainer -PlatformType $platformType
            # 检测宿主机服务（extapi）
            $ujavaHostResults = Test-UjavaHostServices -Server $server

            # 如果检测到对外服务进程 ubains-meeting-api-1.0-SNAPSHOT.jar 未运行，则触发远程修复（新统一平台）
            $global:ExternalServiceRepairResult = $null   # 供报告使用
            if ($ujavaHostResults -and $ujavaHostResults.Count -gt 0) {
                # 根据 Pattern 字段匹配具体 jar 名，或者使用 Service -eq 'extapi'
                $extSvc = $ujavaHostResults | Where-Object { $_.Pattern -eq 'ubains-meeting-api-1.0-SNAPSHOT.jar' }
                if (-not $extSvc) {
                    # 兼容：按 Service 名称退回一档
                    $extSvc = $ujavaHostResults | Where-Object { $_.Service -eq 'extapi' }
                }

                if ($extSvc -and -not $extSvc.Running) {
                    Write-Log -Level "WARN" -Message "[EXT] 检测到对外服务进程未运行，准备执行远程修复 (fix_external_service_disconnect)"
                    $global:ExternalServiceRepairResult = Repair-ExternalMeetingService -Server $server
                }
            }
        }
        else {
            # 如果没有检测到ujava容器，尝试在宿主机检测（可能是传统部署方式）
            Write-Log -Level "WARN" -Message "未检测到 ujava 容器，尝试在宿主机检测服务..."
            $ujavaContainerResults = Test-UjavaServices -Server $server -ContainerName $null -PlatformType $platformType
            $ujavaHostResults = Test-UjavaHostServices -Server $server
        }
        
        # upython 检测
        if ($systemInfo.HasUpython) {
            $upythonResults = Test-ContainerPorts -Server $server -ContainerName $systemInfo.UpythonContainer -PortList $UpythonPorts -ServiceType "upython"
        }
        
        # upython_voice 检测
        if ($systemInfo.HasUpythonVoice) {
            $upythonVoiceResults = Test-ContainerPorts -Server $server -ContainerName $systemInfo.UpythonVoiceContainer -PortList $UpythonVoicePorts -ServiceType "upython_voice"
        }
    }
    else {
        # 传统平台检测：使用传统平台专用检测逻辑
        Write-Log -Level "INFO" -Message "传统平台：使用传统平台检测逻辑"
        
        # ujava 容器内服务检测（传统平台只有nginx和meeting）
        if ($systemInfo.HasUjava) {
            $ujavaContainerResults = Test-UjavaOldPlatformContainerServices -Server $server -ContainerName $systemInfo.UjavaContainer
        }
        else {
            Write-Log -Level "WARN" -Message "未检测到 ujava 容器，跳过容器内服务检测"
        }
        
        # ujava 宿主机服务检测（传统平台extapi在/var/www/java/external-meeting-api）
        $ujavaHostResults = Test-UjavaOldPlatformHostServices -Server $server

        # 如果检测到对外服务进程 ubains-meeting-api-1.0-SNAPSHOT.jar 未运行，则触发远程修复
        $global:ExternalServiceRepairResult = $null   # 供报告使用
        if ($ujavaHostResults -and $ujavaHostResults.Count -gt 0) {
            # 根据 Pattern 字段匹配具体 jar 名，或者使用 Service -eq 'extapi'
            $extSvc = $ujavaHostResults | Where-Object { $_.Pattern -eq 'ubains-meeting-api-1.0-SNAPSHOT.jar' }
            if (-not $extSvc) {
                # 兼容：按 Service 名称退回一档
                $extSvc = $ujavaHostResults | Where-Object { $_.Service -eq 'extapi' }
            }

            if ($extSvc -and -not $extSvc.Running) {
                Write-Log -Level "WARN" -Message "[EXT] 检测到对外服务进程未运行，准备执行远程修复 (fix_external_service_disconnect)"
                $global:ExternalServiceRepairResult = Repair-ExternalMeetingService -Server $server
            }
        }
        
        # upython 检测（传统平台使用不同的端口列表）
        if ($systemInfo.HasUpython) {
            $upythonResults = Test-ContainerPorts -Server $server -ContainerName $systemInfo.UpythonContainer -PortList $UpythonOldPlatformPorts -ServiceType "upython (传统平台)"
        }
        
        # upython_voice 检测（传统平台通常没有，但如果有容器也检测）
        if ($systemInfo.HasUpythonVoice) {
            $upythonVoiceResults = Test-ContainerPorts -Server $server -ContainerName $systemInfo.UpythonVoiceContainer -PortList $UpythonVoicePorts -ServiceType "upython_voice"
        }
    }
    
    # DNS 解析检测（所有平台都需要检测）
    Write-Host ""
    $dnsResults = Test-DNSResolution -Server $server
    
    # 服务器资源分析（所有平台都需要检测）
    Write-Host ""
    $resourceResults = Test-ServerResources -Server $server

    # 容器信息收集（加入到自检报告）
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 容器信息（报告） =========="
    $containerInfo = Test-ContainerInformation -Server $server -PrintDetails
    if (-not $containerInfo -or $containerInfo.Count -eq 0) {
    }
    $md += ""

    # 检测配置文件中的IP地址
    Write-Host ""
    Write-Log -Level "INFO" -Message "========== 开始检测配置文件 IP =========="
    if ($platformType -eq "new") {
        Check-NewPlatformIPs -ServerIP $server.IP -Username $server.User -Password $server.Pass
    } elseif ($platformType -eq "old") {
        # ✅ 关键：把 systemInfo 传进去，才能按 meeting/unified 选路径
        Check-TraditionalPlatformIPs -ServerIP $server.IP -Username $server.User -Password $server.Pass -SystemInfo $systemInfo
    }
    Write-Log -Level "INFO" -Message "========== 结束检测配置文件 IP =========="

    # 检测 NTP 服务
    Write-Log -Level "INFO" -Message "========== 开始检测NTP服务 =========="
    $ntpResults = Check-NTPService -ServerIP $server.IP -Username $server.User -Password $server.Pass
    Write-Log -Level "INFO" -Message "NTP 服务检测完成."
    # 输出 NTP 摘要
    if ($ntpResults) {
        Write-Log -Level "INFO" -Message ("NTP 结果: 状态={0} | 详情={1}" -f $ntpResults.Status, $ntpResults.Detail)
    }
    Write-Log -Level "INFO" -Message "========== 结束检测NTP服务 =========="

    # 文件权限检测
    Write-Log -Level "INFO" -Message "========== 开始检测文件权限 =========="
    $filePermResults = Check-FilePermissions -Server $server -PlatformType $platformType -SystemInfo $systemInfo
    Write-Log -Level "INFO" -Message "========== 结束检测文件权限 =========="

    # 现场数据备份 (可选)
    $bakChoice = Read-Host "是否执行现场数据备份并下载? (y/n) [默认: n]"
    $bakupResults = $null
    if ($bakChoice -eq 'y' -or $bakChoice -eq 'Y') {
        $bakupResults = DataBakup -Server $server -PlatformType $platformType -SystemInfo $systemInfo
    } else {
        Write-Log -Level "INFO" -Message "跳过现场数据备份"
    }

    # 询问是否导出日志
    Write-Host ""
    Write-Host "==================================================================" -ForegroundColor Cyan
    $exportChoice = Read-Host "是否导出服务日志到本地? (y/n) [默认: n]"
    
    $logExportResults = $null
    if ($exportChoice -eq "y" -or $exportChoice -eq "Y") {
        $logExportResults = Export-ServiceLogs -Server $server -PlatformType $platformType -SystemInfo $systemInfo
    }
    else {
        Write-Log -Level "INFO" -Message "跳过日志导出"
    }
    
    # 安卓设备自检（按 PRD 15：手动输入设备IP，连接+拉取日志）
    $androidResults = Test-AndroidDeviceHealth -ScriptDir $SCRIPT_DIR

    # 生成检测报告
    Show-HealthReport -Server $server -PlatformType $platformType -SystemInfo $systemInfo `
        -UjavaContainerResults $ujavaContainerResults `
        -UjavaHostResults $ujavaHostResults `
        -UpythonResults $upythonResults `
        -UpythonVoiceResults $upythonVoiceResults `
        -DNSResults $dnsResults `
        -ResourceResults $resourceResults `
        -LogExportResults $logExportResults `
        -NTPResults $ntpResults `
        -FilePermResults $filePermResults `
        -ContainerInfo $containerInfo `
        -AndroidResults $androidResults
}

# 执行主函数
Main
