﻿<# 
功能：Windows 端远程更新入口脚本（实现 PRD 3.1、3.2）
依赖：PuTTY 的 plink.exe / pscp.exe
#>

Set-StrictMode -Version Latest
$ErrorActionPreference = "Stop"

# 让当前 PowerShell 会话使用 UTF-8（尽量避免中文乱码）
[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

# -------------------- Version --------------------
# 脚本版本号（用于日志/截图回溯；需要时手工递增）
$SCRIPT_VERSION = "1.0.0"
# -------------------- /Version --------------------

function Write-Info($msg) { Write-Host "[INFO] $msg" -ForegroundColor Cyan }
function Write-Warn($msg) { Write-Host "[WARN] $msg" -ForegroundColor Yellow }
function Write-Err ($msg) { Write-Host "[ERROR] $msg" -ForegroundColor Red }

function Read-NonEmpty([string]$prompt, [string]$default = "") {
    while ($true) {
        if ($default) {
            $v = Read-Host "$prompt (默认: $default)"
            if ([string]::IsNullOrWhiteSpace($v)) { return $default }
            return $v
        } else {
            $v = Read-Host $prompt
            if (-not [string]::IsNullOrWhiteSpace($v)) { return $v }
        }
    }
}

function Read-Choice([string]$prompt, [string[]]$choices) {
    Write-Host $prompt
    for ($i=0; $i -lt $choices.Count; $i++) {
        Write-Host "  [$($i+1)] $($choices[$i])"
    }
    while ($true) {
        $sel = Read-Host "请输入序号(1-$($choices.Count))"
        if ($sel -match '^\d+$') {
            $idx = [int]$sel - 1
            if ($idx -ge 0 -and $idx -lt $choices.Count) { return $choices[$idx] }
        }
        Write-Warn "输入无效，请重新输入。"
    }
}

function Find-Tool([string]$name) {
    $cmd = Get-Command $name -ErrorAction SilentlyContinue
    if ($cmd) { return $cmd.Source }

    $local = Join-Path $PSScriptRoot $name
    if (Test-Path $local) { return (Resolve-Path $local).Path }

    return $null
}

function Require-PuttyTools {
    $plink = Find-Tool "plink.exe"
    $pscp  = Find-Tool "pscp.exe"
    if (-not $plink -or -not $pscp) {
        Write-Err "未找到 plink.exe/pscp.exe。请安装 PuTTY 并加入 PATH，或将 plink.exe/pscp.exe 放在脚本同目录。"
        throw "Missing PuTTY tools"
    }
    return @{ plink=$plink; pscp=$pscp }
}

function Invoke-PlinkCommand {
    param(
        [Parameter(Mandatory=$true)][string]$PlinkPath,
        [Parameter(Mandatory=$true)][string]$HostName,
        [Parameter(Mandatory=$true)][int]$Port,
        [Parameter(Mandatory=$true)][string]$User,
        [Parameter(Mandatory=$true)][string]$Password,
        [Parameter(Mandatory=$true)][string]$Command
    )

    $args = @(
        "-batch",
        "-ssh",
        "-P", "$Port",
        "-l", $User,
        "-pw", $Password,
        $HostName,
        $Command
    )

    Write-Info "远端执行: $Command"

    $tmpOut = Join-Path $env:TEMP ("plink_{0}.out.log" -f ([guid]::NewGuid().ToString("N")))
    $tmpErr = Join-Path $env:TEMP ("plink_{0}.err.log" -f ([guid]::NewGuid().ToString("N")))

    $p = Start-Process -FilePath $PlinkPath -ArgumentList $args -NoNewWindow -Wait -PassThru `
        -RedirectStandardOutput $tmpOut -RedirectStandardError $tmpErr

    $outText = ""
    $errText = ""
    if (Test-Path $tmpOut) { $outText = (Get-Content $tmpOut -Raw -Encoding utf8) }
    if (Test-Path $tmpErr) { $errText = (Get-Content $tmpErr -Raw -Encoding utf8) }

    if (-not [string]::IsNullOrWhiteSpace($outText)) { Write-Host $outText }
    if (-not [string]::IsNullOrWhiteSpace($errText)) { Write-Warn $errText }

    Remove-Item -ErrorAction SilentlyContinue $tmpOut, $tmpErr

    if ($p.ExitCode -ne 0) {
        throw "plink 执行失败，ExitCode=$($p.ExitCode)。命令：$Command"
    }

    return $outText
}

function Invoke-PscpUpload {
    param(
        [Parameter(Mandatory=$true)][string]$PscpPath,
        [Parameter(Mandatory=$true)][string]$HostName,
        [Parameter(Mandatory=$true)][int]$Port,
        [Parameter(Mandatory=$true)][string]$User,
        [Parameter(Mandatory=$true)][string]$Password,
        [Parameter(Mandatory=$true)][string[]]$LocalFiles,
        [Parameter(Mandatory=$true)][string]$RemoteDir
    )

    foreach ($lf in $LocalFiles) {
        if (-not (Test-Path $lf)) {
            throw "本地文件不存在：$lf"
        }
    }

    if ($RemoteDir.EndsWith("/") -eq $false) { $RemoteDir = "$RemoteDir/" }

    $remoteTarget = ("{0}@{1}:{2}" -f $User, $HostName, $RemoteDir)

    foreach ($lf in $LocalFiles) {
        $args = @(
            "-batch",
            "-P", "$Port",
            "-pw", $Password,
            $lf,
            $remoteTarget
        )

        Write-Info "上传: $(Split-Path $lf -Leaf) -> $RemoteDir"
        $p = Start-Process -FilePath $PscpPath -ArgumentList $args -NoNewWindow -Wait -PassThru
        if ($p.ExitCode -ne 0) {
            throw "pscp 上传失败，ExitCode=$($p.ExitCode)。文件：$lf"
        }
    }
}

function Invoke-PscpDownload {
    param(
        [Parameter(Mandatory=$true)][string]$PscpPath,
        [Parameter(Mandatory=$true)][string]$HostName,
        [Parameter(Mandatory=$true)][int]$Port,
        [Parameter(Mandatory=$true)][string]$User,
        [Parameter(Mandatory=$true)][string]$Password,
        [Parameter(Mandatory=$true)][string]$RemoteFile,
        [Parameter(Mandatory=$true)][string]$LocalDir
    )

    if (-not (Test-Path $LocalDir)) {
        New-Item -ItemType Directory -Path $LocalDir | Out-Null
    }

    $remoteSpec = ("{0}@{1}:{2}" -f $User, $HostName, $RemoteFile)

    $args = @(
        "-batch",
        "-P", "$Port",
        "-pw", $Password,
        $remoteSpec,
        $LocalDir
    )

    Write-Info "下载: $RemoteFile -> $LocalDir"
    $p = Start-Process -FilePath $PscpPath -ArgumentList $args -NoNewWindow -Wait -PassThru
    if ($p.ExitCode -ne 0) {
        throw "pscp 下载失败，ExitCode=$($p.ExitCode)。远端文件：$RemoteFile"
    }
}

# -------------------- 3.1 采集参数 --------------------
Write-Host "=== 远程程序更新（Windows 端入口）===" -ForegroundColor Green

# 版本记录（控制台输出 + 便于截图）
Write-Info ("remote_program_update.ps1 version={0}" -f $SCRIPT_VERSION)
Write-Info ("PowerShell={0}  Host={1}" -f $PSVersionTable.PSVersion, $env:COMPUTERNAME)
Write-Info ("ScriptPath={0}" -f $PSCommandPath)

$serverIp   = Read-NonEmpty "请输入服务器IP"
$sshPortStr = Read-NonEmpty "请输入SSH端口" "22"
$sshPort    = [int]$sshPortStr
$username   = Read-NonEmpty "请输入用户名" "root"
$securePwd  = Read-Host "请输入密码" -AsSecureString
$plainPwd   = [Runtime.InteropServices.Marshal]::PtrToStringAuto([Runtime.InteropServices.Marshal]::SecureStringToBSTR($securePwd))

$remoteDir  = Read-NonEmpty "请输入远端文件存放路径" "/home/Update/"

$platformType = Read-Choice "请选择平台类型：" @("新统一平台", "传统平台")
$systemType   = Read-Choice "请选择更新系统类型：" @("会议预定系统", "运维集控系统", "讯飞转录系统")
$updateType   = Read-Choice "请选择更新类型：" @("前端更新", "后端更新", "全量更新")

Write-Info "参数确认：IP=$serverIp Port=$sshPort User=$username RemoteDir=$remoteDir 平台=$platformType 系统=$systemType 更新=$updateType"

# -------------------- 工具检测 --------------------
$tools = Require-PuttyTools
$plink = $tools.plink
$pscp  = $tools.pscp

# 记录工具版本（可选但对定位 plink 兼容性很有用）
try {
    $plinkVer = & $plink -V 2>&1 | Select-Object -First 1
    if ($plinkVer) { Write-Info ("plink: {0}" -f $plinkVer) }
} catch { }

try {
    $pscpVer = & $pscp -V 2>&1 | Select-Object -First 1
    if ($pscpVer) { Write-Info ("pscp: {0}" -f $pscpVer) }
} catch { }

# -------------------- 连接测试（3.1） --------------------
Invoke-PlinkCommand -PlinkPath $plink -HostName $serverIp -Port $sshPort -User $username -Password $plainPwd -Command "echo Connected; uname -a"
Write-Info "SSH 连接测试成功。"

# 1) 远端创建目录 + 检查 unzip
Invoke-PlinkCommand -PlinkPath $plink -HostName $serverIp -Port $sshPort -User $username -Password $plainPwd -Command "mkdir -p '$remoteDir'; command -v unzip >/dev/null 2>&1 || (echo '缺少 unzip，请安装：yum install -y unzip 或 apt-get install -y unzip' && exit 2)"

# ====== 3.2 本地准备：找到 zip 包 + program_update.sh ======
# 约定：zip 压缩包与本 remote_program_update.ps1 同级目录
$zipFiles = @(
    Get-ChildItem -Path $PSScriptRoot -Filter *.zip -File |
    Select-Object -ExpandProperty FullName
)

if ($zipFiles.Length -eq 0) {
    throw "未找到 zip 压缩包：请将更新包（*.zip）放到脚本同级目录：$PSScriptRoot"
}
if ($zipFiles.Length -gt 1) {
    Write-Warn "检测到多个 zip，默认使用第一个：$($zipFiles[0])"
}
$zipPath = $zipFiles[0]

$programUpdateSh = Join-Path $PSScriptRoot "program_update.sh"
if (-not (Test-Path $programUpdateSh)) {
    throw "未找到 program_update.sh：$programUpdateSh（请放在脚本同级目录）"
}

Write-Info "将上传更新包：$zipPath"
Write-Info "将上传更新脚本：$programUpdateSh"
# ====== 3.2 本地准备结束 ======

# 2) 上传 zip + program_update.sh
Invoke-PscpUpload -PscpPath $pscp -HostName $serverIp -Port $sshPort -User $username -Password $plainPwd -LocalFiles @($zipPath, $programUpdateSh) -RemoteDir $remoteDir

# 3) 远端解压（解压到 remoteDir）
$zipName = Split-Path $zipPath -Leaf
Invoke-PlinkCommand -PlinkPath $plink -HostName $serverIp -Port $sshPort -User $username -Password $plainPwd -Command "cd '$remoteDir' && unzip -o '$zipName' -d '$remoteDir'"

# ====== 默认执行 program_update.sh（实现 3.3 备份）并下载备份包到本机脚本目录 ======
# 额外打印远端 program_update.sh 版本，便于回溯
Invoke-PlinkCommand -PlinkPath $plink -HostName $serverIp -Port $sshPort -User $username -Password $plainPwd -Command "cd '$remoteDir' && ./program_update.sh --version 2>/dev/null || true"

$execCmd = "cd '$remoteDir' && chmod +x ./program_update.sh && ./program_update.sh --platform '$platformType' --system '$systemType' --update '$updateType' --workdir '$remoteDir'"
$out = Invoke-PlinkCommand -PlinkPath $plink -HostName $serverIp -Port $sshPort -User $username -Password $plainPwd -Command $execCmd

# 从输出解析 BACKUP_TAR
# program_update.sh 输出格式：BACKUP_TAR=/home/Backup/BakYYYYmmdd_HHMMSS.tar.gz
$backupTar = $null
if ($out -match '(?m)^BACKUP_TAR=(.+)\s*$') {
    $backupTar = $Matches[1].Trim()
}

if (-not $backupTar) {
    throw "未从远端输出中解析到 BACKUP_TAR。请确认 program_update.sh 已实现 3.3 且会 echo BACKUP_TAR=..."
}

Write-Info "远端备份包：$backupTar"
Invoke-PscpDownload -PscpPath $pscp -HostName $serverIp -Port $sshPort -User $username -Password $plainPwd -RemoteFile $backupTar -LocalDir $PSScriptRoot
Write-Info "备份包已下载到：$PSScriptRoot"