# -*- coding: utf-8 -*-
"""
根据模板用例 Excel，在同一个工作簿中新建 Sheet，并从 JSON 配置文件中读取用例数据写入。
后续只需要维护 JSON 文件即可复用。
"""

import os
import json
from copy import copy
from openpyxl import load_workbook
from openpyxl.styles import Alignment

# ===== 1. 配置路径（改为相对当前脚本的路径） =====
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
TEMPLATE_PATH = os.path.join(BASE_DIR, "用例文件", "兰州中石化项目测试用例20251203.xlsx")
NEW_SHEET_NAME = "兰州登录MQTT用例"  # 新建的 Sheet 名
CASES_FILE = os.path.join(BASE_DIR, "config", "兰州用例.json")  # 用例配置文件（JSON）

# 与你表头对应的顺序（根据截图）
headers_order = [
    "序号", "功能模块", "功能类别", "用例编号", "功能描述", "用例等级",
    "功能编号", "用例名称", "预置条件", "操作步骤", "JSON", "预期结果",
    "测试结果", "测试结论", "日志截屏", "备注",
]


def load_cases():
    """从 JSON 配置文件加载用例列表。"""
    if not os.path.exists(CASES_FILE):
        raise FileNotFoundError(f"找不到用例配置文件: {CASES_FILE}")
    with open(CASES_FILE, "r", encoding="utf-8") as f:
        data = json.load(f)
    if not isinstance(data, list):
        raise ValueError("用例配置文件的根节点必须是列表(List)")
    return data


def main():
    if not os.path.exists(TEMPLATE_PATH):
        print("找不到模板文件：", TEMPLATE_PATH)
        return

    try:
        cases = load_cases()
    except Exception as e:
        print("加载用例配置失败：", e)
        return

    wb = load_workbook(TEMPLATE_PATH)

    # 以第一个sheet作为模板
    template_sheet = wb.worksheets[0]

    # 如果新sheet已存在就先删除
    if NEW_SHEET_NAME in wb.sheetnames:
        ws_new = wb[NEW_SHEET_NAME]
        wb.remove(ws_new)

    ws_new = wb.create_sheet(NEW_SHEET_NAME)

    # ===== 复制表头（仅值 + 简单样式） =====
    # 假设模板表头在第3行（根据截图），如有偏差可以改成2或其它
    header_row_index = 3
    for col_idx, cell in enumerate(template_sheet[header_row_index], start=1):
        new_cell = ws_new.cell(row=1, column=col_idx, value=cell.value)
        if cell.has_style:
            new_cell.font = copy(cell.font)
            new_cell.fill = copy(cell.fill)
            new_cell.border = copy(cell.border)
            new_cell.alignment = copy(cell.alignment)
            new_cell.number_format = cell.number_format

    ws_new.freeze_panes = "B2"

    # ===== 写入用例数据（预置条件/操作步骤自动换行，JSON列统一留空） =====
    row = 2
    for case in cases:
        for col_idx, header in enumerate(headers_order, start=1):
            val = case.get(header, "")
            if header in ("预置条件", "操作步骤") and isinstance(val, str):
                # 把分号+空格变成换行，Excel 中会显示为多行
                val = val.replace("; ", "\n")
            if header == "JSON":
                # JSON 列统一留空
                val = ""
            ws_new.cell(row=row, column=col_idx, value=val)
        row += 1

    # ===== 对“预置条件”和“操作步骤”开启自动换行 =====
    col_idx_pre = headers_order.index("预置条件") + 1
    col_idx_steps = headers_order.index("操作步骤") + 1
    for r in range(1, row):
        cell_pre = ws_new.cell(row=r, column=col_idx_pre)
        cell_steps = ws_new.cell(row=r, column=col_idx_steps)
        cell_pre.alignment = Alignment(wrap_text=True, vertical="top")
        cell_steps.alignment = Alignment(wrap_text=True, vertical="top")

    # 调整列宽
    for col in ws_new.columns:
        max_len = 0
        col_letter = col[0].column_letter
        for c in col:
            v = c.value
            if v is None:
                continue
            l = len(str(v))
            if l > max_len:
                max_len = l
        ws_new.column_dimensions[col_letter].width = min(max_len + 2, 60)

    wb.save(TEMPLATE_PATH)
    print("已在原文件中创建新Sheet：", NEW_SHEET_NAME)


if __name__ == "__main__":
    main()