#!/usr/bin/env bash
# 远程升级指定服务器的容器版本脚本
# 实现内容：
# 1. 校验目标服务器架构是否为 x86
# 2. 将当前目录下的镜像文件与部署脚本同步至远端目录
# 3. 停止远端旧容器
# 4. 交互询问目标服务器是否属于新统一平台
# 5. 自动递增容器编号并执行远端部署脚本

set -euo pipefail

# -------------------- Version --------------------
# 脚本版本号（用于日志/截图回溯；需要时手工递增）
SCRIPT_VERSION="1.0.0"
# -------------------- /Version --------------------

REMOTE_ARCH_ALLOW_REGEX='^(x86_64|amd64|i386|i686)$'

declare -A SERVER_IP SERVER_USER SERVER_PASS
SERVER_IP["1"]="192.168.5.48"
SERVER_USER["1"]="root"
SERVER_PASS["1"]="Ubains@123"

SERVER_IP["2"]="192.168.5.67"
SERVER_USER["2"]="root"
SERVER_PASS["2"]="Ubains@123"

SERVER_IP["3"]="192.168.5.47"
SERVER_USER["3"]="root"
SERVER_PASS["3"]="Ubains@1234"

declare -A SERVER_DESC
SERVER_DESC["1"]="标准版预定运维服务器"
SERVER_DESC["2"]="阿曼项目预定服务器"
SERVER_DESC["3"]="标准版预定运维测试发布服务器"

log() {
  local level="$1"; shift
  printf '[%(%F %T)T] [%s] %s\n' -1 "${level}" "$*"
}

# 支持 --version / -v
if [[ "${1:-}" == "--version" || "${1:-}" == "-v" ]]; then
  echo "remote_update.sh version: ${SCRIPT_VERSION}"
  exit 0
fi

# 启动即记录版本与调用参数（便于回溯）
log INFO "remote_update.sh version=${SCRIPT_VERSION}"
log INFO "Args: $*"

DEFAULT_REMOTE_DIR="/home/containerUpdate"
LOCAL_EMQX_DIR="/data/middleware/emqx"
LOCAL_PYTHON_DIR="/data/services/api/python-cmdb"
LOCAL_NGINX_DIR="/data/middleware/nginx"

usage() {
  cat <<'EOF'
用法:
  remote_update.sh [远端目录]

参数说明:
  远端目录     在目标服务器上存放镜像与部署脚本的目录，默认 /home/containerUpdate
  部署脚本     固定为 container_update.sh (位于当前目录)

脚本会弹出列表供选择内置服务器 (含 IP / 用户 / 密码) 以及容器
镜像文件将根据容器自动映射，如 ujava→java1.8.0_472.tar.gz
EOF
  exit 1
}

if [[ $# -gt 1 ]]; then
  usage
fi

if [[ $# -eq 1 ]]; then
  REMOTE_DIR="$1"
else
  REMOTE_DIR="${DEFAULT_REMOTE_DIR}"
  log INFO "未指定远端目录，默认使用 ${REMOTE_DIR}"
fi
DEPLOY_SCRIPT="container_update.sh"

if [[ ! -f "${DEPLOY_SCRIPT}" ]]; then
  log ERROR "部署脚本 ${DEPLOY_SCRIPT} 不存在"
  exit 3
fi

DEPLOY_NAME="$(basename "${DEPLOY_SCRIPT}")"

CONTAINER_OPTIONS=(ujava uemqx uredis upython unacos unginx)
declare -A CONTAINER_IMAGE
CONTAINER_IMAGE["ujava"]="java1.8.0_472.tar.gz"
CONTAINER_IMAGE["uemqx"]="uemqx5.8.4.tar.gz"
CONTAINER_IMAGE["uredis"]="redis8.2.2.tar.gz"
CONTAINER_IMAGE["upython"]="python_v15.tar.gz"
CONTAINER_IMAGE["unacos"]="nacos-server-v2.5.2.tar.gz"
CONTAINER_IMAGE["unginx"]="/data/temp/nginx-1.29.3.tar.gz"

# 容器对应的 Docker 镜像名称（用于版本校验）
declare -A CONTAINER_DOCKER_IMAGE
CONTAINER_DOCKER_IMAGE["ujava"]="139.9.60.86:5000/ujava:v6"
CONTAINER_DOCKER_IMAGE["uemqx"]="139.9.60.86:5000/uemqx:v2"
CONTAINER_DOCKER_IMAGE["uredis"]="139.9.60.86:5000/redis:v3"
CONTAINER_DOCKER_IMAGE["upython_new"]="139.9.60.86:5000/upython:v15"
CONTAINER_DOCKER_IMAGE["upython_old"]="139.9.60.86:5000/upython:v14"
CONTAINER_DOCKER_IMAGE["unacos"]="nacos-server:v2.5.2"
CONTAINER_DOCKER_IMAGE["unginx"]="nginx:1.29.3"

log INFO "可选择的容器："
for idx in "${!CONTAINER_OPTIONS[@]}"; do
  option_index=$((idx + 1))
  printf "  [%s] %s\n" "${option_index}" "${CONTAINER_OPTIONS[idx]}"
done

read -rp "请输入容器编号: " CONTAINER_KEY
if ! [[ "${CONTAINER_KEY}" =~ ^[1-6]$ ]]; then
  log ERROR "容器编号无效，请输入 1-6"
  exit 9
fi
CONTAINER_PREFIX="${CONTAINER_OPTIONS[$((CONTAINER_KEY - 1))]}"
log INFO "已选择容器 ${CONTAINER_PREFIX}"

IMAGE_FILE="${CONTAINER_IMAGE[${CONTAINER_PREFIX}]}"
if [[ -z "${IMAGE_FILE}" ]]; then
  log ERROR "容器 ${CONTAINER_PREFIX} 未配置镜像映射"
  exit 10
fi

# nginx 使用绝对路径，其他容器使用相对路径
if [[ "${IMAGE_FILE}" = /* ]]; then
  # 绝对路径
  if [[ ! -f "${IMAGE_FILE}" ]]; then
    log ERROR "镜像文件 ${IMAGE_FILE} 不存在"
    exit 2
  fi
  IMAGE_NAME="$(basename "${IMAGE_FILE}")"
else
  # 相对路径（当前目录）
  if [[ ! -f "${IMAGE_FILE}" ]]; then
    log ERROR "镜像文件 ${IMAGE_FILE} 不存在，请放置于当前目录后重试"
    exit 2
  fi
  IMAGE_NAME="$(basename "${IMAGE_FILE}")"
fi

SERVER_KEYS=(1 2 3)

log INFO "可选择的目标服务器："
for key in "${SERVER_KEYS[@]}"; do
  printf "  [%s] %s (%s %s)\n" "${key}" "${SERVER_DESC[${key}]}" "${SERVER_IP[${key}]}" "${SERVER_USER[${key}]}"
done
printf "  [0] 手动输入服务器信息\n"

read -rp "请输入服务器编号: " SERVER_KEY

# 手动输入服务器信息
if [[ "${SERVER_KEY}" == "0" ]]; then
  log INFO "进入手动输入模式"
  
  read -rp "请输入目标服务器 IP 地址: " REMOTE_HOST
  if [[ -z "${REMOTE_HOST}" ]]; then
    log ERROR "服务器 IP 地址不能为空"
    exit 7
  fi
  
  read -rp "请输入 SSH 端口号 [默认 22]: " SSH_PORT_INPUT
  SSH_PORT="${SSH_PORT_INPUT:-22}"
  
  read -rp "请输入登录用户名 [默认 root]: " REMOTE_USER_INPUT
  REMOTE_USER="${REMOTE_USER_INPUT:-root}"
  
  read -rsp "请输入登录密码: " REMOTE_PASS
  echo ""  # 换行
  if [[ -z "${REMOTE_PASS}" ]]; then
    log ERROR "登录密码不能为空"
    exit 7
  fi
  
  REMOTE_DESC="手动输入服务器"
  log INFO "已配置目标服务器: ${REMOTE_USER}@${REMOTE_HOST}:${SSH_PORT}"
  
elif [[ -z "${SERVER_IP[${SERVER_KEY}]:-}" ]]; then
  log ERROR "编号 ${SERVER_KEY} 不存在，请重新运行脚本"
  exit 7
else
  REMOTE_HOST="${SERVER_IP[${SERVER_KEY}]}"
  REMOTE_USER="${SERVER_USER[${SERVER_KEY}]}"
  REMOTE_PASS="${SERVER_PASS[${SERVER_KEY}]}"
  REMOTE_DESC="${SERVER_DESC[${SERVER_KEY}]}"
  SSH_PORT="22"
  log INFO "已选择 ${REMOTE_DESC} (${REMOTE_HOST})"
fi

if ! command -v sshpass >/dev/null 2>&1; then
  log ERROR "未安装 sshpass，无法使用密码方式登陆"
  exit 8
fi

SSH_TIMEOUT=30

ssh_exec() {
  sshpass -p "${REMOTE_PASS}" ssh -p "${SSH_PORT}" -o StrictHostKeyChecking=no -o ConnectTimeout=${SSH_TIMEOUT} "$@"
}

# 先测试 SSH 连接
log INFO "测试 SSH 连接..."
TEST_OUTPUT="$(sshpass -p "${REMOTE_PASS}" ssh -p "${SSH_PORT}" -o StrictHostKeyChecking=no -o ConnectTimeout=10 "${REMOTE_USER}@${REMOTE_HOST}" "echo CONNECTION_OK" 2>&1)" || TEST_STATUS=$?
TEST_STATUS=${TEST_STATUS:-0}
# 检查输出中是否包含 CONNECTION_OK（忽略 MOTD 等欢迎信息）
if [[ ${TEST_STATUS} -ne 0 ]] || ! echo "${TEST_OUTPUT}" | grep -q "CONNECTION_OK"; then
  log ERROR "SSH 连接失败！"
  log ERROR "输出信息: ${TEST_OUTPUT}"
  log ERROR "请检查: 1) IP地址是否正确 2) 端口是否正确 3) 密码是否正确 4) 网络是否可达"
  exit 4
fi
log INFO "SSH 连接测试通过"

scp_exec() {
  sshpass -p "${REMOTE_PASS}" scp -P "${SSH_PORT}" -o StrictHostKeyChecking=no -o ConnectTimeout=${SSH_TIMEOUT} "$@"
}

sync_emqx_assets() {
  local remote_dir="$1"
  local platform_label="$2"

  if [[ ! -d "${LOCAL_EMQX_DIR}" ]]; then
    log ERROR "本地 EMQX 目录不存在: ${LOCAL_EMQX_DIR}"
    exit 11
  fi

  log INFO "准备同步 EMQX 目录 (${LOCAL_EMQX_DIR} -> ${REMOTE_HOST}:${remote_dir})"
  ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "mkdir -p '${remote_dir}'"

  local backup_cmd
  backup_cmd=$(cat <<'EOF'
set -e
TARGET_DIR="$1"
if [[ -d "${TARGET_DIR}" && -n "$(ls -A "${TARGET_DIR}")" ]]; then
  ts=$(date +%Y%m%d_%H%M%S)
  backup="${TARGET_DIR}_backup_${ts}"
  cp -r "${TARGET_DIR}" "${backup}"
  echo "${backup}"
fi
EOF
)
  local backup_path
  backup_path="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "bash -s '${remote_dir}'" <<<"${backup_cmd}" || true)"
  if [[ -n "${backup_path}" ]]; then
    log INFO "远端目录已备份到 ${backup_path}"
  else
    log INFO "远端目录为空或不存在，跳过备份"
  fi

  if command -v rsync >/dev/null 2>&1; then
    log INFO "使用 rsync 同步 EMQX 目录"
    if ! sshpass -p "${REMOTE_PASS}" rsync -az --delete -e "ssh -p ${SSH_PORT} -o StrictHostKeyChecking=no -o ConnectTimeout=${SSH_TIMEOUT}" "${LOCAL_EMQX_DIR}/" "${REMOTE_USER}@${REMOTE_HOST}:${remote_dir}/"; then
      log ERROR "rsync 同步失败"
      exit 12
    fi
  else
    log WARN "未检测到 rsync，改用 scp 拷贝 EMQX 目录"
    if ! scp_exec -r "${LOCAL_EMQX_DIR}/." "${REMOTE_USER}@${REMOTE_HOST}:${remote_dir}/"; then
      log ERROR "scp 拷贝失败"
      exit 13
    fi
  fi

  log INFO "EMQX 目录同步完成，开始设置远端配置文件权限"
  ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "if [[ -d '${remote_dir}/config' ]]; then cd '${remote_dir}/config' && ls *.conf >/dev/null 2>&1 && chmod +x *.conf || true; else echo 'WARN: config 目录不存在'; fi"
  log INFO "远端 EMQX 配置权限处理完成 (${platform_label} 平台)"
}

sync_python_assets() {
  local remote_dir="$1"
  local platform_label="$2"

  if [[ ! -d "${LOCAL_PYTHON_DIR}" ]]; then
    log ERROR "本地 Python 目录不存在: ${LOCAL_PYTHON_DIR}"
    exit 14
  fi

  log INFO "准备同步 Python 目录 (${LOCAL_PYTHON_DIR} -> ${REMOTE_HOST}:${remote_dir})"
  ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "mkdir -p '${remote_dir}'"

  local backup_cmd
  backup_cmd=$(cat <<'EOF'
set -e
TARGET_DIR="$1"
if [[ -d "${TARGET_DIR}" && -n "$(ls -A "${TARGET_DIR}")" ]]; then
  ts=$(date +%Y%m%d_%H%M%S)
  backup="${TARGET_DIR}_backup_${ts}"
  cp -r "${TARGET_DIR}" "${backup}"
  echo "${backup}"
fi
EOF
)
  local backup_path
  backup_path="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "bash -s '${remote_dir}'" <<<"${backup_cmd}" || true)"
  if [[ -n "${backup_path}" ]]; then
    log INFO "远端 Python 目录已备份到 ${backup_path}"
  else
    log INFO "远端 Python 目录为空或不存在，跳过备份"
  fi

  if command -v rsync >/dev/null 2>&1; then
    log INFO "使用 rsync 同步 Python 目录"
    if ! sshpass -p "${REMOTE_PASS}" rsync -az --delete -e "ssh -p ${SSH_PORT} -o StrictHostKeyChecking=no -o ConnectTimeout=${SSH_TIMEOUT}" "${LOCAL_PYTHON_DIR}/" "${REMOTE_USER}@${REMOTE_HOST}:${remote_dir}/"; then
      log ERROR "rsync 同步 Python 目录失败"
      exit 15
    fi
  else
    log WARN "未检测到 rsync，改用 scp 拷贝 Python 目录"
    if ! scp_exec -r "${LOCAL_PYTHON_DIR}/." "${REMOTE_USER}@${REMOTE_HOST}:${remote_dir}/"; then
      log ERROR "scp 拷贝 Python 目录失败"
      exit 16
    fi
  fi

  log INFO "开始更新 Python 配置中的 IP 地址 (${platform_label} 平台)"
  ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "find '${remote_dir}' -type f -name '*.conf' -exec sed -i 's/192\\.168\\.5\\.44/${REMOTE_HOST}/g' {} +" || true
  log INFO "Python 目录同步完成"

  if [[ -n "${backup_path}" ]]; then
    log INFO "尝试从备份目录保留认证文件 authFile.uas"
    ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "
      set -e
      if [[ -f '${backup_path}/authFile.uas' ]]; then
        cp '${backup_path}/authFile.uas' '${remote_dir}/authFile.uas'
        echo 'INFO: 已将 authFile.uas 回拷至新目录'
      else
        echo 'WARN: 备份目录内未找到 authFile.uas，跳过回拷'
      fi
    " || log WARN "authFile.uas 回拷时出现问题，需人工检查"
  else
    log INFO "无历史备份目录，跳过 authFile.uas 回拷"
  fi
}

# --------------------------------------------------------------------------------------------------
# 校验远端服务器镜像版本，判断是否已更新
# 参数:
#   $1 - 容器前缀 (如 ujava, uredis)
#   $2 - 平台类型 (new/old)，仅 upython 需要区分
# 返回:
#   0 - 远端未更新或用户选择继续
#   1 - 远端已更新且用户选择跳过
# --------------------------------------------------------------------------------------------------
check_remote_image_version() {
  local container_prefix="$1"
  local platform_type="$2"
  local expected_image

  # 根据容器类型和平台类型获取期望的镜像名称
  if [[ "${container_prefix}" == "upython" ]]; then
    if [[ "${platform_type}" == "new" ]]; then
      expected_image="${CONTAINER_DOCKER_IMAGE["upython_new"]}"
    else
      expected_image="${CONTAINER_DOCKER_IMAGE["upython_old"]}"
    fi
  else
    expected_image="${CONTAINER_DOCKER_IMAGE["${container_prefix}"]}"
  fi

  if [[ -z "${expected_image}" ]]; then
    log WARN "未配置容器 ${container_prefix} 的镜像版本信息，跳过版本校验"
    return 0
  fi

  log INFO "检查远端服务器镜像版本..."
  local remote_images
  remote_images="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "docker images --format '{{.Repository}}:{{.Tag}}'" 2>/dev/null || true)"

  if echo "${remote_images}" | grep -qw "${expected_image}"; then
    log WARN "⚠️ 检测到远端服务器已存在目标镜像: ${expected_image}"
    log WARN "⚠️ 目标服务器可能已完成更新"
    
    # 额外检查是否有相关容器在运行
    local running_container
    running_container="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "docker ps --format '{{.Names}}' | grep -E '^${container_prefix}([0-9]+)?$' | head -n1" 2>/dev/null || true)"
    
    if [[ -n "${running_container}" ]]; then
      # 获取运行容器使用的镜像
      local container_image
      container_image="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "docker inspect --format '{{.Config.Image}}' '${running_container}'" 2>/dev/null || true)"
      if [[ "${container_image}" == "${expected_image}" ]]; then
        log WARN "⚠️ 检测到容器 ${running_container} 正在使用目标镜像 ${expected_image}"
        log WARN "⚠️ 目标服务器该容器已是最新版本，无需更新"
        read -rp "是否仍要继续更新? (y/n): " CONTINUE_INPUT
        case "${CONTINUE_INPUT}" in
          y|Y) 
            log INFO "用户选择继续更新"
            return 0 
            ;;
          n|N) 
            log INFO "用户选择跳过更新"
            return 1 
            ;;
          *) 
            log ERROR "输入无效，终止操作"
            exit 6 
            ;;
        esac
      fi
    fi
    
    read -rp "远端已有目标镜像，是否仍要继续更新? (y/n): " CONTINUE_INPUT
    case "${CONTINUE_INPUT}" in
      y|Y) 
        log INFO "用户选择继续更新"
        return 0 
        ;;
      n|N) 
        log INFO "用户选择跳过更新"
        return 1 
        ;;
      *) 
        log ERROR "输入无效，终止操作"
        exit 6 
        ;;
    esac
  else
    log INFO "✅ 远端服务器尚未安装目标镜像 ${expected_image}，继续更新流程"
    return 0
  fi
}

sync_nginx_assets() {
  local remote_dir="$1"
  local platform_label="$2"

  if [[ ! -d "${LOCAL_NGINX_DIR}" ]]; then
    log ERROR "本地 Nginx 目录不存在: ${LOCAL_NGINX_DIR}"
    exit 17
  fi

  log INFO "准备同步 Nginx 目录 (${LOCAL_NGINX_DIR} -> ${REMOTE_HOST}:${remote_dir})"
  ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "mkdir -p '${remote_dir}'"

  local backup_cmd
  backup_cmd=$(cat <<'EOF'
set -e
TARGET_DIR="$1"
if [[ -d "${TARGET_DIR}" && -n "$(ls -A "${TARGET_DIR}")" ]]; then
  ts=$(date +%Y%m%d_%H%M%S)
  backup="${TARGET_DIR}_backup_${ts}"
  cp -r "${TARGET_DIR}" "${backup}"
  echo "${backup}"
fi
EOF
)
  local backup_path
  backup_path="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "bash -s '${remote_dir}'" <<<"${backup_cmd}" || true)"
  if [[ -n "${backup_path}" ]]; then
    log INFO "远端 Nginx 目录已备份到 ${backup_path}"
  else
    log INFO "远端 Nginx 目录为空或不存在，跳过备份"
  fi

  if command -v rsync >/dev/null 2>&1; then
    log INFO "使用 rsync 同步 Nginx 目录"
    if ! sshpass -p "${REMOTE_PASS}" rsync -az --delete -e "ssh -p ${SSH_PORT} -o StrictHostKeyChecking=no -o ConnectTimeout=${SSH_TIMEOUT}" "${LOCAL_NGINX_DIR}/" "${REMOTE_USER}@${REMOTE_HOST}:${remote_dir}/"; then
      log ERROR "rsync 同步 Nginx 目录失败"
      exit 18
    fi
  else
    log WARN "未检测到 rsync，改用 scp 拷贝 Nginx 目录"
    if ! scp_exec -r "${LOCAL_NGINX_DIR}/." "${REMOTE_USER}@${REMOTE_HOST}:${remote_dir}/"; then
      log ERROR "scp 拷贝 Nginx 目录失败"
      exit 19
    fi
  fi

  log INFO "Nginx 目录同步完成 (${platform_label} 平台)"
}

log INFO "开始校验远端架构 (${REMOTE_HOST})"
ARCH_OUTPUT="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "uname -m" 2>&1)" || SSH_STATUS=$?
SSH_STATUS=${SSH_STATUS:-0}
if [[ ${SSH_STATUS} -ne 0 ]]; then
  log ERROR "无法获取远端架构，请检查网络或权限"
  log ERROR "SSH 输出: ${ARCH_OUTPUT}"
  log ERROR "请确认: 1) IP地址是否正确 2) 端口是否正确 3) 密码是否正确 4) 网络是否可达"
  exit 4
fi
ARCH="$(printf '%s' "${ARCH_OUTPUT}" | tr -d '\r' | awk 'NF {last=$0} END {print last}')"
if [[ ! "${ARCH}" =~ ${REMOTE_ARCH_ALLOW_REGEX} ]]; then
  log ERROR "远端架构为 ${ARCH}，非 x86，终止操作"
  exit 5
fi
log INFO "远端架构 ${ARCH} 校验通过"

# 自动检测平台类型（通过检查 /data/services 目录是否存在）
log INFO "自动检测目标服务器平台类型..."
PLATFORM_CHECK_OUTPUT="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "[ -d /data/services ] && echo 'NEW_PLATFORM' || echo 'OLD_PLATFORM'" 2>/dev/null || echo "CHECK_FAILED")"
PLATFORM_CHECK_RESULT="$(echo "${PLATFORM_CHECK_OUTPUT}" | tr -d '\r' | awk 'NF {last=$0} END {print last}')"

if [[ "${PLATFORM_CHECK_RESULT}" == "NEW_PLATFORM" ]]; then
  PLATFORM_FLAG="--new-platform"
  PLATFORM_TYPE="new"
  log INFO "✅ 检测到 /data/services 目录存在，自动识别为新统一平台"
elif [[ "${PLATFORM_CHECK_RESULT}" == "OLD_PLATFORM" ]]; then
  PLATFORM_FLAG=""
  PLATFORM_TYPE="old"
  log INFO "✅ 未检测到 /data/services 目录，自动识别为传统平台"
else
  log WARN "⚠️ 自动检测平台类型失败，切换为手动确认模式"
  read -rp "目标服务器是否为新统一平台? (y/n): " PLATFORM_INPUT
  case "${PLATFORM_INPUT}" in
    y|Y) PLATFORM_FLAG="--new-platform" ; PLATFORM_TYPE="new" ; log INFO "标记为新统一平台" ;;
    n|N) PLATFORM_FLAG="" ; PLATFORM_TYPE="old" ; log INFO "标记为传统平台" ;;
    *) log ERROR "输入无效，请输入 y 或 n"; exit 6 ;;
  esac
fi

# 校验远端镜像版本，判断是否已更新
if ! check_remote_image_version "${CONTAINER_PREFIX}" "${PLATFORM_TYPE}"; then
  log INFO "用户取消更新，流程结束"
  exit 0
fi

log INFO "创建远端目录 ${REMOTE_DIR}"
ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "mkdir -p '${REMOTE_DIR}'"

log INFO "传输镜像与部署脚本到远端目录"
# nginx 使用绝对路径，需要特殊处理
if [[ "${IMAGE_FILE}" = /* ]]; then
  # 绝对路径：先复制到临时位置，再传输
  TEMP_IMAGE="/tmp/$(basename "${IMAGE_FILE}")"
  cp "${IMAGE_FILE}" "${TEMP_IMAGE}"
  scp_exec "${TEMP_IMAGE}" "${DEPLOY_SCRIPT}" "${REMOTE_USER}@${REMOTE_HOST}:${REMOTE_DIR}/"
  rm -f "${TEMP_IMAGE}"
  # 在远端将镜像文件移动到正确位置
  ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "mkdir -p /data/temp && mv '${REMOTE_DIR}/$(basename "${IMAGE_FILE}")' /data/temp/$(basename "${IMAGE_FILE}")"
else
  # 相对路径：直接传输
  scp_exec "${IMAGE_FILE}" "${DEPLOY_SCRIPT}" "${REMOTE_USER}@${REMOTE_HOST}:${REMOTE_DIR}/"
fi

log INFO "查找并停止远端旧容器"
CURRENT_CONTAINER=$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "docker ps --format '{{.Names}}' | grep -E '^${CONTAINER_PREFIX}([0-9]+)?$' | sort -V | tail -n1" || true)
if [[ -n "${CURRENT_CONTAINER}" ]]; then
  log INFO "检测到旧容器 ${CURRENT_CONTAINER}，执行停止"
  ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "docker stop '${CURRENT_CONTAINER}'"
else
  log INFO "未找到匹配 ${CONTAINER_PREFIX} 或 ${CONTAINER_PREFIX}[数字] 的运行容器"
fi

if [[ "${CONTAINER_PREFIX}" == "uemqx" ]]; then
  if [[ "${PLATFORM_TYPE}" == "new" ]]; then
    REMOTE_EMQX_DIR="/data/middleware/emqx"
  else
    REMOTE_EMQX_DIR="/var/www/emqx"
  fi
  sync_emqx_assets "${REMOTE_EMQX_DIR}" "${PLATFORM_TYPE}"
fi
# Python 容器暂不同步文件，功能保留待后续启用
# if [[ "${CONTAINER_PREFIX}" == "upython" ]]; then
#   if [[ "${PLATFORM_TYPE}" == "new" ]]; then
#     REMOTE_PYTHON_DIR="/data/services/api/python-cmdb"
#   else
#     REMOTE_PYTHON_DIR="/var/www/html"
#   fi
#   sync_python_assets "${REMOTE_PYTHON_DIR}" "${PLATFORM_TYPE}"
# fi
if [[ "${CONTAINER_PREFIX}" == "upython" ]]; then
  log INFO "Python 容器暂不执行文件同步，仅部署容器"
fi
if [[ "${CONTAINER_PREFIX}" == "unginx" ]]; then
  if [[ "${PLATFORM_TYPE}" == "new" ]]; then
    REMOTE_NGINX_DIR="/data/middleware/nginx"
    sync_nginx_assets "${REMOTE_NGINX_DIR}" "${PLATFORM_TYPE}"
  else
    log ERROR "Nginx 容器仅支持新统一平台，请重新选择"
    exit 20
  fi
fi
if [[ -n "${CURRENT_CONTAINER}" ]]; then
  LAST_NUM="${CURRENT_CONTAINER##*[!0-9]}"
  if [[ -z "${LAST_NUM}" ]]; then
    LAST_NUM=0
  fi
  NEXT_NUM=$((LAST_NUM + 1))
else
  NEXT_NUM=1
fi
NEW_CONTAINER="${CONTAINER_PREFIX}${NEXT_NUM}"
log INFO "新容器名称确定为 ${NEW_CONTAINER}"

if [[ -n "${PLATFORM_FLAG}" ]]; then
  PLATFORM_TAIL=" '${PLATFORM_FLAG}'"
else
  PLATFORM_TAIL=""
fi

# 根据镜像文件路径类型确定远端镜像路径
if [[ "${IMAGE_FILE}" = /* ]]; then
  # 绝对路径：使用远端绝对路径
  REMOTE_IMAGE_PATH="/data/temp/${IMAGE_NAME}"
else
  # 相对路径：使用远端目录下的文件
  REMOTE_IMAGE_PATH="${REMOTE_DIR}/${IMAGE_NAME}"
fi

REMOTE_CMD=$(
  cat <<EOF
set -euo pipefail
cd '${REMOTE_DIR}'
sed -i 's/\r$//' '${DEPLOY_NAME}' 2>/dev/null || true
chmod +x '${DEPLOY_NAME}'
./'${DEPLOY_NAME}' '${NEW_CONTAINER}' '${REMOTE_IMAGE_PATH}'${PLATFORM_TAIL}
EOF
)

log INFO "开始执行远端部署脚本"
sshpass -p "${REMOTE_PASS}" ssh -tt -p "${SSH_PORT}" -o StrictHostKeyChecking=no -o ConnectTimeout=${SSH_TIMEOUT} "${REMOTE_USER}@${REMOTE_HOST}" "${REMOTE_CMD}"
log INFO "远端部署执行完成"

# 清理远端镜像包和部署脚本
log INFO "开始清理远端镜像包..."
CLEANUP_CMD=$(cat <<EOF
set -e
# 清理镜像包
if [[ -f "${REMOTE_IMAGE_PATH}" ]]; then
  rm -f "${REMOTE_IMAGE_PATH}"
  echo "已删除镜像包: ${REMOTE_IMAGE_PATH}"
fi
# 清理部署脚本
if [[ -f "${REMOTE_DIR}/${DEPLOY_NAME}" ]]; then
  rm -f "${REMOTE_DIR}/${DEPLOY_NAME}"
  echo "已删除部署脚本: ${REMOTE_DIR}/${DEPLOY_NAME}"
fi
# 如果远端目录为空，则删除目录
if [[ -d "${REMOTE_DIR}" ]] && [[ -z "\$(ls -A '${REMOTE_DIR}')" ]]; then
  rmdir "${REMOTE_DIR}"
  echo "已删除空目录: ${REMOTE_DIR}"
fi
EOF
)
CLEANUP_OUTPUT="$(ssh_exec "${REMOTE_USER}@${REMOTE_HOST}" "${CLEANUP_CMD}" 2>&1)" || CLEANUP_STATUS=$?
CLEANUP_STATUS=${CLEANUP_STATUS:-0}
if [[ ${CLEANUP_STATUS} -eq 0 ]]; then
  log INFO "✅ 远端镜像包清理完成"
  # 输出清理详情
  echo "${CLEANUP_OUTPUT}" | while read -r line; do
    [[ -n "${line}" ]] && log INFO "   ${line}"
  done
else
  log WARN "⚠️ 远端镜像包清理过程中出现警告，但不影响部署结果"
fi

log INFO "远程升级流程结束"

# --------------------------------------------------------------------------------------------------
# Java 容器部署函数（摘自 new_auto.sh -> java_x86），供后续本地或远端脚本复用。
# 新增逻辑：如检测到旧容器运行，则执行停止与移除操作后再部署，避免名称冲突。
# --------------------------------------------------------------------------------------------------
java_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Java 服务 (Docker 版, x86)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="ujava2"
  local image_tar="/data/temp/java1.8.0_472.tar.gz"
  local image_name="139.9.60.86:5000/ujava:v6"
  local image_id="a8ba5fa12b8e"

  local host_api="/data/services/api"
  local host_web="/data/services/web"
  local host_nginx_log="/data/middleware/nginx/nginx_log"
  local host_fdfs_data="/var/fdfs/storage/data"

  local port_args=(
    "-p 8085:8085"
    "-p 8993:8993" "-p 8994:8994" "-p 8995:8995"
    "-p 8999:8999"
    "-p 8719:8719" "-p 8720:8720"
    "-p 9204:9204" "-p 9200:9200" "-p 9201:9201"
    "-p 9905:9905" "-p 9911:9911" "-p 9908:9908"
    "-p 9906:9906" "-p 9907:9907" "-p 9909:9909" "-p 9910:9910"
    "-p 30880:30880" "-p 30881:30881" "-p 30882:30882"
    "-p 30883:30883" "-p 30884:30884"
  )

  log "INFO" "🔍 检查 Java 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 正在运行，准备停止"
    $sudoset docker stop "${container_name}"
    log "INFO" "✅ 旧容器 ${container_name} 已停止"
  elif $sudoset docker ps -a --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 存在但未运行，跳过重启以免覆盖"
    log "INFO" "🛈 如需重新部署，请手动处理旧容器"
    return 1
  fi

  log "INFO" "🔍 检查 Java 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在。"
  else
    log "INFO" "❌ 镜像 ${image_name} 不存在，开始加载离线包..."
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if $sudoset docker load -i "${image_tar}"; then
      log "INFO" "🎉 镜像加载成功"
    else
      log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
      return 1
    fi
    $sudoset docker tag "${image_id}" "${image_name}" 2>/dev/null || true
    log "INFO" "🏷️ 镜像已标记为 ${image_name}"
  fi

  log "INFO" "🚀 正在启动 Java 容器: ${container_name} ..."
  $sudoset docker run -itd \
    --privileged \
    -v "${host_api}:/var/www/java/api" \
    -v "${host_web}:/var/www/java/web" \
    -v "${host_nginx_log}:/usr/local/nginx/logs" \
    -v /etc/localtime:/etc/localtime:ro \
    -v "${host_fdfs_data}:/var/fdfs/storage/data" \
    "${port_args[@]}" \
    --restart=always \
    --mac-address="02:42:ac:11:00:02" \
    --name "${container_name}" \
    "${image_name}" \
    /var/www/java/api/start.sh

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  log "INFO" "✅ Java 容器启动成功，等待初始化..."
  sleep 8

  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 Java 服务部署完成！"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}