#!/usr/bin/env bash

#======================================================================
# 自动化部署打包上传脚本
# 脚本名称: pakage_upload.sh
# 功能描述: 根据用户选择的系统类型，执行相应的打包操作
# 
# 功能状态:
#   ✅ 功能点1: 询问用户选择需要打包的系统类型 - 已完成待验证
#   ✅ 功能点2: 会议预定系统 - 前后端服务包打包 - 已完成待验证
#   ✅ 功能点2: 会议预定系统 - 容器镜像包打包 - 已完成待验证
#   ✅ 功能点2: 运维集控系统 - 前后端服务包打包 - 已完成待验证
#   ⏳ 功能点2: 运维集控系统 - 容器镜像包打包 - 待开发
#   ⏳ 功能点2: 语音转录系统 - 待开发
#   ⏳ 功能点2: 电子桌牌系统 - 待开发
#   ⏳ 功能点2: 无纸化信令服务 - 待开发
#   ✅ 功能点3: 打包成tar.gz并上传网盘 - 已完成待验证
#======================================================================

# -------------------- Version --------------------
# 脚本版本号（用于日志/截图回溯；需要时手工递增）
SCRIPT_VERSION="1.0.0"
# -------------------- /Version --------------------

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# 自动化部署包基础路径
OFFLINE_PACKAGE_PATH="/data/offline_auto_unifiedPlatform"

# NAS 网盘配置
NAS_SERVER="192.168.9.9"
NAS_SHARE="研发管理"
NAS_USER="陈泽键"
NAS_PASS_ENCRYPTED="Mm01ZUwuTUA="
NAS_MOUNT_POINT="/mnt/nas_upload"
NAS_UPLOAD_DIR="调试打包路径"

# 临时打包文件存放目录
TEMP_PACK_DIR="/tmp"

# 日志函数
log_info() {
    echo -e "${GREEN}[INFO]${NC} $(date '+%Y-%m-%d %H:%M:%S') - $1"
}

# 支持 --version / -v
if [[ "${1:-}" == "--version" || "${1:-}" == "-v" ]]; then
    echo "pakage_upload.sh version: ${SCRIPT_VERSION}"
    exit 0
fi

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $(date '+%Y-%m-%d %H:%M:%S') - $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $(date '+%Y-%m-%d %H:%M:%S') - $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $(date '+%Y-%m-%d %H:%M:%S') - $1"
}

# 打印分隔线
print_separator() {
    echo -e "${CYAN}========================================${NC}"
}

# 打印标题
print_title() {
    echo ""
    print_separator
    echo -e "${BLUE}  $1${NC}"
    print_separator
    echo ""
}

#======================================================================
# 功能点1: 询问用户选择需要打包的系统类型 [已完成待验证]
#======================================================================
select_system_type() {
    print_title "自动化部署打包上传工具"
    
    echo -e "${CYAN}请选择需要打包的系统类型：${NC}"
    echo ""
    
    # 系统类型列表
    local options=(
        "会议预定系统"
        "运维集控系统"
        "语音转录系统"
        "电子桌牌系统"
        "无纸化信令服务"
        "退出"
    )
    
    # 使用 select 命令让用户选择
    PS3=$'\n请输入选项编号 (1-6): '
    
    select opt in "${options[@]}"; do
        case $opt in
            "会议预定系统")
                log_info "您选择了: 会议预定系统"
                SELECTED_SYSTEM="meeting"
                SELECTED_SYSTEM_NAME="会议预定系统"
                break
                ;;
            "运维集控系统")
                log_info "您选择了: 运维集控系统"
                SELECTED_SYSTEM="devops"
                SELECTED_SYSTEM_NAME="运维集控系统"
                break
                ;;
            "语音转录系统")
                log_info "您选择了: 语音转录系统"
                SELECTED_SYSTEM="voice_transcription"
                SELECTED_SYSTEM_NAME="语音转录系统"
                break
                ;;
            "电子桌牌系统")
                log_info "您选择了: 电子桌牌系统"
                SELECTED_SYSTEM="nameplate"
                SELECTED_SYSTEM_NAME="电子桌牌系统"
                log_warn "该系统打包功能尚未开发"
                break
                ;;
            "无纸化信令服务")
                log_info "您选择了: 无纸化信令服务"
                SELECTED_SYSTEM="paperless"
                SELECTED_SYSTEM_NAME="无纸化信令服务"
                log_warn "该系统打包功能尚未开发"
                break
                ;;
            "退出")
                log_info "用户选择退出"
                exit 0
                ;;
            *)
                log_error "无效的选项，请输入 1-6 之间的数字"
                ;;
        esac
    done
}

#======================================================================
# 功能点2: 会议预定系统 - 前后端服务包打包 [已完成待验证]
#======================================================================

# 复制jar包的通用函数
copy_jar_files() {
    local src_dir="$1"
    local dest_dir="$2"
    local description="$3"
    
    log_info "正在处理: $description"
    
    # 检查源目录是否存在
    if [ ! -d "$src_dir" ]; then
        log_error "源目录不存在: $src_dir"
        return 1
    fi
    
    # 创建目标目录（如果不存在）
    if [ ! -d "$dest_dir" ]; then
        log_info "创建目标目录: $dest_dir"
        mkdir -p "$dest_dir"
        if [ $? -ne 0 ]; then
            log_error "创建目标目录失败: $dest_dir"
            return 1
        fi
    fi
    
    # 查找并复制jar文件
    local jar_count=$(find "$src_dir" -maxdepth 1 -name "*.jar" -type f | wc -l)
    
    if [ "$jar_count" -eq 0 ]; then
        log_warn "未找到jar文件: $src_dir"
        return 0
    fi
    
    # 先删除目标目录中的旧jar文件
    rm -f "$dest_dir"/*.jar 2>/dev/null
    
    # 复制新的jar文件
    cp "$src_dir"/*.jar "$dest_dir/"
    if [ $? -eq 0 ]; then
        log_success "已复制 $jar_count 个jar文件到: $dest_dir"
        return 0
    else
        log_error "复制jar文件失败: $src_dir -> $dest_dir"
        return 1
    fi
}

# 复制前端文件的通用函数
copy_frontend_files() {
    local src_dir="$1"
    local dest_dir="$2"
    local description="$3"
    local include_js="$4"      # 是否包含js文件
    local include_temp="$5"    # 是否包含temp文件夹
    
    log_info "正在处理: $description"
    
    # 检查源目录是否存在
    if [ ! -d "$src_dir" ]; then
        log_error "源目录不存在: $src_dir"
        return 1
    fi
    
    # 创建目标目录（如果不存在）
    if [ ! -d "$dest_dir" ]; then
        log_info "创建目标目录: $dest_dir"
        mkdir -p "$dest_dir"
        if [ $? -ne 0 ]; then
            log_error "创建目标目录失败: $dest_dir"
            return 1
        fi
    fi
    
    local copy_count=0
    
    # 复制 static 文件夹
    if [ -d "$src_dir/static" ]; then
        rm -rf "$dest_dir/static" 2>/dev/null
        cp -r "$src_dir/static" "$dest_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制 static 文件夹"
            ((copy_count++))
        else
            log_error "复制 static 文件夹失败"
        fi
    else
        log_warn "static 文件夹不存在: $src_dir/static"
    fi
    
    # 复制 index.html 文件
    if [ -f "$src_dir/index.html" ]; then
        cp "$src_dir/index.html" "$dest_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制 index.html 文件"
            ((copy_count++))
        else
            log_error "复制 index.html 文件失败"
        fi
    else
        log_warn "index.html 文件不存在: $src_dir/index.html"
    fi
    
    # 如果需要复制js文件
    if [ "$include_js" = "true" ]; then
        local js_count=$(find "$src_dir" -maxdepth 1 -name "*.js" -type f | wc -l)
        if [ "$js_count" -gt 0 ]; then
            cp "$src_dir"/*.js "$dest_dir/" 2>/dev/null
            if [ $? -eq 0 ]; then
                log_success "已复制 $js_count 个js文件"
                ((copy_count++))
            else
                log_error "复制js文件失败"
            fi
        else
            log_warn "未找到js文件: $src_dir"
        fi
    fi
    
    # 如果需要复制temp文件夹
    if [ "$include_temp" = "true" ]; then
        if [ -d "$src_dir/temp" ]; then
            rm -rf "$dest_dir/temp" 2>/dev/null
            cp -r "$src_dir/temp" "$dest_dir/"
            if [ $? -eq 0 ]; then
                log_success "已复制 temp 文件夹"
                ((copy_count++))
            else
                log_error "复制 temp 文件夹失败"
            fi
        else
            log_warn "temp 文件夹不存在: $src_dir/temp"
        fi
    fi
    
    if [ $copy_count -gt 0 ]; then
        log_success "前端文件复制完成: $description"
        return 0
    else
        log_warn "没有文件被复制: $description"
        return 1
    fi
}

# 会议预定系统 - 前后端服务包打包
package_meeting_system_services() {
    print_title "会议预定系统 - 前后端服务包打包"
    
    local success_count=0
    local fail_count=0
    local total_tasks=16
    
    #----------------------------------------------------------------------
    # 后端服务 - JAR包更新 (共9个)
    #----------------------------------------------------------------------
    print_separator
    echo -e "${BLUE}  后端服务 JAR包更新${NC}"
    print_separator
    
    # 1. auth-sso-auth
    if copy_jar_files \
        "/data/services/api/auth/auth-sso-auth" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/auth/auth-sso-auth" \
        "auth-sso-auth"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 2. auth-sso-gatway
    if copy_jar_files \
        "/data/services/api/auth/auth-sso-gatway" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/auth/auth-sso-gatway" \
        "auth-sso-gatway"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 3. auth-sso-system
    if copy_jar_files \
        "/data/services/api/auth/auth-sso-system" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/auth/auth-sso-system" \
        "auth-sso-system"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 4. java-meeting2.0
    if copy_jar_files \
        "/data/services/api/java-meeting/java-meeting2.0" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/java-meeting/java-meeting2.0" \
        "java-meeting2.0"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 5. java-meeting3.0
    if copy_jar_files \
        "/data/services/api/java-meeting/java-meeting3.0" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/java-meeting/java-meeting3.0" \
        "java-meeting3.0"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 6. java-meeting-extapi
    if copy_jar_files \
        "/data/services/api/java-meeting/java-meeting-extapi" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/java-meeting/java-meeting-extapi" \
        "java-meeting-extapi"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 7. java-message-scheduling
    if copy_jar_files \
        "/data/services/api/java-meeting/java-message-scheduling" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/java-meeting/java-message-scheduling" \
        "java-message-scheduling"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 8. java-mqtt
    if copy_jar_files \
        "/data/services/api/java-meeting/java-mqtt" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/java-meeting/java-mqtt" \
        "java-mqtt"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 9. java-quartz
    if copy_jar_files \
        "/data/services/api/java-meeting/java-quartz" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/java-meeting/java-quartz" \
        "java-quartz"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 前端服务 - Web文件更新 (共7个)
    #----------------------------------------------------------------------
    echo ""
    print_separator
    echo -e "${BLUE}  前端服务 Web文件更新${NC}"
    print_separator
    
    # 10. pc-vue2-ai (static + index.html)
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-ai" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-ai" \
        "pc-vue2-ai" \
        "false" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 11. pc-vue2-backstage (static + index.html)
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-backstage" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-backstage" \
        "pc-vue2-backstage" \
        "false" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 12. pc-vue2-main (static + index.html + js)
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-main" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-main" \
        "pc-vue2-main" \
        "true" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 13. pc-vue2-meetingControl (static + index.html)
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-meetingControl" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-meetingControl" \
        "pc-vue2-meetingControl" \
        "false" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 14. pc-vue2-meetngV2 (static + index.html + js)
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-meetngV2" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-meetngV2" \
        "pc-vue2-meetngV2" \
        "true" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 15. pc-vue2-meetngV3 (static + index.html + js)
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-meetngV3" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-meetngV3" \
        "pc-vue2-meetngV3" \
        "true" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    # 16. pc-vue2-platform (static + index.html + temp)
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-platform" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-platform" \
        "pc-vue2-platform" \
        "false" \
        "true"; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 打包结果统计
    #----------------------------------------------------------------------
    echo ""
    print_separator
    echo -e "${BLUE}  前后端服务包打包结果统计${NC}"
    print_separator
    echo ""
    echo -e "  总任务数: ${CYAN}$total_tasks${NC}"
    echo -e "  成功: ${GREEN}$success_count${NC}"
    echo -e "  失败: ${RED}$fail_count${NC}"
    echo ""
    
    if [ $fail_count -eq 0 ]; then
        log_success "会议预定系统 - 前后端服务包打包完成！"
        return 0
    else
        log_warn "会议预定系统 - 前后端服务包打包完成，但有 $fail_count 个任务失败"
        return 1
    fi
}

#======================================================================
# 功能点2: 会议预定系统 - 容器镜像包打包 [已完成待验证]
#======================================================================

# 复制指定文件的通用函数
copy_specific_files() {
    local src_dir="$1"
    local dest_dir="$2"
    shift 2
    local files=("$@")
    local description="${files[0]}"
    
    log_info "正在处理容器镜像文件..."
    
    # 检查源目录是否存在
    if [ ! -d "$src_dir" ]; then
        log_error "源目录不存在: $src_dir"
        return 1
    fi
    
    # 创建目标目录（如果不存在）
    if [ ! -d "$dest_dir" ]; then
        log_info "创建目标目录: $dest_dir"
        mkdir -p "$dest_dir"
        if [ $? -ne 0 ]; then
            log_error "创建目标目录失败: $dest_dir"
            return 1
        fi
    fi
    
    local success_count=0
    local fail_count=0
    
    for file in "${files[@]}"; do
        local src_path="$src_dir/$file"
        local dest_path="$dest_dir/$file"
        
        if [ -f "$src_path" ]; then
            # 复制文件
            cp "$src_path" "$dest_path"
            if [ $? -eq 0 ]; then
                log_success "已复制文件: $file"
                ((success_count++))
            else
                log_error "复制文件失败: $file"
                ((fail_count++))
            fi
        elif [ -d "$src_path" ]; then
            # 复制目录
            rm -rf "$dest_path" 2>/dev/null
            cp -r "$src_path" "$dest_path"
            if [ $? -eq 0 ]; then
                log_success "已复制目录: $file"
                ((success_count++))
            else
                log_error "复制目录失败: $file"
                ((fail_count++))
            fi
        else
            log_warn "文件/目录不存在: $src_path"
            ((fail_count++))
        fi
    done
    
    if [ $fail_count -eq 0 ]; then
        return 0
    else
        return 1
    fi
}

# 会议预定系统 - 容器镜像包打包
package_meeting_system_containers() {
    print_title "会议预定系统 - 容器镜像包打包"
    
    local src_temp_dir="/data/temp"
    local dest_temp_dir="${OFFLINE_PACKAGE_PATH}/data/temp"
    
    local success_count=0
    local fail_count=0
    local total_tasks=10
    
    # 检查源目录是否存在
    if [ ! -d "$src_temp_dir" ]; then
        log_error "源目录不存在: $src_temp_dir"
        return 1
    fi
    
    # 创建目标目录
    if [ ! -d "$dest_temp_dir" ]; then
        log_info "创建目标目录: $dest_temp_dir"
        mkdir -p "$dest_temp_dir"
    fi
    
    print_separator
    echo -e "${BLUE}  容器镜像包文件更新${NC}"
    print_separator
    
    #----------------------------------------------------------------------
    # 1. Docker 相关文件
    #----------------------------------------------------------------------
    log_info "[1/10] 处理 Docker 相关文件..."
    local docker_files=("docker-20.10.7.tgz" "docker.service" "docker")
    local docker_success=0
    
    for item in "${docker_files[@]}"; do
        local src_path="$src_temp_dir/$item"
        local dest_path="$dest_temp_dir/$item"
        
        if [ -f "$src_path" ]; then
            cp "$src_path" "$dest_path"
            if [ $? -eq 0 ]; then
                log_success "已复制: $item"
                ((docker_success++))
            else
                log_error "复制失败: $item"
            fi
        elif [ -d "$src_path" ]; then
            rm -rf "$dest_path" 2>/dev/null
            cp -r "$src_path" "$dest_path"
            if [ $? -eq 0 ]; then
                log_success "已复制目录: $item"
                ((docker_success++))
            else
                log_error "复制目录失败: $item"
            fi
        else
            log_warn "文件/目录不存在: $src_path"
        fi
    done
    
    if [ $docker_success -eq ${#docker_files[@]} ]; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 2. MySQL 相关文件
    #----------------------------------------------------------------------
    log_info "[2/10] 处理 MySQL 相关文件..."
    local mysql_files=("umysql.tar.gz" "devops_voice.sql" "huazhao2.sql" "nacos_mysql.sql" "offline.sql")
    local mysql_success=0
    
    for item in "${mysql_files[@]}"; do
        local src_path="$src_temp_dir/$item"
        local dest_path="$dest_temp_dir/$item"
        
        if [ -f "$src_path" ]; then
            cp "$src_path" "$dest_path"
            if [ $? -eq 0 ]; then
                log_success "已复制: $item"
                ((mysql_success++))
            else
                log_error "复制失败: $item"
            fi
        else
            log_warn "文件不存在: $src_temp_dir/$item"
        fi
    done
    
    if [ $mysql_success -ge 1 ]; then
        ((success_count++))
    else
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 3. Redis
    #----------------------------------------------------------------------
    log_info "[3/10] 处理 Redis 镜像..."
    if [ -f "$src_temp_dir/redis8.2.2.tar.gz" ]; then
        cp "$src_temp_dir/redis8.2.2.tar.gz" "$dest_temp_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制: redis8.2.2.tar.gz"
            ((success_count++))
        else
            log_error "复制失败: redis8.2.2.tar.gz"
            ((fail_count++))
        fi
    else
        log_warn "文件不存在: $src_temp_dir/redis8.2.2.tar.gz"
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 4. EMQX
    #----------------------------------------------------------------------
    log_info "[4/10] 处理 EMQX 镜像..."
    if [ -f "$src_temp_dir/uemqx5.8.4.tar.gz" ]; then
        cp "$src_temp_dir/uemqx5.8.4.tar.gz" "$dest_temp_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制: uemqx5.8.4.tar.gz"
            ((success_count++))
        else
            log_error "复制失败: uemqx5.8.4.tar.gz"
            ((fail_count++))
        fi
    else
        log_warn "文件不存在: $src_temp_dir/uemqx5.8.4.tar.gz"
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 5. FastDFS
    #----------------------------------------------------------------------
    log_info "[5/10] 处理 FastDFS 镜像..."
    if [ -f "$src_temp_dir/ufastdfs.tar.gz" ]; then
        cp "$src_temp_dir/ufastdfs.tar.gz" "$dest_temp_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制: ufastdfs.tar.gz"
            ((success_count++))
        else
            log_error "复制失败: ufastdfs.tar.gz"
            ((fail_count++))
        fi
    else
        log_warn "文件不存在: $src_temp_dir/ufastdfs.tar.gz"
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 6. Nacos
    #----------------------------------------------------------------------
    log_info "[6/10] 处理 Nacos 镜像..."
    if [ -f "$src_temp_dir/nacos-server-v2.5.2.tar.gz" ]; then
        cp "$src_temp_dir/nacos-server-v2.5.2.tar.gz" "$dest_temp_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制: nacos-server-v2.5.2.tar.gz"
            ((success_count++))
        else
            log_error "复制失败: nacos-server-v2.5.2.tar.gz"
            ((fail_count++))
        fi
    else
        log_warn "文件不存在: $src_temp_dir/nacos-server-v2.5.2.tar.gz"
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 7. Nginx
    #----------------------------------------------------------------------
    log_info "[7/9] 处理 Nginx 镜像..."
    if [ -f "$src_temp_dir/nginx-1.29.3.tar.gz" ]; then
        cp "$src_temp_dir/nginx-1.29.3.tar.gz" "$dest_temp_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制: nginx-1.29.3.tar.gz"
            ((success_count++))
        else
            log_error "复制失败: nginx-1.29.3.tar.gz"
            ((fail_count++))
        fi
    else
        log_warn "文件不存在: $src_temp_dir/nginx-1.29.3.tar.gz"
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 8. Chrony 配置
    #----------------------------------------------------------------------
    log_info "[8/9] 处理 Chrony 配置文件..."
    if [ -f "$src_temp_dir/chrony.conf" ]; then
        cp "$src_temp_dir/chrony.conf" "$dest_temp_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制: chrony.conf"
            ((success_count++))
        else
            log_error "复制失败: chrony.conf"
            ((fail_count++))
        fi
    else
        log_warn "文件不存在: $src_temp_dir/chrony.conf"
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 9. JDK
    #----------------------------------------------------------------------
    log_info "[9/10] 处理 JDK 安装包..."
    if [ -f "$src_temp_dir/jdk-8u472-linux-x64.tar.gz" ]; then
        cp "$src_temp_dir/jdk-8u472-linux-x64.tar.gz" "$dest_temp_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制: jdk-8u472-linux-x64.tar.gz"
            ((success_count++))
        else
            log_error "复制失败: jdk-8u472-linux-x64.tar.gz"
            ((fail_count++))
        fi
    else
        log_warn "文件不存在: $src_temp_dir/jdk-8u472-linux-x64.tar.gz"
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 10. Java 镜像包
    #----------------------------------------------------------------------
    log_info "[10/10] 处理 Java 镜像包..."
    if [ -f "$src_temp_dir/java1.8.0_472.tar.gz" ]; then
        cp "$src_temp_dir/java1.8.0_472.tar.gz" "$dest_temp_dir/"
        if [ $? -eq 0 ]; then
            log_success "已复制: java1.8.0_472.tar.gz"
            ((success_count++))
        else
            log_error "复制失败: java1.8.0_472.tar.gz"
            ((fail_count++))
        fi
    else
        log_warn "文件不存在: $src_temp_dir/java1.8.0_472.tar.gz"
        ((fail_count++))
    fi
    
    #----------------------------------------------------------------------
    # 打包结果统计
    #----------------------------------------------------------------------
    echo ""
    print_separator
    echo -e "${BLUE}  容器镜像包打包结果统计${NC}"
    print_separator
    echo ""
    echo -e "  总任务数: ${CYAN}$total_tasks${NC}"
    echo -e "  成功: ${GREEN}$success_count${NC}"
    echo -e "  失败: ${RED}$fail_count${NC}"
    echo ""
    
    if [ $fail_count -eq 0 ]; then
        log_success "会议预定系统 - 容器镜像包打包完成！"
        return 0
    else
        log_warn "会议预定系统 - 容器镜像包打包完成，但有 $fail_count 个任务失败"
        return 1
    fi
}

#======================================================================
# 功能点2: 运维集控系统 - 前后端服务包打包 [已完成待验证]
#======================================================================

# 运维集控系统 - 前后端服务包打包
package_devops_system_services() {
    print_title "运维集控系统 - 前后端服务包打包"

    local success_count=0
    local fail_count=0
    local total_tasks=6

    #----------------------------------------------------------------------
    # 后端服务 - 文件夹更新 (共2个)
    #----------------------------------------------------------------------
    print_separator
    echo -e "${BLUE}  后端服务 文件夹更新${NC}"
    print_separator

    # 1. cmdb
    if copy_specific_files \
        "/data/services/api/python-cmdb" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/python-cmdb" \
        "cmdb"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    # 2. UbainsDevOps
    if copy_specific_files \
        "/data/services/api/python-cmdb" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/python-cmdb" \
        "UbainsDevOps"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #----------------------------------------------------------------------
    # 前端服务 - Web文件更新 (共4个)
    #----------------------------------------------------------------------
    echo ""
    print_separator
    echo -e "${BLUE}  前端服务 Web文件更新${NC}"
    print_separator

    # 3. pc-vue2-moniter
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-moniter" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-moniter" \
        "pc-vue2-moniter" \
        "true" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    # 4. pc-vue2-backstage
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-backstage" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-backstage" \
        "pc-vue2-backstage" \
        "false" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    # 5. pc-vue2-main
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-main" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-main" \
        "pc-vue2-main" \
        "true" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    # 6. pc-vue2-platform
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-platform" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-platform" \
        "pc-vue2-platform" \
        "false" \
        "true"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #----------------------------------------------------------------------
    # 打包结果统计
    #----------------------------------------------------------------------
    echo ""
    print_separator
    echo -e "${BLUE}  前后端服务包打包结果统计${NC}"
    print_separator
    echo ""
    echo -e "  总任务数: ${CYAN}$total_tasks${NC}"
    echo -e "  成功: ${GREEN}$success_count${NC}"
    echo -e "  失败: ${RED}$fail_count${NC}"
    echo ""

    if [ $fail_count -eq 0 ]; then
        log_success "运维集控系统 - 前后端服务包打包完成！"
        return 0
    else
        log_warn "运维集控系统 - 前后端服务包打包完成，但有 $fail_count 个任务失败"
        return 1
    fi
}

#======================================================================
# 功能点2: 语音转录系统 - 前后端服务包打包 [待补充]
#======================================================================

# 讯飞转录系统 - 前后端服务包打包
package_voice_transcription_system_services() {
    print_title "讯飞转录系统 - 前后端服务包打包"

    local success_count=0
    local fail_count=0
    local total_tasks=7

    #----------------------------------------------------------------------
    # 后端服务 - 文件夹更新 (共1个)
    #----------------------------------------------------------------------
    print_separator
    echo -e "${BLUE}  后端服务 文件夹更新${NC}"
    print_separator

    # 1. python-voice
    if copy_specific_files \
        "/data/services/api/python-voice" \
        "${OFFLINE_PACKAGE_PATH}/data/services/api/python-voice" \
        "python-voice"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #----------------------------------------------------------------------
    # 前端服务 - Web文件更新 (共6个)
    #----------------------------------------------------------------------
    echo ""
    print_separator
    echo -e "${BLUE}  前端服务 Web文件更新${NC}"
    print_separator

    # 2. pc-vue2-voice
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-voice/pc-vue2-voice" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-voice/pc-vue2-voice" \
        "pc-vue2-voice" \
        "true" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    # 3. pc-vue2-backstage
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-backstage" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-backstage" \
        "pc-vue2-backstage" \
        "false" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    # 4. pc-vue2-main
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-main" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-main" \
        "pc-vue2-main" \
        "true" \
        "false"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    # 5. pc-vue2-platform
    if copy_frontend_files \
        "/data/services/web/pc/pc-vue2-platform" \
        "${OFFLINE_PACKAGE_PATH}/data/services/web/pc/pc-vue2-platform" \
        "pc-vue2-platform" \
        "false" \
        "true"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    # 6. iFlyTrans_server_bag
    if copy_specific_files \
        "/data/third_party/iFlyTrans" \
        "${OFFLINE_PACKAGE_PATH}/data/third_party/iFlyTrans" \
        "iFlyTrans_server_bag"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #----------------------------------------------------------------------
    # 打包结果统计
    #----------------------------------------------------------------------
    echo ""
    print_separator
    echo -e "${BLUE}  前后端服务包打包结果统计${NC}"
    print_separator
    echo ""
    echo -e "  总任务数: ${CYAN}$total_tasks${NC}"
    echo -e "  成功: ${GREEN}$success_count${NC}"
    echo -e "  失败: ${RED}$fail_count${NC}"
    echo ""

    if [ $fail_count -eq 0 ]; then
        log_success "讯飞转录系统 - 前后端服务包打包完成！"
        return 0
    else
        log_warn "讯飞转录系统 - 前后端服务包打包完成，但有 $fail_count 个任务失败"
        return 1
    fi
}

#======================================================================
# 功能点2: 语音转录系统 - 容器镜像包打包 [待补充]
#======================================================================

# 讯飞转录系统 - 容器镜像包打包
package_voice_transcription_containers() {
    print_title "讯飞转录系统 - 容器镜像包打包"

    local src_temp_dir="/data/temp"
    local dest_temp_dir="${OFFLINE_PACKAGE_PATH}/data/temp"

    local success_count=0
    local fail_count=0
    local total_tasks=9

    # 检查源目录是否存在
    if [ ! -d "$src_temp_dir" ]; then
        log_error "源目录不存在: $src_temp_dir"
        return 1
    fi

    # 创建目标目录
    if [ ! -d "$dest_temp_dir" ]; then
        log_info "创建目标目录: $dest_temp_dir"
        mkdir -p "$dest_temp_dir"
    fi

    print_separator
    echo -e "${BLUE}  容器镜像包文件更新${NC}"
    print_separator

    #------------------------------------------------------------------
    # 1. Docker 相关文件
    #------------------------------------------------------------------
    log_info "[1/9] 处理 Docker 相关文件..."
    local docker_files=("docker-20.10.7.tgz" "docker.service" "docker")
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "${docker_files[@]}"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 2. MySQL 相关文件
    #------------------------------------------------------------------
    log_info "[2/9] 处理 MySQL 相关文件..."
    local mysql_files=("umysql.tar.gz" "devops_voice.sql" "huazhao2.sql" "nacos_mysql.sql" "offline.sql")
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "${mysql_files[@]}"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 3. Redis 文件
    #------------------------------------------------------------------
    log_info "[3/9] 处理 Redis 文件..."
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "redis8.2.2.tar.gz"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 4. EMQX 文件
    #------------------------------------------------------------------
    log_info "[4/9] 处理 EMQX 文件..."
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "uemqx5.8.4.tar.gz"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 5. FastDFS 文件
    #------------------------------------------------------------------
    log_info "[5/9] 处理 FastDFS 文件..."
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "ufastdfs.tar.gz"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 6. Nacos 文件
    #------------------------------------------------------------------
    log_info "[6/9] 处理 Nacos 文件..."
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "nacos-server-v2.5.2.tar.gz"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 7. Nginx 文件
    #------------------------------------------------------------------
    log_info "[7/9] 处理 Nginx 文件..."
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "nginx-1.29.3.tar.gz"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 8. UVoice 文件
    #------------------------------------------------------------------
    log_info "[8/9] 处理 UVoice 文件..."
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "uvoice3.tar.gz"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 9. iFlyTrans 文件
    #------------------------------------------------------------------
    log_info "[9/9] 处理 iFlyTrans 文件..."
    if copy_specific_files "$src_temp_dir" "$dest_temp_dir" "iFlyTrans_server_bag"; then
        ((success_count++))
    else
        ((fail_count++))
    fi

    #------------------------------------------------------------------
    # 打包结果统计
    #------------------------------------------------------------------
    echo ""
    print_separator
    echo -e "${BLUE}  容器镜像包打包结果统计${NC}"
    print_separator
    echo ""
    echo -e "  总任务数: ${CYAN}$total_tasks${NC}"
    echo -e "  成功: ${GREEN}$success_count${NC}"
    echo -e "  失败: ${RED}$fail_count${NC}"
    echo ""

    if [ $fail_count -eq 0 ]; then
        log_success "讯飞转录系统 - 容器镜像包打包完成！"
        return 0
    else
        log_warn "讯飞转录系统 - 容器镜像包打包完成，但有 $fail_count 个任务失败"
        return 1
    fi
}

#======================================================================
# 功能点3: 打包成tar.gz格式并增加md5校验后上传网盘 [已完成待验证]
#======================================================================

# 解密 NAS 密码
decrypt_password() {
    echo "${NAS_PASS_ENCRYPTED}" | base64 -d
}

# 检查依赖
check_dependencies() {
    local missing_deps=()
    
    # 检查 cifs 支持
    if ! command -v mount.cifs >/dev/null 2>&1; then
        if ! grep -q cifs /proc/filesystems 2>/dev/null; then
            missing_deps+=("cifs-utils")
        fi
    fi
    
    # 检查 pv 工具（可选，用于显示进度）
    if ! command -v pv >/dev/null 2>&1; then
        log_warn "未安装 pv 工具，将使用 cp 命令（无进度显示）"
    fi
    
    # 检查 md5sum 工具
    if ! command -v md5sum >/dev/null 2>&1; then
        missing_deps+=("coreutils")
    fi
    
    if [[ ${#missing_deps[@]} -gt 0 ]]; then
        log_error "缺少依赖: ${missing_deps[*]}"
        log_info "请执行: yum install -y ${missing_deps[*]} 或 apt install -y ${missing_deps[*]}"
        return 1
    fi
    
    return 0
}

# 挂载 NAS
mount_nas() {
    local nas_pass
    nas_pass="$(decrypt_password)"
    
    # 创建挂载点
    if [[ ! -d "${NAS_MOUNT_POINT}" ]]; then
        log_info "创建挂载点: ${NAS_MOUNT_POINT}"
        sudo mkdir -p "${NAS_MOUNT_POINT}"
    fi
    
    # 检查是否已挂载
    if mountpoint -q "${NAS_MOUNT_POINT}" 2>/dev/null; then
        log_info "NAS 已挂载，跳过挂载步骤"
        return 0
    fi
    
    log_info "正在挂载 NAS: //${NAS_SERVER}/${NAS_SHARE}"
    
    # 挂载 SMB/CIFS 共享
    if ! sudo mount -t cifs "//${NAS_SERVER}/${NAS_SHARE}" "${NAS_MOUNT_POINT}" \
        -o "username=${NAS_USER},password=${nas_pass},iocharset=utf8,vers=2.0"; then
        log_error "NAS 挂载失败"
        log_info "请检查: 1) 网络连接 2) 账号密码 3) 共享路径"
        return 1
    fi
    
    log_success "NAS 挂载成功"
    return 0
}

# 卸载 NAS
unmount_nas() {
    if mountpoint -q "${NAS_MOUNT_POINT}" 2>/dev/null; then
        log_info "正在卸载 NAS..."
        sudo umount "${NAS_MOUNT_POINT}" || true
    fi
}

# 打包目录为 tar.gz 格式
pack_directory() {
    local source_dir="$1"
    local pack_file="$2"
    
    log_info "开始打包目录: ${source_dir}"
    log_info "目标文件: ${pack_file}"
    
    # 获取目录大小用于进度估算
    local dir_size
    dir_size=$(du -sb "${source_dir}" 2>/dev/null | awk '{print $1}')
    
    if command -v pv >/dev/null 2>&1; then
        # 使用 pv 显示打包进度
        tar -cf - -C "$(dirname "${source_dir}")" "$(basename "${source_dir}")" | \
            pv -s "${dir_size}" -p -t -e -r | \
            gzip > "${pack_file}"
    else
        # 无 pv 时直接打包
        log_info "正在打包（无进度显示）..."
        tar -czvf "${pack_file}" -C "$(dirname "${source_dir}")" "$(basename "${source_dir}")"
    fi
    
    if [[ -f "${pack_file}" ]]; then
        local file_size
        file_size=$(du -h "${pack_file}" | awk '{print $1}')
        log_success "打包完成，文件大小: ${file_size}"
        return 0
    else
        log_error "打包失败"
        return 1
    fi
}

# 生成 MD5 校验文件
generate_md5() {
    local source_file="$1"
    local md5_file="${source_file}.md5"
    
    log_info "正在生成 MD5 校验文件..."
    
    # 生成 MD5 校验值
    local md5_value
    md5_value=$(md5sum "${source_file}" | awk '{print $1}')
    
    if [[ -z "${md5_value}" ]]; then
        log_error "MD5 校验值生成失败"
        return 1
    fi
    
    # 写入 MD5 文件
    local file_name
    file_name=$(basename "${source_file}")
    echo "${md5_value}  ${file_name}" > "${md5_file}"
    
    if [[ -f "${md5_file}" ]]; then
        log_success "MD5 校验文件已生成: ${md5_file}"
        log_info "MD5 值: ${md5_value}"
        return 0
    else
        log_error "MD5 校验文件写入失败"
        return 1
    fi
}

# 上传文件到 NAS
upload_to_nas() {
    local source_file="$1"
    local target_dir="${NAS_MOUNT_POINT}/${NAS_UPLOAD_DIR}"
    
    # 确保目标目录存在
    if [[ ! -d "${target_dir}" ]]; then
        log_info "创建目标目录: ${target_dir}"
        sudo mkdir -p "${target_dir}"
    fi
    
    local file_name
    file_name=$(basename "${source_file}")
    local target_file="${target_dir}/${file_name}"
    
    log_info "开始上传文件到 NAS..."
    log_info "源文件: ${source_file}"
    log_info "目标: //${NAS_SERVER}/${NAS_SHARE}/${NAS_UPLOAD_DIR}/${file_name}"
    
    if command -v pv >/dev/null 2>&1; then
        # 使用 pv 显示上传进度
        pv -p -t -e -r "${source_file}" | sudo tee "${target_file}" > /dev/null
    else
        # 无 pv 时使用 cp
        log_info "正在上传（无进度显示）..."
        sudo cp "${source_file}" "${target_file}"
    fi
    
    # 验证上传结果
    if [[ -f "${target_file}" ]]; then
        local uploaded_size
        uploaded_size=$(sudo du -h "${target_file}" | awk '{print $1}')
        log_success "上传完成！"
        log_info "   文件路径: \\\\${NAS_SERVER}\\${NAS_SHARE}\\${NAS_UPLOAD_DIR}\\${file_name}"
        log_info "   文件大小: ${uploaded_size}"
        return 0
    else
        log_error "上传失败，目标文件不存在"
        return 1
    fi
}

# 清理函数
cleanup_nas() {
    unmount_nas
}

# 打包并上传到网盘
package_and_upload() {
    print_title "打包并上传到公司网盘"

    # 动态设置网盘上传路径
    set_nas_upload_dir

    # 生成打包文件名
    local timestamp
    timestamp=$(date +%Y%m%d_%H%M%S)
    local pack_name="offline_auto_unifiedPlatform"
    local pack_file="${TEMP_PACK_DIR}/${pack_name}.tar.gz"
    local md5_file="${pack_file}.md5"
    
    log_info "=================================================================="
    log_info "打包上传信息"
    log_info "=================================================================="
    log_info "源目录: ${OFFLINE_PACKAGE_PATH}"
    log_info "打包文件: ${pack_file}"
    log_info "目标网盘: \\\\${NAS_SERVER}\\${NAS_SHARE}\\${NAS_UPLOAD_DIR}"
    log_info "=================================================================="
    
    # 设置清理钩子
    trap cleanup_nas EXIT
    
    # 步骤2: 打包目录
    log_info "[步骤 2/5] 打包目录..."
    if ! pack_directory "${OFFLINE_PACKAGE_PATH}" "${pack_file}"; then
        log_error "打包失败，终止上传"
        return 1
    fi
    
    # 步骤3: 生成 MD5 校验文件
    log_info "[步骤 3/5] 生成 MD5 校验文件..."
    if ! generate_md5 "${pack_file}"; then
        log_error "MD5 校验文件生成失败，终止上传"
        return 1
    fi
    
    # 步骤4: 挂载 NAS
    log_info "[步骤 4/5] 挂载 NAS..."
    if ! mount_nas; then
        log_error "NAS 挂载失败，终止上传"
        return 1
    fi
    
    # 步骤5: 上传文件
    log_info "[步骤 5/5] 上传文件..."
    
    # 上传 tar.gz 文件
    if ! upload_to_nas "${pack_file}"; then
        log_error "tar.gz 文件上传失败"
        return 1
    fi
    
    # 上传 MD5 校验文件
    if ! upload_to_nas "${md5_file}"; then
        log_error "MD5 校验文件上传失败"
        return 1
    fi
    
    # 清理临时文件
    log_info "清理临时文件..."
    rm -f "${pack_file}" "${md5_file}"
    
    echo ""
    print_separator
    echo -e "${GREEN}  🎉 打包上传全部完成！${NC}"
    print_separator
    echo ""
    echo -e "  打包文件: ${CYAN}${pack_name}.tar.gz${NC}"
    echo -e "  MD5校验: ${CYAN}${pack_name}.tar.gz.md5${NC}"
    echo -e "  网盘路径: ${CYAN}\\\\${NAS_SERVER}\\${NAS_SHARE}\\${NAS_UPLOAD_DIR}\\${NC}"
    echo ""
    
    return 0
}

# 动态设置网盘上传路径
set_nas_upload_dir() {
    case $SELECTED_SYSTEM in
        "meeting")
            NAS_UPLOAD_DIR="调试打包路径/会议预定系统"
            ;;
        "devops")
            NAS_UPLOAD_DIR="调试打包路径/运维集控系统"
            ;;
        "voice_transcription")
            NAS_UPLOAD_DIR="调试打包路径/语音转录系统"
            ;;
        "nameplate")
            NAS_UPLOAD_DIR="调试打包路径/电子桌牌系统"
            ;;
        "paperless")
            NAS_UPLOAD_DIR="调试打包路径/无纸化信令服务"
            ;;
        *)
            log_error "未知的系统类型: $SELECTED_SYSTEM"
            exit 1
            ;;
    esac

    log_info "网盘上传路径已设置为: $NAS_UPLOAD_DIR"
}

#======================================================================
# 根据选择执行打包操作
#======================================================================
execute_package() {
    case $SELECTED_SYSTEM in
        "meeting")
            # 会议预定系统
            echo ""
            echo -e "${CYAN}请选择打包类型：${NC}"
            echo "  1) 前后端服务包"
            echo "  2) 容器镜像包"
            echo "  3) 全部打包"
            echo "  4) 全部打包并上传网盘"
            echo ""
            read -p "请输入选项编号 (1-4): " package_type
            
            case $package_type in
                1)
                    package_meeting_system_services
                    ;;
                2)
                    package_meeting_system_containers
                    ;;
                3)
                    package_meeting_system_services
                    package_meeting_system_containers
                    ;;
                4)
                    package_meeting_system_services
                    package_meeting_system_containers
                    # 询问是否上传
                    echo ""
                    read -p "打包完成，是否上传到公司网盘? (y/n): " do_upload
                    if [ "$do_upload" = "y" ] || [ "$do_upload" = "Y" ]; then
                        # Ensure the NAS upload directory is set before packaging and uploading
                        set_nas_upload_dir
                        package_and_upload
                    else
                        log_info "用户取消上传"
                    fi
                    ;;
                *)
                    log_error "无效的选项"
                    exit 1
                    ;;
            esac
            ;;
        "devops")
            # 运维集控系统
            echo ""
            echo -e "${CYAN}请选择打包类型：${NC}"
            echo "  1) 前后端服务包"
            echo "  2) 容器镜像包"
            echo "  3) 全部打包"
            echo "  4) 全部打包并上传网盘"
            echo ""
            read -p "请输入选项编号 (1-4): " package_type
            
            case $package_type in
                1)
                    package_devops_system_services
                    ;;
                2)
                    # package_devops_docker_images  # Removed invalid reference
                    ;;
                3)
                    package_devops_system_services
                    # package_devops_docker_images  # Removed invalid reference
                    ;;
                4)
                    package_devops_system_services
                    # package_devops_docker_images  # Removed invalid reference
                    # 询问是否上传
                    echo ""
                    read -p "打包完成，是否上传到公司网盘? (y/n): " do_upload
                    if [ "$do_upload" = "y" ] || [ "$do_upload" = "Y" ]; then
                        package_and_upload
                    else
                        log_info "用户取消上传"
                    fi
                    ;;
                *)
                    log_error "无效的选项"
                    exit 1
                    ;;
            esac
            ;;
        "voice_transcription")
            # 语音转录系统
            echo ""
            echo -e "${CYAN}请选择打包类型：${NC}"
            echo "  1) 前后端服务包"
            echo "  2) 容器镜像包"
            echo "  3) 全部打包"
            echo "  4) 全部打包并上传网盘"
            echo ""
            read -p "请输入选项编号 (1-4): " package_type
            
            case $package_type in
                1)
                    package_voice_transcription_system_services
                    ;;
                2)
                    package_voice_transcription_containers
                    ;;
                3)
                    package_voice_transcription_system_services
                    package_voice_transcription_containers
                    ;;
                4)
                    package_voice_transcription_system_services
                    package_voice_transcription_containers
                    # 询问是否上传
                    echo ""
                    read -p "打包完成，是否上传到公司网盘? (y/n): " do_upload
                    if [ "$do_upload" = "y" ] || [ "$do_upload" = "Y" ]; then
                        package_and_upload
                    else
                        log_info "用户取消上传"
                    fi
                    ;;
                *)
                    log_error "无效的选项"
                    exit 1
                    ;;
            esac
            ;;
        "nameplate")
            log_warn "电子桌牌系统打包功能尚未开发"
            ;;
        "paperless")
            log_warn "无纸化信令服务打包功能尚未开发"
            ;;
        *)
            log_error "未知的系统类型"
            exit 1
            ;;
    esac
}

#======================================================================
# 主函数
#======================================================================
main() {
    # 启动即记录版本与调用参数（便于回溯）
    log_info "pakage_upload.sh version=${SCRIPT_VERSION}"
    log_info "Args: $*"

    # 检查是否以root权限运行
    if [ "$EUID" -ne 0 ]; then
        log_warn "建议使用root权限运行此脚本"
    fi
    
    # 检查自动化部署包目录是否存在
    if [ ! -d "$OFFLINE_PACKAGE_PATH" ]; then
        log_error "自动化部署包目录不存在: $OFFLINE_PACKAGE_PATH"
        read -p "是否创建该目录? (y/n): " create_dir
        if [ "$create_dir" = "y" ] || [ "$create_dir" = "Y" ]; then
            mkdir -p "$OFFLINE_PACKAGE_PATH"
            if [ $? -eq 0 ]; then
                log_success "目录创建成功: $OFFLINE_PACKAGE_PATH"
            else
                log_error "目录创建失败"
                exit 1
            fi
        else
            log_info "用户取消操作"
            exit 0
        fi
    fi
    
    # 功能点1: 询问用户选择系统类型
    select_system_type
    
    # 功能点2: 执行打包操作
    execute_package
    
    echo ""
    print_separator
    log_success "脚本执行完成！"
    print_separator
}

# 执行主函数
main "$@"

