#!/bin/bash

#===============================================================================
# 脚本名称：monitor_inner_api_services.sh
# 功能描述：预定系统后端服务监测与自愈脚本
# 版本：V1.0
# 创建日期：2026-01-27
# 基于文档：_PRD_预定系统后端服务监测需求文档.md
#
# 监测对象：
#   1. 平台API可用性 (https://SERVER_IP/api/system/getVerifyCode)
#   2. 容器ujava*内服务进程 (meeting2.0)，支持ujava2/ujava3/ujava6等模糊匹配
#   3. 宿主机6060端口malan服务
#
# 使用方法：
#   chmod +x monitor_inner_api_services.sh
#   ./monitor_inner_api_services.sh
#
# 定时任务示例：
#   */5 * * * * /path/to/monitor_inner_api_services.sh
#===============================================================================

#-------------------------------------------------------------------------------
# 配置变量段
#-------------------------------------------------------------------------------

# 服务器配置（由get_server_ip()函数自动获取）
SERVER_IP=""

# 容器配置（支持模糊匹配：ujava2/ujava3/ujava6等）
CONTAINER_NAME="ujava2"  # 脚本将自动查找匹配的容器

# API配置（将在main函数中根据SERVER_IP动态生成）
API_URL=""

# Malan服务配置
MALAN_DIR="/var/www/malan"
MALAN_PORT="6060"

# 日志配置
LOG_FILE="/var/log/monitor-inner-api-services.log"

# 超时配置
START_TIMEOUT=30
RESTART_WAIT=30

# 调试模式（设为true启用详细参数打印）
DEBUG_MODE="true"

# 服务配置（关联数组：服务key -> jar包标识）
declare -A SERVICES
SERVICES["meeting2.0"]="ubains-meeting-inner-api"

# 服务路径配置（关联数组：服务key -> 服务路径）
declare -A SERVICE_PATHS
SERVICE_PATHS["meeting2.0"]="/var/www/java/api-java-meeting2.0"

# 启动脚本名称
START_SCRIPT="run.sh"

#-------------------------------------------------------------------------------
# 工具函数段
#-------------------------------------------------------------------------------

# 获取格式化时间戳
# 返回格式：YYYY-MM-DD HH:MM:SS
timestamp() {
    date '+%Y-%m-%d %H:%M:%S'
}

# 统一日志输出函数
# 参数：$1 - 日志内容（必须包含标识符）
# 示例：log "[RUNNING] service_key (PID: 1234)"
log() {
    local message="$1"
    echo "$(timestamp) $message" | tee -a "$LOG_FILE"
}

# 动态查找匹配的容器
# 支持模糊匹配：ujava2、ujava3、ujava6等
# 返回：找到的容器名称，未找到返回空
find_container() {
    local container
    # 查找所有以ujava开头后跟数字的运行中容器
    container=$(docker ps --format "{{.Names}}" 2>/dev/null | grep -E "^ujava[0-9]+$" | head -n1)

    if [ -z "$container" ]; then
        echo ""
        return 1
    fi

    echo "$container"
}

# 自动获取本机IP地址
# 支持多种方法，优先使用hostname -I
# 返回：检测到的IP地址，失败返回空
get_server_ip() {
    local ip

    # 方法1：通过hostname -I获取（推荐，支持多网卡）
    ip=$(hostname -I 2>/dev/null | awk '{print $1}')
    if [ -n "$ip" ] && [ "$ip" != "127.0.0.1" ]; then
        echo "$ip"
        return 0
    fi

    # 方法2：通过ip命令获取
    if command -v ip &> /dev/null; then
        ip=$(ip route get 1 2>/dev/null | awk '{for(i=1;i<=NF;i++) if($i=="src") print $(i+1)}' | head -n1)
        if [ -n "$ip" ] && [ "$ip" != "127.0.0.1" ]; then
            echo "$ip"
            return 0
        fi
    fi

    # 方法3：通过ifconfig获取
    if command -v ifconfig &> /dev/null; then
        ip=$(ifconfig 2>/dev/null | grep "inet " | grep -v "127.0.0.1" | awk '{print $2}' | head -n1)
        if [ -n "$ip" ]; then
            echo "$ip"
            return 0
        fi
    fi

    # 方法4：通过解析/proc/net/route（适用于没有ifconfig的系统）
    if [ -f /proc/net/route ]; then
        ip=$(awk 'NR==2 {ip=strtonum("0x"$2); printf "%d.%d.%d.%d\n", and(ip>>24,255), and(ip>>16,255), and(ip>>8,255), and(ip,255)}' /proc/net/route)
        if [ -n "$ip" ] && [ "$ip" != "0.0.0.0" ]; then
            echo "$ip"
            return 0
        fi
    fi

    echo ""
    return 1
}

#-------------------------------------------------------------------------------
# 检查函数段
#-------------------------------------------------------------------------------

# 检查平台API可用性
# 返回：0 - API正常，1 - API异常
check_platform_api() {
    log "[INFO] 正在检查平台API可用性..."

    local response
    local http_code

    # 发送请求并获取响应和HTTP状态码
    response=$(curl -k -m 10 -s -w "\n%{http_code}" "$API_URL" 2>&1)
    http_code=$(echo "$response" | tail -n1)
    response_body=$(echo "$response" | head -n-1)

    # 验证HTTP状态码为200
    if [ "$http_code" != "200" ]; then
        log "[ERROR] API检查失败，HTTP状态码: $http_code"
        return 1
    fi

    # 验证响应内容包含"操作成功"
    if ! echo "$response_body" | grep -q "操作成功"; then
        log "[ERROR] API检查失败，响应内容未包含'操作成功'"
        return 1
    fi

    # 验证响应内容包含code:200（允许空格）
    if ! echo "$response_body" | grep -q "code.*:.*200"; then
        log "[ERROR] API检查失败，响应内容未包含'code:200'"
        return 1
    fi

    log "[SUCCESS] 平台API检查通过"
    return 0
}

# 检查6060端口是否监听
# 返回：0 - 端口已监听，1 - 端口未监听
check_port_6060() {
    log "[INFO] 正在检查6060端口状态..."

    # 优先使用netstat
    if command -v netstat &> /dev/null; then
        if netstat -tuln 2>/dev/null | grep -q ":${MALAN_PORT} "; then
            log "[SUCCESS] 6060端口已监听"
            return 0
        fi
    # 其次使用ss
    elif command -v ss &> /dev/null; then
        if ss -tuln 2>/dev/null | grep -q ":${MALAN_PORT} "; then
            log "[SUCCESS] 6060端口已监听"
            return 0
        fi
    # 最后使用lsof
    elif command -v lsof &> /dev/null; then
        if lsof -i:"${MALAN_PORT}" &> /dev/null; then
            log "[SUCCESS] 6060端口已监听"
            return 0
        fi
    else
        log "[WARNING] 未找到端口检查命令 (netstat/ss/lsof)"
    fi

    log "[WARNING] 6060端口未监听"
    return 1
}

# 获取容器内服务进程PID
# 参数：$1 - jar包标识
# 返回：PID字符串（失败返回空）
get_service_pid() {
    local jar_identifier="$1"
    local pid

    # 在容器内执行ps命令查找进程
    pid=$(docker exec "$CONTAINER_NAME" ps aux 2>/dev/null | grep "$jar_identifier" | grep -v grep | awk '{print $2}' | head -n1)

    echo "$pid"
}

# 检查服务运行状态
# 参数：$1 - 服务key
# 输出：状态信息到stdout（不含时间戳）
# 返回：0 - 服务运行中，1 - 服务未运行
check_service_status() {
    local service_key="$1"
    local jar_identifier="${SERVICES[$service_key]}"

    # 获取PID
    local pid
    pid=$(get_service_pid "$jar_identifier")

    # PID为空，服务未运行
    if [ -z "$pid" ]; then
        echo "[STOPPED] $service_key (未找到进程)"
        return 1
    fi

    # 验证PID是否有效
    if ! docker exec "$CONTAINER_NAME" ps -p "$pid" &> /dev/null; then
        echo "[STOPPED] $service_key (PID: $pid 无效)"
        return 1
    fi

    echo "[RUNNING] $service_key (PID: $pid)"
    return 0
}

#-------------------------------------------------------------------------------
# 操作函数段
#-------------------------------------------------------------------------------

# 启动单个服务
# 参数：$1 - 服务key
# 返回：0 - 启动成功，1 - 启动失败
start_service() {
    local service_key="$1"
    local service_path="${SERVICE_PATHS[$service_key]}"
    local start_script_path="${service_path}/${START_SCRIPT}"

    log "[INFO] 正在启动服务: $service_key"
    if [ "$DEBUG_MODE" = "true" ]; then
        log "[DEBUG] 服务路径: $service_path"
        log "[DEBUG] 启动脚本: $start_script_path"
        log "[DEBUG] jar标识: ${SERVICES[$service_key]}"
    fi

    # 先检查服务目录是否存在
    if ! docker exec "$CONTAINER_NAME" test -d "$service_path"; then
        log "[ERROR] 服务目录不存在: $service_path"
        log "[ERROR] 请检查脚本中 SERVICE_PATHS[$service_key] 的配置是否正确"
        log "[ERROR] 排查命令: docker exec $CONTAINER_NAME ls -la /var/www/java/"
        return 1
    fi

    # 检查run.sh是否存在
    if ! docker exec "$CONTAINER_NAME" test -f "$start_script_path"; then
        log "[ERROR] 启动脚本不存在: $start_script_path"
        log "[ERROR] 目录内容如下:"
        docker exec "$CONTAINER_NAME" ls -la "$service_path" 2>&1 | while read line; do
            log "[ERROR]   $line"
        done
        return 1
    fi

    # 在容器内执行启动脚本
    if [ "$DEBUG_MODE" = "true" ]; then
        log "[DEBUG] 执行命令: cd $service_path && ./$START_SCRIPT"
    fi
    docker exec "$CONTAINER_NAME" bash -c "cd '$service_path' && ./'$START_SCRIPT'" &> /dev/null

    # 等待服务启动
    local wait_time=0
    while [ $wait_time -lt $START_TIMEOUT ]; do
        local jar_identifier="${SERVICES[$service_key]}"
        local pid
        pid=$(get_service_pid "$jar_identifier")

        if [ -n "$pid" ]; then
            log "[SUCCESS] $service_key (PID: $pid)"
            return 0
        fi

        sleep 2
        wait_time=$((wait_time + 2))
    done

    log "[FAILED] $service_key 启动超时（${START_TIMEOUT}秒）"
    return 1
}

# 强制重启关键服务
# 参数：$1 - 服务标识（如"inner"）
restart_critical_services() {
    local critical_service="$1"
    log "[WARNING] ===== 开始强制重启关键服务: $critical_service ====="

    # 查找meeting2.0服务并重启
    for service_key in "${!SERVICES[@]}"; do
        log "[INFO] 强制重启: $service_key"

        # 先尝试停止进程
        local jar_identifier="${SERVICES[$service_key]}"
        local pid
        pid=$(get_service_pid "$jar_identifier")

        if [ -n "$pid" ]; then
            docker exec "$CONTAINER_NAME" kill -9 "$pid" &> /dev/null
            sleep 2
        fi

        # 启动服务
        if start_service "$service_key"; then
            log "[SUCCESS] $service_key 重启成功"
        else
            log "[FAILED] $service_key 重启失败"
        fi
    done

    log "[WARNING] ===== 关键服务重启完成，等待${RESTART_WAIT}秒后继续 ====="
    sleep "$RESTART_WAIT"
}

# 批量检查所有服务
# 返回：停止的服务数量
check_services() {
    local problems=0
    local status_output

    for service_key in "${!SERVICES[@]}"; do
        status_output=$(check_service_status "$service_key")
        log "$status_output"  # 记录服务状态日志
        if [[ "$status_output" =~ \[STOPPED\] ]]; then
            problems=$((problems + 1))
        fi
    done

    echo "$problems"
}

# 重启问题服务
# 参数：$1 - 问题服务数量
restart_problem_services() {
    local problems="$1"
    log "[WARNING] 检测到 $problems 个服务停止，将重新启动"

    for service_key in "${!SERVICES[@]}"; do
        local jar_identifier="${SERVICES[$service_key]}"
        local pid
        pid=$(get_service_pid "$jar_identifier")

        # 如果服务未运行，则启动
        if [ -z "$pid" ]; then
            start_service "$service_key"
        fi
    done
}

# 启动malan服务
# 返回：0 - 启动成功，1 - 启动失败
start_malan_service() {
    log "[INFO] 正在启动malan服务..."

    # 检查malan目录是否存在
    if [ ! -d "$MALAN_DIR" ]; then
        log "[ERROR] malan目录不存在: $MALAN_DIR"
        return 1
    fi

    # 检查malan可执行文件是否存在
    if [ ! -f "${MALAN_DIR}/malan" ]; then
        log "[ERROR] malan可执行文件不存在: ${MALAN_DIR}/malan"
        return 1
    fi

    # 启动malan
    cd "$MALAN_DIR" || {
        log "[ERROR] 无法进入目录: $MALAN_DIR"
        return 1
    }

    bash -c "source /etc/profile && ./malan &" &> /dev/null

    # 等待端口就绪
    local wait_time=0
    while [ $wait_time -lt $START_TIMEOUT ]; do
        if check_port_6060 &> /dev/null; then
            log "[SUCCESS] malan服务启动成功，6060端口已监听"
            return 0
        fi

        sleep 2
        wait_time=$((wait_time + 2))
    done

    log "[FAILED] malan服务启动超时，6060端口未监听"
    return 1
}

#-------------------------------------------------------------------------------
# 主流程段
#-------------------------------------------------------------------------------

# 主函数
main() {
    # 确保日志目录存在
    local log_dir
    log_dir=$(dirname "$LOG_FILE")
    if [ ! -d "$log_dir" ]; then
        mkdir -p "$log_dir" 2>/dev/null || {
            echo "[ERROR] 无法创建日志目录: $log_dir" >&2
            exit 1
        }
    fi

    log "[START] ===== 服务监测开始 ====="

    # 自动获取服务器IP
    SERVER_IP=$(get_server_ip)
    if [ -z "$SERVER_IP" ]; then
        log "[ERROR] 无法自动获取服务器IP，脚本退出"
        exit 1
    fi
    log "[INFO] 自动获取服务器IP: $SERVER_IP"
    API_URL="https://${SERVER_IP}/api/system/getVerifyCode"

    # 动态查找匹配的容器
    local found_container
    found_container=$(find_container)
    if [ -z "$found_container" ]; then
        log "[ERROR] 无法找到匹配的容器 (ujava*)，脚本退出"
        exit 1
    fi
    CONTAINER_NAME="$found_container"
    log "[INFO] 找到容器: $CONTAINER_NAME"

    # 打印运行参数（调试模式）
    if [ "$DEBUG_MODE" = "true" ]; then
        log "[DEBUG] ========== 运行参数 =========="
        log "[DEBUG] 服务器IP: $SERVER_IP"
        log "[DEBUG] 容器名: $CONTAINER_NAME"
        log "[DEBUG] API_URL: $API_URL"
        log "[DEBUG] MALAN_DIR: $MALAN_DIR"
        log "[DEBUG] MALAN_PORT: $MALAN_PORT"
        log "[DEBUG] START_TIMEOUT: ${START_TIMEOUT}秒"
        log "[DEBUG] RESTART_WAIT: ${RESTART_WAIT}秒"
        for svc in "${!SERVICES[@]}"; do
            log "[DEBUG] 服务[$svc]: jar=${SERVICES[$svc]}, path=${SERVICE_PATHS[$svc]}"
        done
        log "[DEBUG] =============================="
    fi

    # 步骤1：平台API检查
    if ! check_platform_api; then
        restart_critical_services "inner"
    fi

    # 步骤2：容器内服务检查
    log "[INFO] ===== $(timestamp) 服务状态检查 ====="
    local problems
    problems=$(check_services)
    if [ "$problems" -gt 0 ]; then
        restart_problem_services "$problems"
    fi

    # 步骤3：6060端口检查
    if ! check_port_6060; then
        start_malan_service
    fi

    log "[SUMMARY] ===== $(timestamp) 操作完成 ====="
}

#-------------------------------------------------------------------------------
# 脚本入口
#-------------------------------------------------------------------------------

# 执行主函数
main "$@"
