#!/bin/bash
# clear_deleted_files.sh - 定时清理被进程占用的已删除文件

LOG_FILE="/var/log/cleanup_deleted_files.log"
DATE=$(date '+%Y-%m-%d %H:%M:%S')

echo "[$DATE] 开始检查被占用的已删除文件..." >> "$LOG_FILE"

# 使用 lsof +L1 查找 link count 为 0（即已删除）但仍被打开的文件
DELETED_ENTRIES=$(lsof +L1 2>/dev/null | grep -v "COMMAND")

if [ -z "$DELETED_ENTRIES" ]; then
    echo "[$DATE] 未发现被占用的已删除文件。" >> "$LOG_FILE"
    exit 0
fi

echo "[$DATE] 发现以下被占用的已删除文件：" >> "$LOG_FILE"
echo "$DELETED_ENTRIES" >> "$LOG_FILE"

# 提取唯一 PID 列表
PIDS=$(echo "$DELETED_ENTRIES" | awk 'NR>1 {print $2}' | sort -u)

for PID in $PIDS; do
    if ! kill -0 "$PID" 2>/dev/null; then
        echo "[$DATE] PID $PID 已不存在，跳过。" >> "$LOG_FILE"
        continue
    fi

    CMD=$(ps -p "$PID" -o comm= 2>/dev/null | tr -d ' ')
    echo "[$DATE] 处理进程: PID=$PID, CMD=$CMD" >> "$LOG_FILE"

    # 尝试发送 SIGHUP（适用于 Nginx、rsyslog 等支持 reload 的服务）
    if kill -0 "$PID" 2>/dev/null; then
        echo "[$DATE] 向 PID $PID 发送 SIGHUP 信号..." >> "$LOG_FILE"
        kill -HUP "$PID" 2>/dev/null && continue
    fi

    # 若 SIGHUP 无效或不支持，记录警告（不强制 kill，避免服务中断）
    echo "[$DATE] WARNING: 无法通过 SIGHUP 释放 PID $PID 的文件句柄。建议手动处理或配置 logrotate。" >> "$LOG_FILE"
done

echo "[$DATE] 检查完成。" >> "$LOG_FILE"
