#!/bin/bash
#------------------------------工具类模块---------------------------------------------------
# 日志打印函数
LOG_FILE="/var/log/new_auto_script.log"
mkdir -p "$(dirname "$LOG_FILE")" 2>/dev/null
# 用户输入的东西也同步记录到日志信息里面
function log() {
    # 获取当前时间
    local timestamp
    timestamp=$(date "+%Y-%m-%d %H:%M:%S")
    # 日志级别对应的颜色代码
    local INFO_COLOR="\033[32m"      # 绿色
    local WARN_COLOR="\033[33m"      # 黄色
    local ERROR_COLOR="\033[31m"     # 红色
    local DEBUG_COLOR="\033[36m"     # 青色
    local RESET_COLOR="\033[0m"      # 重置颜色
    # 参数判断
    if [ $# -lt 2 ]; then
        echo "Usage: log <level> <message>"
        return 1
    fi
    local level=$1
    local message=$2
    # 根据日志级别选择颜色
    case $level in
        "INFO")
            color=$INFO_COLOR
            ;;
        "WARN")
            color=$WARN_COLOR
            ;;
        "ERROR")
            color=$ERROR_COLOR
            ;;
        "DEBUG")
            color=$DEBUG_COLOR
            ;;
        *)
            echo "Invalid log level: $level"
            return 1
            ;;
    esac
    # 输出带颜色的日志到屏幕
    echo -e "${color}[$timestamp] [$level] $message${RESET_COLOR}"
    # 追加纯文本日志到文件
    echo "[$timestamp] [$level] $message" >> "$LOG_FILE"
}



#------------------------------检测模块-----------------------------------------------------

#------------------------------服务器检测-start---------------------------------------------
function detectNetType() {
  log "INFO" "准备获取所有网口信息，排除 loopback 和虚拟接口"
  INTERFACES=$(find /sys/class/net -mindepth 1 -maxdepth 1 -type d -not -name 'lo' -not -name 'docker*' -not -name 'br-*' -not -name 'veth*' -not -name 'virbr*' -not -name 'vmnet*' -exec basename {} \;)
  # 遍历每个网口，检查是否配置了 IP 地址，并输出网口信息
  log "INFO" "检测到的网口信息如下："
for iface in $INTERFACES; do
    log "INFO" "-----------------------------"
    log "INFO" "接口: $iface"

    log "INFO" "  检查接口 $iface 是否配置了 IP 地址"
    IP_ADDR=$(ip addr show "$iface" | grep "inet " | awk '{print $2}')

    if [[ -n "$IP_ADDR" ]]; then
        log  "INFO" "  配置的 IP 地址: $IP_ADDR"
    else
        log  "INFO"  "  没有配置 IP 地址"
    fi

    # 获取接口的连接状态（carrier）
    log "INFO" "  检查接口 $iface 的连接状态"
    if [[ -f /sys/class/net/$iface/carrier ]]; then
        CARRIER=$(cat /sys/class/net/$iface/carrier)
        [[ "$CARRIER" == "1" ]] && LINK="connected" || LINK="disconnected"
    else
        LINK="unknown"
    fi
    log "INFO" "  状态: $LINK"

    # 使用 ethtool 获取接口的类型（光口、电口等）
    log "INFO" "  获取接口 $iface 的类型"
    ETHTOOL_OUTPUT=$(ethtool "$iface" 2>/dev/null)
    TYPE=$(echo "$ETHTOOL_OUTPUT" | grep -i "Port:" | awk '{print tolower($2)}')
    [[ -z "$TYPE" ]] && TYPE="unknown" && TYPE="other"

    # 获取速率和驱动信息
    log "INFO" "  获取接口 $iface 的速率和驱动信息"
    SPEED=$(echo "$ETHTOOL_OUTPUT" | grep -i "Speed:" | awk '{print $2}')
    [[ -z "$SPEED" ]] && SPEED="unknown"
    DRIVER=$(ethtool -i "$iface" 2>/dev/null | grep "driver:" | awk '{print $2}')
    [[ -z "$DRIVER" ]] && DRIVER="unknown"

    # 获取 MAC 地址
    log "INFO" "  获取接口 $iface 的 MAC 地址"
    MAC=$(cat /sys/class/net/$iface/address)

#    转换 TYPE 为中文描述
    case "$TYPE" in
        "copper")
            TYPE="电口"
            ;;
        "fiber")
            TYPE="光口"
            ;;
        "other")
            TYPE="其他类型"
            ;;
        *)
            TYPE="未知类型"
            ;;
    esac
    # 输出接口信息
    log "INFO" "  光电口类型: ${TYPE:-unknown}"
    log "INFO" "  速率: $SPEED"
    log "INFO"  "  MAC 地址: $MAC"
    log "INFO" "  驱动: $DRIVER"
done
}


function checkOS() {
log "INFO" "开始检测服务器环境"
log "INFO" "**************《请检查当前系统是否适配》**************************************************"
# 检测当前的操作系统是否为X86架构
log "INFO" "开始检查当前操作系统架构"
if [ "$(uname -m)" != "x86_64" ]; then
    log "ERROR" "当前操作系统不是X86架构，请使用X86架构的操作系统"
    exit 1
else
    log "INFO" "当前操作系统为X86架构，继续执行脚本"
fi


server_available_mem=`free -g |awk 'NR==2' | awk '{print$7}'`
server_max_mem=`free -g|awk 'NR==2' |awk '{print$2}'`

log "INFO" "当前系统最大内存为:${server_max_mem}GB 可用内存为:${server_available_mem}GB"

#判断服务器的最大内存值小于约定的内存大小min_memory_value=15，就返回。否则成功。
log "INFO" "开始检查服务器最大内存"
if [ $server_max_mem -lt $min_memory_value ];then
  log "WARN" "当前系统最大内存为:${server_max_mem}G,建议将内存至少添加为: ${min_memory_value}G"
    #  需要用户确认信息，支持默认空格和y或Y，并日志记录用户输入情况
    read -p "是否继续执行脚本？(y/n): " yn
    if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
#        记录用户如情况
        log "INFO" "用户选择继续执行脚本,输入为${yn} "
    else
        log "ERROR" "脚本执行终止，请调整服务器内存后重新运行脚本"
        exit 1
    fi
else
  log "INFO" "检查当前系统最大内存为:${server_max_mem}G 满足系统运行！"
fi
#todo:考虑把交换关闭掉

#判断服务器的内存值小于约定的内存大小min_memory_available=10，就返回。否则成功。
log "INFO" "开始检查服务器可用内存"
if [ $server_available_mem -lt $min_memory_available ];then
      log  "WARN"  "当前系统docker可用内存为:${server_available_mem}G,建议将内存至少添加为： ${min_memory_available}G ！"
      #  需要用户确认信息，支持默认空格和y或Y，并日志记录用户输入情况
      read -p "是否继续执行脚本？(y/n): " yn
      if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
      #   记录用户如情况
          log "INFO" "用户选择继续执行脚本,输入为${yn} "
      else
          log "ERROR" "脚本执行终止，请调整服务器内存后重新运行脚本"
          exit 1
      fi
else
      log  "INFO"  "检查当前系统docker可用内存为:${server_available_mem}G 满足系统运行!"
fi

#判断服务的根分区的硬盘大小是否超过预设的硬盘资源。成功继续，否则退出。
log "INFO" "开始检查服务器硬盘空间"
server_disk_available=`df -B G /|awk '/\//{print$4}' | awk '{sub(/.{1}$/,"")}1'`

if [ $server_disk_available -lt $min_disk_available ];then
    log "WARN"  "当前系统的硬盘空间为:${server_disk_available}G,建议将硬盘空间至少添加为: ${min_disk_available}G!"
        #  需要用户确认信息，支持默认空格和y或Y，并日志记录用户输入情况
        read -p "是否继续执行脚本？(y/n): " yn
        if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
        #   记录用户如情况
            log "INFO" "用户选择继续执行脚本,输入为${yn} "
        else
            log "ERROR" "脚本执行终止，请调整服务器磁盘后重新运行脚本"
            exit 1
        fi
else
    log "INFO"  "检查当前系统硬盘空间为:${server_disk_available}G,满足系统运行!"
fi

# 检测当前用户是否有root权限
log "INFO" "开始检查当前用户权限"
if [ "$(id -u)" -ne 0 ]; then
    log "ERROR" "当前用户没有root权限，请使用root用户执行脚本，获者使用sudo命令执行脚本"
    exit 1
else
    log "INFO" "当前用户有root权限，继续执行脚本"
fi
log "INFO" "当前用户权限检查完成"

# 检测当前服务器是否存在多张网卡并且ipv4不唯一的需要列出来
log "INFO" "开始检查服务器网卡信息"
network_interfaces=$(ip -o -4 addr show | awk '{print $2}' | sort -u)
if [ -z "$network_interfaces" ]; then
    log "ERROR" "当前服务器没有检测到任何网卡，请检查网络配置"
    exit 1
else
    log "INFO" "当前服务器检测到的网卡信息如下：$network_interfaces"
#    获取对应的IP信息
    for interface in $network_interfaces; do
        ip_address=$(ip -o -4 addr show "$interface" | awk '{print $4}' | cut -d/ -f1)
        if [ -n "$ip_address" ]; then
            log "INFO" "网卡 $interface 的 IP 地址为: $ip_address"
            IP=$ip_address
        else
            log "WARN" "网卡 $interface 没有配置 IPv4 地址"
        fi
    done
fi
log "INFO" "服务器网卡信息检查完成"

log "INFO" "开始检测服务器网口类型"
detectNetType
log "INFO" "服务器网口类型检测完成"
# 用户确认以后继续往下
read -p "请确认网口信息是否正确,特别注意需要仔细观察你的项目是要光口还是电口 ，输入 y 或 Y 确认，输入 n 或 N 退出脚本: " yn
if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
    log "INFO" "用户确认网口信息正确，继续执行脚本"
else
    log "ERROR" "用户确认网口信息不正确，脚本执行终止"
    exit 1
fi


  
echo -e "\033[36m 
*******************************************************
*                  可适配的操作系统(X86)              *
*                                                     *
*                     cento7.5                        *                  
*                     cento7.6                        *
*                     cento7.9                        *
*                     统信系统                        *
*                     鲲鹏系统                        *
*                     Ubuntu                          *
*                     redhat7                         *
*                     银河麒麟V10                     *
*                     openEuler 24.03 (LTS-SP1)       *
*                                                     *
*******************************************************
         \033[0m"

echo -e "\033[36m\n检测到当前操作系统如下，请确认是否适配：\n         \033[0m"
# 优先获取PRETTY_NAME，没有则拼接NAME和VERSION
log "INFO" "开始获取当前操作系统版本信息"
if grep -q '^PRETTY_NAME=' /etc/os-release; then
    os_version=$(grep '^PRETTY_NAME=' /etc/os-release | cut -d= -f2 | tr -d '"')
else
    name=$(grep '^NAME=' /etc/os-release | cut -d= -f2 | tr -d '"')
    version=$(grep '^VERSION=' /etc/os-release | cut -d= -f2 | tr -d '"')
    os_version="$name $version"
fi
log "INFO" "当前操作系统版本为: $os_version"

case "$os_version" in
    *"CentOS Linux 7"*|*"UOS"*|*"Kylin V10"*|*"Ubuntu"*|*"Red Hat Enterprise Linux Server 7"*|*"openEuler"*)
        log "INFO" "当前操作系统版本适配，继续执行脚本" ;;
    *)
        log "ERROR" "当前操作系统版本不适配，请使用CentOS 7、UOS、麒麟V10、Ubuntu、Red Hat Enterprise Linux Server 7 或 openEuler 24.03 (LTS-SP1)"
        #  需要用户确认信息，支持默认空格和y或Y，并日志记录用户输入情况
        read -p "是否继续执行脚本？(y/n): " yn
        if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
        #   记录用户输入情况
            log "INFO" "用户选择继续执行脚本,输入为${yn} "
        else
            log "ERROR" "脚本执行终止，请调整服务器操作系统后重新运行脚本"
            exit 1
        fi;;
esac
log "INFO" "当前操作系统版本适配，继续执行脚本"
#检查服务器时间和时区
log "INFO" "开始检查服务器时间和时区"
log "INFO" "**************《请检查当前时间，时区是否正确》*****************************************************************"
log "INFO" "当前服务器时区、时间为: $(date)"

read -p "确认无误请按 y 或 Y 或直接回车，不正确请按 n 修改时间:     " yn
if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
    log "INFO" "下面进入系统部署。。。"
    sleep 3
else
    read -p "输入当前日期，格式如2021/02/06——" da
    echo "="
    date -s $da
    read -p "输入当前时间，格式如18:00:00——" sa
    echo "="
    date -s $sa
    timedatectl set-timezone Asia/Shanghai
    clock -w
    echo "==================================================="
    echo "查看修改后的时间"
    date
fi
}
log "INFO" "服务器检测完成"

#------------------------------服务器检测-end------------------------------------------------------------------------------------------------------------------------
#todo: 对文件的检测每次部署的时候根据功能，这里检测的文件是要不一样的，这个先记录，不着急实现
#------------------------------部署文件检测-start--------------------------------------------------------------------------------------------------------------------
function File_detection_java()
{
log "INFO" "************************************"
log "INFO" "开始检测预定系统部署文件是否完整"
log "INFO" "************************************"
ds1=$auto_java"/java/api-java-meeting2.0/conf"
ds2=$auto_java"/java/api-java-meeting2.0/config"
ds3=$auto_java"/java/api-java-meeting2.0/update"
ds4=$auto_java"/java/external-meeting-api"
ds5=$auto_java"/java/nginx-conf.d"
ds6=$auto_java"/redis"
ds7=$auto_java"/java/ubains-web-2.0"
ds8=$auto_java"/java/cert"
ds9=$auto_java"/server_bag/docker"
ds10=$auto_java"/server_bag/ntp"
ds11=$auto_java"/server_bag/"

fs1=$auto_java"/java/api-java-meeting2.0/run.sh"
fs2=$auto_java"/java/api-java-meeting2.0/update.sh"
fs3=$auto_java"/java/api-java-meeting2.0/ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
fs4=$auto_java"/java/start.sh"
fs5=$auto_java"/server_bag/docker.service"
fs6=$auto_java"/server_bag/docker-20.10.7.tgz"
fs7=$auto_java"/server_bag/uemqx5.8.7.tar.gz"
fs8=$auto_java"/server_bag/java1.8.0_321.tar"
fs9=$auto_java"/server_bag/mysql-v8.4.7.tar.gz"
fs10=$auto_java"/server_bag/redis6.2.6.tar"
fs11=$auto_java"/server_bag/ufastdfs.tar"
fs12=$auto_java"/new_auto.sh"


ds=("$ds1" "$ds2" "$ds3" "$ds4" "$ds5" "$ds6" "$ds7" "$ds8" "$ds9" "$ds10" "$ds11")
fs=("$fs1" "$fs2" "$fs3" "$fs4" "$fs5" "$fs6" "$fs7" "$fs8" "$fs9" "$fs10" "$fs11" "$fs12")
log "INFO" "准备检查预定系统文件"
for j in 0 1 2 3 4 5 6 7 8 9 10;
 do
        if [ -d "${ds[j]}" ] ; then
        log "INFO" "${ds[j]} 文件校验成功! "
        
        sleep 1
        else
          log "ERROR" "${ds[j]} 文件不存在,请重新上传文件再执行脚本"
        exit
        fi
done
log "INFO" "预定系统文件检测成功"
log "INFO" "准备检查预定系统部署文件"
for j in 0 1 2 3 4 5 6 7 8 9 10 11 ;
 do
        if [ -f "${fs[j]}" ] ; then
          log  "INFO" "${fs[j]} 文件校验成功！"
          sleep 1
        else
          log "ERROR" "${fs[j]} 文件不存在,请重新上传文件再执行脚本"
          exit
        fi
done
}
log "INFO" "预定系统部署文件检测成功"
log "INFO" "************************************"
log "INFO" "检测预定系统部署文件完成"
log "INFO" "************************************"
#------------------------------部署文件检测-end--------------------------------------------------------------------------------------------------------------------


#创建用户(预定安装需要:该用户与系统启动用户有关)
function add_user()
{
log "INFO" "创建用户user"
user=user

log "INFO" "判断是否存在用户$user"
egrep "^$user" /etc/passwd >& /dev/null
if [ $? -ne 0 ]; then
    log "INFO" "用户$user不存在，开始创建用户"
    username="user"
    password="Ubains@135"
    useradd -m "$username"
    echo "$username:$password" | chpasswd
    log "INFO" "用户$user创建成功"
else
    log "INFO" "用户user已创建! "
fi
}


#------------------------------开放防火墙端口-start------------------------------------------------------------------------------------------------------------------------

#--------开放预定端口---------------------
function firewalldjava()
{
#ssh登录（基本）22
#预定mqtt（基本）1883、8083
#预定后台（基本）8999
#预定对外接口后台（门口屏3.0）8996
#预定系统前端，非https（基本）8080
#预定系统前端，https（基本）443
#预定redis（基本）6379
#图片服务器（基本）8888
#预定后台调试（基本）8997
#fastdfs/tracker（基本）22122
#fastdfs/storage（基本）23000
#数据库端口（基本）8306
#无纸化同步环境  62121 554 1935 10000
#时间同步端口123(门口屏使用)
#预定会控页面（java容器内8083）
#预定登录系统后台8998
#登录系统前端8079
#smc视频会议8085
#cloudlink视频会议8088
#企业微信前端8086
#微信公众号前端8087
#微信公众号后端80
#桌牌系统9990
#安卓远程控制8892

log "INFO" "***************************************"
log "INFO" "检查firewalld状态......"
log "INFO" "***************************************"

$sudoset firewall-cmd --state
if [ $? -eq  0 ]; then
  log  "INFO" "firewalld已开启，开始检查tcp端口"
#	for j in 22 1883 8883 18083 18082 443 8888 8997 22122 23000 8306 62121 554 1935 10000  8998 8079 8085 8088 8086 8087 6060 9990
	for j in 22 1883 8883 443 8997  62121 554 1935 10000 8892
      	do
             #firewall-cmd --list-port |grep $j
             a=`firewall-cmd --list-port |grep $j|wc -l`
             if [ "$a" -gt "0" ]; then
		            log  "INFO" "$j 已开放"
             else
	              log  "INFO" "开放端口 $j"
                $sudoset firewall-cmd --zone=public --add-port=$j/tcp --permanent;
             fi
        done
  log  "INFO" "tcp端口已经完成检测,开始检查udp端口"
	for i in 123 323
 	do
             #firewall-cmd --list-port |grep $j
             a=`firewall-cmd --list-port |grep $i|wc -l`
             if [ "$a" -gt "0" ]; then
		           log "INFO" "$i 已开放"
             else
		           log "INFO" "开放端口"$j
               $sudoset firewall-cmd --zone=public --add-port=$i/udp --permanent;
             fi
        done
  log "INFO" "准备重新加载防火墙"
	$sudoset firewall-cmd --reload
	log "INFO" "防火墙重载成功端口已开放完成"
  log "INFO" "firewall-cmd --list-port查看已开放的端口"
  $sudoset firewall-cmd --list-port
else
  log  "WARN" $j" firewalld已关闭!"
fi
log "INFO" "***************************************"
log "INFO" "firewalld状态检测完成......"
log "INFO" "***************************************"
}
# 这个323在门口屏的时候要考虑这个
#------------------------------开放防火墙端口-end------------------------------------------------------------------------------------------------------------------------


#------------------------------服务安装-start------------------------------------------------------------------------------------------------------------------------

#docker安装
function docker_x86()
{
  log "INFO" "***************************************"
  log "INFO" "检查Docker...... "
  log "INFO" "***************************************"
	docker -v
    if [ $? -eq  0 ]; then
        log "INFO" "检查到Docker已安装! "
    else
        log "WARN" "docker 未安装 "
        log "INFO" "docker安装中  "

	$sudoset mkdir -p $PWD/soft
        $sudoset cp -rf $auto_java/server_bag/docker-20.10.7.tgz $PWD/soft
        sleep 3
        $sudoset tar -xvf $PWD/soft/docker-20.10.7.tgz -C $PWD/soft
        $sudoset cp $PWD/soft/docker/*  /usr/bin/
        $sudoset rm -rf /etc/systemd/system/docker.service
        sleep 3
        $sudoset cp -rf $auto_java/server_bag/docker.service /etc/systemd/system/docker.service
        sleep 1
        $sudoset chmod +x /etc/systemd/system/docker.service
        sleep 3
        $sudoset systemctl daemon-reload 
        sleep 3
        $sudoset systemctl start docker  
        sleep 3
        $sudoset systemctl enable docker.service  

        log "INFO" "***************************************"
        log "INFO" "docker安装完成！！！"
        log "INFO" "***************************************"

        $sudoset docker -v
    fi
}


#数据库配置

#x86架构mysql安装
function mysql_x86()
{
log "INFO" "检查mysql...... "
$sudoset docker images |grep mysql
$sudoset docker ps |grep umysql
if [ $? -eq  0 ]; then
log "INFO" "检查到mysql已安装！  "
else
    log "WARN" "mysql 未安装 "
    log "WARN" "mysql安装中。。。 "
    $sudoset docker load -i $auto_java/server_bag/mysql-v8.4.7.tar.gz
    $sudoset cp -rf $auto_java/mysql /opt/
    $sudoset chmod -R 777 /opt/mysql
    $sudoset chown -R 27:27 /opt/mysql/logs
    $sudoset chmod 644 /opt/mysql/conf.d/my.cnf
    $sudoset chown root:root /opt/mysql/conf.d/my.cnf
    sleep 3
	#$sudoset docker run -p 8306:3306 --name umysql --mac-address="02:42:ac:11:00:05" -v /usr/local/docker/mysql/conf:/etc/mysql/conf -v /usr/local/docker/mysql/my.cnf:/etc/mysql/my.cnf -v /usr/local/docker/mysql/conf.d:/etc/mysql/conf.d -v /usr/local/docker/mysql/logs:/var/log/mysql -v /usr/local/docker/mysql/data:/var/lib/mysql -v /usr/local/docker/mysql/mysql-files:/var/lib/mysql-files -v /etc/localtime:/etc/localtime:ro --restart=always -e MYSQL_ROOT_PASSWORD="dNrprU&2S" -itd 139.9.60.86:5000/umysql:v4
    $sudoset docker run -d --name umysql -p 8306:3306 --restart=always --security-opt seccomp=unconfined --mac-address="02:42:ac:11:00:25" -e MYSQL_ROOT_PASSWORD="dNrprU&2S" -v /etc/localtime:/etc/localtime:ro -v /opt/mysql/data:/var/lib/mysql -v /opt/mysql/conf.d:/etc/mysql/conf.d -v /opt/mysql/logs:/var/log/mysql mysql:v8.4.7 --mysql-native-password=ON
    $sudoset docker ps
fi
sleep 30
#修改数据库密码
log "INFO" "准备修改数据库密码"
NEW_PASSWORD="dNrprU&2S"
CONTAINER_NAME="umysql"
USERS=("root" "mysqluser")
docker exec "$CONTAINER_NAME" mysql -u"root" -p"$NEW_PASSWORD" -e "ALTER USER 'root'@'localhost' IDENTIFIED BY '$NEW_PASSWORD'; FLUSH PRIVILEGES;"
docker exec "$CONTAINER_NAME" mysql -u"root" -p"$NEW_PASSWORD" -e "ALTER USER 'root'@'%' IDENTIFIED BY '$NEW_PASSWORD'; FLUSH PRIVILEGES;"
docker exec "$CONTAINER_NAME" mysql -u"root" -p"$NEW_PASSWORD" -e "create user mysqluser identified by '$NEW_PASSWORD';"
docker exec "$CONTAINER_NAME" mysql -u"root" -p"$NEW_PASSWORD" -e "GRANT ALL ON ubains.* TO 'mysqluser'@'%';GRANT ALL ON devops.* TO 'mysqluser'@'%';"
docker exec "$CONTAINER_NAME" mysql -u"root" -p"$NEW_PASSWORD" -e "alter user 'mysqluser'@'%' identified with mysql_native_password by '$NEW_PASSWORD';"


for user in "${USERS[@]}"
do
   docker exec "$CONTAINER_NAME" mysql -u"root" -p"$NEW_PASSWORD" -e "update mysql.user set Grant_priv='Y',Super_priv='Y' where user = '${user}' and host = '%';"
done

log "INFO" "数据库密码修改成功！"
}

#x86架构安装mqtt(预定2.0)
function emqx_x86()
{
log "INFO" "正在检查emqx是否安装..."
$sudoset docker images |grep emqx
$sudoset docker ps |grep uemqx
if [ $? -eq  0 ]; then
  log "INFO" "检查到eqmx已安装！正在进入下一步..."
else
  log "WARN" "检查到eqmx未安装！正在安装..."
  $sudoset docker load -i $auto_java/server_bag/uemqx5.8.7.tar.gz
  $sudoset docker run -d \
    --name uemqx \
    --privileged \
    --restart always \
    --mac-address="02:42:ac:12:00:06" \
    -p 1883:1883 \
    -p 8883:8883 \
    -p 8083:8083 \
    -v /var/www/emqx/config/emqx.conf:/opt/emqx/etc/emqx.conf:ro \
    -v /var/www/emqx/config/auth-built-in-db-bootstrap.csv:/opt/emqx/etc/auth-built-in-db-bootstrap.csv:ro \
    -v /var/www/emqx/config/acl.conf:/opt/emqx/etc/acl.conf:ro \
    -v /var/www/emqx/data:/opt/emqx/data \
    -v /var/www/emqx/log:/opt/emqx/log \
    emqx/emqx:5.8.7
  sleep 1
  log "INFO" "完成emqx的安装"
fi
}


#x86架构redis安装
function redis_x86()
{
  log "INFO" "正在检查redis服务是否安装..."
  $sudoset docker images |grep redis
  $sudoset docker ps |grep uredis
if [ $? -eq  0 ]; then
  log "INFO" "检查到redis服务已安装！"
else
  log "WARN" "检查到redis服务未安装，正在安装..."
    $sudoset docker load -i $auto_java/server_bag/redis6.2.6.tar
    sleep 3
    $sudoset docker tag f1b6973564e9 139.9.60.86:5000/redis:v2
    $sudoset docker run -itd --restart=always --mac-address="02:42:ac:11:00:04" -p 6379:6379 --name uredis -v /var/www/redis/redis-6.0.3.conf:/etc/redis/redis.conf -v /var/www/redis/data:/data -v /etc/localtime:/etc/localtime:ro 139.9.60.86:5000/redis:v2 redis-server /etc/redis/redis.conf --appendonly yes
  log "INFO" "完成redis服务的安装"
fi
}

#x86架构java安装
function java_x86()
{
  log "INFO" "检查java......"
$sudoset docker images |grep java
$sudoset docker ps |grep java
if [ $? -eq  0 ]; then
  log "INFO" "检查到java已安装！"
else
  log "WARN" "检查到java未安装，正在安装..."
  $sudoset docker load -i $auto_java/server_bag/java1.8.0_321.tar
  $sudoset docker tag 9d7e103413e2 139.9.60.86:5000/ujava:v4
  $sudoset docker run -itd --privileged -v $install_dir_java/java:/var/www/java -v $install_dir_java/java/nginx-conf.d/nginx_log:/usr/local/nginx/logs -v /etc/localtime:/etc/localtime:ro -v /var/fdfs/storage/data:/var/fdfs/storage/data -p 554:554 -p 1935:1935 -p 10000:10000 -p 2333:2333  -p 2334:2334 -p 8997:8997 -p 8998:8998 -p 8079:8079 -p 443:443 -p 8080:8080 -p 8085:8085 -p 8086:8086 -p 8087:8087 -p 8088:8088 -p 8889:8889 -p 8999:8999 --restart=always --mac-address="02:42:ac:11:00:02" --name ujava2 139.9.60.86:5000/ujava:v4 /var/www/java/start.sh
  log "INFO" "完成java的安装"
fi
}

#x86架构ntp安装
function ntp_x86()
{
#  判断是否为centos7 否则跳出函数
if [ ! -f /etc/redhat-release ]; then
    log "ERROR" "当前系统推荐使用NTP服务"
    return 1
fi
log "INFO" "检查ntp服务是否安装......"
$sudoset systemctl status ntpd |grep running
#ps -aux | grep ntp
if [ $? -eq  0 ]; then
    log "INFO" "检查到ntp已安装！"
else
    log "WARN" "检查到ntp未安装，正在安装..."
    $sudoset cd $auto_java/server_bag/ntp
    $sudoset rpm -Uvh *.rpm --nodeps --force
    $sudoset systemctl enable ntpd
    $sudoset systemctl enable ntpdate
    $sudoset systemctl start ntpd
    $sudoset systemctl status ntpd
    $sudoset mv /etc/ntp.conf    /etc/ntp.confbak
    $sudoset cp $auto_java/server_bag/ntp/ntp.conf   /etc/
    $sudoset systemctl restart  ntpd
    $sudoset systemctl disable  chronyd
    log "INFO" "完成ntp服务的安装"
fi
}

#x86架构统信、麒麟系统的ntp安装
function ntp_uos() {
    #    判断如果是centos7就退出安装
    if [ -f /etc/redhat-release ]; then
        log "ERROR" "当前系统已经安装NTP服务"
        return 1
    fi

#    判断文件是否存在ntp1.aliyun.com地址
    if grep -q "ntp1.aliyun.com" /etc/chrony.conf; then
        log "INFO" "NTP配置文件中已存在ntp1.aliyun.com地址，无需重复添加"
        return 0
    fi

    local config_dir="$auto_java/server_bag/ntp"
    local backup="/etc/chrony.confbak"
    local target="/etc/chrony.conf"

    # 切换目录（不需要 sudo）
    cd "$config_dir" || { echo "无法进入目录: $config_dir"; return 1; }

    # 备份原有配置
    if [ -f "$target" ]; then
        log "INFO" "正在备份旧配置文件到 $backup"
        sudo cp "$target" "$backup"
    fi

#    检查是否可以通外网，可以则配置外网的ntp服务器地址，不可以则询问是否有企业ntp服务器地址
    if ! ping -c 1 ntp1.aliyun.com &> /dev/null; then
          log "ERROR" "无法连接到外网NTP服务器 ntp1.aliyun.com，请检查网络连接"
          read -p "是否有企业NTP服务器地址？(y/n): " yn
          if [[ -z "$yn" || "$yn" =~ ^[yY]$ ]]; then
              read -p "请输入企业NTP服务器地址: " custom_ntp
              echo "server $custom_ntp iburst" | sudo tee -a "$target"
              echo "allow all" | sudo tee -a "$target"
          else
              log "ERROR" "没有可用的NTP服务器地址，配置默认地址，请联系管理员"
              echo "server ntp1.aliyun.com iburst" | sudo tee -a "$target"
              echo "allow all" | sudo tee -a "$target"
          fi
    else
        log "INFO" "可以连接到外网NTP服务器 ntp1.aliyun.com，继续配置"
        # 如果可以连接到外网NTP服务器，则继续添加默认的NTP服务器地址
          echo "server ntp1.aliyun.com iburst" | sudo tee -a "$target"
          echo "server ntp2.aliyun.com iburst" | sudo tee -a "$target"
          echo "server ntp3.aliyun.com iburst" | sudo tee -a "$target"
          echo "allow all" | sudo tee -a "$target"
    fi


    # 输出同步地址到日志
    log "INFO" "正在写入同步地址到 $target"

    # 重启服务
    log "INFO" "正在重启 chronyd 服务..."
    sudo systemctl daemon-reload
    sudo systemctl restart chronyd
    sudo systemctl enable chronyd

    log "INFO" "chronyd 服务已重启并设置为开机自启"
}





#安装fastfds
function fastfds_X86()
{
  log "INFO" "检查fastfds...... "
  $sudoset docker images |grep fastfds
  $sudoset docker ps |grep utracker
  $sudoset docker ps |grep ustorage
if [ $? -eq  0 ]; then
  log "INFO" "检查到fastfds已安装！"
else
  log "WARN" "检查到fastfds未安装，正在安装..."
  $sudoset docker load -i $auto_java/server_bag/ufastdfs.tar
  sleep 3
  $sudoset docker run -dti --network=host --name utracker --restart=always -v /var/fdfs/tracker:/var/fdfs -v /etc/localtime:/etc/localtime 139.9.60.86:5000/ufastdfs:v1 tracker
  sleep 3
  $sudoset docker run -dti --network=host --name ustorage --restart=always -e TRACKER_SERVER=$IP:22122 -v /var/fdfs/storage:/var/fdfs  -v /etc/localtime:/etc/localtime  139.9.60.86:5000/ufastdfs:v1 storage
  log "INFO" "完成fastfds的安装"
fi
}

#x86架构检测服务安装
function monitor_x86()
{
    log "INFO" "检查jdk服务......"
    java -version
    if [ $? -eq 0 ]; then
        log "INFO" "检查到 jdk已安装，卸载已安装的java版本"
        $sudoset sed -i '/JAVA_HOME/d' /etc/profile
        $sudoset sed -i '/CLASSPATH/d' /etc/profile
        $sudoset sed -i '/^export JAVA_HOME/d' /etc/profile
        $sudoset sed -i '/^export CLASSPATH/d' /etc/profile
        $sudoset sed -i '/^export PATH/d' /etc/profile
        source /etc/profile
	      log "INFO" "清空profile文件完成"
        log "INFO" "正在安装 jdk......"

        $sudoset mkdir -p /opt/deploy/java
        $sudoset cp $auto_java/server_bag/jdk-8u361-linux-x64.tar.gz /opt/deploy/
        $sudoset tar -zxvf /opt/deploy/jdk-8u361-linux-x64.tar.gz -C /opt/deploy/java
        $sudoset sed -i '$a#set java environment' /etc/profile
        $sudoset sed -i '$aJAVA_HOME=/opt/deploy/java/jdk1.8.0_361' /etc/profile
        $sudoset sed -i '$aCLASSPATH=.:$JAVA_HOME/lib.tools.jar' /etc/profile
        $sudoset sed -i '$aPATH=$JAVA_HOME/bin:$PATH'   /etc/profile
        $sudoset sed -i '$aexport JAVA_HOME CLASSPATH PATH'  /etc/profile
        $sudoset source /etc/profile
        $sudoset java -version
        log "INFO" "java安装完成"

    else
        log "WARN" "检查到 jdk未安装，开始安装jdk"
        $sudoset mkdir -p /opt/deploy/java
        $sudoset cp $auto_java/server_bag/jdk-8u361-linux-x64.tar.gz /opt/deploy/
        $sudoset tar -zxvf /opt/deploy/jdk-8u361-linux-x64.tar.gz -C /opt/deploy/java
        $sudoset sed -i '$a#set java environment' /etc/profile
        $sudoset sed -i '$aJAVA_HOME=/opt/deploy/java/jdk1.8.0_361' /etc/profile
        $sudoset sed -i '$aCLASSPATH=.:$JAVA_HOME/lib.tools.jar' /etc/profile
        $sudoset sed -i '$aPATH=$JAVA_HOME/bin:$PATH'   /etc/profile
        $sudoset sed -i '$aexport JAVA_HOME CLASSPATH PATH'  /etc/profile
        sleep 2
        log "INFO" "开始source /etc/profile"
        $sudoset source /etc/profile
        $sudoset java -version
        log "INFO" "java安装完成"
        $sudoset chmod 755 /var/www/java/external-meeting-api/run.sh
        $sudoset echo 'su - root -c "cd /var/www/java/external-meeting-api && ./run.sh"' | sudo tee -a /etc/rc.local > /dev/null
        $sudoset echo 'su - root -c "cd /var/www/malan && ./run.sh"' | sudo tee -a /etc/rc.local > /dev/null
        $sudoset echo 'su - root -c "cd /var/www/java/ws-scrcpy-deploy && ./rebuild-native.sh && start.sh &"' | sudo tee -a /etc/rc.local > /dev/null
	    $sudoset chmod 777 /etc/rc.local
        $sudoset chmod 777 /etc/rc.d/rc.local
        $sudoset sh /var/www/malan/run.sh
        $sudoset sh /var/www/java/external-meeting-api/run.sh
        $sudoset sh /var/www/java/ws-scrcpy-deploy/rebuild-native.sh
        $sudoset sh /var/www/java/ws-scrcpy-deploy/start.sh &
        sleep 2
    fi
}

# x86架构 malan 安装/启动
function malan_x86()
{
  log "INFO" "检查并启动 malan 服务..."
  local malan_dir="/var/www/malan"
  local malan_bin="${malan_dir}/malan"
  local malan_run="${malan_dir}/run.sh"

  if [ ! -d "$malan_dir" ]; then
    log "ERROR" "未找到目录: $malan_dir（请确认文件已上传）"
    return 1
  fi

  if [ ! -f "$malan_bin" ]; then
    log "ERROR" "未找到文件: $malan_bin"
    return 1
  fi

  if [ ! -f "$malan_run" ]; then
    log "ERROR" "未找到文件: $malan_run"
    return 1
  fi

  log "INFO" "授予可执行权限: $malan_bin"
  $sudoset chmod 755 "$malan_bin" || { log "ERROR" "chmod 失败: $malan_bin"; return 1; }

  log "INFO" "授予可执行权限: $malan_run"
  $sudoset chmod 755 "$malan_run" || { log "ERROR" "chmod 失败: $malan_run"; return 1; }

  log "INFO" "进入目录并执行: $malan_run"
  $sudoset bash -c "cd '$malan_dir' && ./run.sh" || { log "ERROR" "执行失败: ${malan_dir}/run.sh"; return 1; }

  log "INFO" "malan 启动命令执行完成"
  return 0
}

function getLatestVersion() {
      log "INFO" "检查中间件最新版本信息"
#     判断是否能够访问外网，不能就跳过联网检查
      if ! curl -s --head "https://www.baidu.com" | grep "200 OK" > /dev/null; then
          log "WARN" "无法访问外网，跳过最新版本检查"
          return
      fi
#      todo:如果访问不到，就跳过,后面改成用自己的接口服务
     log "INFO" "准备获取最新版本信息"
     latest_nginx_version=$(curl -s "https://nginx.org/en/download.html" | grep -oP 'nginx-\K[0-9]+\.[0-9]+\.[0-9]+' | head -n 1)
     latest_redis_version=$(curl -s "https://api.github.com/repos/redis/redis/releases/latest" | \
                         grep -m 1 '"tag_name"' | \
                         grep -oP '[0-9]+\.[0-9]+\.[0-9]+' )
     latest_mysql_version=$(curl -s https://repo.mysql.com/yum/mysql-8.0-community/el/7/x86_64/ \
        | grep -oP 'mysql-community-server-[0-9]+\.[0-9]+\.[0-9]+' \
        | sort -V \
        | tail -n 1 \
        | grep -oP '[0-9]+\.[0-9]+\.[0-9]+'
        )
      log "INFO" "最新版本信息获取成功"
      log "INFO" "准备获取本地版本信息"
     current_nginx_version=$(docker exec ujava2  sh -c  "/usr/local/nginx/sbin/nginx -v " 2>&1 | grep -oP 'nginx/\K[0-9\.]+')
     current_redis_version=$(docker exec uredis  sh -c  "redis-server --version " 2>&1 | grep -oP 'Redis server v=\K[0-9\.]+')
     current_mysql_version=$(docker exec umysql  sh -c  "mysql --version " 2>&1 | grep -oP 'mysql  Ver \K[0-9\.]+')
      log "INFO" "本地版本信息获取成功, 准备进行版本比较"
#   比较不同的版本情况，不一样的给出告警提示
    if [ "$latest_nginx_version" != "$current_nginx_version" ]; then
        log "WARN" "Nginx版本不一致，最新版本: $latest_nginx_version, 当前版本: $current_nginx_version"
    else
        log "INFO" "Nginx版本一致，当前版本: $current_nginx_version"
    fi

    if [ "$latest_redis_version" != "$current_redis_version" ]; then
        log "WARN" "Redis版本不一致，最新版本: $latest_redis_version, 当前版本: $current_redis_version"
    else
        log "INFO" "Redis版本一致，当前版本: $current_redis_version"
    fi
    if [ "$latest_mysql_version" != "$current_mysql_version" ]; then
        log "WARN" "MySQL版本不一致，最新版本: $latest_mysql_version, 当前版本: $current_mysql_version"
    else
        log "INFO" "MySQL版本一致，当前版本: $current_mysql_version"
    fi
    log "INFO" "版本检查完成"
}



#------------------------------服务安装-end------------------------------------------------------------------------------------------------------------------------


#------------------------------文件上传&更改ip-start------------------------------------------------------------------------------------------------------------------------
#todo:针对更新对外包，如果存在，需要判断是否更新
#-----------------------------------预定系统--------------------------------------------
function java_files()
{
    log "INFO" "检查预定配置文件 "
    $sudoset ls $install_dir_java | grep java
    if [ $? -eq 0 ]; then
        sleep 1
        log "INFO" "检查到java已上传，准备进行更细致的判断"
        # 进行更细致的判断
        subdirs=("api-java-dubbo" "api-java-meeting2.0" "cert" "external-meeting-api" "nginx-conf.d" "ubains-video-web-3.0" "ubains-web-2.0" "ubains-web-h5")
        all_exist=true

        for subdir in "${subdirs[@]}"; do
            if [ ! -d "$install_dir_java/java/$subdir" ]; then
                echo -e "\033[31m ${install_dir_java}/java/$subdir 文件不存在，准备上传... \033[0m"
                all_exist=false
                break # 找到一个不存在的目录后就可以停止检查
            fi
        done

        if [ "$all_exist" = true ]; then
            log "INFO" "所有必要的子目录都存在，无需上传。"
        else
            log "WARN" "部分必要的子目录不存在，准备上传缺失的文件..."
            $sudoset mkdir -p $install_dir_java/java
            $sudoset cp -rf $auto_java/java $install_dir_java
            $sudoset cp -rf $auto_java/emqx $install_dir_java
            $sudoset cp -rf $auto_java/redis $install_dir_java
            $sudoset cp -rf $auto_java/malan $install_dir_java
        fi
    else
        sleep 1
        log "WARN" "检查到java未上传，准备上传文件..."
        $sudoset mkdir -p $install_dir_java/java
        $sudoset cp -rf $auto_java/java $install_dir_java
        $sudoset cp -rf $auto_java/emqx $install_dir_java
        $sudoset cp -rf $auto_java/redis $install_dir_java
        $sudoset cp -rf $auto_java/malan $install_dir_java
    fi
        log "INFO" "文件上传完毕！正在校验需要修改配置的文件....！"
dj1=$install_dir_java"/java/api-java-meeting2.0/conf"
dj2=$install_dir_java"/java/api-java-meeting2.0/config"
dj3=$install_dir_java"/java/api-java-meeting2.0/update"
dj4=$install_dir_java"/java/external-meeting-api"
dj5=$install_dir_java"/java/nginx-conf.d"
dj6=$install_dir_java"/redis"
dj7=$install_dir_java"/java/ubains-web-2.0"
dj8=$install_dir_java"/java/cert"
dj9=$install_dir_java"/emqx"
fj1=$install_dir_java"/java/api-java-meeting2.0/run.sh"
fj2=$install_dir_java"/java/api-java-meeting2.0/update.sh"
fj3=$install_dir_java"/java/api-java-meeting2.0/ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
fj4=$install_dir_java"/java/start.sh"

dj=("$dj1" "$dj2" "$dj3" "$dj4" "$dj5" "$dj6" "$dj7" "$dj8" "$dj9")
fj=("$fj1" "$fj2" "$fj3" "$fj4")
for j in 0 1 2 3 4 5 6 7 8;
 do
        if [ -d "${dj[j]}" ] ; then
        echo -e "\033[32m ${dj[j]} 文件校验成功！\033[0m"
        else
        echo "${dj[j]} 文件不存在,请重新上传文件再执行脚本 "
        fi
done

for j in 0 1 2 3 ;
 do
        if [ -f "${fj[j]}" ] ; then
        echo -e "\033[32m ${fj[j]} 文件校验成功！\033[0m"
        sleep 1
        else
        echo "${fj[j]} 文件不存在,请重新上传文件再执行脚本 "
        fi
done

#更改配置文件ip
log "INFO" "开始更改配置文件ip"
file1=$install_dir_java'/java/api-java-meeting2.0/conf/fdfs_client.conf'
file2=$install_dir_java'/java/api-java-meeting2.0/config/config.js'
file3=$install_dir_java'/java/api-java-meeting2.0/config/application-prod.properties'
file4=$install_dir_java'/java/api-java-meeting2.0/config/application-mqtt.yml'
file5=$install_dir_java'/java/api-java-meeting2.0/config/application-thirdParty.properties'
file6=$install_dir_java'/java/ubains-web-2.0/static/config.json'
file7=$install_dir_java'/java/nginx-conf.d/meeting443.conf'
file8=$install_dir_java'/java/nginx-conf.d/video8083.conf'
file9=$install_dir_java'/java/nginx-conf.d/h5-8086.conf'
file10=$install_dir_java'/java/external-meeting-api/config/application-prod.properties'
file11=$install_dir_java'/java/external-meeting-api/config/application-mqtt.yml'
file12=$install_dir_java'/java/external-meeting-api/config/application-thirdParty.properties'
file13=$install_dir_java'/java/api-java-dubbo/dingding/config/application-prod.properties'
file14=$install_dir_java'/java/api-java-dubbo/dingding/config/bootstrap.yml'
file15=$install_dir_java'/java/api-java-dubbo/welink/config/application-prod.properties'
file16=$install_dir_java'/java/api-java-dubbo/welink/config/bootstrap.yml'
file=("$file1" "$file2" "$file3" "$file4" "$file5" "$file6" "$file7" "$file8" "$file9" "$file10" "$file11" "$file12" "$file13" "$file14" "$file15" "$file16")
for j in 0 1 2 3 4 5 7 8 9 10 11 12 13 14 15;
 do
        if [ -f "${file[j]}" ] ; then
          log "INFO" "${file[j]} 文件校验成功！"

          sleep 1
        else
          log "ERROR" "${file[j]} 文件不存在！"
        fi
done

log "INFO" "开始修改配置文件中的ip地址"
log "INFO" "准备修改配置文件中的ip地址为$IP"

for i in "${file[@]}";

 do
    $sudoset sed -i "s/$ip/$IP/g"  $i
    echo $IP
    log "INFO" "已修改 $i"
    sleep 1
done

sleep 3
log "INFO" "文件配置修改完毕，正在授予文件可执行权限...！"
$sudoset chmod 755 $install_dir_java/java/start.sh
$sudoset chmod 755 $install_dir_java/java/api-java-meeting2.0/run.sh
$sudoset chmod 755 $install_dir_java/java/api-java-meeting2.0/update.sh
$sudoset chmod 755 $install_dir_java/java/api-java-meeting2.0/script.sh
$sudoset chmod 755 $install_dir_java/java/external-meeting-api/run.sh
$sudoset chmod 755 $install_dir_java/java/external-meeting-api/check.sh
$sudoset chmod 755 $install_dir_java/java/external-meeting-api/startExternalProject.sh
$sudoset chmod 755 $install_dir_java/java/api-java-dubbo/welink/run.sh
$sudoset chmod 755 $install_dir_java/java/api-java-dubbo/dingding/run.sh
$sudoset chmod 755 -R $install_dir_java/emqx/config/*
$sudoset chown -R 1000:1000 $install_dir_java/emqx/data
$sudoset chown -R 1000:1000 $install_dir_java/emqx/log
$sudoset chmod 755 $install_dir_java/redis/redis-6.0.3.conf
$sudoset chmod 755 $install_dir_java/malan/malan
$sudoset chmod 755 $install_dir_java/malan/run.sh
log "INFO" "已授予文件可执行权限！准备进入服务安装...！"
sleep 3
}

#------------------------------文件上传&更改ip-end--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

function check_crontab() { 
    log "INFO" "开始配置数据库备份、删除进程检测、对外服务检测的定时任务.."
    $sudoset mkdir -p /opt/scripts/

    # 从定时脚本目录复制脚本到 /opt/scripts/
    $sudoset cp -rf $auto_java/scripts/* /opt/scripts/
    $sudoset chmod 755 /opt/scripts/*.sh
    
    # 添加定时脚本目录中的脚本定时任务
    # 监控 Redis 服务 - 每5分钟检查一次
    if ! crontab -l 2>/dev/null | grep -q "monitor_redis_service.sh"; then
        log "INFO" "创建监控Redis服务的定时任务..."
        (crontab -l 2>/dev/null; echo "*/5 * * * * /opt/scripts/monitor_redis_service.sh >> /var/log/monitor_redis_service.log 2>&1") | crontab -
    else
        log "INFO" "监控Redis服务的定时任务已存在"
    fi
    
    # 监控 EMQX 服务 - 每5分钟检查一次
    if ! crontab -l 2>/dev/null | grep -q "monitor_emqx_service.sh"; then
        log "INFO" "创建监控EMQX服务的定时任务..."
        (crontab -l 2>/dev/null; echo "*/5 * * * * /opt/scripts/monitor_emqx_service.sh >> /var/log/monitor_emqx_service.log 2>&1") | crontab -
    else
        log "INFO" "监控EMQX服务的定时任务已存在"
    fi
    
    # 监控 MySQL 服务 - 每5分钟检查一次
    if ! crontab -l 2>/dev/null | grep -q "monitor_mysql_service.sh"; then
        log "INFO" "创建监控MySQL服务的定时任务..."
        (crontab -l 2>/dev/null; echo "*/5 * * * * /opt/scripts/monitor_mysql_service.sh >> /var/log/monitor_mysql_service.log 2>&1") | crontab -
    else
        log "INFO" "监控MySQL服务的定时任务已存在"
    fi
    
    # 监控外部API服务 - 每5分钟检查一次
    if ! crontab -l 2>/dev/null | grep -q "monitor_external_api_services.sh"; then
        log "INFO" "创建监控外部API服务的定时任务..."
        (crontab -l 2>/dev/null; echo "*/5 * * * * /opt/scripts/monitor_external_api_services.sh >> /var/log/monitor_external_api_services.log 2>&1") | crontab -
    else
        log "INFO" "监控外部API服务的定时任务已存在"
    fi

    # 监控内部API服务 - 每5分钟检查一次
    if ! crontab -l 2>/dev/null | grep -q "monitor_inner_api_services.sh"; then
        log "INFO" "创建监控内部API服务的定时任务..."
        (crontab -l 2>/dev/null; echo "*/5 * * * * /opt/scripts/monitor_inner_api_services.sh >> /var/log/monitor-inner-api-services.log 2>&1") | crontab -
    else
        log "INFO" "监控内部API服务的定时任务已存在"
    fi
    
    # 数据库备份 - 每天凌晨2点执行
    if ! crontab -l 2>/dev/null | grep -q "backup_mysql_databases.sh"; then
        log "INFO" "创建数据库备份定时任务..."
        (crontab -l 2>/dev/null; echo "0 2 * * * /opt/scripts/backup_mysql_databases.sh >> /var/log/backup_mysql_databases.log 2>&1") | crontab -
    else
        log "INFO" "数据库备份定时任务已存在"
    fi
    
    # 清理已删除文件 - 每天凌晨1点执行
    if ! crontab -l 2>/dev/null | grep -q "cleanup_deleted_files.sh"; then
        log "INFO" "创建清理已删除文件定时任务..."
        (crontab -l 2>/dev/null; echo "0 1 * * * /opt/scripts/cleanup_deleted_files.sh >> /var/log/cleanup_deleted_files.log 2>&1") | crontab -
    else
        log "INFO" "清理已删除文件定时任务已存在"
    fi
    
    # 重启cron服务
    sleep 3
    if command -v systemctl >/dev/null 2>&1; then
        sudo systemctl restart cron 2>/dev/null || sudo systemctl restart crond 2>/dev/null
    else
        sudo service cron restart 2>/dev/null || sudo service crond restart 2>/dev/null
    fi
    
    log "INFO" "定时任务配置成功"
}

#------------------------------全局配置-start--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
###服务器检测配置项
#mem-check
#最小内存大小G
min_memory_value=8
#最小有效内存大小G
min_memory_available=8
#disk-check
#最小硬盘资源大小G
#min_disk_value=200
#最小有效硬盘大小G
min_disk_available=50

#默认为空，如果为非root用户执行，则需要配置为sudo
sudoset=""
#------------------------------全局配置-end--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
#-------------------------------脚本执行-start--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
sys1="ubains.txt"
sys2="devops.txt"
sys3="ubains_devops.txt"
checkOS
if [ -e "$PWD/$sys1" ]; then
  txt="检测该脚本为【预定系统 部署】"
  version="预定系统版本"
elif [ -e "$PWD/$sys2" ]; then
  txt="检测该脚本为【运维系统 部署】"
  version="运维系统版本"
elif [ -e "$PWD/$sys3" ]; then
  txt="检测该脚本为【预定系统+运维系统 部署】"
  version="预定系统+运维系统版本"
fi

log "INFO" "$txt"

#-------------------------------获取服务器ip-start--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
#文件默认ip
ip='192.168.1.141'
IP=$(hostname -I | awk '{print $1}')
function server_ip()
{
	# 弹出弹框，并根据用户选择执行相应操作
	choice=$(whiptail --title "请选择操作" \
					  --menu "请确认服务器ip是否正确:$IP" 15 50 4 \
					  1 "是" \
					  2 "否" \
					  3 "退出" \
					  3>&1 1>&2 2>&3)

	# 根据用户选择执行相应操作
	case $choice in
		1)
			# 在这里编写操作1的代码逻辑
			;;
		2)
			# 设置对话框的标题和提示信息
			dialog_title="请输入服务器IP："
			dialog_prompt="请输入服务器IP："
			# 使用whiptail显示输入框，并将用户输入的值保存到变量input_value
			IP=$(whiptail --title "$dialog_title" --inputbox "$dialog_prompt" 10 50 3>&1 1>&2 2>&3)
			# 在这里编写操作2的代码逻辑
			;;
		3)
			exit
			# 在这里编写操作3的代码逻辑
			;;
		*)
			echo "无效的选择"
			;;
	esac
}
#-------------------------------获取服务器的ip-end-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#-------------------------------部署预定系统-------------------------------
    echo -e "\033[36m

---------------------------------------------开始部署预定系统---------------------------------------------------


    \033[0m"
	    server_ip
        #自动化脚本路径
        auto_java=$PWD
        add_user
        if [ -d "$PWD/java" ]; then
          echo ""
        else
          unzip $PWD/java.zip -d $PWD/
          unzip $PWD/server_bag.zip -d $PWD/
        fi
        #检测文件
        File_detection_java
        rm -rf $PWD/java.zip
        rm -rf $PWD/server_bag.zip
        #开放防火墙
        firewalldjava
        #传输文件更改ip
        install_dir_java="/var/www"
        java_files
        #安装服务
        docker_x86
        mysql_x86
        redis_x86
        emqx_x86
        java_x86
        fastfds_X86
        monitor_x86
        malan_x86
        ntp_x86
        ntp_uos
        getLatestVersion
#___________________________________________________________________________________

log "INFO" "查看服务是否启动成功——————docker ps"
log "INFO" "预定系统服务：umysql,uredis,ujava2，uemqx，ustorage，utracker "

$sudoset docker ps

#------------------------------脚本执行-end------------------------------------------------------------------
#tail -1000f /var/www/java/api-java-meeting2.0/logs/ubains-INFO-AND-ERROR.log