#!/bin/bash

# --- 配置区域 ---
# 日志文件路径
LOG_FILE="/var/log/monitor_external_api_services.log"

# 定义要监控的服务及其相关信息
# 格式: "进程名:目录路径:启动脚本路径"
SERVICES=(
    "ubains-meeting-api-1.0-SNAPSHOT.jar:/var/www/java/external-meeting-api:/var/www/java/external-meeting-api/run.sh"
    "malan:/var/www/malan:/var/www/malan/run.sh"
)

# --- 函数定义 ---

# 记录日志的函数
log_message() {
    local message="$1"
    echo "$(date '+%Y-%m-%d %H:%M:%S') - $message" >> "$LOG_FILE"
}

# 检查进程是否运行
is_process_running() {
    local process_name="$1"
    # 使用 pgrep -f 搜索命令行中包含该名称的进程
    # 对于 .jar 文件，通常搜索的是 java -jar ...<jar_name>
    # 对于其他程序，搜索其进程名
    if [[ "$process_name" == *.jar ]]; then
        # 如果是 JAR 文件，搜索 java 进程中包含该 jar 名称的
        pgrep -f "java.*$process_name" > /dev/null
    else
        # 否则直接搜索进程名
        pgrep -x "$process_name" > /dev/null
    fi
    
    if [ $? -eq 0 ]; then
        return 0 # 进程运行中
    else
        return 1 # 进程未运行
    fi
}

# 检查目录是否存在
is_directory_exists() {
    local dir_path="$1"
    [ -d "$dir_path" ]
}

# 检查启动脚本是否存在且可执行
is_script_executable() {
    local script_path="$1"
    [ -x "$script_path" ]
}

# 启动服务
start_service() {
    local service_name="$1"
    local dir_path="$2"
    local script_path="$3"

    log_message "尝试启动服务 '$service_name'..."

    # 切换到服务目录并执行启动脚本
    # 使用子shell (cd ...) 以确保不影响当前脚本的工作目录
    (
        cd "$dir_path" || { log_message "错误: 无法切换到目录 '$dir_path'"; return 1; }
        
        # 检查启动脚本是否存在且可执行
        if ! is_script_executable "$script_path"; then
             log_message "错误: 启动脚本 '$script_path' 不存在或不可执行。"
             return 1
        fi

        # 执行启动脚本
        # 使用 nohup 将进程与终端脱离，使其在后台持续运行
        # 输出重定向到 startup.log 或 /dev/null
        nohup "./$(basename "$script_path")" > startup.log 2>&1 &
        
        # 等待一小段时间，让进程有机会启动
        sleep 3
        
        # 再次检查进程是否启动成功
        if is_process_running "$service_name"; then
            log_message "服务 '$service_name' 启动成功。"
            return 0
        else
            log_message "错误: 服务 '$service_name' 启动失败。"
            return 1
        fi
    )
}

# --- 主逻辑 ---

# 初始化日志
log_message "=== 服务监控脚本开始执行 ==="

# 遍历所有服务
for service_info in "${SERVICES[@]}"; do
    # 解析服务信息
    IFS=':' read -r service_name dir_path script_path <<< "$service_info"
    # 去除可能的前后空格
    service_name=$(echo "$service_name" | xargs)
    dir_path=$(echo "$dir_path" | xargs)
    script_path=$(echo "$script_path" | xargs)

    log_message "检查服务: $service_name (目录: $dir_path, 启动脚本: $script_path)"

    # 检查目录是否存在
    if ! is_directory_exists "$dir_path"; then
        log_message "跳过服务 '$service_name': 目录 '$dir_path' 不存在。"
        continue # 跳过当前循环，检查下一个服务
    fi

    # 检查进程是否运行
    if is_process_running "$service_name"; then
        log_message "服务 '$service_name' 正在运行。"
    else
        log_message "服务 '$service_name' 未运行。"
        # 尝试启动服务
        start_service "$service_name" "$dir_path" "$script_path"
    fi
done

log_message "=== 服务监控脚本执行完毕 ==="

