#!/bin/bash

# 宿主机上的脚本：检查 Redis，如果容器未运行则重启 uredis 容器
LOG_FILE="/var/log/monitor_redis_service.log"

log() {
    local message
    message="$(date '+%Y-%m-%d %H:%M:%S') - $1"
    echo "$message"
    # 同时写入日志文件
    echo "$message" >> "$LOG_FILE"
}

# 直接在脚本中定义 Redis 密码
REDIS_PASSWORD="dNrprU&2S"

check_redis() {
    # 检查 uredis 容器是否正在运行
    if docker ps --format '{{.Names}}' | grep -Fxq "uredis"; then
        # 首先尝试不使用密码的 ping 命令
        if docker exec uredis redis-cli ping >/dev/null 2>&1; then
            # 无密码情况下能 ping 通
            log "Redis 服务状态正常"
            return 0
        else
            # 检查是否是认证问题
            auth_output=$(docker exec uredis redis-cli ping 2>&1)
            if [[ "$auth_output" == *"NOAUTH"* ]] || [[ "$auth_output" == *"Authentication required"* ]]; then
                log "检测到需要认证的 Redis 服务，正在尝试使用密码..."
                
                # 使用硬编码的密码进行认证测试
                if docker exec uredis redis-cli -a "$REDIS_PASSWORD" --no-auth-warning ping >/dev/null 2>&1; then
                    ping_output=$(docker exec uredis redis-cli -a "$REDIS_PASSWORD" --no-auth-warning ping 2>&1)
                    if [ "$ping_output" = "PONG" ]; then
                        log "Redis 服务状态正常（已通过密码认证）"
                        return 0
                    else
                        log "警告: 使用密码认证成功，但 ping 返回意外状态: $ping_output"
                        return 1
                    fi
                else
                    log "警告: 需要认证，但无法使用预设密码连接到 Redis"
                    return 1
                fi
            else
                # 不是认证问题，可能是其他错误
                log "警告: Redis 连接失败，错误信息: $auth_output"
                return 1
            fi
        fi
    else
        log "信息: uredis 容器未运行"
        return 1
    fi
}

restart_redis_container() {
    log "Redis 未运行。正在尝试重启容器 'uredis'..."

    # 检查容器是否存在（不仅仅是运行状态）
    if docker ps -a --format '{{.Names}}' | grep -Fxq "uredis"; then
        # 停止可能挂起的容器
        docker stop uredis >/dev/null 2>&1 || true
        
        # 容器存在但未运行，尝试启动
        if docker start uredis; then
            log "成功: Redis 容器已成功启动。"
            
            # 等待几秒让服务启动
            log "信息: 等待 20 秒让 Redis 服务完全启动..."
            for i in {1..20}; do
                echo -n "." >> "$LOG_FILE"
                sleep 1
            done
            echo "" >> "$LOG_FILE"
            
            # 检查启动后容器是否仍在运行
            if docker ps --format '{{.Names}}' | grep -Fxq "uredis"; then
                log "信息: Redis 容器现在正在运行。"
                
                # 等待一段时间后再次检查服务状态
                log "信息: 等待 10 秒后检查服务状态..."
                sleep 10
                
                # 再次检查是否需要认证
                if docker exec uredis redis-cli ping >/dev/null 2>&1; then
                    # 不需要认证
                    if docker exec uredis redis-cli ping >/dev/null 2>&1; then
                        ping_output=$(docker exec uredis redis-cli ping 2>&1)
                        if [ "$ping_output" = "PONG" ]; then
                            log "信息: Redis 服务状态正常。"
                            return 0
                        else
                            log "错误: Redis 服务状态异常: $ping_output"
                            return 1
                        fi
                    else
                        log "错误: Redis 服务状态检查失败。"
                        return 1
                    fi
                else
                    # 需要认证，尝试使用密码
                    auth_output=$(docker exec uredis redis-cli ping 2>&1)
                    if [[ "$auth_output" == *"NOAUTH"* ]] || [[ "$auth_output" == *"Authentication required"* ]]; then
                        # 使用硬编码的密码进行认证测试
                        if docker exec uredis redis-cli -a "$REDIS_PASSWORD" --no-auth-warning ping >/dev/null 2>&1; then
                            ping_output=$(docker exec uredis redis-cli -a "$REDIS_PASSWORD" --no-auth-warning ping 2>&1)
                            if [ "$ping_output" = "PONG" ]; then
                                log "信息: Redis 服务状态正常（已通过密码认证）。"
                                return 0
                            else
                                log "错误: Redis 服务状态异常: $ping_output"
                                return 1
                            fi
                        else
                            log "错误: Redis 需要认证，但无法使用预设密码连接到 Redis。"
                            return 1
                        fi
                    else
                        log "错误: Redis 服务状态检查失败。"
                        return 1
                    fi
                fi
            else
                log "错误: Redis 容器在启动后不久就停止了。"
                return 1
            fi
        else
            log "错误: 无法启动 Redis 容器。"
            
            # 尝试清理数据目录并重启
            log "信息: 尝试清理数据目录并重新启动 Redis 容器..."
            
            # 检查并删除可能存在的数据目录
            if [ -d "/var/www/java/redis/data" ]; then
                log "信息: 清理 /var/www/java/redis/data 目录..."
                rm -rf /var/www/java/redis/data/*
            fi
            
            if [ -d "/var/www/redis/data" ]; then
                log "信息: 清理 /var/www/redis/data 目录..."
                rm -rf /var/www/redis/data/*
            fi
            
            # 再次尝试启动容器
            log "信息: 再次尝试启动 Redis 容器..."
            if docker start uredis; then
                log "成功: Redis 容器在清理数据后已成功启动。"
                
                # 等待几秒让服务启动
                log "信息: 等待 20 秒让 Redis 服务完全启动..."
                for i in {1..20}; do
                    echo -n "." >> "$LOG_FILE"
                    sleep 1
                done
                echo "" >> "$LOG_FILE"
                
                # 检查启动后容器是否仍在运行
                if docker ps --format '{{.Names}}' | grep -Fxq "uredis"; then
                    log "信息: Redis 容器现在正在运行。"
                    
                    # 等待一段时间后再次检查服务状态
                    log "信息: 等待 10 秒后检查服务状态..."
                    sleep 10
                    
                    # 再次检查是否需要认证
                    if docker exec uredis redis-cli ping >/dev/null 2>&1; then
                        # 不需要认证
                        if docker exec uredis redis-cli ping >/dev/null 2>&1; then
                            ping_output=$(docker exec uredis redis-cli ping 2>&1)
                            if [ "$ping_output" = "PONG" ]; then
                                log "信息: Redis 服务状态正常。"
                                return 0
                            else
                                log "错误: Redis 服务状态异常: $ping_output"
                                return 1
                            fi
                        else
                            log "错误: Redis 服务状态检查失败。"
                            return 1
                        fi
                    else
                        # 需要认证，尝试使用密码
                        auth_output=$(docker exec uredis redis-cli ping 2>&1)
                        if [[ "$auth_output" == *"NOAUTH"* ]] || [[ "$auth_output" == *"Authentication required"* ]]; then
                            # 使用硬编码的密码进行认证测试
                            if docker exec uredis redis-cli -a "$REDIS_PASSWORD" --no-auth-warning ping >/dev/null 2>&1; then
                                ping_output=$(docker exec uredis redis-cli -a "$REDIS_PASSWORD" --no-auth-warning ping 2>&1)
                                if [ "$ping_output" = "PONG" ]; then
                                    log "信息: Redis 服务状态正常（已通过密码认证）。"
                                    return 0
                                else
                                    log "错误: Redis 服务状态异常: $ping_output"
                                    return 1
                                fi
                            else
                                log "错误: Redis 需要认证，但无法使用预设密码连接到 Redis。"
                                return 1
                            fi
                        else
                            log "错误: Redis 服务状态检查失败。"
                            return 1
                        fi
                    fi
                else
                    log "错误: Redis 容器在启动后不久就停止了。"
                    return 1
                fi
            else
                log "错误: 即使清理了数据目录，仍然无法启动 Redis 容器。"
                return 1
            fi
        fi
    else
        log "错误: 容器 'uredis' 不存在！"
        log "信息: 您可能需要手动重新创建 Redis 容器。"
        return 1
    fi
}

# 主逻辑
log "开始检查 Redis 服务状态..."
if check_redis; then
    log "Redis 正在运行且状态正常。"
else
    log "Redis 无响应或容器未运行。"
    restart_redis_container
    
    # 检查重启后是否正常工作
    if check_redis; then
        log "Redis 已成功重启，现在状态正常。"
    else
        log "错误: 重启尝试后 Redis 仍无响应。"
        log "信息: 请使用以下命令检查容器日志: docker logs uredis"
    fi
fi