#!/bin/bash

# 宿主机上的脚本：检查 MySQL，如果容器未运行则重启 umysql 容器
LOG_FILE="/var/log/monitor_mysql_service.log"

log() {
    local message
    message="$(date '+%Y-%m-%d %H:%M:%S') - $1"
    echo "$message"
    # 同时写入日志文件
    echo "$message" >> "$LOG_FILE"
}

get_mysql_password() {
    docker exec umysql printenv MYSQL_ROOT_PASSWORD 2>/dev/null
}

check_mysql() {
    if docker ps --format '{{.Names}}' | grep -Fxq "umysql"; then
        local mysql_password
        mysql_password=$(get_mysql_password)
        
        if [ -z "$mysql_password" ]; then
            return 1
        fi
        
        # 尝试连接数据库
        if docker exec umysql mysqladmin ping -h localhost -u root -p"$mysql_password" 2>/dev/null; then
            # 尝试执行一个简单的SQL查询来确认数据库功能正常
            if docker exec umysql mysql -u root -p"$mysql_password" -e "SELECT 1;" >/dev/null 2>&1; then
                log "MySQL 服务连接正常，可以执行SQL查询"
                return 0
            else
                log "警告: MySQL 服务可以ping通，但无法执行SQL查询"
                return 1
            fi
        else
            log "警告: 无法连接到 MySQL 服务"
            return 1
        fi
    else
        log "信息: umysql 容器未运行"
        return 1
    fi
}

restart_mysql_container() {
    log "MySQL 未运行。正在尝试重启容器 'umysql'..."

    # 检查容器是否存在（不仅仅是运行状态）
    if docker ps -a --format '{{.Names}}' | grep -Fxq "umysql"; then
        # 停止可能挂起的容器
        docker stop umysql >/dev/null 2>&1 || true
        
        # 容器存在但未运行，尝试启动
        if docker start umysql; then
            log "成功: MySQL 容器已成功启动。"
            
            # 等待几秒让服务启动
            log "信息: 等待 60 秒让 MySQL 服务完全启动..."
            for i in {1..60}; do
                echo -n "." >> "$LOG_FILE"
                sleep 1
            done
            echo "" >> "$LOG_FILE"
            
            # 检查启动后容器是否仍在运行
            if docker ps --format '{{.Names}}' | grep -Fxq "umysql"; then
                log "信息: MySQL 容器现在正在运行。"
                
                # 等待一段时间后检查服务状态
                log "信息: 等待 20 秒后检查服务状态..."
                sleep 60
                
                local mysql_password
                mysql_password=$(get_mysql_password)
                
                if [ -z "$mysql_password" ]; then
                    return 1
                fi
                
                if docker exec umysql mysqladmin ping -h localhost -u root -p"$mysql_password" >/dev/null 2>&1; then
                    if docker exec umysql mysql -u root -p"$mysql_password" -e "SELECT 1;" >/dev/null 2>&1; then
                        log "信息: MySQL 服务状态正常。"
                        return 0
                    else
                        log "错误: MySQL 服务运行中，但无法执行SQL查询。"
                        return 1
                    fi
                else
                    log "错误: MySQL 服务状态检查失败。"
                    return 1
                fi
            else
                log "错误: MySQL 容器在启动后不久就停止了。"
                return 1
            fi
        else
            log "错误: 无法启动 MySQL 容器。"
            return 1
        fi
    else
        log "错误: 容器 'umysql' 不存在！"
        log "信息: 您可能需要手动重新创建 MySQL 容器。"
        return 1
    fi
}

# 主逻辑
log "开始检查 MySQL 服务状态..."
if check_mysql; then
    log "MySQL 正在运行且状态正常。"
else
    log "MySQL 无响应或容器未运行。"
    restart_mysql_container
    
    # 检查重启后是否正常工作
    if check_mysql; then
        log "MySQL 已成功重启，现在状态正常。"
    else
        log "错误: 重启尝试后 MySQL 仍无响应。"
        log "信息: 请使用以下命令检查容器日志: docker logs umysql"
    fi
fi