#!/bin/bash

LOG_FILE="/data/logs/ujava2-service-manager.log"
unacos_LOG="/data/logs/unacos_restart.log"
timestamp() {
  date "+%Y-%m-%d %H:%M:%S"
}

declare -A SERVICE_CMD_MAP=(
  ["auth"]="ubains-auth.jar"
  ["gateway"]="ubains-gateway.jar"
  ["system"]="ubains-modules-system.jar"
  ["meeting2.0"]="java-meeting2.0/ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
  ["meeting3.0"]="java-meeting3.0/ubains-meeting-inner-api-1.0-SNAPSHOT.jar"
  ["mqtt"]="ubains-meeting-mqtt-1.0-SNAPSHOT.jar"
  ["quartz"]="ubains-meeting-quartz-1.0-SNAPSHOT.jar"
  ["message"]="ubains-meeting-message-scheduling-1.0-SNAPSHOT.jar"
)

declare -A SERVICE_SCRIPT_MAP=(
  ["auth"]="/var/www/java/api/auth/auth-sso-auth/run.sh"
  ["gateway"]="/var/www/java/api/auth/auth-sso-gatway/run.sh"
  ["system"]="/var/www/java/api/auth/auth-sso-system/run.sh"
  ["meeting2.0"]="/var/www/java/api/java-meeting/java-meeting2.0/run.sh"
  ["meeting3.0"]="/var/www/java/api/java-meeting/java-meeting3.0/run.sh"
  ["mqtt"]="/var/www/java/api/java-meeting/java-mqtt/run.sh"
  ["quartz"]="/var/www/java/api/java-meeting/java-quartz/run.sh"
  ["message"]="/var/www/java/api/java-meeting/java-message-scheduling/run.sh"
)

# 平台API检查函数（通用方法，不依赖jq）
check_platform_api() {
  local api_url="https://192.168.5.44/platform/api/code"
  local http_code
  local response
  
  # 获取HTTP状态码和响应内容
  response=$(curl -s -k -m 10 -w "\n%{http_code}" "$api_url" 2>/dev/null)
  http_code=$(echo "$response" | tail -1)
  response=$(echo "$response" | sed '$d')
  
  # 首先检查HTTP状态码
  if [ "$http_code" != "200" ]; then
    echo "$(timestamp) [ERROR] 平台API检查失败: HTTP状态码 $http_code" | tee -a "$LOG_FILE"
    return 1
  fi
  
  # 检查响应是否为空
  if [ -z "$response" ]; then
    echo "$(timestamp) [ERROR] 平台API检查失败: 响应为空" | tee -a "$LOG_FILE"
    return 1
  fi
  
  # 使用grep检查JSON响应中的关键字段（所有Linux系统都有grep）
  # 检查是否包含 "操作成功" 和 "code":200 或 "code": 200
  if echo "$response" | grep -q '"操作成功"' && \
     echo "$response" | grep -qE '"code"[[:space:]]*:[[:space:]]*200'; then
    echo "$(timestamp) [INFO] 平台API检查正常" | tee -a "$LOG_FILE"
    return 0
  else
    echo "$(timestamp) [ERROR] 平台API检查失败: ${response:0:200}..." | tee -a "$LOG_FILE"
    return 1
  fi
}


# 修改后的重启关键服务函数
restart_critical_services() {
  echo "$(timestamp) [WARNING] 开始强制重启gateway、auth、system服务" | tee -a "$LOG_FILE"
  
  local services=("gateway" "auth" "system")
  local restarted=0
  
  for service in "${services[@]}"; do
    # 无论服务是否运行，都先停止再启动
    stop_service "$service"
    if start_service "$service"; then
      ((restarted++))
    fi
  done
  
  echo "$(timestamp) [SUMMARY] 已强制重启 $restarted 个关键服务" | tee -a "$LOG_FILE"
}

# 原有unacos健康检查函数
check_unacos_health() {
  # 检查unacos容器是否运行
  if ! docker inspect unacos2 &>/dev/null; then
    echo "$(timestamp) [ERROR] unacos2容器未找到" | tee -a "$unacos_LOG"
    return 1
  fi

  # 检查unacos容器是否处于运行状态
  if [ "$(docker inspect -f '{{.State.Status}}' unacos2)" != "running" ]; then
    echo "$(timestamp) [ERROR] unacos2容器未运行" | tee -a "$unacos_LOG"
    return 1
  fi

  # 简单检查unacos API是否可访问
  if ! curl -s -m 5 "http://localhost:8848/nacos/" &>/dev/null; then
    echo "$(timestamp) [WARNING] unacos2服务不可访问" | tee -a "$unacos_LOG"
    return 1
  fi

  return 0
}

get_service_pid() {
  local service_key="$1"
  local jar_path="${SERVICE_CMD_MAP[$service_key]}"
  
  # 通过工作目录和jar文件名组合查找
  case "$service_key" in
    "meeting2.0")
     docker exec ujava2 sh -c "ps aux | grep '[j]ava' | grep -F 'ubains-meeting-inner-api' | grep -F 'java-meeting2.0/config' | grep -v 'java-meeting3.0' | awk '{print \$2}' | head -1"
     ;;
    "meeting3.0")
      docker exec ujava2 sh -c "ps aux | grep '[j]ava' | grep -F 'ubains-meeting-inner-api' | grep -F 'java-meeting3.0//config' | awk '{print \$2}' | head -1"
      ;;
    *)
      docker exec ujava2 sh -c "ps aux | grep -F '$jar_path' | grep -v grep | awk '{print \$2}' | head -1"
      ;;
  esac
}

is_service_running() {
  local pid=$(get_service_pid "$1")
  [ -n "$pid" ] && docker exec ujava2 sh -c "ps -p $pid >/dev/null 2>&1"
}

start_service() {
  local service_key="$1"
  echo "$(timestamp) [START] $service_key" | tee -a "$LOG_FILE"
  
  stop_service "$service_key"
  
  docker exec -d ujava2 bash -c "nohup ${SERVICE_SCRIPT_MAP[$service_key]} > /var/log/${service_key}.log 2>&1 & echo \$! > /tmp/${service_key}.pid"
  
  local timeout=30
  while ((timeout-- > 0)); do
    if is_service_running "$service_key"; then
      local pid=$(get_service_pid "$service_key")
      echo "$(timestamp) [SUCCESS] $service_key (PID: $pid)" | tee -a "$LOG_FILE"
      return 0
    fi
    sleep 1
  done
  
  echo "$(timestamp) [FAILED] $service_key 启动超时" | tee -a "$LOG_FILE"
  return 1
}

stop_service() {
  local service_key="$1"
  local pid=$(get_service_pid "$service_key")
  
  [ -z "$pid" ] && return 0
  
  echo "$(timestamp) [STOP] $service_key (PID: $pid)" | tee -a "$LOG_FILE"
  docker exec ujava2 sh -c "kill $pid && sleep 1 || kill -9 $pid"
  sleep 2
}

restart_unacos_if_needed() {
  local stopped_count=$1
  
  # 先检查unacos健康状态
  if ! check_unacos_health; then
    echo "$(timestamp) [WARNING] unacos2服务异常，将尝试重启" | tee -a "$unacos_LOG"
    docker restart unacos2 >> "$unacos_LOG" 2>&1
    
    if [ $? -eq 0 ]; then
      echo "$(timestamp) [SUCCESS] unacos2容器重启成功" | tee -a "$unacos_LOG"
      sleep 30
    else
      echo "$(timestamp) [ERROR] unacos2容器重启失败" | tee -a "$unacos_LOG"
    fi
    return
  fi
  
  # 原有逻辑：当停止的服务超过3个时重启unacos
  if [ "$stopped_count" -gt 3 ]; then
    echo "$(timestamp) [WARNING] 检测到 $stopped_count 个服务停止，将重启unacos2容器" | tee -a "$unacos_LOG"
    docker restart unacos2 >> "$unacos_LOG" 2>&1
    
    if [ $? -eq 0 ]; then
      echo "$(timestamp) [SUCCESS] unacos2容器重启成功" | tee -a "$unacos_LOG"
      sleep 30
    else
      echo "$(timestamp) [ERROR] unacos2容器重启失败" | tee -a "$unacos_LOG"
    fi
  fi
}

check_services() {
  local problems=0
  
  echo "===== $(timestamp) 服务状态检查 =====" | tee -a "$LOG_FILE"
  
  # 首先检查unacos状态
  if ! check_unacos_health; then
    echo "$(timestamp) [WARNING] unacos2服务异常，可能会影响其他服务" | tee -a "$LOG_FILE"
  fi
  
  for service_key in "${!SERVICE_CMD_MAP[@]}"; do
    if is_service_running "$service_key"; then
      local pid=$(get_service_pid "$service_key")
      echo "$(timestamp) [RUNNING] $service_key (PID: $pid)" | tee -a "$LOG_FILE"
    else
      echo "$(timestamp) [STOPPED] $service_key" | tee -a "$LOG_FILE"
      ((problems++))
    fi
  done
  
  restart_unacos_if_needed $problems
  
  return $problems
}

restart_problem_services() {
  local restarted=0
  
  echo "===== $(timestamp) 服务恢复启动 =====" | tee -a "$LOG_FILE"
  
  # 检查unacos是否正常，如果不正常先重启unacos
  if ! check_unacos_health; then
    echo "$(timestamp) [WARNING] unacos2服务异常，将先重启unacos2" | tee -a "$LOG_FILE"
    docker restart unacos2 >> "$unacos_LOG" 2>&1
    sleep 30
  fi
  
  for service_key in "${!SERVICE_CMD_MAP[@]}"; do
    if ! is_service_running "$service_key"; then
      if start_service "$service_key"; then
        ((restarted++))
      fi
    fi
  done
  
  echo "$(timestamp) [SUMMARY] 已尝试重启 $restarted 个服务" | tee -a "$LOG_FILE"
}

# ========== 宿主机对外服务监测函数（新增） ==========
# 获取宿主机对外服务的PID
get_host_extapi_pid() {
  ps aux | grep -F 'ubains-meeting-api-1.0-SNAPSHOT.jar' | grep -v grep | awk '{print $2}' | head -1
}

# 检查宿主机对外服务是否运行
is_host_extapi_running() {
  local pid=$(get_host_extapi_pid)
  [ -n "$pid" ] && ps -p "$pid" >/dev/null 2>&1
}

# 检查宿主机对外服务状态
check_host_extapi_service() {
  local service_name="extapi"
  local service_path="/data/services/api/java-meeting/java-meeting-extapi"
  local jar_file="ubains-meeting-api-1.0-SNAPSHOT.jar"
  
  echo "===== $(timestamp) 宿主机对外服务状态检查 =====" | tee -a "$LOG_FILE"
  
  if is_host_extapi_running; then
    local pid=$(get_host_extapi_pid)
    echo "$(timestamp) [RUNNING] $service_name (PID: $pid)" | tee -a "$LOG_FILE"
    return 0
  else
    echo "$(timestamp) [STOPPED] $service_name" | tee -a "$LOG_FILE"
    return 1
  fi
}

# 重启宿主机对外服务
restart_host_extapi_service() {
  local service_name="extapi"
  local service_path="/data/services/api/java-meeting/java-meeting-extapi"
  local jar_file="ubains-meeting-api-1.0-SNAPSHOT.jar"
  local run_script="${service_path}/run.sh"
  
  echo "$(timestamp) [START] 宿主机服务 $service_name" | tee -a "$LOG_FILE"
  
  # 停止服务
  local pid=$(get_host_extapi_pid)
  if [ -n "$pid" ]; then
    echo "$(timestamp) [STOP] $service_name (PID: $pid)" | tee -a "$LOG_FILE"
    kill "$pid" 2>/dev/null && sleep 1 || kill -9 "$pid" 2>/dev/null
    sleep 2
  fi
  
  # 检查启动脚本是否存在
  if [ ! -f "$run_script" ]; then
    echo "$(timestamp) [ERROR] 启动脚本不存在: $run_script" | tee -a "$LOG_FILE"
    return 1
  fi
  
  # 启动服务
  cd "$service_path" || {
    echo "$(timestamp) [ERROR] 无法进入服务目录: $service_path" | tee -a "$LOG_FILE"
    return 1
  }
  
  # 加载 /etc/profile 中的环境变量（包括 Java 环境变量），然后执行 run.sh
  # 使用 bash -c 确保环境变量在同一 shell 中生效并传递到 run.sh
  bash -c "source /etc/profile && ./run.sh" &
  local start_pid=$!
  
  # 等待服务启动
  local timeout=30
  while ((timeout-- > 0)); do
    if is_host_extapi_running; then
      local pid=$(get_host_extapi_pid)
      echo "$(timestamp) [SUCCESS] $service_name (PID: $pid)" | tee -a "$LOG_FILE"
      return 0
    fi
    sleep 1
  done
  
  echo "$(timestamp) [FAILED] $service_name 启动超时" | tee -a "$LOG_FILE"
  return 1
}

# ========== 6060端口监测函数（新增） ==========
# 检查6060端口是否在监听
check_port_6060() {
  if netstat -tuln 2>/dev/null | grep -q ':6060 ' || \
     ss -tuln 2>/dev/null | grep -q ':6060 ' || \
     lsof -i:6060 >/dev/null 2>&1; then
    return 0  # 端口正在监听
  else
    return 1  # 端口未监听
  fi
}

# 启动malan服务
start_malan_service() {
  local malan_path="/data/middleware/monitor"
  local service_name="malan"
  
  echo "$(timestamp) [START] 启动 $service_name 服务" | tee -a "$LOG_FILE"
  
  # 检查目录是否存在
  if [ ! -d "$malan_path" ]; then
    echo "$(timestamp) [ERROR] malan目录不存在: $malan_path" | tee -a "$LOG_FILE"
    return 1
  fi
  
  # 进入目录并启动malan（后台运行）
  cd "$malan_path" || {
    echo "$(timestamp) [ERROR] 无法进入目录: $malan_path" | tee -a "$LOG_FILE"
    return 1
  }
  
  # 加载环境变量，然后启动malan
  bash -c "source /etc/profile && ./malan" &
  local start_pid=$!
  
  # 等待端口启动
  local timeout=30
  local waited=0
  while ((timeout-- > 0)); do
    if check_port_6060; then
      echo "$(timestamp) [SUCCESS] $service_name 启动成功，6060端口已监听" | tee -a "$LOG_FILE"
      return 0
    fi
    ((waited++))
    sleep 1
  done
  
  echo "$(timestamp) [FAILED] $service_name 启动超时，6060端口未监听" | tee -a "$LOG_FILE"
  return 1
}

# 检查并启动malan服务（如果需要）
check_and_start_malan() {
  echo "===== $(timestamp) 6060端口状态检查 =====" | tee -a "$LOG_FILE"
  
  if check_port_6060; then
    echo "$(timestamp) [INFO] 6060端口正在监听" | tee -a "$LOG_FILE"
    return 0
  else
    echo "$(timestamp) [WARNING] 6060端口未监听，将启动malan服务" | tee -a "$LOG_FILE"
    start_malan_service
    return $?
  fi
}

# 主流程
# 先检查平台API
if ! check_platform_api; then
  restart_critical_services
fi

# 原有服务检查逻辑
check_services || restart_problem_services

# 宿主机对外服务检查（新增）
if ! check_host_extapi_service; then
  restart_host_extapi_service
fi

# 6060端口监测（新增）
check_and_start_malan

echo "===== $(timestamp) 操作完成 =====" | tee -a "$LOG_FILE"
