#!/bin/bash

#------------------------------工具类模块---------------------------------------------------
LOG_FILE="/var/log/new_auto_script.log"
mkdir -p "$(dirname "$LOG_FILE")" 2>/dev/null

function log() {
    local timestamp=$(date "+%Y-%m-%d %H:%M:%S")
    local INFO_COLOR="\033[32m"
    local WARN_COLOR="\033[33m"
    local ERROR_COLOR="\033[31m"
    local DEBUG_COLOR="\033[36m"
    local RESET_COLOR="\033[0m"
    
    if [ $# -lt 2 ]; then
        echo "Usage: log <level> <message>"
        return 1
    fi
    
    local level=$1
    local message=$2
    
    case $level in
        "INFO") color=$INFO_COLOR ;;
        "WARN") color=$WARN_COLOR ;;
        "ERROR") color=$ERROR_COLOR ;;
        "DEBUG") color=$DEBUG_COLOR ;;
        *) echo "Invalid log level: $level"; return 1 ;;
    esac
    
    echo -e "${color}[$timestamp] [$level] $message${RESET_COLOR}"
    echo "[$timestamp] [$level] $message" >> "$LOG_FILE"
}

# ===========================================
# 脚本: replace_ip_interactive.sh
# 功能: 扫描配置文件，查找并替换IP（完全自动化版本）
# ===========================================

# --- 配置区域 ---
TARGET_DIR="/data"
BACKUP_DIR="/data/temp"
OLD_IP_INPUT="192.168.9.84"

# 检查命令行参数
if [ $# -ge 1 ]; then
    NEW_IP_INPUT="$1"
    log "INFO" "使用传入的IP地址: $NEW_IP_INPUT"
else
    # 自动获取本机IP
    NEW_IP_INPUT=$(hostname -I | awk '{print $1}' | cut -d' ' -f1)
    if [[ -z "$NEW_IP_INPUT" ]]; then
        NEW_IP_INPUT=$(ip addr show | grep -E 'inet\s+(192\.168|172\.1[6789]\.|172\.2[0-9]\.|172\.3[01]\.|10\.)' | awk '{print $2}' | cut -d/ -f1 | head -n1)
        if [[ -z "$NEW_IP_INPUT" ]]; then
            log "ERROR" "无法自动获取IP地址，请手动指定"
            exit 1
        fi
    fi
    log "INFO" "自动获取IP地址: $NEW_IP_INPUT"
fi

# 验证IP地址格式
if ! [[ $NEW_IP_INPUT =~ ^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$ ]]; then
    log "ERROR" "无效的IP地址格式: $NEW_IP_INPUT"
    exit 1
fi

# 支持的配置文件扩展名
ALLOWED_EXTENSIONS=(
    "\.js$"
    "\.yml$"
    "\.yaml$"
    "\.json$"
    "\.properties$"
    "\.conf$"
    "\.config$"
    "\.ini$"
    "\.env$"
    "\.py$"
    "\.xml$"
    "\.txt$"
    "\.sh$"
    "\.cnf$"
)

EXT_REGEX=$(IFS='|'; echo "${ALLOWED_EXTENSIONS[*]}")

# 检查目标目录
if [[ ! -d "$TARGET_DIR" ]]; then
    log "ERROR" "目录 '$TARGET_DIR' 不存在！"
    exit 1
fi

# 确保备份目录存在
if [[ ! -d "$BACKUP_DIR" ]]; then
    log "INFO" "备份目录 $BACKUP_DIR 不存在，正在创建..."
    mkdir -p "$BACKUP_DIR" || {
        log "ERROR" "无法创建备份目录 $BACKUP_DIR，请检查权限"
        exit 1
    }
    log "INFO" "已创建 $BACKUP_DIR"
fi

# 转义旧IP用于sed
OLD_IP_ESCAPED=$(echo "$OLD_IP_INPUT" | sed 's/\./\\./g')

log "INFO" "开始自动化IP替换流程"
log "INFO" "扫描目录: $TARGET_DIR"
log "INFO" "查找旧IP: $OLD_IP_INPUT"
log "INFO" "替换为新IP: $NEW_IP_INPUT"

# 临时文件记录匹配的文件路径
MATCHED_FILES="/tmp/matched_files.$$"
> "$MATCHED_FILES"

# 使用 find 查找文件并处理
log "INFO" "正在扫描配置文件..."
find "$TARGET_DIR" -type f -regextype posix-extended -regex ".*($EXT_REGEX)" -print0 2>/dev/null | \
while IFS= read -r -d '' file; do
    [[ -L "$file" ]] && continue
    if ! file -b "$file" 2>/dev/null | grep -qi "text"; then
        continue
    fi

    if grep -Fq "$OLD_IP_INPUT" "$file" 2>/dev/null; then
        log "INFO" "发现匹配文件: $file"
        echo "$file" >> "$MATCHED_FILES"
    fi
done

# 检查是否有匹配的文件
if [[ ! -s "$MATCHED_FILES" ]]; then
    log "INFO" "在指定类型的配置文件中未找到 '$OLD_IP_INPUT'。"
    rm -f "$MATCHED_FILES"
    exit 0
fi

MATCH_COUNT=$(wc -l < "$MATCHED_FILES")
log "INFO" "扫描完成，共找到 $MATCH_COUNT 个配置文件包含 '$OLD_IP_INPUT'。"

# 执行替换操作
log "INFO" "开始执行替换操作..."
SUCCESS_COUNT=0
FAIL_COUNT=0

while IFS= read -r file; do
    if [[ -f "$file" ]]; then
        # 创建备份文件名（替换斜杠为下划线）
        BACKUP_NAME=$(echo "$file" | sed 's|^/||; s|/|_|g')".bak"
        BACKUP_PATH="$BACKUP_DIR/$BACKUP_NAME"
        
        # 创建备份
        if cp "$file" "$BACKUP_PATH" 2>/dev/null; then
            # 执行替换
            if sed -i "s/$OLD_IP_ESCAPED/$NEW_IP_INPUT/g" "$file" 2>/dev/null; then
                log "INFO" "✅ 已替换: $file"
                ((SUCCESS_COUNT++))
            else
                log "ERROR" "❌ 替换失败: $file"
                ((FAIL_COUNT++))
            fi
        else
            log "ERROR" "❌ 备份失败: $file → $BACKUP_PATH"
            ((FAIL_COUNT++))
        fi
    fi
done < "$MATCHED_FILES"

# 清理临时文件
rm -f "$MATCHED_FILES"

# 输出结果统计
log "INFO" "=================================================="
log "INFO" "IP替换完成: 成功 $SUCCESS_COUNT 个文件，失败 $FAIL_COUNT 个文件"
log "INFO" "备份文件保存在: $BACKUP_DIR"
log "INFO" "=================================================="

if [ $FAIL_COUNT -eq 0 ]; then
    exit 0
else
    exit 1
fi
