import paho.mqtt.client as mqtt
import logging
import threading
import time
from queue import Queue
from collections import defaultdict

# 配置日志输出到控制台
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[logging.StreamHandler()]
)

# 全局配置
THREAD_COUNT = 5  # 线程数量
message_queue = Queue()  # 消息队列
message_counter = defaultdict(int)  # 消息计数器
message_details = defaultdict(list)  # 消息详细记录


class MessageStats:
    def __init__(self):
        self.total_count = 0
        self.last_received = None
        self.first_received = None


# 全局消息统计
global_stats = MessageStats()


def print_banner(title):
    """打印分隔横幅"""
    banner = f"\n=== {title} ===\n"
    print(banner)


def test_progress():
    """测试进度显示"""
    print_banner("系统测试")
    for i in range(1, 6):
        time.sleep(0.5)
        logging.info(f"系统测试中... {i}/5")
    print_banner("测试完成")


def on_connect(client, userdata, flags, rc):
    """MQTT连接回调"""
    if rc == 0:
        logging.info("MQTT连接成功")
        client.subscribe("/androidPanel/#")  # 订阅所有相关主题
    else:
        logging.error(f"MQTT连接失败，错误码: {rc}")


def on_message(client, userdata, msg):
    """MQTT消息回调"""
    try:
        payload = msg.payload.decode()
        logging.info(f"收到原始消息 - 主题: {msg.topic}")
        message_queue.put((msg.topic, payload))
    except Exception as e:
        logging.error(f"消息处理失败: {e}")


def worker_thread():
    """工作线程函数 - 只记录消息不进行其他操作"""
    while True:
        topic, payload = message_queue.get()
        try:
            # 记录消息统计
            message_counter[topic] += 1
            global_stats.total_count += 1

            # 记录详细消息
            timestamp = time.strftime('%Y-%m-%d %H:%M:%S')
            message_details[topic].append({
                'timestamp': timestamp,
                'payload': payload
            })

            # 更新首次/最后接收时间
            if global_stats.first_received is None:
                global_stats.first_received = timestamp
            global_stats.last_received = timestamp

            # 打印消息摘要
            logging.info(
                f"消息记录 - 主题: {topic} | "
                f"次数: {message_counter[topic]} | "
                f"时间: {timestamp}"
            )

            # 定期打印统计摘要
            if global_stats.total_count % 10 == 0:
                print_banner("统计摘要")
                logging.info(f"运行时间: {time.strftime('%H:%M:%S', time.gmtime(time.time() - start_time))}")
                logging.info(f"总消息数: {global_stats.total_count}")
                for topic, count in message_counter.items():
                    logging.info(f"主题 '{topic}': {count}次")

        except Exception as e:
            logging.error(f"消息记录失败: {e}")
        finally:
            message_queue.task_done()


if __name__ == "__main__":
    start_time = time.time()
    print_banner("MQTT消息监控程序启动")
    logging.info(f"启动时间: {time.strftime('%Y-%m-%d %H:%M:%S')}")
    logging.info(f"工作线程数: {THREAD_COUNT}")

    # 测试系统功能
    test_progress()

    # 启动工作线程池
    for i in range(THREAD_COUNT):
        t = threading.Thread(
            target=worker_thread,
            name=f"Worker-{i + 1}",
            daemon=True
        )
        t.start()
        logging.info(f"已启动工作线程 {i + 1}/{THREAD_COUNT}")

    # 创建MQTT客户端
    client = mqtt.Client()
    client.username_pw_set("mqtt@cmdb", "mqtt@webpassw0RD")
    client.on_connect = on_connect
    client.on_message = on_message

    try:
        # 连接MQTT服务器
        client.connect("192.168.5.229", 1883)
        logging.info("MQTT客户端已启动，等待消息...")

        # 保持运行
        client.loop_forever()
    except KeyboardInterrupt:
        print_banner("程序终止")
        logging.info("最终统计:")
        logging.info(f"总运行时间: {time.strftime('%H:%M:%S', time.gmtime(time.time() - start_time))}")
        logging.info(f"总接收消息数: {global_stats.total_count}")
        for topic, count in message_counter.items():
            logging.info(f"主题 '{topic}': {count}次")
    except Exception as e:
        logging.error(f"程序发生错误: {e}")
    finally:
        client.disconnect()
        logging.info("MQTT客户端已断开连接")