# _PRD_根据PRD文档生成测试用例_计划执行.md

> 版本：V1.0
> 创建日期：2026-01-28
> 关联需求：`_PRD_根据PRD文档生成测试用例.md`
> 目标：基于现有脚本实现，输出完整的开发计划与技术规范

---

## 1. 开发概述

### 1.1 脚本基本信息

| 项目 | 内容 |
|------|------|
| 脚本路径 | `AuxiliaryTool/用例生成/create_test_xlsx.py` |
| 编程语言 | Python 3.x |
| 核心依赖 | `openpyxl`（Excel 操作） |
| 入口函数 | `main()` |

### 1.2 功能模块划分

```
create_test_xlsx.py
├── 1) 路径与常量定义
├── 2) PRD 解析与用例生成
│   ├── RequirementItem 数据类
│   ├── normalize_md() - Markdown 预处理
│   ├── extract_requirement_items() - 需求条目抽取
│   ├── classify_category() - 功能类别判定
│   ├── decide_priority() - 用例等级判定
│   ├── build_steps() - 操作步骤生成
│   ├── build_expected() - 预期结果生成
│   └── prd_to_cases() - PRD 转用例
├── 3) Excel 写入
│   ├── find_header_row() - 表头查找
│   ├── safe_sheet_name() - Sheet 名称安全处理
│   └── write_cases_to_sheet() - 写入 Sheet
└── 4) 交互式操作
    ├── pick_prds_interactively() - PRD 选择
    ├── ask_sheet_name_interactively() - Sheet 名称输入
    └── ask_output_name_interactively() - 输出文件名输入
```

---

## 2. 开发任务清单

### 2.1 阶段一：基础框架（优先级：高）

#### 任务 1.1：路径与常量定义

**目标**：定义所有路径常量和全局配置

**实现要点**：
```python
# 路径计算
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
REPO_ROOT = os.path.abspath(os.path.join(BASE_DIR, "..", ".."))

# PRD 目录候选（兼容两种路径）
PRD_DIR_CANDIDATES = [
    os.path.join(REPO_ROOT, "Docs", "开发PRD"),
    os.path.join(REPO_ROOT, "Docs", "PRD"),
]

def resolve_prd_dir() -> str:
    """按优先级检测 PRD 目录，返回存在的路径"""
    for d in PRD_DIR_CANDIDATES:
        if os.path.isdir(d):
            return d
    return PRD_DIR_CANDIDATES[0]  # 默认第一个
```

**表头顺序常量**：
```python
headers_order = [
    "序号", "功能模块", "功能类别", "用例编号", "功能描述", "用例等级",
    "功能编号", "用例名称", "预置条件", "操作步骤", "JSON", "预期结果",
    "测试结果", "测试结论", "日志截屏", "备注",
]
```

**验收标准**：
- 路径计算正确，跨平台兼容
- `resolve_prd_dir()` 能正确检测存在的目录

---

#### 任务 1.2：命令行参数解析

**目标**：支持命令行参数覆盖默认配置

**实现要点**：
```python
import argparse

parser = argparse.ArgumentParser(description="根据PRD生成用例JSON，并写入测试用例Excel")
parser.add_argument("--template", default=TEMPLATE_PATH_DEFAULT, help="模板Excel路径")
parser.add_argument("--module", default="通用模块", help="功能模块字段值")
parser.add_argument("--prefix", default="TC", help="用例编号前缀")
parser.add_argument("--overwrite", action="store_true", help="是否覆盖保存")
args = parser.parse_args()
```

**验收标准**：
- `--help` 能正确显示帮助信息
- 各参数能正确覆盖默认值

---

### 2.2 阶段二：PRD 解析（优先级：高）

#### 任务 2.1：数据类定义

**目标**：定义需求条目数据结构

**实现要点**：
```python
from dataclasses import dataclass
from typing import List

@dataclass
class RequirementItem:
    code: str      # 编号（如 "1.1"）
    title: str     # 标题
    detail_lines: List[str]  # 详情内容行列表
```

---

#### 任务 2.2：正则表达式定义

**目标**：定义编号识别的正则表达式

**实现要点**：
```python
# 点分编号：1.1、2.3.1 等
RE_SECTION = re.compile(r"^\s*(?P<code>\d+(?:\.\d+)*)\s*[、\.\-]\s*(?P<title>.+?)\s*[:：]?\s*$")

# Markdown 标题：# ## ### 等
RE_MD_HEADING = re.compile(r"^\s*(#{1,6})\s+(?P<title>.+?)\s*$")
```

**测试用例**：
| 输入 | 匹配结果 | code | title |
|------|----------|------|-------|
| `1.1 用户登录` | ✓ | `1.1` | `用户登录` |
| `2.3.1-异常处理` | ✓ | `2.3.1` | `异常处理` |
| `## 功能描述` | ✓ | - | `功能描述` |
| `### 接口定义` | ✓ | - | `接口定义` |

---

#### 任务 2.3：Markdown 预处理

**目标**：过滤代码块和 HTML 注释

**实现要点**：
```python
def normalize_md(text: str) -> str:
    # 统一换行符
    text = text.replace("\r\n", "\n").replace("\r", "\n")
    # 移除代码块
    text = re.sub(r"```.*?```", "", text, flags=re.S)
    # 移除 HTML 注释
    text = re.sub(r"<!--.*?-->", "", text, flags=re.S)
    return text
```

**验收标准**：
- 代码块被正确移除
- HTML 注释被正确移除
- 换行符统一为 `\n`

---

#### 任务 2.4：需求条目抽取

**目标**：从 Markdown 文本中抽取需求条目

**核心逻辑**：
```python
def extract_requirement_items(md_text: str) -> List[RequirementItem]:
    lines = md_text.split("\n")
    items: List[RequirementItem] = []

    current_code = ""
    current_title = ""
    current_detail: List[str] = []

    def flush():
        nonlocal current_code, current_title, current_detail
        if current_title:
            detail = [clean_line(x) for x in current_detail if clean_line(x)]
            items.append(RequirementItem(
                code=current_code,
                title=clean_line(current_title),
                detail_lines=detail
            ))
        current_code = ""
        current_title = ""
        current_detail = []

    for raw in lines:
        line = raw.rstrip("\n")

        # 匹配点分编号
        m = RE_SECTION.match(line)
        if m:
            flush()
            current_code = m.group("code").strip()
            current_title = m.group("title").strip()
            continue

        # 匹配 Markdown 标题
        mh = RE_MD_HEADING.match(line)
        if mh:
            if current_title:
                # 已有标题，嵌套标题作为详情
                current_detail.append(line)
            else:
                # 新标题
                flush()
                current_code = ""
                current_title = mh.group("title").strip()
            continue

        # 详情行
        if current_title:
            current_detail.append(line)

    flush()

    # 过滤：标题长度 >= 2
    items = [it for it in items if it.title and len(it.title) >= 2]
    return items
```

**关键点**：
1. 遇到点分编号时，`flush()` 当前条目，开始新条目
2. 遇到 Markdown 标题时：
   - 如果已有标题 → 作为详情追加（嵌套处理）
   - 如果无标题 → `flush()` 并创建新条目
3. 最后必须 `flush()` 一次
4. 过滤标题长度 < 2 的条目

---

#### 任务 2.5：行清洗

**目标**：清洗单行文本

**实现要点**：
```python
def clean_line(line: str) -> str:
    line = line.strip()
    # 移除列表符号
    line = re.sub(r"^[\-\*\+]\s+", "", line)
    # 合并多余空格
    line = re.sub(r"\s+", " ", line)
    return line
```

---

### 2.3 阶段三：用例生成规则（优先级：高）

#### 任务 3.1：功能类别判定

**目标**：根据关键词判定功能类别

**实现要点**：
```python
def classify_category(title: str, detail_lines: List[str]) -> str:
    text = (title + " " + " ".join(detail_lines)).lower()

    if any(k in text for k in ["异常", "失败", "错误", "告警", "报警", "超时", "断开", "暴涨"]):
        return "异常场景"
    if any(k in text for k in ["安全", "权限", "鉴权", "加密", "脱敏"]):
        return "安全/鉴权"
    if any(k in text for k in ["报告", "输出", "word", "markdown", "邮件", "钉钉", "通知"]):
        return "运维可观测"
    return "功能测试"
```

**判定顺序**：异常场景 → 安全/鉴权 → 运维可观测 → 功能测试（默认）

---

#### 任务 3.2：用例等级判定

**目标**：根据关键词判定用例等级

**实现要点**：
```python
def decide_priority(title: str, detail_lines: List[str]) -> str:
    text = title + " " + " ".join(detail_lines)

    if any(k in text for k in ["必须", "报警", "告警", "峰值", "暴涨", "发送", "对接"]):
        return "高"
    if any(k in text for k in ["待实现", "可以", "建议", "优化"]):
        return "中"
    return "中"  # 默认
```

**判定顺序**：高 → 中（待实现） → 中（默认）

---

#### 任务 3.3：操作步骤生成

**目标**：从详情中提取操作步骤

**实现要点**：
```python
def build_steps(detail_lines: List[str]) -> str:
    keywords = ["检查", "监测", "记录", "输出", "发送", "查询", "进入", "生成", "判定", "调用", "执行"]

    candidates: List[str] = []
    for ln in detail_lines:
        s = clean_line(ln)
        if not s:
            continue
        if any(k in s for k in keywords):
            candidates.append(s)

    # 去重
    uniq: List[str] = []
    for c in candidates:
        if c not in uniq:
            uniq.append(c)

    if not uniq:
        return "1. 按 PRD 描述执行该功能/流程; 2. 采集相关日志/输出; 3. 记录实际结果"

    uniq = uniq[:6]  # 最多 6 条
    return "; ".join([f"{i+1}. {x}" for i, x in enumerate(uniq)])
```

**关键点**：
- 关键词列表必须完整
- 去重逻辑保证不重复
- 最多 6 条
- 默认值必须准确

---

#### 任务 3.4：预期结果生成

**目标**：从详情中提取预期结果

**实现要点**：
```python
def build_expected(title: str, detail_lines: List[str]) -> str:
    keywords = ["需要", "应", "必须", "输出", "记录", "发送", "判定", "成功", "失败"]

    candidates: List[str] = []
    for ln in detail_lines:
        s = clean_line(ln)
        if not s:
            continue
        if any(k in s for k in keywords):
            candidates.append(s)

    # 去重
    uniq: List[str] = []
    for c in candidates:
        if c not in uniq:
            uniq.append(c)

    if not uniq:
        return f"{title} 按 PRD 约定产出正确结果（请补充具体断言点）"

    uniq = uniq[:6]  # 最多 6 条
    return "; ".join(uniq)
```

---

#### 任务 3.5：用例编号生成

**目标**：生成格式化的用例编号

**实现要点**：
```python
def make_case_id(prefix: str, idx: int) -> str:
    return f"{prefix}-{idx:03d}"  # TC-001, TC-002, ...
```

---

#### 任务 3.6：用例记录生成

**目标**：将需求条目转换为用例记录

**实现要点**：
```python
def to_case_record(idx: int, module: str, prefix: str, req: RequirementItem) -> Dict[str, str]:
    record: Dict[str, str] = {
        "序号": idx,
        "功能模块": module,
        "功能类别": classify_category(req.title, req.detail_lines),
        "用例编号": make_case_id(prefix, idx),
        "功能描述": req.title,
        "用例等级": decide_priority(req.title, req.detail_lines),
        "功能编号": req.code or "",
        "用例名称": req.title,
        "预置条件": "1. 环境已部署; 2. 具备执行权限; 3. 必要依赖/配置已准备",
        "操作步骤": build_steps(req.detail_lines),
        "JSON": "",
        "预期结果": build_expected(req.title, req.detail_lines),
        "测试结果": "",
        "测试结论": "",
        "日志截屏": "",
        "备注": "",
    }
    # 确保所有字段存在
    for k in headers_order:
        record.setdefault(k, "")
    return record
```

---

#### 任务 3.7：PRD 转用例（主函数）

**目标**：读取 PRD 文件并转换为用例列表

**实现要点**：
```python
def prd_to_cases(prd_path: str, module: str, prefix: str, start_idx: int = 1) -> List[Dict[str, str]]:
    with open(prd_path, "r", encoding="utf-8") as f:
        md = normalize_md(f.read())

    items = extract_requirement_items(md)
    cases: List[Dict[str, str]] = []

    idx = start_idx
    for it in items:
        # 过滤无效标题
        if any(k in it.title for k in ["说明", "目录", "背景", "概述"]):
            continue
        cases.append(to_case_record(idx=idx, module=module, prefix=prefix, req=it))
        idx += 1

    return cases
```

**关键点**：
- `start_idx` 支持多 PRD 全局递增
- 过滤"说明、目录、背景、概述"等无效标题

---

### 2.4 阶段四：Excel 写入（优先级：高）

#### 任务 4.1：表头查找

**目标**：在模板中查找表头行

**实现要点**：
```python
def find_header_row(template_sheet) -> int:
    must_keys = {"序号", "用例名称", "操作步骤", "预期结果"}
    max_scan = min(30, template_sheet.max_row or 30)

    for r in range(1, max_scan + 1):
        values = []
        for c in range(1, min(40, template_sheet.max_column or 40) + 1):
            v = template_sheet.cell(row=r, column=c).value
            if v is None:
                continue
            values.append(str(v).strip())
        # 至少匹配 2 个关键字
        if len(must_keys.intersection(values)) >= 2:
            return r
    return 3  # 默认第 3 行
```

**关键点**：
- 扫描前 30 行
- 匹配至少 2 个关键字
- 默认返回 3

---

#### 任务 4.2：Sheet 名称安全处理

**目标**：确保 Sheet 名称符合 Excel 规范

**实现要点**：
```python
def safe_sheet_name(name: str) -> str:
    # Excel 不允许的字符: [ ] : * ? / \
    name = re.sub(r"[\[\]\:\*\?\/\\]", "_", name)
    return name[:31]  # 最大 31 字符
```

**Excel Sheet 名称限制**：
- 最大长度：31 字符
- 不允许字符：`[ ] : * ? / \`

---

#### 任务 4.3：写入 Sheet

**目标**：将用例写入 Excel Sheet

**实现要点**：
```python
from copy import copy
from openpyxl.styles import Alignment

def write_cases_to_sheet(wb, template_sheet, sheet_name: str, cases: List[Dict[str, str]]):
    # Sheet 存在检查
    if sheet_name in wb.sheetnames:
        ws = wb[sheet_name]
        start_row = (ws.max_row or 1) + 1  # 追加模式
    else:
        ws = wb.create_sheet(sheet_name)
        start_row = 2

        # 复制表头
        header_row_index = find_header_row(template_sheet)
        for col_idx, cell in enumerate(template_sheet[header_row_index], start=1):
            new_cell = ws.cell(row=1, column=col_idx, value=cell.value)
            if cell.has_style:
                new_cell.font = copy(cell.font)
                new_cell.fill = copy(cell.fill)
                new_cell.border = copy(cell.border)
                new_cell.alignment = copy(cell.alignment)
                new_cell.number_format = cell.number_format
        ws.freeze_panes = "B2"  # 冻结首行

    # 写入数据
    row = start_row
    for case in cases:
        for col_idx, header in enumerate(headers_order, start=1):
            val = case.get(header, "")
            # 分号转换行（预置条件、操作步骤）
            if header in ("预置条件", "操作步骤") and isinstance(val, str):
                val = val.replace("; ", "\n")
            # JSON 强制清空
            if header == "JSON":
                val = ""
            ws.cell(row=row, column=col_idx, value=val)
        row += 1

    # 自动换行（预置条件、操作步骤）
    col_idx_pre = headers_order.index("预置条件") + 1
    col_idx_steps = headers_order.index("操作步骤") + 1
    for r in range(1, row):
        ws.cell(row=r, column=col_idx_pre).alignment = Alignment(wrap_text=True, vertical="top")
        ws.cell(row=r, column=col_idx_steps).alignment = Alignment(wrap_text=True, vertical="top")

    # 列宽自适应
    for col in ws.columns:
        max_len = 0
        col_letter = col[0].column_letter
        for c in col:
            v = c.value
            if v is None:
                continue
            l = len(str(v))
            if l > max_len:
                max_len = l
        ws.column_dimensions[col_letter].width = min(max_len + 2, 60)

    return ws
```

**关键点**：
1. Sheet 复用：存在则追加，不存在则创建
2. 表头样式复制：font、fill、border、alignment、number_format
3. 分号转换行：预置条件、操作步骤
4. JSON 强制清空
5. 自动换行：预置条件、操作步骤列
6. 列宽最大 60

---

### 2.5 阶段五：交互式操作（优先级：中）

#### 任务 5.1：列出 PRD 文件

**目标**：列出 PRD 目录下的所有 .md 文件

**实现要点**：
```python
def list_prd_files(prd_dir: str) -> List[str]:
    if not os.path.isdir(prd_dir):
        return []
    fs = [f for f in os.listdir(prd_dir) if f.lower().endswith(".md")]
    fs.sort()
    return fs
```

---

#### 任务 5.2：解析多选输入

**目标**：解析用户输入的序号

**实现要点**：
```python
def parse_multi_input(s: str) -> List[str]:
    s = (s or "").strip()
    if not s:
        return []
    parts = re.split(r"[,\s]+", s)  # 逗号或空格分隔
    return [p for p in (x.strip() for x in parts) if p]
```

**支持格式**：`1,2,5` 或 `1 2 5` 或 `1, 2, 5`

---

#### 任务 5.3：交互式选择 PRD

**目标**：引导用户选择 PRD 文件

**实现要点**：
```python
def pick_prds_interactively(prd_dir: str) -> List[str]:
    files = list_prd_files(prd_dir)
    if not files:
        raise RuntimeError(f"PRD目录为空或不存在：{prd_dir}")

    print(f"PRD目录：{prd_dir}")
    for i, f in enumerate(files, start=1):
        print(f"{i:>2}. {f}")
    print("支持多选：输入多个序号（如 1,2,5 或 1 2 5）。")

    while True:
        s = input("请输入PRD序号(可多选)：").strip()
        parts = parse_multi_input(s)
        if not parts or not all(p.isdigit() for p in parts):
            print("输入无效，请输入序号（可多选）。")
            continue

        idxs: List[int] = []
        ok = True
        for p in parts:
            n = int(p)
            if not (1 <= n <= len(files)):
                ok = False
                break
            idxs.append(n)
        if not ok:
            print("序号超出范围，请重试。")
            continue

        # 去重并构建路径
        seen = set()
        paths: List[str] = []
        for n in idxs:
            if n in seen:
                continue
            seen.add(n)
            paths.append(os.path.join(prd_dir, files[n - 1]))
        return paths
```

**关键点**：
1. 空目录检测
2. 输入验证（数字、范围）
3. 自动去重

---

#### 任务 5.4：输入 Sheet 名称

**目标**：获取 Sheet 名称并安全处理

**实现要点**：
```python
def ask_sheet_name_interactively(default_name: str) -> str:
    s = input(f"请输入Sheet名称（回车使用默认：{default_name}）：").strip()
    return safe_sheet_name(s or default_name)
```

---

#### 任务 5.5：输入输出文件名

**目标**：获取输出文件名

**实现要点**：
```python
def ask_output_name_interactively(default_name: str) -> str:
    s = input(f"请输入生成的测试用例文件名称（回车使用默认：{default_name}）：").strip()
    return s or default_name
```

---

### 2.6 阶段六：主流程集成（优先级：高）

#### 任务 6.1：主函数

**目标**：整合所有模块，实现完整流程

**实现要点**：
```python
from datetime import datetime

def main():
    # 1. 解析命令行参数
    parser = argparse.ArgumentParser(description="根据PRD生成用例JSON，并写入测试用例Excel")
    parser.add_argument("--template", default=TEMPLATE_PATH_DEFAULT, help="模板Excel路径")
    parser.add_argument("--module", default="通用模块", help="功能模块字段值")
    parser.add_argument("--prefix", default="TC", help="用例编号前缀")
    parser.add_argument("--overwrite", action="store_true", help="是否覆盖保存")
    args = parser.parse_args()

    # 2. 定位 PRD 目录
    prd_dir = resolve_prd_dir()

    # 3. 检查模板文件
    if not os.path.exists(args.template):
        print("找不到模板文件：", args.template)
        return

    # 4. 交互式选择 PRD
    prd_paths = pick_prds_interactively(prd_dir)

    # 5. 确定 Sheet 名称
    if len(prd_paths) == 1:
        default_sheet = f"{os.path.splitext(os.path.basename(prd_paths[0]))[0]}_用例"
    else:
        default_sheet = "多PRD_用例"
    sheet_name = ask_sheet_name_interactively(default_sheet)

    # 6. 确定输出文件名
    if len(prd_paths) == 1:
        default_out_base = os.path.splitext(os.path.basename(prd_paths[0]))[0]
    else:
        default_out_base = "MultiPRD"
    out_base = ask_output_name_interactively(default_out_base)

    # 7. 打开模板
    wb = load_workbook(args.template)
    template_sheet = wb.worksheets[0]

    # 8. 多 PRD 合并到同一个 sheet（序号全局递增）
    os.makedirs(OUTPUT_JSON_DIR, exist_ok=True)
    total_cases = 0
    next_idx = 1

    for prd_path in prd_paths:
        prd_base = os.path.splitext(os.path.basename(prd_path))[0]

        # 生成用例
        cases = prd_to_cases(prd_path=prd_path, module=args.module, prefix=args.prefix, start_idx=next_idx)
        if not cases:
            print(f"未从PRD抽取到可生成用例的条目：{prd_path}")
            continue

        # 输出 JSON
        json_path = os.path.join(OUTPUT_JSON_DIR, f"{prd_base}_用例.json")
        with open(json_path, "w", encoding="utf-8") as f:
            json.dump(cases, f, ensure_ascii=False, indent=4)
        print(f"已生成 JSON：{json_path}（{len(cases)}条）")

        # 写入 Sheet
        write_cases_to_sheet(wb, template_sheet, sheet_name, cases)

        total_cases += len(cases)
        next_idx += len(cases)

    # 9. 保存 Excel
    if args.overwrite:
        out_xlsx = args.template
    else:
        os.makedirs(OUTPUT_XLSX_DIR, exist_ok=True)
        ts = datetime.now().strftime("%Y%m%d_%H%M%S")
        out_xlsx = os.path.join(OUTPUT_XLSX_DIR, f"{out_base}_{ts}.xlsx")

    wb.save(out_xlsx)
    print("已生成Excel：", out_xlsx)
    print("Sheet：", sheet_name)
    print("总用例条数：", total_cases)


if __name__ == "__main__":
    main()
```

---

## 3. 测试验证

### 3.1 单元测试

| 模块 | 测试项 | 预期结果 |
|------|--------|----------|
| `resolve_prd_dir()` | 目录存在检测 | 返回存在的目录路径 |
| `normalize_md()` | 代码块过滤 | ````...```` 被移除 |
| `normalize_md()` | HTML 注释过滤 | `<!--...-->` 被移除 |
| `extract_requirement_items()` | 点分编号解析 | 正确提取 code 和 title |
| `extract_requirement_items()` | Markdown 标题解析 | 正确提取 title |
| `extract_requirement_items()` | 嵌套标题处理 | 嵌套标题作为详情 |
| `classify_category()` | 异常场景 | 包含"异常"返回"异常场景" |
| `classify_category()` | 安全/鉴权 | 包含"安全"返回"安全/鉴权" |
| `decide_priority()` | 高等级 | 包含"必须"返回"高" |
| `build_steps()` | 无匹配内容 | 返回默认值 |
| `build_steps()` | 有匹配内容 | 返回分号分隔的步骤 |
| `build_expected()` | 无匹配内容 | 返回默认值 |
| `safe_sheet_name()` | 非法字符过滤 | `[ ] : * ? / \` 替换为 `_` |
| `safe_sheet_name()` | 长度限制 | 超过 31 字符截断 |
| `find_header_row()` | 表头查找 | 返回正确行号 |

### 3.2 集成测试

| 场景 | 操作 | 预期结果 |
|------|------|----------|
| 单 PRD 生成 | 选择 1 个 PRD | 生成 JSON 和 Excel |
| 多 PRD 合并 | 选择多个 PRD | 合并在同一 Sheet，序号递增 |
| Sheet 复用 | 重复生成到同一 Sheet | 数据追加而非覆盖 |
| 覆盖模式 | 使用 `--overwrite` | 覆盖模板文件 |
| 命令行参数 | 使用 `--module`、`--prefix` | 参数生效 |

### 3.3 边界测试

| 场景 | 操作 | 预期结果 |
|------|------|----------|
| 空 PRD 目录 | 无 .md 文件 | 报错提示 |
| 模板文件不存在 | 指定不存在的模板 | 报错提示 |
| 无效输入 | 输入非数字序号 | 提示重试 |
| 序号超范围 | 输入超出范围的序号 | 提示重试 |
| 无可用条目 | PRD 无有效需求 | 跳过该 PRD |

---

## 4. 依赖关系

### 4.1 模块依赖图

```
main()
├── argparse (命令行解析)
├── resolve_prd_dir() (目录定位)
├── pick_prds_interactively()
│   └── list_prd_files()
│       └── parse_multi_input()
├── ask_sheet_name_interactively()
│   └── safe_sheet_name()
├── ask_output_name_interactively()
├── load_workbook() (openpyxl)
├── prd_to_cases()
│   ├── normalize_md()
│   ├── extract_requirement_items()
│   │   ├── clean_line()
│   │   └── RequirementItem
│   └── to_case_record()
│       ├── classify_category()
│       ├── decide_priority()
│       ├── build_steps()
│       ├── build_expected()
│       └── make_case_id()
├── write_cases_to_sheet()
│   ├── find_header_row()
│   └── Alignment() (openpyxl)
└── wb.save() (openpyxl)
```

### 4.2 开发顺序建议

1. **第一批**：基础框架
   - 路径与常量定义
   - 命令行参数解析
   - 数据类定义

2. **第二批**：PRD 解析
   - 正则表达式定义
   - Markdown 预处理
   - 需求条目抽取
   - 行清洗

3. **第三批**：用例生成
   - 功能类别判定
   - 用例等级判定
   - 操作步骤生成
   - 预期结果生成
   - 用例编号生成
   - 用例记录生成
   - PRD 转用例主函数

4. **第四批**：Excel 写入
   - 表头查找
   - Sheet 名称安全处理
   - 写入 Sheet

5. **第五批**：交互式操作
   - 列出 PRD 文件
   - 解析多选输入
   - 交互式选择 PRD
   - 输入处理函数

6. **第六批**：主流程集成
   - 主函数

---

## 5. 关键常量与配置

### 5.1 正则表达式

| 变量 | 表达式 | 用途 |
|------|--------|------|
| `RE_SECTION` | `^\s*(\d+(?:\.\d+)*)\s*[、\.\-]\s*(.+?)\s*[:：]?\s*$` | 匹配点分编号 |
| `RE_MD_HEADING` | `^\s*(#{1,6})\s+(.+?)\s*$` | 匹配 Markdown 标题 |

### 5.2 关键词列表

| 用途 | 关键词 |
|------|--------|
| 异常场景 | `["异常", "失败", "错误", "告警", "报警", "超时", "断开", "暴涨"]` |
| 安全/鉴权 | `["安全", "权限", "鉴权", "加密", "脱敏"]` |
| 运维可观测 | `["报告", "输出", "word", "markdown", "邮件", "钉钉", "通知"]` |
| 高等级 | `["必须", "报警", "告警", "峰值", "暴涨", "发送", "对接"]` |
| 中等级 | `["待实现", "可以", "建议", "优化"]` |
| 操作步骤 | `["检查", "监测", "记录", "输出", "发送", "查询", "进入", "生成", "判定", "调用", "执行"]` |
| 预期结果 | `["需要", "应", "必须", "输出", "记录", "发送", "判定", "成功", "失败"]` |
| 过滤标题 | `["说明", "目录", "背景", "概述"]` |

### 5.3 默认值

| 配置项 | 默认值 |
|--------|--------|
| 模板文件 | `兰州中石化项目测试用例20251203.xlsx` |
| 功能模块 | `通用模块` |
| 用例编号前缀 | `TC` |
| 预置条件 | `1. 环境已部署; 2. 具备执行权限; 3. 必要依赖/配置已准备` |
| 操作步骤默认值 | `1. 按 PRD 描述执行该功能/流程; 2. 采集相关日志/输出; 3. 记录实际结果` |
| 预期结果默认值 | `{标题} 按 PRD 约定产出正确结果（请补充具体断言点）` |
| Sheet 单 PRD 默认 | `{PRD文件名}_用例` |
| Sheet 多 PRD 默认 | `多PRD_用例` |
| 输出文件单 PRD 默认 | `{PRD文件名}` |
| 输出文件多 PRD 默认 | `MultiPRD` |

### 5.4 限制值

| 配置项 | 限制值 |
|--------|--------|
| Sheet 名称最大长度 | 31 字符 |
| 表头扫描最大行数 | 30 行 |
| 表头扫描最大列数 | 40 列 |
| 操作步骤最大条数 | 6 条 |
| 预期结果最大条数 | 6 条 |
| 列宽最大值 | 60 |
| 标题最小长度 | 2 字符 |

---

## 6. 验收清单

### 6.1 功能验收

- [ ] 能正确解析点分编号格式（如 `1.1`、`2.3.1`）
- [ ] 能正确解析 Markdown 标题格式（如 `## 标题`）
- [ ] 能处理 Markdown 标题嵌套场景
- [ ] 功能类别判定符合规则
- [ ] 用例等级判定符合规则
- [ ] 操作步骤生成符合规则（分号分隔、最多 6 条）
- [ ] 预期结果生成符合规则（分号分隔、最多 6 条）
- [ ] 支持单 PRD 生成
- [ ] 支持多 PRD 合并（序号全局递增）
- [ ] Sheet 复用功能正常（追加模式）
- [ ] JSON 文件正确生成
- [ ] Excel 文件正确生成
- [ ] 分号在 Excel 中转换为换行
- [ ] JSON 列在 Excel 中强制清空
- [ ] Sheet 名称非法字符被过滤
- [ ] Sheet 名称长度限制为 31 字符
- [ ] 命令行参数生效
- [ ] 覆盖模式正常工作

### 6.2 异常验收

- [ ] PRD 目录为空时正确报错
- [ ] 模板文件不存在时正确报错
- [ ] 无效输入时提示重试
- [ ] 序号超范围时提示重试
- [ ] 无可用需求条目时跳过该 PRD
- [ ] 写入失败时有明确提示

---

## 7. 参考资料

### 7.1 需求文档
- `_PRD_根据PRD文档生成测试用例.md`

### 7.2 现有实现
- `AuxiliaryTool/用例生成/create_test_xlsx.py`

### 7.3 依赖库
- `openpyxl`：Excel 文件操作
- Python 标准库：`os`, `re`, `json`, `argparse`, `datetime`, `dataclasses`, `typing`, `copy`

### 7.4 需求规范
- 代码规范：`Docs/PRD/01规范文档/_PRD_规范文档_代码规范.md`
- 问题总结：`Docs/PRD/01规范文档/_PRD_问题总结_记录文档.md`
- 方法总结：`Docs/PRD/01规范文档/_PRD_方法总结_记录文档.md`
- 文档规范：`Docs/PRD/01规范文档/_PRD_规范文档_文档规范.md`
- 测试规范：`Docs/PRD/01规范文档/_PRD_规范文档_测试规范.md`
