#!/usr/bin/env bash
# 容器镜像打包上传至公司网盘脚本
# 功能：
# 1. 将当前目录压缩成 tar.gz 格式文件
# 2. 上传到公司 NAS 网盘（SMB 协议）
# 3. 显示上传进度

set -euo pipefail

# ================================
# 配置区（密码已加密存储）
# ================================
NAS_SERVER="192.168.9.9"
NAS_SHARE="研发管理"
NAS_USER="陈泽键"
NAS_PASS_ENCRYPTED="Mm01ZUwuTUA="
NAS_MOUNT_POINT="/mnt/nas_upload"

# 上传目录（网盘上的目标目录）
NAS_UPLOAD_DIR="调试打包路径"

log() {
  local level="$1"; shift
  printf '[%(%F %T)T] [%s] %s\n' -1 "${level}" "$*"
}

# 解密密码
decrypt_password() {
  echo "${NAS_PASS_ENCRYPTED}" | base64 -d
}

# 检查依赖
check_dependencies() {
  local missing_deps=()
  
  if ! command -v cifs-utils >/dev/null 2>&1 && ! command -v mount.cifs >/dev/null 2>&1; then
    # 检查是否可以挂载 cifs
    if ! grep -q cifs /proc/filesystems 2>/dev/null; then
      missing_deps+=("cifs-utils")
    fi
  fi
  
  if ! command -v pv >/dev/null 2>&1; then
    log "WARN" "未安装 pv 工具，将使用 cp 命令（无进度显示）"
  fi
  
  if [[ ${#missing_deps[@]} -gt 0 ]]; then
    log "ERROR" "缺少依赖: ${missing_deps[*]}"
    log "INFO" "请执行: yum install -y ${missing_deps[*]} 或 apt install -y ${missing_deps[*]}"
    return 1
  fi
  
  return 0
}

# 挂载 NAS
mount_nas() {
  local nas_pass
  nas_pass="$(decrypt_password)"
  
  # 创建挂载点
  if [[ ! -d "${NAS_MOUNT_POINT}" ]]; then
    log "INFO" "创建挂载点: ${NAS_MOUNT_POINT}"
    sudo mkdir -p "${NAS_MOUNT_POINT}"
  fi
  
  # 检查是否已挂载
  if mountpoint -q "${NAS_MOUNT_POINT}" 2>/dev/null; then
    log "INFO" "NAS 已挂载，跳过挂载步骤"
    return 0
  fi
  
  log "INFO" "正在挂载 NAS: //${NAS_SERVER}/${NAS_SHARE}"
  
  # 挂载 SMB/CIFS 共享
  if ! sudo mount -t cifs "//${NAS_SERVER}/${NAS_SHARE}" "${NAS_MOUNT_POINT}" \
    -o "username=${NAS_USER},password=${nas_pass},iocharset=utf8,vers=2.0"; then
    log "ERROR" "NAS 挂载失败"
    log "INFO" "请检查: 1) 网络连接 2) 账号密码 3) 共享路径"
    return 1
  fi
  
  log "INFO" "✅ NAS 挂载成功"
  return 0
}

# 卸载 NAS
unmount_nas() {
  if mountpoint -q "${NAS_MOUNT_POINT}" 2>/dev/null; then
    log "INFO" "正在卸载 NAS..."
    sudo umount "${NAS_MOUNT_POINT}" || true
  fi
}

# 打包当前目录
pack_current_dir() {
  local pack_name="$1"
  local source_dir="$2"
  
  log "INFO" "开始打包目录: ${source_dir}"
  log "INFO" "目标文件: ${pack_name}"
  
  # 获取目录大小用于进度估算
  local dir_size
  dir_size=$(du -sb "${source_dir}" 2>/dev/null | awk '{print $1}')
  
  if command -v pv >/dev/null 2>&1; then
    # 使用 pv 显示打包进度
    tar -cf - -C "$(dirname "${source_dir}")" "$(basename "${source_dir}")" | \
      pv -s "${dir_size}" -p -t -e -r | \
      gzip > "${pack_name}"
  else
    # 无 pv 时直接打包
    tar -czvf "${pack_name}" -C "$(dirname "${source_dir}")" "$(basename "${source_dir}")"
  fi
  
  if [[ -f "${pack_name}" ]]; then
    local file_size
    file_size=$(du -h "${pack_name}" | awk '{print $1}')
    log "INFO" "✅ 打包完成，文件大小: ${file_size}"
    return 0
  else
    log "ERROR" "打包失败"
    return 1
  fi
}

# 上传文件到 NAS（带进度显示）
upload_to_nas() {
  local source_file="$1"
  local target_dir="${NAS_MOUNT_POINT}/${NAS_UPLOAD_DIR}"
  
  # 确保目标目录存在
  if [[ ! -d "${target_dir}" ]]; then
    log "INFO" "创建目标目录: ${target_dir}"
    sudo mkdir -p "${target_dir}"
  fi
  
  local file_name
  file_name=$(basename "${source_file}")
  local target_file="${target_dir}/${file_name}"
  
  log "INFO" "开始上传文件到 NAS..."
  log "INFO" "源文件: ${source_file}"
  log "INFO" "目标: //${NAS_SERVER}/${NAS_SHARE}/${NAS_UPLOAD_DIR}/${file_name}"
  
  # 获取文件大小
  local file_size
  file_size=$(stat -c%s "${source_file}" 2>/dev/null || stat -f%z "${source_file}" 2>/dev/null)
  
  if command -v pv >/dev/null 2>&1; then
    # 使用 pv 显示上传进度
    pv -p -t -e -r "${source_file}" | sudo tee "${target_file}" > /dev/null
  else
    # 无 pv 时使用 cp 并显示简单进度
    log "INFO" "正在上传（无进度显示）..."
    sudo cp "${source_file}" "${target_file}"
  fi
  
  # 验证上传结果
  if [[ -f "${target_file}" ]]; then
    local uploaded_size
    uploaded_size=$(sudo du -h "${target_file}" | awk '{print $1}')
    log "INFO" "✅ 上传完成！"
    log "INFO" "   文件路径: \\\\${NAS_SERVER}\\${NAS_SHARE}\\${NAS_UPLOAD_DIR}\\${file_name}"
    log "INFO" "   文件大小: ${uploaded_size}"
    return 0
  else
    log "ERROR" "上传失败，目标文件不存在"
    return 1
  fi
}

# 显示使用说明
usage() {
  cat <<'EOF'
用法:
  upload_to_nas.sh [选项] [目录路径]

选项:
  -h, --help      显示帮助信息
  -n, --name      指定打包文件名（不含扩展名）
  -d, --dir       指定上传到网盘的子目录

参数:
  目录路径        要打包上传的目录，默认为当前目录

示例:
  # 打包并上传当前目录
  ./upload_to_nas.sh

  # 打包并上传指定目录
  ./upload_to_nas.sh /data/temp/container-images

  # 指定打包文件名
  ./upload_to_nas.sh -n java_container_v6 /data/temp/java

网盘信息:
  服务器: \\192.168.9.9\home
  账号: 陈泽键
  上传目录: 容器镜像包/
EOF
}

# 清理函数
cleanup() {
  unmount_nas
}

# 主函数
main() {
  local source_dir="."
  local pack_name=""
  local custom_upload_dir=""
  
  # 解析参数
  while [[ $# -gt 0 ]]; do
    case "$1" in
      -h|--help)
        usage
        exit 0
        ;;
      -n|--name)
        pack_name="$2"
        shift 2
        ;;
      -d|--dir)
        custom_upload_dir="$2"
        shift 2
        ;;
      -*)
        log "ERROR" "未知选项: $1"
        usage
        exit 1
        ;;
      *)
        source_dir="$1"
        shift
        ;;
    esac
  done
  
  # 设置自定义上传目录
  if [[ -n "${custom_upload_dir}" ]]; then
    NAS_UPLOAD_DIR="${custom_upload_dir}"
  fi
  
  # 转换为绝对路径
  source_dir=$(cd "${source_dir}" && pwd)
  
  if [[ ! -d "${source_dir}" ]]; then
    log "ERROR" "目录不存在: ${source_dir}"
    exit 1
  fi
  
  # 生成打包文件名
  if [[ -z "${pack_name}" ]]; then
    local dir_name
    dir_name=$(basename "${source_dir}")
    local timestamp
    timestamp=$(date +%Y%m%d_%H%M%S)
    pack_name="${dir_name}_${timestamp}"
  fi
  
  local pack_file="/tmp/${pack_name}.tar.gz"
  
  log "INFO" "=================================================================="
  log "INFO" "容器镜像打包上传工具"
  log "INFO" "=================================================================="
  log "INFO" "源目录: ${source_dir}"
  log "INFO" "打包文件: ${pack_file}"
  log "INFO" "目标网盘: \\\\${NAS_SERVER}\\${NAS_SHARE}\\${NAS_UPLOAD_DIR}"
  log "INFO" "=================================================================="
  
  # 设置清理钩子
  trap cleanup EXIT
  
  # 检查依赖
  if ! check_dependencies; then
    exit 1
  fi
  
  # 步骤1: 打包目录
  log "INFO" "[步骤 1/3] 打包目录..."
  if ! pack_current_dir "${pack_file}" "${source_dir}"; then
    exit 1
  fi
  
  # 步骤2: 挂载 NAS
  log "INFO" "[步骤 2/3] 挂载 NAS..."
  if ! mount_nas; then
    exit 1
  fi
  
  # 步骤3: 上传文件
  log "INFO" "[步骤 3/3] 上传文件..."
  if ! upload_to_nas "${pack_file}"; then
    exit 1
  fi
  
  # 清理临时文件
  log "INFO" "清理临时文件..."
  rm -f "${pack_file}"
  
  log "INFO" "=================================================================="
  log "INFO" "🎉 全部完成！"
  log "INFO" "=================================================================="
  
  return 0
}

main "$@"

