#!/usr/bin/env bash
set -euo pipefail

# ================================
# 全局变量
# ================================
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
LOG_FILE="${SCRIPT_DIR}/container_update_$(date +%Y%m%d_%H%M%S).log"
INTERACTIVE_LOG_ENABLED=false  # 交互式模式下启用日志

# ================================
# 日志函数（支持同时输出到控制台和文件）
# ================================
log() {
  local level="$1"; shift
  local message="$*"
  local timestamp
  timestamp=$(printf '%(%F %T)T' -1)
  local log_line="[${timestamp}] [${level}] ${message}"
  
  # 输出到控制台
  printf '%s\n' "${log_line}"
  
  # 如果启用了日志文件，同时写入文件
  if [[ "${INTERACTIVE_LOG_ENABLED}" == "true" && -n "${LOG_FILE}" ]]; then
    printf '%s\n' "${log_line}" >> "${LOG_FILE}"
  fi
}

# 仅写入日志文件（不输出到控制台）
log_to_file() {
  local level="$1"; shift
  local message="$*"
  local timestamp
  timestamp=$(printf '%(%F %T)T' -1)
  local log_line="[${timestamp}] [${level}] ${message}"
  
  if [[ "${INTERACTIVE_LOG_ENABLED}" == "true" && -n "${LOG_FILE}" ]]; then
    printf '%s\n' "${log_line}" >> "${LOG_FILE}"
  fi
}

usage() {
  cat <<'EOF'
用法:
  container_update.sh [选项] [容器名称] [镜像包]

运行模式:
  1. 交互式模式（无参数）:
     ./container_update.sh
     脚本会引导选择容器类型、平台类型，自动检测镜像文件
     适用于技术人员从网盘下载压缩包后在项目服务器上执行

  2. 命令行模式（带参数）:
     ./container_update.sh <容器名称> <镜像包> [--new-platform]
     适用于远程调用或自动化脚本

参数说明:
  容器名称       如 ujava3，用于指定 docker 容器名称
  镜像包         离线镜像 tar 包路径，若为相对路径则基于当前目录
  --new-platform 目标服务器为新统一平台时添加该参数
  -i, --interactive  强制进入交互式模式

脚本会根据容器前缀与平台类型调用对应部署函数，并自动使用提供的镜像包执行 docker load。
EOF
}

# ================================
# 容器与镜像映射配置
# ================================
declare -A CONTAINER_IMAGE_MAP
CONTAINER_IMAGE_MAP["ujava"]="java1.8.0_472.tar.gz"
CONTAINER_IMAGE_MAP["uemqx"]="uemqx5.8.4.tar.gz"
CONTAINER_IMAGE_MAP["uredis"]="redis8.2.2.tar.gz"
CONTAINER_IMAGE_MAP["upython"]="python_v15.tar.gz"
CONTAINER_IMAGE_MAP["unacos"]="nacos-server-v2.5.2.tar.gz"
CONTAINER_IMAGE_MAP["unginx"]="nginx-1.29.3.tar.gz"

# 容器描述信息
declare -A CONTAINER_DESC_MAP
CONTAINER_DESC_MAP["ujava"]="Java 服务容器"
CONTAINER_DESC_MAP["uemqx"]="EMQX 消息队列容器"
CONTAINER_DESC_MAP["uredis"]="Redis 缓存容器"
CONTAINER_DESC_MAP["upython"]="Python 服务容器"
CONTAINER_DESC_MAP["unacos"]="Nacos 注册中心容器"
CONTAINER_DESC_MAP["unginx"]="Nginx 反向代理容器"

# ================================
# 步骤1：询问更新哪个容器
# ================================
select_container() {
  log "INFO" "=================================================================="
  log "INFO" "步骤 1/4：选择要更新的容器"
  log "INFO" "=================================================================="
  
  local container_options=(ujava uemqx uredis upython unacos unginx)
  echo ""
  log "INFO" "可选择的容器类型："
  for idx in "${!container_options[@]}"; do
    local option_index=$((idx + 1))
    local container_name="${container_options[idx]}"
    local container_desc="${CONTAINER_DESC_MAP[${container_name}]}"
    printf "  [%s] %-10s - %s\n" "${option_index}" "${container_name}" "${container_desc}"
  done
  echo ""
  
  local container_key
  read -rp "请输入容器编号 (1-6): " container_key
  log_to_file "INFO" "用户输入容器编号: ${container_key}"
  
  if ! [[ "${container_key}" =~ ^[1-6]$ ]]; then
    log "ERROR" "容器编号无效，请输入 1-6"
    return 1
  fi
  
  SELECTED_CONTAINER="${container_options[$((container_key - 1))]}"
  log "INFO" "✅ 已选择容器类型: ${SELECTED_CONTAINER} (${CONTAINER_DESC_MAP[${SELECTED_CONTAINER}]})"
  return 0
}

# ================================
# 步骤2：检查压缩包中是否存在所需镜像和配置文件
# ================================
check_required_files() {
  log "INFO" "=================================================================="
  log "INFO" "步骤 2/4：检查所需镜像和配置文件"
  log "INFO" "=================================================================="
  
  local container_prefix="$1"
  local image_file="${CONTAINER_IMAGE_MAP[${container_prefix}]}"
  
  log "INFO" "检查 ${container_prefix} 容器所需文件..."
  log "INFO" "  所需镜像文件: ${image_file}"
  
  # 搜索镜像文件的可能位置
  local search_paths=(
    "${SCRIPT_DIR}/${image_file}"
    "$(pwd)/${image_file}"
    "/data/temp/${image_file}"
    "/home/containerUpdate/${image_file}"
  )
  
  FOUND_IMAGE_PATH=""
  for path in "${search_paths[@]}"; do
    if [[ -f "${path}" ]]; then
      FOUND_IMAGE_PATH="${path}"
      break
    fi
  done
  
  # 如果没找到，尝试在当前目录递归搜索
  if [[ -z "${FOUND_IMAGE_PATH}" ]]; then
    log "INFO" "在常用路径未找到，尝试在当前目录搜索..."
    local found_file
    found_file=$(find "${SCRIPT_DIR}" -maxdepth 2 -name "${image_file}" -type f 2>/dev/null | head -n1)
    if [[ -n "${found_file}" ]]; then
      FOUND_IMAGE_PATH="${found_file}"
    fi
  fi
  
  if [[ -z "${FOUND_IMAGE_PATH}" ]]; then
    log "ERROR" "❌ 未找到镜像文件: ${image_file}"
    log "ERROR" "请确保压缩包中包含该镜像文件，或手动指定路径"
    echo ""
    read -rp "是否手动输入镜像文件路径? (y/n) [默认 n]: " manual_input
    manual_input="${manual_input:-n}"
    if [[ "${manual_input}" =~ ^[yY]$ ]]; then
      read -rp "请输入镜像文件完整路径: " FOUND_IMAGE_PATH
      if [[ ! -f "${FOUND_IMAGE_PATH}" ]]; then
        log "ERROR" "镜像文件不存在: ${FOUND_IMAGE_PATH}"
        return 1
      fi
    else
      return 1
    fi
  fi
  
  log "INFO" "✅ 找到镜像文件: ${FOUND_IMAGE_PATH}"
  
  # 检查镜像文件大小
  local file_size
  file_size=$(du -h "${FOUND_IMAGE_PATH}" | awk '{print $1}')
  log "INFO" "   文件大小: ${file_size}"
  
  # 检查部署脚本自身
  if [[ -f "${SCRIPT_DIR}/container_update.sh" ]]; then
    log "INFO" "✅ 部署脚本存在: ${SCRIPT_DIR}/container_update.sh"
  fi
  
  # 根据容器类型检查额外配置文件（可选）
  case "${container_prefix}" in
    uemqx)
      log "INFO" "📋 EMQX 容器将使用远端已有配置或默认配置"
      ;;
    unginx)
      log "INFO" "📋 Nginx 容器将使用远端已有配置或默认配置"
      ;;
    uredis)
      log "INFO" "📋 Redis 容器将使用远端已有配置或默认配置"
      ;;
    *)
      log "INFO" "📋 ${container_prefix} 容器配置检查完成"
      ;;
  esac
  
  return 0
}

# ================================
# 步骤3：判断当前服务器是传统平台还是新统一平台
# ================================
detect_platform() {
  log "INFO" "=================================================================="
  log "INFO" "步骤 3/4：检测服务器平台类型"
  log "INFO" "=================================================================="
  
  local container_prefix="$1"
  
  log "INFO" "自动检测平台类型..."
  log "INFO" "  检测依据: /data/services 目录是否存在"
  
  if [[ -d "/data/services" ]]; then
    DETECTED_PLATFORM="new"
    log "INFO" "✅ 检测到 /data/services 目录存在"
    log "INFO" "✅ 识别为: 新统一平台 (使用 /data/ 目录结构)"
  else
    DETECTED_PLATFORM="old"
    log "INFO" "✅ 未检测到 /data/services 目录"
    log "INFO" "✅ 识别为: 传统平台 (使用 /var/www/ 目录结构)"
  fi
  
  # 确认平台类型
  echo ""
  local platform_display
  if [[ "${DETECTED_PLATFORM}" == "new" ]]; then
    platform_display="新统一平台"
  else
    platform_display="传统平台"
  fi
  
  read -rp "当前识别为 [${platform_display}]，是否正确? (y/n) [默认 y]: " platform_confirm
  platform_confirm="${platform_confirm:-y}"
  log_to_file "INFO" "用户确认平台类型: ${platform_confirm}"
  
  if [[ "${platform_confirm}" =~ ^[nN]$ ]]; then
    if [[ "${DETECTED_PLATFORM}" == "new" ]]; then
      DETECTED_PLATFORM="old"
      log "INFO" "⚠️ 已手动切换为: 传统平台"
    else
      DETECTED_PLATFORM="new"
      log "INFO" "⚠️ 已手动切换为: 新统一平台"
    fi
  fi
  
  # 检查 Nacos 和 Nginx 平台限制
  if [[ "${container_prefix}" == "unacos" || "${container_prefix}" == "unginx" ]]; then
    if [[ "${DETECTED_PLATFORM}" == "old" ]]; then
      log "ERROR" "❌ ${container_prefix} 仅支持新统一平台，无法在传统平台部署"
      return 1
    fi
  fi
  
  log "INFO" "✅ 平台类型确认: ${DETECTED_PLATFORM}"
  return 0
}

# ================================
# 步骤4：检查并停止正在运行的同类型容器
# ================================
check_and_stop_running_container() {
  log "INFO" "=================================================================="
  log "INFO" "步骤 4/4：检查并停止正在运行的容器"
  log "INFO" "=================================================================="
  
  local container_prefix="$1"
  
  log "INFO" "检查是否存在 ${container_prefix} 类型的容器在运行..."
  
  # 查找所有匹配的容器（包括运行中和已停止的）
  local all_containers
  all_containers=$(docker ps -a --format '{{.Names}}' 2>/dev/null | grep -E "^${container_prefix}[0-9]*$" | sort -V || true)
  
  if [[ -z "${all_containers}" ]]; then
    log "INFO" "✅ 未发现 ${container_prefix} 类型的容器"
    RUNNING_CONTAINER=""
    NEW_CONTAINER_NAME="${container_prefix}1"
    log "INFO" "   新容器将命名为: ${NEW_CONTAINER_NAME}"
    return 0
  fi
  
  log "INFO" "发现以下 ${container_prefix} 类型的容器:"
  echo "${all_containers}" | while read -r name; do
    local status
    status=$(docker inspect --format '{{.State.Status}}' "${name}" 2>/dev/null || echo "unknown")
    printf "     - %s (状态: %s)\n" "${name}" "${status}"
  done
  
  # 查找正在运行的容器
  local running_containers
  running_containers=$(docker ps --format '{{.Names}}' 2>/dev/null | grep -E "^${container_prefix}[0-9]*$" | sort -V || true)
  
  if [[ -n "${running_containers}" ]]; then
    # 获取最新运行的容器
    RUNNING_CONTAINER=$(echo "${running_containers}" | tail -n1)
    log "WARN" "⚠️ 检测到容器 ${RUNNING_CONTAINER} 正在运行"
    
    echo ""
    read -rp "是否停止该容器以便进行更新? (y/n) [默认 y]: " stop_confirm
    stop_confirm="${stop_confirm:-y}"
    log_to_file "INFO" "用户确认停止容器: ${stop_confirm}"
    
    if [[ "${stop_confirm}" =~ ^[yY]$ ]]; then
      log "INFO" "正在停止容器 ${RUNNING_CONTAINER}..."
      if docker stop "${RUNNING_CONTAINER}" >/dev/null 2>&1; then
        log "INFO" "✅ 容器 ${RUNNING_CONTAINER} 已停止"
      else
        log "ERROR" "❌ 停止容器 ${RUNNING_CONTAINER} 失败"
        return 1
      fi
    else
      log "WARN" "⚠️ 用户选择不停止容器，部署可能会因端口冲突而失败"
    fi
  else
    RUNNING_CONTAINER=""
    log "INFO" "✅ 没有正在运行的 ${container_prefix} 类型容器"
  fi
  
  # 确定新容器名称（自动递增编号）
  local latest_container
  latest_container=$(echo "${all_containers}" | tail -n1)
  
  if [[ -n "${latest_container}" ]]; then
    local last_num="${latest_container##*[!0-9]}"
    if [[ -z "${last_num}" ]]; then
      last_num=0
    fi
    local next_num=$((last_num + 1))
    NEW_CONTAINER_NAME="${container_prefix}${next_num}"
  else
    NEW_CONTAINER_NAME="${container_prefix}1"
  fi
  
  log "INFO" "✅ 新容器将命名为: ${NEW_CONTAINER_NAME}"
  return 0
}

# ================================
# 交互式模式主函数
# ================================
interactive_mode() {
  # 启用日志记录
  INTERACTIVE_LOG_ENABLED=true
  
  log "INFO" "=================================================================="
  log "INFO" "容器部署工具 - 交互式模式"
  log "INFO" "=================================================================="
  log "INFO" "日志文件: ${LOG_FILE}"
  log "INFO" "脚本目录: ${SCRIPT_DIR}"
  log "INFO" "=================================================================="
  echo ""
  
  # 步骤1：选择容器
  if ! select_container; then
    log "ERROR" "容器选择失败，退出"
    exit 1
  fi
  echo ""
  
  # 步骤2：检查所需文件
  if ! check_required_files "${SELECTED_CONTAINER}"; then
    log "ERROR" "文件检查失败，退出"
    exit 2
  fi
  echo ""
  
  # 步骤3：检测平台类型
  if ! detect_platform "${SELECTED_CONTAINER}"; then
    log "ERROR" "平台检测失败，退出"
    exit 3
  fi
  echo ""
  
  # 步骤4：检查并停止运行中的容器
  if ! check_and_stop_running_container "${SELECTED_CONTAINER}"; then
    log "ERROR" "容器检查失败，退出"
    exit 4
  fi
  echo ""
  
  # 显示部署信息摘要
  log "INFO" "=================================================================="
  log "INFO" "部署信息确认"
  log "INFO" "=================================================================="
  log "INFO" "  容器类型: ${SELECTED_CONTAINER} (${CONTAINER_DESC_MAP[${SELECTED_CONTAINER}]})"
  log "INFO" "  新容器名: ${NEW_CONTAINER_NAME}"
  log "INFO" "  平台类型: ${DETECTED_PLATFORM}"
  log "INFO" "  镜像文件: ${FOUND_IMAGE_PATH}"
  if [[ -n "${RUNNING_CONTAINER}" ]]; then
    log "INFO" "  旧容器:   ${RUNNING_CONTAINER} (已停止)"
  fi
  log "INFO" "=================================================================="
  echo ""
  
  read -rp "确认开始部署? (y/n) [默认 y]: " deploy_confirm
  deploy_confirm="${deploy_confirm:-y}"
  log_to_file "INFO" "用户确认部署: ${deploy_confirm}"
  
  if [[ ! "${deploy_confirm}" =~ ^[yY]$ ]]; then
    log "INFO" "用户取消部署"
    exit 0
  fi
  
  # 设置全局变量并执行部署
  DEPLOY_CONTAINER_NAME="${NEW_CONTAINER_NAME}"
  DEPLOY_IMAGE_TAR="${FOUND_IMAGE_PATH}"
  
  # 调用部署函数
  local deploy_fn
  case "${SELECTED_CONTAINER}" in
    ujava)
      deploy_fn="java_${DETECTED_PLATFORM}platform_x86"
      ;;
    uredis)
      deploy_fn="redis_${DETECTED_PLATFORM}platform_x86"
      ;;
    uemqx)
      deploy_fn="emqx_${DETECTED_PLATFORM}platform_x86"
      ;;
    upython)
      deploy_fn="python_${DETECTED_PLATFORM}platform_x86"
      ;;
    unacos)
      deploy_fn="nacos_newplatform_x86"
      ;;
    unginx)
      deploy_fn="nginx_newplatform_x86"
      ;;
  esac
  
  log "INFO" "=================================================================="
  log "INFO" "开始执行部署"
  log "INFO" "=================================================================="
  log "INFO" "开始部署 ${SELECTED_CONTAINER} (容器: ${DEPLOY_CONTAINER_NAME}, 平台: ${DETECTED_PLATFORM})"
  
  # 执行部署
  "${deploy_fn}"
  
  log "INFO" "=================================================================="
  log "INFO" "部署流程结束"
  log "INFO" "=================================================================="
  log "INFO" "日志文件已保存到: ${LOG_FILE}"
}

main() {
  # 检查是否为交互式模式
  if [[ $# -eq 0 ]] || [[ "$1" == "-i" ]] || [[ "$1" == "--interactive" ]]; then
    interactive_mode
    return 0
  fi
  
  # 命令行模式
  if [[ $# -lt 2 ]]; then
    usage
    exit 1
  fi

  local container_name="$1"
  local image_archive="$2"
  shift 2

  local platform_type="old"
  while [[ $# -gt 0 ]]; do
    case "$1" in
      --new-platform)
        platform_type="new"
        ;;
      *)
        usage
        exit 1
        ;;
    esac
    shift
  done

  if [[ -z "${container_name}" ]]; then
    log ERROR "容器名称不能为空"
    exit 2
  fi

  local image_tar_path
  if [[ "${image_archive}" = /* ]]; then
    image_tar_path="${image_archive}"
  else
    image_tar_path="$(pwd)/${image_archive}"
  fi

  if [[ ! -f "${image_tar_path}" ]]; then
    log ERROR "镜像文件不存在: ${image_tar_path}"
    exit 3
  fi

  local container_prefix
  if [[ "${container_name}" =~ ^([a-zA-Z]+)[0-9]+$ ]]; then
    container_prefix="${BASH_REMATCH[1]}"
  else
    container_prefix="${container_name}"
  fi

  local deploy_fn
  case "${container_prefix}" in
    ujava)
      if [[ "${platform_type}" == "new" ]]; then
        deploy_fn="java_newplatform_x86"
      else
        deploy_fn="java_oldplatform_x86"
      fi
      ;;
    uredis)
      if [[ "${platform_type}" == "new" ]]; then
        deploy_fn="redis_newplatform_x86"
      else
        deploy_fn="redis_oldplatform_x86"
      fi
      ;;
    uemqx)
      if [[ "${platform_type}" == "new" ]]; then
        deploy_fn="emqx_newplatform_x86"
      else
        deploy_fn="emqx_oldplatform_x86"
      fi
      ;;
    upython)
      if [[ "${platform_type}" == "new" ]]; then
        deploy_fn="python_newplatform_x86"
      else
        deploy_fn="python_oldplatform_x86"
      fi
      ;;
    nacos|unacos)
      if [[ "${platform_type}" == "new" ]]; then
        deploy_fn="nacos_newplatform_x86"
      else
        log ERROR "Nacos 服务仅支持新统一平台，请使用 --new-platform 参数"
        exit 4
      fi
      ;;
    unginx)
      if [[ "${platform_type}" == "new" ]]; then
        deploy_fn="nginx_newplatform_x86"
      else
        log ERROR "Nginx 服务仅支持新统一平台，请使用 --new-platform 参数"
        exit 4
      fi
      ;;
    *)
      log ERROR "容器 ${container_prefix} 暂未实现部署逻辑"
      exit 4
      ;;
  esac

  DEPLOY_CONTAINER_NAME="${container_name}"
  DEPLOY_IMAGE_TAR="${image_tar_path}"

  log INFO "开始部署 ${container_prefix} (容器: ${DEPLOY_CONTAINER_NAME}, 平台: ${platform_type}, 镜像: ${DEPLOY_IMAGE_TAR})"
  "${deploy_fn}"
  log INFO "部署流程结束"
}

release_port_if_busy() {
  local target_port="$1"
  local publish_info

  publish_info="$($sudoset lsof -i TCP:${target_port} -sTCP:LISTEN -t || true)"
  if [[ -z "${publish_info}" ]]; then
    log "INFO" "端口 ${target_port} 当前空闲"
    return 0
  fi

  log "WARN" "检测到端口 ${target_port} 被占用，尝试释放"
  while read -r pid; do
    [[ -z "${pid}" ]] && continue
    if $sudoset kill -9 "${pid}"; then
      log "INFO" "已结束占用端口 ${target_port} 的进程 PID=${pid}"
    else
      log "ERROR" "结束进程 ${pid} 失败"
      return 1
    fi
  done <<< "${publish_info}"
}

# --------------------------------------------------------------------------------------------------
# Java 容器部署函数（摘自 new_auto.sh -> java_x86），供后续本地或远端脚本复用。
# 新增逻辑：如检测到旧容器运行，则执行停止与移除操作后再部署，避免名称冲突。
# --------------------------------------------------------------------------------------------------
# 新统一平台的部署函数
java_newplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Java 服务 (Docker 版, x86)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-ujava2}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/java1.8.0_472.tar.gz}"
  local image_name="139.9.60.86:5000/ujava:v6"
  local image_id="a8ba5fa12b8e"

  local host_api="/data/services/api"
  local host_web="/data/services/web"
  local host_nginx_log="/data/middleware/nginx/nginx_log"
  local host_fdfs_data="/var/fdfs/storage/data"

  local port_args=(
    -p 8085:8085
    -p 8993:8993 -p 8994:8994 -p 8995:8995
    -p 8999:8999
    -p 8719:8719 -p 8720:8720
    -p 9204:9204 -p 9200:9200 -p 9201:9201
    -p 9905:9905 -p 9911:9911 -p 9908:9908
    -p 9906:9906 -p 9907:9907 -p 9909:9909 -p 9910:9910
    -p 30880:30880 -p 30881:30881 -p 30882:30882
    -p 30883:30883 -p 30884:30884
  )

  log "INFO" "🔍 检查 Java 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 正在运行，准备停止"
    $sudoset docker stop "${container_name}"
    log "INFO" "✅ 旧容器 ${container_name} 已停止"
  elif $sudoset docker ps -a --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 存在但未运行，跳过重启以免覆盖"
    log "INFO" "🛈 如需重新部署，请手动处理旧容器"
    return 1
  fi

  log "INFO" "🔍 检查 Java 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在。"
  else
    log "INFO" "❌ 镜像 ${image_name} 不存在，开始加载离线包..."
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if $sudoset docker load -i "${image_tar}"; then
      log "INFO" "🎉 镜像加载成功"
    else
      log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
      return 1
    fi
    $sudoset docker tag "${image_id}" "${image_name}" 2>/dev/null || true
    log "INFO" "🏷️ 镜像已标记为 ${image_name}"
  fi

  log "INFO" "🚀 正在启动 Java 容器: ${container_name} ..."
  $sudoset docker run -itd \
    --privileged \
    -v "${host_api}:/var/www/java/api" \
    -v "${host_web}:/var/www/java/web" \
    -v "${host_nginx_log}:/usr/local/nginx/logs" \
    -v /etc/localtime:/etc/localtime:ro \
    -v "${host_fdfs_data}:/var/fdfs/storage/data" \
    "${port_args[@]}" \
    --restart=always \
    --mac-address="02:42:ac:11:00:02" \
    --name "${container_name}" \
    --entrypoint "/var/www/java/api/start.sh" \
    "${image_name}"

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  log "INFO" "✅ Java 容器启动成功，等待初始化..."
  sleep 8

  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 Java 服务部署完成！"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}

# 传统平台的部署函数
java_oldplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Java 服务 (Docker 版, x86)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-ujava2}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/java1.8.0_472.tar.gz}"
  local image_name="139.9.60.86:5000/ujava:v6"
  local image_id="a8ba5fa12b8e"

  local host_java="/var/www/java"
  local host_nginx_log="/var/www/java/nginx-conf.d/nginx_log"
  local host_fdfs_data="/var/fdfs/storage/data"

  local port_args=(
    -p 554:554
    -p 1935:1935
    -p 10000:10000
    -p 2333:2333
    -p 2334:2334
    -p 8997:8997
    -p 8998:8998
    -p 8079:8079
    -p 443:443
    -p 8080:8080
    -p 8085:8085
    -p 8086:8086
    -p 8087:8087
    -p 8088:8088
    -p 8889:8889
    -p 8999:8999
  )

  log "INFO" "🔍 检查 Java 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 正在运行，准备停止"
    $sudoset docker stop "${container_name}"
    log "INFO" "✅ 旧容器 ${container_name} 已停止"
  elif $sudoset docker ps -a --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 存在但未运行，跳过重启以免覆盖"
    log "INFO" "🛈 如需重新部署，请手动处理旧容器"
    return 1
  fi

  log "INFO" "🔍 检查 Java 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在。"
  else
    log "INFO" "❌ 镜像 ${image_name} 不存在，开始加载离线包..."
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if $sudoset docker load -i "${image_tar}"; then
      log "INFO" "🎉 镜像加载成功"
    else
      log "ERROR" "⛔ 镜像加载失败，请检查文件完整性"
      return 1
    fi
    $sudoset docker tag "${image_id}" "${image_name}" 2>/dev/null || true
    log "INFO" "🏷️ 镜像已标记为 ${image_name}"
  fi

  log "INFO" "🚀 正在启动 Java 容器: ${container_name} ..."
  $sudoset docker run -itd \
    --privileged \
    -v "${host_java}:/var/www/java" \
    -v "${host_nginx_log}:/usr/local/nginx/logs" \
    -v /etc/localtime:/etc/localtime:ro \
    -v "${host_fdfs_data}:/var/fdfs/storage/data" \
    "${port_args[@]}" \
    --restart=always \
    --mac-address="02:42:ac:11:00:02" \
    --name "${container_name}" \
    "${image_name}" \
    /var/www/java/start.sh

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  log "INFO" "✅ Java 容器启动成功，等待初始化..."
  sleep 8

  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 Java 服务部署完成！"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}

# --------------------------------------------------------------------------------------------------
# Redis 容器部署函数（新平台 + 传统平台）
# --------------------------------------------------------------------------------------------------
redis_oldplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Redis 服务 (Docker 版, x86, 传统平台)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-uredis}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/redis8.2.2.tar.gz}"
  local image_name="139.9.60.86:5000/redis:v3"
  local image_id="3bd8c109f88b"
  local redis_conf_host="/var/www/redis/redis-8.2.2.conf"
  local redis_data_host="/var/www/redis/data"
  local redis_port="6379"
  local old_redis_conf="/var/www/redis/redis-6.0.3.conf"
  local new_redis_conf="/var/www/redis/redis-8.2.2.conf"

  log "INFO" "🔍 检查 Redis 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "✅ 容器 '${container_name}' 已在运行"
    return 0
  fi

  log "INFO" "🔍 检查 Redis 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在"
  else
    log "WARN" "❌ 镜像不存在，开始加载离线包"
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if $sudoset docker load -i "${image_tar}"; then
      log "INFO" "🎉 镜像加载成功"
    else
      log "ERROR" "⛔ 镜像加载失败"
      return 1
    fi
    $sudoset docker tag "${image_id}" "${image_name}" 2>/dev/null || true
  fi

  #检查端口是否被占用，如果有就杀死
  release_port_if_busy "${redis_port}"

  # 处理配置文件迁移（传统平台需要将旧版本配置重命名为新版本）
  if [[ -f "${new_redis_conf}" ]]; then
    log "INFO" "✅ 新版本配置文件 ${new_redis_conf} 已存在，跳过配置迁移"
  elif [[ -f "${old_redis_conf}" ]]; then
    log "INFO" "备份旧配置，并将旧配置命名为新版本配置"
    cp "${old_redis_conf}" "${old_redis_conf}.bak"
    mv "${old_redis_conf}" "${new_redis_conf}"
    log "INFO" "✅ 配置文件迁移完成"
  else
    log "ERROR" "⛔ 未找到 Redis 配置文件 (${old_redis_conf} 或 ${new_redis_conf})"
    log "ERROR" "请确保 /var/www/redis/ 目录下存在有效的 Redis 配置文件"
    return 1
  fi

  log "INFO" "🚀 正在启动 Redis 容器: ${container_name}"
  $sudoset docker run -itd \
    --name "${container_name}" \
    --restart=always \
    --security-opt seccomp=unconfined \
    --network host \
    -v "${new_redis_conf}:/etc/redis/redis.conf" \
    -v "${redis_data_host}:/data" \
    -v "/etc/localtime:/etc/localtime:ro" \
    "${image_name}" \
    redis-server /etc/redis/redis.conf --appendonly yes

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  log "INFO" "✅ Redis 容器启动成功，等待初始化..."
  sleep 5

  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 Redis 部署完成！"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}

redis_newplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Redis 服务 (Docker 版, x86, 新平台)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-uredis}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/redis8.2.2.tar.gz}"
  local image_name="139.9.60.86:5000/redis:v3"
  local image_id="3bd8c109f88b"
  local redis_conf_host="/data/middleware/redis/config/redis.conf"
  local redis_data_host="/data/middleware/redis/data"
  local redis_port="6379"

  log "INFO" "🔍 检查 Redis 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "✅ 容器 '${container_name}' 已在运行"
    return 0
  fi

  log "INFO" "🔍 检查 Redis 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在"
  else
    log "WARN" "❌ 镜像不存在，开始加载离线包"
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if $sudoset docker load -i "${image_tar}"; then
      log "INFO" "🎉 镜像加载成功"
    else
      log "ERROR" "⛔ 镜像加载失败"
      return 1
    fi
    $sudoset docker tag "${image_id}" "${image_name}" 2>/dev/null || true
  fi

  #检查端口是否被占用，如果有就杀死
  release_port_if_busy "${redis_port}"

  log "INFO" "🚀 正在启动 Redis 容器: ${container_name}"
  $sudoset docker run -itd \
    --name "${container_name}" \
    --restart=always \
    --security-opt seccomp=unconfined \
    --network host \
    -v "${redis_conf_host}:/etc/redis/redis.conf" \
    -v "${redis_data_host}:/data" \
    -v "/etc/localtime:/etc/localtime:ro" \
    "${image_name}" \
    redis-server /etc/redis/redis.conf --appendonly yes

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  log "INFO" "✅ Redis 容器启动成功，等待初始化..."
  sleep 5

  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 Redis 部署完成！"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}

# --------------------------------------------------------------------------------------------------
# EMQX 容器部署函数（新平台 + 传统平台）
# --------------------------------------------------------------------------------------------------
emqx_oldplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 EMQX 服务 (Docker 版, x86, 传统平台)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-uemqx2}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/uemqx5.8.4.tar.gz}"
  local image_name="139.9.60.86:5000/uemqx:v2"
  local emqx_config_host="/var/www/emqx/config"
  local emqx_data_host="/var/www/emqx/data"
  local emqx_log_host="/var/www/emqx/log"
  local source_emqx_dir="/data/middleware/emqx"
  local target_emqx_dir="/var/www/emqx"
  local backup_dir="${target_emqx_dir}_backup_$(date +%Y%m%d_%H%M%S)"

  log "INFO" "🔍 检查 EMQX 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 正在运行，准备停止"
    $sudoset docker stop "${container_name}"
    log "INFO" "✅ 旧容器 ${container_name} 已停止"
  fi

  log "INFO" "📂 假设远端 EMQX 目录已同步完成 (由 remote_update.sh 负责)"

  log "INFO" "🔍 检查 EMQX 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在"
  else
    log "WARN" "❌ 镜像不存在，开始加载离线包"
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if $sudoset docker load -i "${image_tar}"; then
      log "INFO" "🎉 镜像加载成功"
    else
      log "ERROR" "⛔ 镜像加载失败"
      return 1
    fi
  fi

  # 检查并释放端口
  local ports=(1883 8083 8084 8883 18083)
  for port in "${ports[@]}"; do
    release_port_if_busy "${port}"
  done

  log "INFO" "🚀 正在启动 EMQX 容器: ${container_name}"
  $sudoset docker run -d \
    --name "${container_name}" \
    --mac-address="02:42:ac:12:00:06" \
    --privileged \
    --cap-add ALL \
    --security-opt seccomp=unconfined \
    --security-opt apparmor=unconfined \
    --user root \
    -p 1883:1883 \
    -p 8083:8083 \
    -p 8084:8084 \
    -p 8883:8883 \
    -p 18083:18083 \
    -v "${emqx_config_host}:/opt/emqx/etc" \
    -v "${emqx_data_host}:/opt/emqx/data" \
    -v "${emqx_log_host}:/opt/emqx/log" \
    --restart=always \
    "${image_name}"

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  log "INFO" "✅ EMQX 容器启动成功，等待初始化..."
  sleep 5

  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 EMQX 部署完成！"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}

emqx_newplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 EMQX 服务 (Docker 版, x86, 新统一平台)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-uemqx2}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/uemqx5.8.4.tar.gz}"
  local image_name="139.9.60.86:5000/uemqx:v2"
  local emqx_config_host="/data/middleware/emqx/config"
  local emqx_data_host="/data/middleware/emqx/data"
  local emqx_log_host="/data/middleware/emqx/log"
  local source_emqx_dir="/data/middleware/emqx"
  local target_emqx_dir="/data/middleware/emqx"
  local backup_dir="${target_emqx_dir}_backup_$(date +%Y%m%d_%H%M%S)"

  log "INFO" "🔍 检查 EMQX 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 正在运行，准备停止"
    $sudoset docker stop "${container_name}"
    log "INFO" "✅ 旧容器 ${container_name} 已停止"
  fi

  log "INFO" "📂 假设远端 EMQX 目录已同步完成 (由 remote_update.sh 负责)"

  log "INFO" "🔍 检查 EMQX 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在"
  else
    log "WARN" "❌ 镜像不存在，开始加载离线包"
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if $sudoset docker load -i "${image_tar}"; then
      log "INFO" "🎉 镜像加载成功"
    else
      log "ERROR" "⛔ 镜像加载失败"
      return 1
    fi
  fi

  # 检查并释放端口
  local ports=(1883 8083 8084 8883 18083)
  for port in "${ports[@]}"; do
    release_port_if_busy "${port}"
  done

  log "INFO" "🚀 正在启动 EMQX 容器: ${container_name}"
  $sudoset docker run -d \
    --name "${container_name}" \
    --mac-address="02:42:ac:12:00:06" \
    --privileged \
    --cap-add ALL \
    --security-opt seccomp=unconfined \
    --security-opt apparmor=unconfined \
    --user root \
    -p 1883:1883 \
    -p 8083:8083 \
    -p 8084:8084 \
    -p 8883:8883 \
    -p 18083:18083 \
    -v "${emqx_config_host}:/opt/emqx/etc" \
    -v "${emqx_data_host}:/opt/emqx/data" \
    -v "${emqx_log_host}:/opt/emqx/log" \
    --restart=always \
    "${image_name}"

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  log "INFO" "✅ EMQX 容器启动成功，等待初始化..."
  sleep 5

  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 EMQX 部署完成！"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}

# --------------------------------------------------------------------------------------------------
# Python 容器部署函数（新平台 + 传统平台）
# --------------------------------------------------------------------------------------------------
python_newplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Python 服务 (Docker 版, x86, 新平台)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-upython}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/python_v15.tar.gz}"
  local image_name="139.9.60.86:5000/upython:v15"
  local host_app_dir="/data/services/api/python-cmdb"
  local container_app_dir="/var/www/html"
  local container_start_cmd="/var/www/html/start.sh"
  local mac_addr="02:42:ac:11:00:07"
  local ports=(-p 8000:8000 -p 8002:8002)

  if [[ ! -d "${host_app_dir}" ]]; then
    log "ERROR" "⛔ 应用目录不存在: ${host_app_dir}"
    return 1
  fi
  if [[ ! -f "${host_app_dir}/start.sh" ]]; then
    log "ERROR" "⛔ 启动脚本缺失: ${host_app_dir}/start.sh"
    return 1
  fi

  log "INFO" "🔍 检查 Python 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 正在运行，准备停止"
    $sudoset docker stop "${container_name}"
    log "INFO" "✅ 旧容器 ${container_name} 已停止"
  elif $sudoset docker ps -a --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 存在但未运行，避免覆盖"
    log "INFO" "🛈 如需重新部署，请手动清理旧容器"
    return 1
  fi

  log "INFO" "🔍 检查 Python 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在"
  else
    log "INFO" "📦 镜像缺失，开始加载: ${image_tar}"
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if ! $sudoset docker load -i "${image_tar}"; then
      log "ERROR" "⛔ 镜像加载失败"
      return 1
    fi
    log "INFO" "🎉 镜像加载完成"
  fi

  log "INFO" "🚀 正在启动 Python 容器: ${container_name}"
  $sudoset docker run -d \
    --name "${container_name}" \
    --restart=always \
    --mac-address="${mac_addr}" \
    --privileged \
    "${ports[@]}" \
    -v "${host_app_dir}:${container_app_dir}" \
    -v "/etc/localtime:/etc/localtime:ro" \
    "${image_name}" \
    "${container_start_cmd}"

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  sleep 5
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 Python 服务部署完成 (新平台)"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}

python_oldplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Python 服务 (Docker 版, x86, 传统平台)"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-upython}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/python_v15.tar.gz}"
  local image_name="139.9.60.86:5000/upython:v15"
  local host_html_dir="/var/www/html"
  local container_start_cmd="/var/www/html/start.sh"
  local mac_addr="02:42:ac:11:00:06"
  local ports=(-p 8000:8000 -p 8002:8002 -p 8443:8443 -p 9009:9009)

  if [[ ! -d "${host_html_dir}" ]]; then
    log "ERROR" "⛔ 目录不存在: ${host_html_dir}"
    return 1
  fi
  if [[ ! -f "${host_html_dir}/start.sh" ]]; then
    log "ERROR" "⛔ 启动脚本缺失: ${host_html_dir}/start.sh"
    return 1
  fi

  log "INFO" "🔍 检查 Python 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 正在运行，准备停止"
    $sudoset docker stop "${container_name}"
    log "INFO" "✅ 旧容器 ${container_name} 已停止"
  elif $sudoset docker ps -a --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 存在但未运行，避免覆盖"
    log "INFO" "🛈 如需重新部署，请手动清理旧容器"
    return 1
  fi

  log "INFO" "🔍 检查 Python 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "${image_name}"; then
    log "INFO" "✅ 镜像 ${image_name} 已存在"
  else
    log "INFO" "📦 镜像缺失，开始加载: ${image_tar}"
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if ! $sudoset docker load -i "${image_tar}"; then
      log "ERROR" "⛔ 镜像加载失败"
      return 1
    fi
    log "INFO" "🎉 镜像加载完成"
  fi

  log "INFO" "🚀 正在启动 Python 容器: ${container_name}"
  $sudoset docker run -itd  -p 8002:8002 -p 8000:8000 -p 8443:8443 -p 9009:9009 -v /var/www/html:/var/www/html -v /etc/localtime:/etc/localtime:ro --restart=always --mac-address="02:42:ac:11:00:06" --privileged --name=upython ${image_name} /var/www/html/start.sh

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ 容器启动失败"
    return 1
  fi

  sleep 5
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "INFO" "🎉 Python 服务部署完成 (传统平台)"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
    log "INFO" "▶️ 触发容器内应用启动命令"
    $sudoset docker exec "${container_name}" /bin/bash -c "cd /var/www/html && nohup python manage.py runserver 0.0.0.0:8000 >/tmp/python_manage.log 2>&1 &" || true
  else
    log "ERROR" "⛔ 容器启动后未运行，请检查日志: docker logs ${container_name}"
    return 1
  fi

  return 0
}

# --------------------------------------------------------------------------------------------------
# Nacos 容器部署函数（仅新平台）
# --------------------------------------------------------------------------------------------------
nacos_newplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Nacos Server (Standalone) - x86, 新统一平台"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-unacos}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/nacos-server-v2.5.2.tar.gz}"
  local image_tag="nacos-server:v2.5.2"
  local host_data_dir="/data/middleware/nacos"

  log "INFO" "🔍 检查 Nacos 容器是否已运行..."
  if $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 正在运行，准备停止"
    $sudoset docker stop "${container_name}"
    log "INFO" "✅ 旧容器 ${container_name} 已停止"
  elif $sudoset docker ps -a --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "WARN" "⚠️ 检测到旧容器 '${container_name}' 存在但未运行，跳过部署以免覆盖"
    log "INFO" "🛈 如需重新部署，请手动处理旧容器"
    return 1
  fi

  log "INFO" "🔍 检查 Nacos 镜像是否存在..."
  if $sudoset docker images --format '{{.Repository}}:{{.Tag}}' | grep -wq "^${image_tag}$"; then
    log "INFO" "✅ 镜像 ${image_tag} 已存在"
  else
    log "WARN" "❌ 镜像不存在，开始加载离线包"
    if [[ ! -f "${image_tar}" ]]; then
      log "ERROR" "⛔ 镜像文件不存在: ${image_tar}"
      return 1
    fi
    if $sudoset docker load -i "${image_tar}"; then
      log "INFO" "🎉 镜像加载成功"
    else
      log "ERROR" "⛔ 镜像加载失败，请检查文件是否完整或已损坏"
      return 1
    fi
  fi

  # 检查并释放端口
  local ports=(8848 9848)
  for port in "${ports[@]}"; do
    release_port_if_busy "${port}"
  done

  log "INFO" "🚀 正在启动 Nacos 容器: ${container_name}"
  $sudoset docker run -d \
    --name "${container_name}" \
    --restart=always \
    -p 8848:8848 \
    -p 9848:9848 \
    -e MODE=standalone \
    -v "${host_data_dir}:/home/nacos" \
    --mac-address="02:42:ac:11:00:10" \
    "${image_tag}"

  if [[ $? -ne 0 ]]; then
    log "ERROR" "⛔ Nacos 容器启动失败"
    return 1
  fi

  log "INFO" "✅ Nacos 容器已启动，名称: ${container_name}"

  # 状态确认（容器是否仍在运行）
  sleep 5
  if ! $sudoset docker ps --format '{{.Names}}' | grep -wq "^${container_name}$"; then
    log "ERROR" "⛔ 容器启动后退出，请检查日志: $sudoset docker logs ${container_name}"
    return 1
  fi

  # 获取本机 IP 地址
  local server_ip=$(ip addr show | grep -E 'inet\s+(192\.168|172\.1[6789]\.|172\.2[0-9]\.|172\.3[01]\.|10\.)' | awk '{print $2}' | cut -d/ -f1 | head -n1)
  
  if [[ -z "${server_ip}" ]]; then
    # 回退到 hostname -I
    server_ip=$(hostname -I | awk '{print $1}')
  fi

  # 输出访问信息
  if [[ -n "${server_ip}" ]]; then
    log "INFO" "✅ Nacos 部署完成！"
    log "🎉 可通过浏览器访问: http://${server_ip}:8848/nacos"
    log "💡 默认账号密码: nacos / nacos"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  else
    log "INFO" "🎉 Nacos 部署完成。"
    log "💡 默认账号密码: nacos / nacos"
    log "📌 无法获取服务器 IP，请手动确认访问地址: http://<your-ip>:8848/nacos"
    $sudoset docker ps --filter "name=${container_name}" --format "table {{.Names}}\t{{.Status}}\t{{.Ports}}"
  fi

  return 0
}

# --------------------------------------------------------------------------------------------------
# Nginx 容器部署函数（仅新平台）
# --------------------------------------------------------------------------------------------------
nginx_newplatform_x86() {
  log "INFO" "=================================================================="
  log "INFO" "开始部署 Nginx (离线编译安装) - x86"
  log "INFO" "=================================================================="

  : "${sudoset:=}"

  local container_name="${DEPLOY_CONTAINER_NAME:-unginx}"
  local image_tar="${DEPLOY_IMAGE_TAR:-/data/temp/nginx-1.29.3.tar.gz}"
  local nginx_version="1.29.3"
  local nginx_image="nginx:${nginx_version}"
  local required_dirs=(
    "/data/middleware/nginx/log"
    "/data/middleware/nginx/data/cache"
    "/data/middleware/nginx/data/html"
    "/data/middleware/nginx/config"
    "/data/services/web"
    "/data/security/nginx_cert"
  )

  # 目录预检查，缺失则创建
  for dir in "${required_dirs[@]}"; do
    if [ ! -d "$dir" ]; then
      log "WARN" "目录 $dir 不存在，自动创建"
      $sudoset mkdir -p "$dir" || {
        log "ERROR" "目录 $dir 创建失败"
        return 1
      }
    fi
  done

  # 创建 nginx 用户和组（如果不存在）
  if ! getent group nginx >/dev/null 2>&1; then
    log "INFO" "创建 nginx 组..."
    $sudoset groupadd -r nginx
  else
    log "INFO" "nginx 组已存在，跳过创建"
  fi

  if ! id nginx >/dev/null 2>&1; then
    log "INFO" "创建 nginx 用户..."
    $sudoset useradd -r -g nginx -s /sbin/nologin -d /var/cache/nginx nginx
  else
    log "INFO" "nginx 用户已存在，跳过创建"
  fi

  # 确认 nginx 用户存在
  if ! id nginx >/dev/null 2>&1; then
    log "ERROR" "nginx 用户创建失败"
    return 1
  fi
  log "INFO" "nginx 用户验证通过"
  #赋予权限
  $sudoset chown -R nginx:nginx /data/middleware/nginx/data/cache
  $sudoset chmod -R 755 /data/middleware/nginx/data/cache

  # 赋予权限
  $sudoset chown -R 1000:1000 /data/middleware/nginx/log
  $sudoset chown -R 1000:1000 /data/middleware/nginx/data/cache
  $sudoset chmod -R 700 /data/middleware/nginx/data/cache

  # 校验镜像包
  if [ ! -s "$image_tar" ]; then
    log "ERROR" "离线包 $image_tar 不存在或为空"
    return 1
  fi

  # 加载镜像文件
  if ! $sudoset docker load -i "$image_tar"; then
    log "ERROR" "镜像加载失败"
    return 1
  fi
  log "INFO" "镜像加载成功"

  # 检查容器是否存在，如果存在则停止并删除
  if $sudoset docker ps --format '{{.Names}}' | grep -qw "$container_name"; then
    log "WARN" "检测到容器 $container_name 正在运行，准备停止"
    $sudoset docker stop "$container_name"
    log "INFO" "容器 $container_name 已停止"
  fi

  # 生成容器
  if ! $sudoset docker run -d \
    --name "$container_name" \
    --mac-address="02:42:ac:11:00:25" \
    --restart=always \
    -p 443:443 \
    -v /data/middleware/nginx/config:/etc/nginx/conf.d \
    -v /data/middleware/nginx/data/html:/usr/share/nginx/html \
    -v /data/middleware/nginx/log:/var/log/nginx \
    -v /data/middleware/nginx/data/cache:/var/cache/nginx \
    -v /data/services/web:/data/services/web:rw \
    -v /data/security/nginx_cert:/data/security/nginx_cert:ro \
    "$nginx_image"; then
    log "ERROR" "容器 $container_name 启动失败"
    return 1
  fi

  log "INFO" "容器 $container_name 启动成功"
  return 0
}

main "$@"