#!/usr/bin/env bash
set -euo pipefail

# 生成自签名 SSL 证书（含 SAN）脚本
# 用法示例：
#  ./generate_ssl.sh --domain ubains.com --out /etc/ssl/ubains --nginx-conf /etc/nginx/conf.d/unified443.conf --yes

DOMAIN_DEFAULT="ubains.com"
DAYS_DEFAULT=36500
OUT_DIR="."
DOMAIN=""
IP=""
NGINX_CONF=""
FORCE_NO_PROMPT=0

print_usage(){
  cat <<EOF
Usage: $0 [--domain DOMAIN] [--ip IP] [--out DIR] [--days N] [--nginx-conf FILE] [--yes]

Options:
  --domain DOMAIN     主机名 (默认: ${DOMAIN_DEFAULT})
  --ip IP             指定 SAN 中的 IP（默认自动检测）
  --out DIR           输出目录，默认当前目录
  --days N            证书有效天数 (默认 ${DAYS_DEFAULT})
  --nginx-conf FILE   可选：将证书路径写入指定的 nginx 配置并重载（脚本会备份原文件）
  --yes               不交互确认，直接执行替换 nginx 配置
  -h, --help          显示本帮助
EOF
}

# 解析参数
while [[ $# -gt 0 ]]; do
  case "$1" in
    --domain) DOMAIN="$2"; shift 2;;
    --ip) IP="$2"; shift 2;;
    --out) OUT_DIR="$2"; shift 2;;
    --days) DAYS="$2"; shift 2;;
    --nginx-conf) NGINX_CONF="$2"; shift 2;;
    --yes) FORCE_NO_PROMPT=1; shift 1;;
    -h|--help) print_usage; exit 0;;
    *) echo "Unknown arg: $1"; print_usage; exit 1;;
  esac
done

DOMAIN=${DOMAIN:-$DOMAIN_DEFAULT}
DAYS=${DAYS:-$DAYS_DEFAULT}

# 检测主机 IPv4（非回环）
detect_ip(){
  local ip
  ip=$(hostname -I 2>/dev/null | awk '{print $1}' || true)
  if [ -z "$ip" ]; then
    ip=$(ip route get 1.1.1.1 2>/dev/null | awk '/src/ {for(i=1;i<=NF;i++) if($i=="src") print $(i+1)}' | head -n1 || true)
  fi
  if [ -z "$ip" ]; then
    ip=$(ip addr show scope global 2>/dev/null | awk '/inet /{print $2}' | cut -d/ -f1 | grep -v '^127\.' | head -n1 || true)
  fi
  echo "${ip:-127.0.0.1}"
}

if [ -z "$IP" ]; then
  IP=$(detect_ip)
fi

mkdir -p "$OUT_DIR"
cd "$OUT_DIR"

# 日志文件
LOG_FILE="$(pwd)/generate_ssl.log"
touch "$LOG_FILE"

# 颜色与日志函数
COL_RESET="\033[0m"
COL_RED="\033[31m"
COL_GREEN="\033[32m"
COL_YELLOW="\033[33m"
COL_BLUE="\033[34m"

log_to_file(){
  echo "$(date +'%F %T') $*" >> "$LOG_FILE"
}

info(){
  echo -e "${COL_BLUE}[INFO]${COL_RESET} $*"
  log_to_file "[INFO] $*"
}

success(){
  echo -e "${COL_GREEN}[OK]${COL_RESET} $*"
  log_to_file "[OK] $*"
}

warn(){
  echo -e "${COL_YELLOW}[WARN]${COL_RESET} $*" >&2
  log_to_file "[WARN] $*"
}

err(){
  echo -e "${COL_RED}[ERROR]${COL_RESET} $*" >&2
  log_to_file "[ERROR] $*"
}

SAN_CONF="san.cnf"
KEY_FILE="server.key"
CRT_FILE="server.crt"

cat > "$SAN_CONF" <<EOF
[req]
default_bits = 2048
prompt = no
default_md = sha256
distinguished_name = dn
req_extensions = v3_req

[dn]
CN = ${DOMAIN}
O = UBAINS
OU = DevOps
C = CN
ST = Guangdong
L = Shenzhen
emailAddress = admin@ubains.com

[v3_req]
basicConstraints = CA:TRUE
keyUsage = critical, digitalSignature, keyEncipherment, keyCertSign
extendedKeyUsage = serverAuth
subjectAltName = @alt_names

[alt_names]
DNS.1 = ${DOMAIN}
DNS.2 = localhost
IP.1 = ${IP}
EOF

info "生成配置文件: $(pwd)/${SAN_CONF} (CN=${DOMAIN}, IP=${IP})"

# 生成私钥与自签名证书
if ! command -v openssl >/dev/null 2>&1; then
  err "未找到 openssl，请先安装 OpenSSL。"
  exit 2
fi

info "开始生成私钥与自签名证书（这可能需要几秒）..."
if openssl req -x509 \
  -newkey rsa:2048 \
  -nodes \
  -keyout "$KEY_FILE" \
  -out "$CRT_FILE" \
  -days "$DAYS" \
  -config "$SAN_CONF" \
  -extensions v3_req; then
  chmod 600 "$KEY_FILE"
  success "已生成： 私钥: $(pwd)/${KEY_FILE}  证书: $(pwd)/${CRT_FILE}"
else
  err "OpenSSL 生成证书失败。请检查 OpenSSL 输出与 $SAN_CONF 的配置。详细日志见 $LOG_FILE"
  exit 3
fi

# 可选：更新 nginx 配置，将证书路径写入并重载
inject_into_nginx_conf(){
  local conf="$1"
  if [ ! -f "$conf" ]; then
    warn "跳过：配置文件不存在： $conf"
    return 1
  fi
  local backup
  backup="${conf}.bak.$(date +%s)"
  cp "$conf" "$backup"
  info "已备份原配置到： $backup"

  sed -E -i \
    -e "s|^\s*ssl_certificate\s+.+;|    ssl_certificate      $(pwd)/${CRT_FILE};|g" \
    -e "s|^\s*ssl_certificate_key\s+.+;|    ssl_certificate_key  $(pwd)/${KEY_FILE};|g" \
    "$conf"

  if ! grep -q "ssl_certificate" "$conf"; then
    awk -v crt="$(pwd)/${CRT_FILE}" -v key="$(pwd)/${KEY_FILE}" '
      /server[[:space:]]*\{/ {print; in_server=1; next}
      in_server && /\}/ {print "    ssl_certificate      " crt ";"; print "    ssl_certificate_key  " key ";"; in_server=0}
      {print}
    ' "$conf" > "${conf}.tmp" && mv "${conf}.tmp" "$conf"
  fi

  info "已将证书路径写入： $conf"

  # 在配置文件目录下创建证书存放目录，并把证书移动/复制到该目录
  local conf_dir
  conf_dir=$(dirname "$conf")
  local cert_store_dir="$conf_dir/certs"
  mkdir -p "$cert_store_dir"
  local src_key
  src_key="$(pwd)/${KEY_FILE}"
  local src_crt
  src_crt="$(pwd)/${CRT_FILE}"
  local dest_key="$cert_store_dir/server.key"
  local dest_crt="$cert_store_dir/server.crt"
  if [ -f "$src_key" ]; then
    cp -f "$src_key" "$dest_key"
    chmod 600 "$dest_key" || true
  else
    warn "未找到源私钥 $src_key，跳过复制私钥。"
  fi
  if [ -f "$src_crt" ]; then
    cp -f "$src_crt" "$dest_crt"
    chmod 644 "$dest_crt" || true
  else
    warn "未找到源证书 $src_crt，跳过复制证书。"
  fi

  info "已将证书复制到： $cert_store_dir"

  # 替换 nginx 配置，使用证书目标路径（而非脚本所在路径）
  sed -E -i \
    -e "s|^\s*ssl_certificate\s+.+;|    ssl_certificate      ${dest_crt};|g" \
    -e "s|^\s*ssl_certificate_key\s+.+;|    ssl_certificate_key  ${dest_key};|g" \
    "$conf"

  if ! grep -q "ssl_certificate" "$conf"; then
    awk -v crt="${dest_crt}" -v key="${dest_key}" '
      /server[[:space:]]*\{/ {print; in_server=1; next}
      in_server && /\}/ {print "    ssl_certificate      " crt ";"; print "    ssl_certificate_key  " key ";"; in_server=0}
      {print}
    ' "$conf" > "${conf}.tmp" && mv "${conf}.tmp" "$conf"
  fi

  info "已将证书路径写入： $conf (指向 $cert_store_dir)"
  local RELOAD_OK=0

  # 1) 优先尝试本地 nginx 可用时的 reload
  if command -v nginx >/dev/null 2>&1; then
    info "检测到本地 nginx 可执行文件，尝试 nginx -t && nginx -s reload"
    if nginx -t >/dev/null 2>&1; then
      if nginx -s reload >/dev/null 2>&1; then
        RELOAD_OK=1
      else
        warn "nginx -s reload 返回非零，继续尝试其他重载方式。"
      fi
    else
      warn "nginx -t 检测失败，跳过本地 reload。"
    fi
  fi

  # 2) 尝试 systemctl reload
  if [ $RELOAD_OK -eq 0 ] && command -v systemctl >/dev/null 2>&1; then
    info "尝试使用 systemctl reload nginx"
    if systemctl reload nginx >/dev/null 2>&1; then
      RELOAD_OK=1
    else
      warn "systemctl reload nginx 失败。"
    fi
  fi

  # 3) 尝试 Docker 环境中的常见容器名或 nginx 镜像
  if [ $RELOAD_OK -eq 0 ] && command -v docker >/dev/null 2>&1; then
    info "尝试在 Docker 容器中重载 nginx（寻找常见容器名或 nginx 镜像）"
    # 首先检查常见容器名
    DOCKER_TARGETS=()
    for name in nginx ujava2 upython; do
      if docker ps --format '{{.Names}}' | grep -wq "$name"; then
        DOCKER_TARGETS+=("$name")
      fi
    done
    # 再查找镜像名包含 nginx 的容器
    while read -r line; do
      c_name=$(echo "$line" | awk '{print $1}')
      c_image=$(echo "$line" | awk '{print $2}')
      if echo "$c_image" | grep -qi nginx; then
        DOCKER_TARGETS+=("$c_name")
      fi
    done < <(docker ps --format '{{.Names}} {{.Image}}')

    # 去重
    if [ ${#DOCKER_TARGETS[@]} -gt 0 ]; then
      mapfile -t unique_targets < <(printf "%s\n" "${DOCKER_TARGETS[@]}" | awk '!x[$0]++')
      for ct in "${unique_targets[@]}"; do
        info "尝试在容器 $ct 内重载 nginx（优先尝试 nginx -t && nginx -s reload）"
        if docker exec "$ct" /usr/local/nginx/sbin/nginx -t >/dev/null 2>&1; then
          if docker exec "$ct" /usr/local/nginx/sbin/nginx -s reload >/dev/null 2>&1; then
            RELOAD_OK=1
            info "容器 $ct 内 nginx 重载成功。"
            break
          else
            warn "容器 $ct 内 nginx -s reload 失败，尝试 docker restart $ct"
            if docker restart "$ct" >/dev/null 2>&1; then
              RELOAD_OK=1
              info "已重启容器 $ct。"
              break
            fi
          fi
        else
          warn "容器 $ct 中未找到 /usr/local/nginx/sbin/nginx 或检测失败，尝试 docker restart $ct"
          if docker restart "$ct" >/dev/null 2>&1; then
            RELOAD_OK=1
            info "已重启容器 $ct。"
            break
          fi
        fi
      done
    else
      warn "未发现疑似 nginx 的 Docker 容器。"
    fi
  fi

  if [ $RELOAD_OK -eq 1 ]; then
    success "nginx 重载成功。"
  else
    err "未能自动重载 nginx。请根据实际运行环境手动重载或重启相应容器/服务。"
    # 提示建议命令
    if command -v docker >/dev/null 2>&1; then
      err "如果使用 Docker，请运行: docker ps --format '{{.Names}} {{.Image}}' 查看容器，然后对相应容器执行 docker restart <name> 或 docker exec <name> /usr/local/nginx/sbin/nginx -s reload"
    else
      err "可尝试： nginx -t && nginx -s reload 或 systemctl reload nginx 或 docker restart <container>"
    fi
  fi
  return 0
}

# 预定义候选 nginx 配置路径（根据 PRD）
CANDIDATES=(
  "/data/middleware/nginx/config/unified443.conf"
  "/var/www/java/nginx-conf.d/meeting443.conf"
  "/var/www/html/nginx-conf/moblie8081.conf"
  "/var/www/html/nginx-conf/rms8443.conf"
)

FOUND=()
for p in "${CANDIDATES[@]}"; do
  if [ -f "$p" ]; then
    FOUND+=("$p")
  fi
done

if [ -n "$NGINX_CONF" ]; then
  # 用户指定单个配置文件，优先处理它
  inject_into_nginx_conf "$NGINX_CONF" || true
else
  if [ ${#FOUND[@]} -gt 0 ]; then
    info "检测到以下可能的 nginx 配置文件："
    for f in "${FOUND[@]}"; do
      echo -e "  - ${COL_YELLOW}$f${COL_RESET}"
    done
    if [ $FORCE_NO_PROMPT -eq 0 ]; then
      read -r -p "是否一键替换以上所有配置并重载 nginx？ [y/N]: " ans
    else
      ans="y"
    fi
    case "$ans" in
      y|Y|yes|YES)
        for f in "${FOUND[@]}"; do
          inject_into_nginx_conf "$f" || true
        done
        ;;
      *) info "已取消自动替换。你仍然可以手动运行脚本并指定 --nginx-conf 来替换。";;
    esac
  else
    warn "未检测到预定义的 nginx 配置文件。若要手动指定，请使用 --nginx-conf PATH。"
  fi
fi

info "下一步：将 $(pwd)/${CRT_FILE} 导入到受信任的根证书颁发机构，以在浏览器中建立信任。"
echo "Windows: 双击证书并安装到 '受信任的根证书颁发机构'。"
echo "macOS: 双击并在钥匙串访问中设为始终信任。"
echo "Linux(Ubuntu/Debian): sudo cp ${CRT_FILE} /usr/local/share/ca-certificates/ubains-ca.crt && sudo update-ca-certificates"
echo "Firefox: 在设置 → 隐私与安全 → 证书 → 证书机构 中导入并信任此证书。"
success "完成。详细操作与错误请查看日志: $LOG_FILE"