#!/bin/bash

# 统信系统安全配置检测脚本
# 功能：检测系统安全配置状态并输出详细日志
# 作者：Security Team
# 日期：$(date +%Y-%m-%d)

# 日志配置
LOG_FILE="/tmp/security_check_$(date +%Y%m%d_%H%M%S).log"
SCRIPT_DIR=$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)

# 日志函数
log() {
    local level=$1
    shift
    local message="$*"
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')
    echo "[$timestamp] [$level] $message" | tee -a "$LOG_FILE"
}

# 初始化检查结果
declare -A check_results

# 检查OpenSSH安全配置
check_ssh_security() {
    log INFO "开始检查OpenSSH安全配置..."
    
    local checks=(
        "X11Forwarding=no"
        "MaxAuthTries=4"
        "IgnoreRhosts=yes"
        "HostbasedAuthentication=no"
        "PermitEmptyPasswords=no"
        "PermitRootLogin=no"
        "Protocol=2"
    )
    
    local total_checks=${#checks[@]}
    local passed_checks=0
    
    for check in "${checks[@]}"; do
        param="${check%=*}"
        expected="${check#*=}"
        
        actual=$(grep "^$param" /etc/ssh/sshd_config 2>/dev/null | head -n1 | cut -d' ' -f2)
        if [[ "$actual" == "$expected" ]]; then
            log INFO "  ✓ $param = $expected (当前值: $actual)"
            ((passed_checks++))
        else
            log WARN "  ✗ $param = $expected (当前值: $actual)"
        fi
    done
    
    # 打印当前SSH配置详情，用于说明配置状态
    log INFO "  当前SSH配置详情："
    for check in "${checks[@]}"; do
        param="${check%=*}"
        grep_result=$(grep "^$param" /etc/ssh/sshd_config 2>/dev/null | head -n1)
        if [[ -n "$grep_result" ]]; then
            log INFO "    $grep_result"
        else
            log INFO "    $param 参数未找到"
        fi
    done
    
    log INFO "OpenSSH安全配置检查完成: $passed_checks/$total_checks 项通过"
    check_results["ssh_security"]="$passed_checks/$total_checks"
}

# 检查命令行界面超时设置
check_timeout_setting() {
    log INFO "开始检查命令行界面超时设置..."
    
    local tmout_value=$(grep -E '^TMOUT=' /etc/profile 2>/dev/null | cut -d'=' -f2)
    local export_tmout=$(grep -E 'export TMOUT' /etc/profile 2>/dev/null)
    
    if [[ -n "$tmout_value" ]] && [[ -n "$export_tmout" ]] && [[ "$tmout_value" -le 300 ]]; then
        log INFO "  ✓ TMOUT 设置为 $tmout_value 秒"
        check_results["timeout_setting"]="1/1"
    else
        log WARN "  ✗ TMOUT 未正确设置 (当前值: $tmout_value)"
        check_results["timeout_setting"]="0/1"
    fi
    
    # 打印当前TMOUT配置详情
    log INFO "  当前TMOUT配置详情："
    tmout_line=$(grep -E '^TMOUT=|^export TMOUT' /etc/profile 2>/dev/null)
    if [[ -n "$tmout_line" ]]; then
        while IFS= read -r line; do
            log INFO "    $line"
        done <<< "$tmout_line"
    else
        log INFO "    未找到TMOUT相关配置"
    fi
    
}

# 检查PAM认证模块配置
check_pam_auth() {
    log INFO "开始检查PAM认证模块配置..."
    
    local auth_lines=$(grep -E "auth\s+sufficient\s+pam_rootok\.so|auth\s+required\s+pam_wheel\.so" /etc/pam.d/su 2>/dev/null | wc -l)
    
    if [[ $auth_lines -ge 2 ]]; then
        log INFO "  ✓ PAM认证模块配置正确"
        check_results["pam_auth"]="1/1"
    else
        log WARN "  ✗ PAM认证模块配置不正确"
        check_results["pam_auth"]="0/1"
    fi
    
    # 打印当前PAM配置详情
    log INFO "  当前PAM su配置详情："
    pam_lines=$(grep -E "auth.*(pam_rootok|pam_wheel)" /etc/pam.d/su 2>/dev/null)
    if [[ -n "$pam_lines" ]]; then
        while IFS= read -r line; do
            log INFO "    $line"
        done <<< "$pam_lines"
    else
        log INFO "    未找到PAM认证相关配置"
    fi
}

# 检查Telnet服务状态
check_telnet_status() {
    log INFO "开始检查Telnet服务状态..."
    
    # 检查telnet是否已禁用
    local telnet_disabled=$(grep -E "^disable" /etc/xinetd.d/telnet 2>/dev/null | grep -o "yes" || echo "")
    
    if [[ -n "$telnet_disabled" ]]; then
        log INFO "  ✓ Telnet服务已禁用"
        check_results["telnet_status"]="1/1"
    else
        log WARN "  ✗ Telnet服务未禁用"
        check_results["telnet_status"]="0/1"
    fi
    
    # 打印当前Telnet配置详情
    log INFO "  当前Telnet配置详情："
    telnet_conf=$(grep -E "^disable" /etc/xinetd.d/telnet 2>/dev/null)
    if [[ -n "$telnet_conf" ]]; then
        log INFO "    $telnet_conf"
    else
        log INFO "    Telnet配置未找到或服务未安装"
    fi
}

# 检查SSH服务状态
check_ssh_service() {
    log INFO "开始检查SSH服务状态..."
    
    if systemctl is-active --quiet sshd; then
        log INFO "  ✓ SSH服务正在运行"
        check_results["ssh_service"]="1/1"
    else
        log WARN "  ✗ SSH服务未运行"
        check_results["ssh_service"]="0/1"
    fi
    
    # 打印SSH服务状态详情
    log INFO "  当前SSH服务状态："
    ssh_status=$(systemctl is-active sshd 2>/dev/null)
    ssh_enabled=$(systemctl is-enabled sshd 2>/dev/null)
    log INFO "    Active: $ssh_status"
    log INFO "    Enabled: $ssh_enabled"
}

# 检查登录日志记录配置
check_login_logging() {
    log INFO "开始检查登录日志记录配置..."
    
    local secure_log_exists=$(stat -c %a /var/log/secure 2>/dev/null | grep "^600$")
    local rsyslog_conf=$(grep -E "authpriv\.\*" /etc/rsyslog.conf 2>/dev/null | grep "/var/log/secure")
    
    if [[ -n "$secure_log_exists" ]] && [[ -n "$rsyslog_conf" ]]; then
        log INFO "  ✓ 登录日志记录配置正确"
        check_results["login_logging"]="1/1"
    else
        log WARN "  ✗ 登录日志记录配置不正确"
        check_results["login_logging"]="0/1"
    fi
    
    # 打印当前rsyslog配置详情
    log INFO "  当前rsyslog配置详情："
    rsyslog_lines=$(grep -E "authpriv\.\*" /etc/rsyslog.conf 2>/dev/null)
    if [[ -n "$rsyslog_lines" ]]; then
        while IFS= read -r line; do
            log INFO "    $line"
        done <<< "$rsyslog_lines"
    else
        log INFO "    未找到authpriv日志配置"
    fi
    
    # 打印secure日志文件权限
    secure_perms=$(stat -c %a /var/log/secure 2>/dev/null)
    log INFO "  /var/log/secure 权限: $secure_perms"
}

# 检查用户目录默认访问权限
check_umask_setting() {
    log INFO "开始检查用户目录默认访问权限设置..."
    
    local umask_value=$(grep -E "^umask" /etc/profile 2>/dev/null | grep -o "[0-9]\{3,\}" | head -n1)
    
    if [[ -n "$umask_value" ]] && [[ "$umask_value" == "027" ]]; then
        log INFO "  ✓ umask 设置为 $umask_value"
        check_results["umask_setting"]="1/1"
    else
        log WARN "  ✗ umask 未正确设置 (当前值: $umask_value)"
        check_results["umask_setting"]="0/1"
    fi
    
    # 打印当前umask配置详情
    log INFO "  当前umask配置详情："
    umask_line=$(grep -E "^umask" /etc/profile 2>/dev/null)
    if [[ -n "$umask_line" ]]; then
        log INFO "    $umask_line"
    else
        log INFO "    未找到umask配置"
    fi
}

# 检查口令过期前警告天数
check_password_warn_age() {
    log INFO "开始检查口令过期前警告天数..."
    
    local warn_age=$(grep -E "^PASS_WARN_AGE" /etc/login.defs 2>/dev/null | awk '{print $2}')
    
    if [[ -n "$warn_age" ]] && [[ "$warn_age" -le 7 ]]; then
        log INFO "  ✓ 口令过期前警告天数设置为 $warn_age 天"
        check_results["password_warn_age"]="1/1"
    else
        log WARN "  ✗ 口令过期前警告天数未正确设置 (当前值: $warn_age)"
        check_results["password_warn_age"]="0/1"
    fi
    
    # 打印当前PASS_WARN_AGE配置详情
    log INFO "  当前PASS_WARN_AGE配置详情："
    warn_age_line=$(grep -E "^PASS_WARN_AGE" /etc/login.defs 2>/dev/null)
    if [[ -n "$warn_age_line" ]]; then
        log INFO "    $warn_age_line"
    else
        log INFO "    未找到PASS_WARN_AGE配置"
    fi
}

# 检查密码复杂度策略
check_password_complexity() {
    log INFO "开始检查密码复杂度策略..."
    
    local complexity_rule=$(grep -E "pam_cracklib\.so" /etc/pam.d/system-auth 2>/dev/null | \
                          grep -E "minlen=8.*dcredit=-1.*ucredit=-1.*ocredit=-1.*lcredit=-1")
    
    if [[ -n "$complexity_rule" ]]; then
        log INFO "  ✓ 密码复杂度策略配置正确"
        check_results["password_complexity"]="1/1"
    else
        log WARN "  ✗ 密码复杂度策略配置不正确"
        check_results["password_complexity"]="0/1"
    fi
    
    # 打印当前密码复杂度配置详情
    log INFO "  当前密码复杂度配置详情："
    complexity_lines=$(grep -E "pam_cracklib\.so" /etc/pam.d/system-auth 2>/dev/null)
    if [[ -n "$complexity_lines" ]]; then
        while IFS= read -r line; do
            log INFO "    $line"
        done <<< "$complexity_lines"
    else
        log INFO "    未找到pam_cracklib配置"
    fi
}

# 检查口令更改最小间隔天数
check_password_min_days() {
    log INFO "开始检查口令更改最小间隔天数..."
    
    local min_days=$(grep -E "^PASS_MIN_DAYS" /etc/login.defs 2>/dev/null | awk '{print $2}')
    
    if [[ -n "$min_days" ]] && [[ "$min_days" -ge 7 ]]; then
        log INFO "  ✓ 口令更改最小间隔天数设置为 $min_days 天"
        check_results["password_min_days"]="1/1"
    else
        log WARN "  ✗ 口令更改最小间隔天数未正确设置 (当前值: $min_days)"
        check_results["password_min_days"]="0/1"
    fi
    
    # 打印当前PASS_MIN_DAYS配置详情
    log INFO "  当前PASS_MIN_DAYS配置详情："
    min_days_line=$(grep -E "^PASS_MIN_DAYS" /etc/login.defs 2>/dev/null)
    if [[ -n "$min_days_line" ]]; then
        log INFO "    $min_days_line"
    else
        log INFO "    未找到PASS_MIN_DAYS配置"
    fi
}

# 检查口令生存周期
check_password_max_days() {
    log INFO "开始检查口令生存周期..."
    
    local max_days=$(grep -E "^PASS_MAX_DAYS" /etc/login.defs 2>/dev/null | awk '{print $2}')
    
    if [[ -n "$max_days" ]] && [[ "$max_days" -le 90 ]]; then
        log INFO "  ✓ 口令生存周期设置为 $max_days 天"
        check_results["password_max_days"]="1/1"
    else
        log WARN "  ✗ 口令生存周期未正确设置 (当前值: $max_days)"
        check_results["password_max_days"]="0/1"
    fi
    
    # 打印当前PASS_MAX_DAYS配置详情
    log INFO "  当前PASS_MAX_DAYS配置详情："
    max_days_line=$(grep -E "^PASS_MAX_DAYS" /etc/login.defs 2>/dev/null)
    if [[ -n "$max_days_line" ]]; then
        log INFO "    $max_days_line"
    else
        log INFO "    未找到PASS_MAX_DAYS配置"
    fi
}

# 检查重要目录及文件权限
check_critical_permissions() {
    log INFO "开始检查重要目录及文件权限..."
    
    local passwd_perm=$(stat -c %a /etc/passwd 2>/dev/null)
    local shadow_perm=$(stat -c %a /etc/shadow 2>/dev/null)
    local group_perm=$(stat -c %a /etc/group 2>/dev/null)
    
    local checks_passed=0
    local total_checks=3
    
    if [[ "$passwd_perm" == "644" ]]; then
        log INFO "  ✓ /etc/passwd 权限正确 ($passwd_perm)"
        ((checks_passed++))
    else
        log WARN "  ✗ /etc/passwd 权限不正确 (当前: $passwd_perm)"
    fi
    
    if [[ "$shadow_perm" == "400" ]]; then
        log INFO "  ✓ /etc/shadow 权限正确 ($shadow_perm)"
        ((checks_passed++))
    else
        log WARN "  ✗ /etc/shadow 权限不正确 (当前: $shadow_perm)"
    fi
    
    if [[ "$group_perm" == "644" ]]; then
        log INFO "  ✓ /etc/group 权限正确 ($group_perm)"
        ((checks_passed++))
    else
        log WARN "  ✗ /etc/group 权限不正确 (当前: $group_perm)"
    fi
    
    log INFO "重要目录及文件权限检查完成: $checks_passed/$total_checks 项通过"
    check_results["critical_permissions"]="$checks_passed/$total_checks"
    
    # 打印当前权限详情
    log INFO "  当前关键文件权限详情："
    log INFO "    /etc/passwd 权限: $passwd_perm (期望: 644)"
    log INFO "    /etc/shadow 权限: $shadow_perm (期望: 400)"
    log INFO "    /etc/group 权限: $group_perm (期望: 644)"
}

# 检查账户认证失败次数限制
check_auth_failure_limit() {
    log INFO "开始检查账户认证失败次数限制..."
    
    local tally2_auth=$(grep -E "pam_tally2\.so.*deny=5" /etc/pam.d/system-auth 2>/dev/null)
    local tally2_account=$(grep -E "pam_tally2\.so" /etc/pam.d/system-auth 2>/dev/null | grep "account")
    
    if [[ -n "$tally2_auth" ]] && [[ -n "$tally2_account" ]]; then
        log INFO "  ✓ 账户认证失败次数限制配置正确"
        check_results["auth_failure_limit"]="1/1"
    else
        log WARN "  ✗ 账户认证失败次数限制配置不正确"
        check_results["auth_failure_limit"]="0/1"
    fi
    
    # 打印当前pam_tally2配置详情
    log INFO "  当前pam_tally2配置详情："
    tally2_lines=$(grep -E "pam_tally2\.so" /etc/pam.d/system-auth 2>/dev/null)
    if [[ -n "$tally2_lines" ]]; then
        while IFS= read -r line; do
            log INFO "    $line"
        done <<< "$tally2_lines"
    else
        log INFO "    未找到pam_tally2配置"
    fi
}

# 检查历史命令数量限制
check_history_limit() {
    log INFO "开始检查历史命令数量限制..."
    
    local hist_file_size=$(grep -E "^HISTFILESIZE=" /etc/profile 2>/dev/null | cut -d'=' -f2)
    local hist_size=$(grep -E "^HISTSIZE=" /etc/profile 2>/dev/null | cut -d'=' -f2)
    
    local checks_passed=0
    local total_checks=2
    
    if [[ -n "$hist_file_size" ]] && [[ "$hist_file_size" -le 5 ]]; then
        log INFO "  ✓ HISTFILESIZE 设置为 $hist_file_size"
        ((checks_passed++))
    else
        log WARN "  ✗ HISTFILESIZE 未正确设置 (当前值: $hist_file_size)"
    fi
    
    if [[ -n "$hist_size" ]] && [[ "$hist_size" -le 5 ]]; then
        log INFO "  ✓ HISTSIZE 设置为 $hist_size"
        ((checks_passed++))
    else
        log WARN "  ✗ HISTSIZE 未正确设置 (当前值: $hist_size)"
    fi
    
    log INFO "历史命令数量限制检查完成: $checks_passed/$total_checks 项通过"
    check_results["history_limit"]="$checks_passed/$total_checks"
    
    # 打印当前历史命令配置详情
    log INFO "  当前历史命令配置详情："
    hist_file_size_line=$(grep -E "^HISTFILESIZE=" /etc/profile 2>/dev/null)
    hist_size_line=$(grep -E "^HISTSIZE=" /etc/profile 2>/dev/null)
    
    if [[ -n "$hist_file_size_line" ]]; then
        log INFO "    $hist_file_size_line"
    else
        log INFO "    未找到HISTFILESIZE配置"
    fi
    
    if [[ -n "$hist_size_line" ]]; then
        log INFO "    $hist_size_line"
    else
        log INFO "    未找到HISTSIZE配置"
    fi
}

# 检查密码重复使用次数限制
check_password_reuse_limit() {
    log INFO "开始检查密码重复使用次数限制..."
    
    local reuse_check=$(grep -E "pam_unix\.so.*remember=5" /etc/pam.d/system-auth 2>/dev/null)
    
    if [[ -n "$reuse_check" ]]; then
        log INFO "  ✓ 密码重复使用次数限制配置正确"
        check_results["password_reuse_limit"]="1/1"
    else
        log WARN "  ✗ 密码重复使用次数限制配置不正确"
        check_results["password_reuse_limit"]="0/1"
    fi
    
    # 打印当前密码重复使用配置详情
    log INFO "  当前密码重复使用配置详情："
    reuse_lines=$(grep -E "pam_unix\.so.*remember" /etc/pam.d/system-auth 2>/dev/null)
    if [[ -n "$reuse_lines" ]]; then
        while IFS= read -r line; do
            log INFO "    $line"
        done <<< "$reuse_lines"
    else
        log INFO "    未找到密码重复使用限制配置"
    fi
}

# 检查Ctrl+Alt+Delete组合键状态
check_ctrl_alt_del() {
    log INFO "开始检查Ctrl+Alt+Delete组合键状态..."
    
    local masked=$(systemctl is-enabled ctrl-alt-del.target 2>/dev/null | grep -o "masked")
    
    if [[ -n "$masked" ]]; then
        log INFO "  ✓ Ctrl+Alt+Delete组合键已被禁用"
        check_results["ctrl_alt_del"]="1/1"
    else
        log WARN "  ✗ Ctrl+Alt+Delete组合键未被禁用"
        check_results["ctrl_alt_del"]="0/1"
    fi
    
    # 打印当前Ctrl+Alt+Delete服务状态
    log INFO "  当前Ctrl+Alt+Delete服务状态："
    cad_status=$(systemctl is-active ctrl-alt-del.target 2>/dev/null)
    cad_enabled=$(systemctl is-enabled ctrl-alt-del.target 2>/dev/null)
    log INFO "    Active: $cad_status"
    log INFO "    Enabled: $cad_enabled"
}

# 输出汇总报告
output_summary() {
    log INFO ""
    log INFO "=================================="
    log INFO "安全配置检测汇总报告"
    log INFO "=================================="
    
    local total_items=0
    local passed_items=0
    
    for key in "${!check_results[@]}"; do
        IFS='/' read -r passed total <<< "${check_results[$key]}"
        ((total_items += total))
        ((passed_items += passed))
        
        log INFO "$(printf '%-20s' "$key:") ${check_results[$key]}"
    done
    
    log INFO "=================================="
    log INFO "总计: $passed_items/$total_items 项通过"
    
    local percentage=$((passed_items * 100 / total_items))
    if [[ $percentage -ge 90 ]]; then
        log INFO "安全配置达标率: $percentage% (优秀)"
    elif [[ $percentage -ge 70 ]]; then
        log INFO "安全配置达标率: $percentage% (良好)"
    else
        log INFO "安全配置达标率: $percentage% (需要改进)"
    fi
    
    log INFO "=================================="
    log INFO "检测完成，详细日志请查看: $LOG_FILE"
}

# 主函数
main() {
    log INFO "开始执行统信系统安全配置检测脚本"
    log INFO "脚本位置: $SCRIPT_DIR"
    log INFO "检测开始时间: $(date '+%Y-%m-%d %H:%M:%S')"
    log INFO ""
    
    # 执行各项检查
    check_ssh_security
    check_timeout_setting
    check_pam_auth
    check_telnet_status
    check_ssh_service
    check_login_logging
    check_umask_setting
    check_password_warn_age
    check_password_complexity
    check_password_min_days
    check_password_max_days
    check_critical_permissions
    check_auth_failure_limit
    check_history_limit
    check_password_reuse_limit
    check_ctrl_alt_del
    
    # 输出汇总报告
    output_summary
}

# 运行主函数
main "$@"